from collections import namedtuple
from typing import NamedTuple

import pytest

from libecalc.common.serializable_chart import ChartCurveDTO, ChartDTO
from libecalc.domain.process.value_objects.fluid_stream.fluid_model import EoSModel, FluidComposition, FluidModel
from libecalc.presentation.yaml.mappers.fluid_mapper import MEDIUM_MW_19P4

StreamData = namedtuple(
    "StreamData",
    [
        "mass_flow_kg_per_h",
        "std_gas_flow_std_m3_per_h",
        "z_factor",
        "temperature_C",
        "pressure_kPa",
        "mass_density_kg_per_m3",
        "actual_volume_flow_m3_per_h",
    ],
)

CompressorData = namedtuple(
    "CompressorData",
    ["speed_rpm", "pressure_ratio", "polytropic_exponent", "power_kW", "polytropic_efficiency", "polytropic_head_m"],
)


class ProcessSimulationSingleCompressor(NamedTuple):
    compressor_chart: ChartDTO
    fluid: FluidModel
    compressor: CompressorData
    inlet_stream: StreamData
    outlet_stream: StreamData


@pytest.fixture
def process_simulator_variable_compressor_data() -> ProcessSimulationSingleCompressor:
    medium_fluid = FluidModel(eos_model=EoSModel.SRK, composition=FluidComposition.model_validate(MEDIUM_MW_19P4))

    chart_curves = [
        ChartCurveDTO(
            polytropic_head_joule_per_kg=[82531.50, 78440.70, 72240.80, 60105.80],
            rate_actual_m3_hour=[2900, 3504, 4003, 4595],
            efficiency_fraction=[0.72, 0.74, 0.74, 0.70],
            speed_rpm=7689,
        ),
        ChartCurveDTO(
            polytropic_head_joule_per_kg=[135819, 129325, 121889, 110617, 98629.7],
            rate_actual_m3_hour=[3709, 4502, 4994, 5508, 5924],
            efficiency_fraction=[0.72, 0.74, 0.74, 0.73, 0.70],
            speed_rpm=9886,
        ),
        ChartCurveDTO(
            polytropic_head_joule_per_kg=[185232, 178885, 171988, 161766, 147512, 133602],
            rate_actual_m3_hour=[4328, 4999, 5506, 6028, 6507, 6908],
            efficiency_fraction=[0.72, 0.74, 0.74, 0.74, 0.72, 0.70],
            speed_rpm=11533,
        ),
    ]

    compressor = CompressorData(
        speed_rpm=[
            7689,
            7689,
            7689,
            7689,
            7689,
            7689,
            7689,
            7689,
            7689,
            7689,
            8500,
            8500,
            8500,
            8500,
            8500,
            11533,
            11533,
            11533,
            11533,
            11533,
            11000,
            11000,
            11000,
            11000,
            11000,
        ],
        pressure_ratio=[
            1.964873133,
            1.93036092,
            1.873537842,
            1.78481191,
            1.669466132,
            2.192754688,
            2.168714352,
            2.154434628,
            2.163993436,
            2.199489753,
            2.171775362,
            2.118456366,
            2.045472946,
            1.943803734,
            1.820177406,
            3.636223591,
            3.607619694,
            3.578534056,
            3.547072177,
            3.507145328,
            3.147891662,
            3.057055679,
            2.900977658,
            2.668852536,
            2.384909108,
        ],
        polytropic_exponent=[
            1.874306988,
            1.854928933,
            1.842470877,
            1.849512331,
            1.878562533,
            1.966101585,
            1.932915101,
            1.876755831,
            1.790176056,
            1.678604433,
            1.57617845,
            1.564222249,
            1.560152602,
            1.568841722,
            1.58447724,
            1.617559279,
            1.611946799,
            1.606401599,
            1.600873472,
            1.59690581,
            1.409988841,
            1.398820814,
            1.397253559,
            1.408202173,
            1.433659293,
        ],
        power_kW=[
            8891.755112,
            9662.154294,
            10205.53513,
            10518.76934,
            10509.97247,
            12015.48302,
            13014.76427,
            13716.24246,
            14118.44692,
            14103.56431,
            6577.987927,
            7151.031336,
            7619.806686,
            7954.44064,
            8032.405163,
            16452.91141,
            16882.20895,
            17290.18063,
            17667.66744,
            18040.91378,
            4387.113298,
            4729.266526,
            5000.286426,
            5165.301507,
            5187.683386,
        ],
        polytropic_efficiency=[
            72.067721,
            73.36578316,
            73.99904748,
            72.85377401,
            70.04565815,
            72.17604378,
            73.45705964,
            73.99488971,
            72.80672917,
            70.07747203,
            72.19119497,
            73.26339642,
            73.44587641,
            72.20672276,
            70.31912887,
            71.96405214,
            72.52333074,
            73.08262865,
            73.64194631,
            73.99565192,
            72.39607683,
            74.01167144,
            74.28685314,
            72.83343881,
            69.74959751,
        ],
        polytropic_head_m=[
            8401.408314,
            8132.037872,
            7700.896039,
            7032.971446,
            6142.030937,
            8377.849466,
            8114.57129,
            7682.047596,
            7020.39979,
            6149.463132,
            10254.39965,
            9862.866014,
            9338.355654,
            8606.024847,
            7679.555034,
            18897.84887,
            18727.40494,
            18554.76731,
            18370.16079,
            18150.23828,
            17146.12355,
            16579.60341,
            15673.62866,
            14311.38518,
            12531.15597,
        ],
    )

    outlet_stream = StreamData(
        mass_flow_kg_per_h=[
            280000,
            320000,
            360000,
            400000,
            440000,
            380000,
            432500,
            485000,
            537500,
            590000,
            170000,
            195000,
            220000,
            245000,
            270000,
            230000,
            240000,
            250000,
            260000,
            270000,
            68000,
            77500,
            87000,
            96500,
            106000,
        ],
        std_gas_flow_std_m3_per_h=[
            340387.3951,
            389014.1658,
            437640.9366,
            486267.7073,
            534894.478,
            461954.3219,
            525776.9585,
            589599.5951,
            653422.2317,
            717244.8682,
            206663.7756,
            237055.5073,
            267447.239,
            297838.9707,
            328230.7024,
            279603.9317,
            291760.6244,
            303917.3171,
            316074.0097,
            328230.7024,
            82665.51024,
            94214.36829,
            105763.2263,
            117312.0844,
            128860.9424,
        ],
        z_factor=[
            0.919748679,
            0.914780915,
            0.907901617,
            0.898797754,
            0.887714004,
            0.948033347,
            0.941873674,
            0.932615343,
            0.919591403,
            0.903245213,
            0.928686241,
            0.924950958,
            0.921046572,
            0.916878167,
            0.911997198,
            1.002928426,
            1.000902736,
            0.998856671,
            0.996696789,
            0.994341827,
            0.973621004,
            0.971620508,
            0.969891584,
            0.968474988,
            0.967370003,
        ],
        temperature_C=[
            90.4198903,
            88.05851563,
            84.99737231,
            81.11168409,
            76.1890121,
            87.09397601,
            84.91283952,
            82.08289658,
            78.53038852,
            74.05208891,
            103.5250182,
            100.2633357,
            96.69235877,
            92.49679499,
            87.15329976,
            156.8709425,
            155.2592059,
            153.6526624,
            151.9953995,
            150.3698581,
            141.4935077,
            136.2261599,
            130.5291078,
            123.8089839,
            115.7518324,
        ],
        pressure_kPa=[
            19648.73133,
            19303.6092,
            18735.37842,
            17848.1191,
            16694.66132,
            25559.3206,
            25127.89632,
            24397.8258,
            23272.28873,
            21821.85763,
            13030.65217,
            12710.73819,
            12272.83768,
            11662.8224,
            10921.06443,
            21817.34155,
            21645.71816,
            21471.20434,
            21282.43306,
            21042.87197,
            6295.783324,
            6114.111358,
            5801.955316,
            5337.705072,
            4769.818216,
        ],
        mass_density_kg_per_m3=[
            137.4561634,
            136.6627723,
            134.7872139,
            131.1272881,
            125.9343601,
            175.0716197,
            174.2974396,
            172.2749683,
            168.3382581,
            162.77605,
            87.13991034,
            86.0892707,
            84.28174818,
            81.37988007,
            77.74814064,
            118.3393198,
            118.0886381,
            117.8183392,
            117.4917895,
            116.8913331,
            36.48142165,
            35.95844553,
            34.66583783,
            32.47934892,
            29.6589661,
        ],
        actual_volume_flow_m3_per_h=[
            2037.01306,
            2341.530137,
            2670.876485,
            3050.471078,
            3493.883636,
            2170.540266,
            2481.390438,
            2815.266807,
            3192.975893,
            3624.611853,
            1950.885643,
            2265.090625,
            2610.29232,
            3010.572144,
            3472.75186,
            1943.563647,
            2032.371648,
            2121.910746,
            2212.920589,
            2309.837632,
            1863.962448,
            2155.265581,
            2509.675388,
            2971.118671,
            3573.9614,
        ],
    )

    inlet_stream = StreamData(
        std_gas_flow_std_m3_per_h=[
            340387.3951,
            389014.1658,
            437640.9366,
            486267.7073,
            534894.478,
            461954.3219,
            525776.9585,
            589599.5951,
            653422.2317,
            717244.8682,
            206663.7756,
            237055.5073,
            267447.239,
            297838.9707,
            328230.7024,
            279603.9317,
            291760.6244,
            303917.3171,
            316074.0097,
            328230.7024,
            82665.51024,
            94214.36829,
            105763.2263,
            117312.0844,
            128860.9424,
        ],
        z_factor=[
            0.804948163,
            0.804948163,
            0.804948163,
            0.804948163,
            0.804948163,
            0.779928943,
            0.779928943,
            0.779928943,
            0.779928943,
            0.779928943,
            0.869068878,
            0.869068878,
            0.869068878,
            0.869068878,
            0.869068878,
            0.869068878,
            0.869068878,
            0.869068878,
            0.869068878,
            0.869068878,
            0.954104108,
            0.954104108,
            0.954104108,
            0.954104108,
            0.954104108,
        ],
        temperature_C=[
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
            30,
        ],
        pressure_kPa=[
            10000,
            10000,
            10000,
            10000,
            10000,
            13000,
            13000,
            13000,
            13000,
            13000,
            6000,
            6000,
            6000,
            6000,
            6000,
            6000,
            6000,
            6000,
            6000,
            6000,
            2000,
            2000,
            2000,
            2000,
            2000,
        ],
        mass_flow_kg_per_h=[
            280000,
            320000,
            360000,
            400000,
            440000,
            380000,
            432500,
            485000,
            537500,
            590000,
            170000,
            195000,
            220000,
            245000,
            270000,
            230000,
            240000,
            250000,
            260000,
            270000,
            68000,
            77500,
            87000,
            96500,
            106000,
        ],
        mass_density_kg_per_m3=[
            95.86526986,
            95.86526986,
            95.86526986,
            95.86526986,
            95.86526986,
            128.6226722,
            128.6226722,
            128.6226722,
            128.6226722,
            128.6226722,
            53.27534437,
            53.27534437,
            53.27534437,
            53.27534437,
            53.27534437,
            53.27534437,
            53.27534437,
            53.27534437,
            53.27534437,
            53.27534437,
            16.17571338,
            16.17571338,
            16.17571338,
            16.17571338,
            16.17571338,
        ],
        actual_volume_flow_m3_per_h=[
            2920.765783,
            3338.018038,
            3755.270292,
            4172.522547,
            4589.774802,
            2954.378054,
            3362.548706,
            3770.719358,
            4178.890011,
            4587.060663,
            3190.969519,
            3660.229743,
            4129.489966,
            4598.750189,
            5068.010413,
            4317.194055,
            4504.898145,
            4692.602234,
            4880.306323,
            5068.010413,
            4203.833141,
            4791.13336,
            5378.433578,
            5965.733796,
            6553.034015,
        ],
    )

    test_data_set = ProcessSimulationSingleCompressor(
        compressor_chart=ChartDTO(curves=chart_curves),
        compressor=compressor,
        fluid=medium_fluid,
        inlet_stream=inlet_stream,
        outlet_stream=outlet_stream,
    )

    return test_data_set
