"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCustomSynthesis = exports.invokeAspects = exports.StackSynthesizer = void 0;
const path = require("path");
const fs = require("fs");
const annotations_1 = require("../annotations");
const constructs_1 = require("constructs");
const aspect_1 = require("../aspect");
class StackSynthesizer {
    /**
     * @param stack the stack to synthesize
     * @param continueOnErrorAnnotations if set to true, `synthesize()` will not throw an error
     * if it encounters an error Annotation. However, it will still add all errors to the manifest
     * file. The cdktf-cli uses this flag to print the errors itself.
     */
    constructor(stack, continueOnErrorAnnotations = false) {
        this.stack = stack;
        this.continueOnErrorAnnotations = continueOnErrorAnnotations;
    }
    synthesize(session) {
        invokeAspects(this.stack);
        if (!session.skipValidation) {
            const errors = this.stack.node
                .findAll()
                .map((node) => node.node
                .validate()
                .map((error) => ({ message: error, source: node })))
                .reduce((prev, curr) => [...prev, ...curr], []);
            if (errors.length > 0) {
                const errorList = errors
                    .map((e) => `[${e.source.node.path}] ${e.message}`)
                    .join("\n  ");
                throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
            }
        }
        const manifest = session.manifest;
        const stackManifest = manifest.forStack(this.stack);
        const workingDirectory = path.join(session.outdir, stackManifest.workingDirectory);
        if (!fs.existsSync(workingDirectory))
            fs.mkdirSync(workingDirectory);
        // call custom synthesis on child nodes (leafs first)
        this.stack.node
            .findAll(constructs_1.ConstructOrder.POSTORDER)
            .forEach((node) => { var _a; return (_a = getCustomSynthesis(node)) === null || _a === void 0 ? void 0 : _a.onSynthesize(session); });
        // collect Annotations into Manifest
        const annotations = this.stack.node
            .findAll()
            .map((node) => ({
            node,
            metadatas: node.node.metadata.filter(isAnnotationMetadata),
        }))
            .map(({ node, metadatas }) => metadatas.map((metadata) => ({
            constructPath: node.node.path,
            level: metadata.type,
            message: metadata.data,
            stacktrace: metadata.trace,
        })))
            .reduce((list, metadatas) => [...list, ...metadatas], []); // Array.flat()
        // it is readonly but this is the place where we are allowed to write to it
        stackManifest.annotations = annotations;
        // abort if one or more error annotations have been encountered
        if (!this.continueOnErrorAnnotations &&
            annotations.some(isErrorAnnotation)) {
            throw new Error(`Encountered Annotations with level "ERROR":\n${annotations
                .filter(isErrorAnnotation)
                .map((a) => `[${a.constructPath}] ${a.message}`)
                .join("\n")}`);
        }
        const tfConfig = this.stack.toTerraform();
        fs.writeFileSync(path.join(session.outdir, stackManifest.synthesizedStackPath), JSON.stringify(tfConfig, undefined, 2));
    }
}
exports.StackSynthesizer = StackSynthesizer;
/**
 * Invoke aspects on the given construct tree.
 *
 * originally from https://github.com/aws/aws-cdk/blob/dcae3eead0dbf9acb1ed80ba95bb104c64cb1bd7/packages/@aws-cdk/core/lib/private/synthesis.ts#L99-L137
 */
function invokeAspects(root) {
    const invokedByPath = {};
    let nestedAspectWarning = false;
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        const node = construct.node;
        const aspects = aspect_1.Aspects.of(construct);
        const allAspectsHere = [...(inheritedAspects !== null && inheritedAspects !== void 0 ? inheritedAspects : []), ...aspects.all];
        const nodeAspectsCount = aspects.all.length;
        for (const aspect of allAspectsHere) {
            let invoked = invokedByPath[node.path];
            if (!invoked) {
                invoked = invokedByPath[node.path] = [];
            }
            if (invoked.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            // if an aspect was added to the node while invoking another aspect it will not be invoked, emit a warning
            // the `nestedAspectWarning` flag is used to prevent the warning from being emitted for every child
            if (!nestedAspectWarning && nodeAspectsCount !== aspects.all.length) {
                annotations_1.Annotations.of(construct).addWarning("We detected an Aspect was added via another Aspect, and will not be applied");
                nestedAspectWarning = true;
            }
            // mark as invoked for this node
            invoked.push(aspect);
        }
        for (const child of construct.node.children) {
            recurse(child, allAspectsHere);
        }
    }
}
exports.invokeAspects = invokeAspects;
const annotationMetadataEntryTypes = [
    annotations_1.AnnotationMetadataEntryType.INFO,
    annotations_1.AnnotationMetadataEntryType.WARN,
    annotations_1.AnnotationMetadataEntryType.ERROR,
];
function isAnnotationMetadata(metadata) {
    return annotationMetadataEntryTypes.includes(metadata.type);
}
function isErrorAnnotation(annotation) {
    return annotation.level === annotations_1.AnnotationMetadataEntryType.ERROR;
}
// originally from https://github.com/aws/aws-cdk/blob/dcae3eead0dbf9acb1ed80ba95bb104c64cb1bd7/packages/%40aws-cdk/core/lib/private/synthesis.ts#L52
const CUSTOM_SYNTHESIS_SYM = Symbol.for("cdktf/customSynthesis");
function addCustomSynthesis(construct, synthesis) {
    Object.defineProperty(construct, CUSTOM_SYNTHESIS_SYM, {
        value: synthesis,
        enumerable: false,
    });
}
exports.addCustomSynthesis = addCustomSynthesis;
function getCustomSynthesis(construct) {
    return construct[CUSTOM_SYNTHESIS_SYM];
}
//# sourceMappingURL=data:application/json;base64,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