"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Fn = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const token_1 = require("./tokens/token");
const tfExpression_1 = require("./tfExpression");
const token_map_1 = require("./tokens/private/token-map");
const encoding_1 = require("./tokens/private/encoding");
const _1 = require(".");
function hasUnescapedDoubleQuotes(str) {
    return /\\([\s\S])|(")/.test(str);
}
// Validators
function anyValue(value) {
    return value;
}
function mapValue(value) {
    return value;
}
function stringValue(value) {
    if (typeof value !== "string" && !token_1.Tokenization.isResolvable(value)) {
        throw new Error(`'${value}' is not a valid string nor a token`);
    }
    if (typeof value === "string" && hasUnescapedDoubleQuotes(value)) {
        throw new Error(`'${value}' can not be used as value directly since it has unescaped double quotes in it. To safely use the value please use Fn.rawString on your string.`);
    }
    return value;
}
function numericValue(value) {
    if (typeof value !== "number" && !token_1.Tokenization.isResolvable(value)) {
        throw new Error(`${value} is not a valid number nor a token`);
    }
    return value;
}
function listOf(type) {
    return (value) => {
        if (token_1.Tokenization.isResolvable(value)) {
            return value;
        }
        if (!Array.isArray(value)) {
            //   throw new Error(`${value} is not a valid list`);
            return value;
        }
        return value
            .filter((item) => item !== undefined && item !== null)
            .map((item, i) => {
            if (token_1.Tokenization.isResolvable(item)) {
                return item;
            }
            if (encoding_1.TokenString.forListToken(item).test()) {
                return item;
            }
            if (typeof item === "string") {
                const tokenList = token_1.Tokenization.reverseString(item);
                const numberOfTokens = tokenList.tokens.length + tokenList.intrinsic.length;
                if (numberOfTokens === 1 && tokenList.literals.length === 0) {
                    return item;
                }
            }
            try {
                type(item);
                return typeof item === "string" ? `"${item}"` : item;
            }
            catch (error) {
                throw new Error(`Element in list ${value} at position ${i} is not of the right type: ${error}`);
            }
        });
    };
}
// Tokenization
function asString(value) {
    return token_map_1.TokenMap.instance().registerString(value);
}
function asNumber(value) {
    return token_map_1.TokenMap.instance().registerNumber(value);
}
function asList(value) {
    return token_map_1.TokenMap.instance().registerList(value);
}
function asBoolean(value) {
    return value; // Booleans can not be represented as a token
}
function asAny(value) {
    return asString(value);
}
function terraformFunction(name, argValidators) {
    return function (...args) {
        if (Array.isArray(argValidators)) {
            if (args.length !== argValidators.length) {
                throw new Error(`${name} takes ${argValidators.length} arguments, but ${args.length} were provided`);
            }
            return tfExpression_1.call(name, args.map((arg, i) => argValidators[i](arg)));
        }
        else {
            return tfExpression_1.call(name, argValidators(args));
        }
    };
}
/**
 * @experimental
 */
class Fn {
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/anytrue.html anytrue} returns true if all elements in a given collection are true or "true".
     *
     * @experimental
     */
    static alltrue(values) {
        return asBoolean(terraformFunction("alltrue", [listOf(anyValue)])(values));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/anytrue.html anytrue} returns true if any element in a given collection is true or "true".
     *
     * @experimental
     */
    static anytrue(value) {
        return asBoolean(terraformFunction("anytrue", [listOf(anyValue)])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/chunklist.html chunklist} splits a single list into fixed-size chunks, returning a list of lists.
     *
     * @experimental
     */
    static chunklist(value, chunkSize) {
        return asList(terraformFunction("chunklist", [listOf(anyValue), numericValue])(value, chunkSize));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/coalesce.html coalesce} takes any number of arguments and returns the first one that isn't null or an empty string.
     *
     * @param value - Arguments are passed in an array.
     * @experimental
     */
    static coalesce(value) {
        return terraformFunction("coalesce", listOf(anyValue))(...value);
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/coalescelist.html coalescelist} takes any number of list arguments and returns the first one that isn't empty.
     *
     * @experimental
     */
    static coalescelist(value) {
        return asList(terraformFunction("coalescelist", listOf(anyValue))(...value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/compact.html compact} takes a list of strings and returns a new list with any empty string elements removed.
     *
     * @experimental
     */
    static compact(value) {
        return asList(terraformFunction("compact", [listOf(anyValue)])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/concat.html concat} takes two or more lists and combines them into a single list.
     *
     * @experimental
     */
    static concat(value) {
        return asList(terraformFunction("concat", listOf(anyValue))(...value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/contains.html contains} determines whether a given list or set contains a given single value as one of its elements.
     *
     * @experimental
     */
    static contains(list, value) {
        return asBoolean(terraformFunction("contains", [listOf(anyValue), anyValue])(list, value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/distinct.html distinct} takes a list and returns a new list with any duplicate elements removed.
     *
     * @experimental
     */
    static distinct(list) {
        return asList(terraformFunction("distinct", [listOf(anyValue)])(list));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/element.html element} retrieves a single element from a list.
     *
     * @experimental
     */
    static element(list, index) {
        return asAny(terraformFunction("element", [listOf(anyValue), numericValue])(list, index));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/flatten.html flatten} takes a list and replaces any elements that are lists with a flattened sequence of the list contents.
     *
     * @experimental
     */
    static flatten(list) {
        return asList(terraformFunction("flatten", [listOf(anyValue)])(list));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/index.html index} finds the element index for a given value in a list.
     *
     * @experimental
     */
    static index(list, value) {
        return asNumber(terraformFunction("index", [listOf(anyValue), anyValue])(list, value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/keys.html keys} takes a map and returns a list containing the keys from that map.
     *
     * @experimental
     */
    static keys(map) {
        return asList(terraformFunction("keys", [mapValue])(map));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/length.html length} determines the length of a given list, map, or string.
     *
     * @experimental
     */
    static lengthOf(value) {
        return asNumber(terraformFunction("length", [anyValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/lookup.html lookup} retrieves the value of a single element from a map, given its key. If the given key does not exist, the given default value is returned instead.
     *
     * @experimental
     */
    static lookup(value, key, defaultValue) {
        return asAny(terraformFunction("lookup", [anyValue, anyValue, anyValue])(value, key, defaultValue));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/matchkeys.html matchkeys} constructs a new list by taking a subset of elements from one list whose indexes match the corresponding indexes of values in another list.
     *
     * @experimental
     */
    static matchkeys(valuesList, keysList, searchSet) {
        return asList(terraformFunction("matchkeys", [anyValue, anyValue, anyValue])(valuesList, keysList, searchSet));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/merge.html merge} takes an arbitrary number of maps or objects, and returns a single map or object that contains a merged set of elements from all arguments.
     *
     * @experimental
     */
    static merge(values) {
        return asList(terraformFunction("merge", listOf(anyValue))(...values));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/one.html one} takes a list, set, or tuple value with either zero or one elements.
     *
     * @experimental
     */
    static one(list) {
        return asAny(terraformFunction("one", [listOf(anyValue)])(list));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/range.html range} generates a list of numbers using a start value, a limit value, and a step value.
     *
     * @experimental
     */
    static range(start, limit, step = 1) {
        return asList(terraformFunction("range", listOf(anyValue))(start, limit, step));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/reverse.html reverse} takes a sequence and produces a new sequence of the same length with all of the same elements as the given sequence but in reverse order.
     *
     * @experimental
     */
    static reverse(values) {
        return asList(terraformFunction("reverse", [listOf(anyValue)])(values));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/setintersection.html setintersection} function takes multiple sets and produces a single set containing only the elements that all of the given sets have in common.
     *
     * @experimental
     */
    static setintersection(values) {
        return asList(terraformFunction("setintersection", listOf(anyValue))(...values));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/setproduct.html setproduct} function finds all of the possible combinations of elements from all of the given sets by computing the Cartesian product.
     *
     * @experimental
     */
    static setproduct(values) {
        return asList(terraformFunction("setproduct", listOf(anyValue))(...values));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/slice.html setsubtract} function returns a new set containing the elements from the first set that are not present in the second set.
     *
     * @experimental
     */
    static setsubtract(minuend, subtrahend) {
        return asList(terraformFunction("setsubtract", [listOf(anyValue), listOf(anyValue)])(minuend, subtrahend));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/setunion.html setunion} function takes multiple sets and produces a single set containing the elements from all of the given sets.
     *
     * @param values - Arguments are passed in an array.
     * @experimental
     */
    static setunion(values) {
        return asList(terraformFunction("setunion", listOf(anyValue))(...values));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/slice.html slice} extracts some consecutive elements from within a list.
     *
     * @experimental
     */
    static slice(list, startindex, endindex) {
        return asList(terraformFunction("slice", [
            listOf(anyValue),
            numericValue,
            numericValue,
        ])(list, startindex, endindex));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/sort.html sort} takes a list of strings and returns a new list with those strings sorted lexicographically.
     *
     * @experimental
     */
    static sort(list) {
        return asList(terraformFunction("sort", [listOf(anyValue)])(list));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/sum.html sum} takes a list or set of numbers and returns the sum of those numbers.
     *
     * @experimental
     */
    static sum(list) {
        return asNumber(terraformFunction("sum", [listOf(anyValue)])(list));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/transpose.html transpose} takes a map of lists of strings and swaps the keys and values to produce a new map of lists of strings.
     *
     * @experimental
     */
    static transpose(value) {
        return asAny(terraformFunction("transpose", [mapValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/values.html values} takes a map and returns a list containing the values of the elements in that map.
     *
     * @experimental
     */
    static values(value) {
        return asList(terraformFunction("values", [mapValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/zipmap.html zipmap} constructs a map from a list of keys and a corresponding list of values.
     *
     * @experimental
     */
    static zipmap(keyslist, valueslist) {
        return asAny(terraformFunction("zipmap", [mapValue])(keyslist, valueslist));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/base64sha256.html base64sha256} computes the SHA256 hash of a given string and encodes it with Base64.
     *
     * @experimental
     */
    static base64sha256(value) {
        return asString(terraformFunction("base64sha256", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/base64sha512.html base64sha512} computes the SHA512 hash of a given string and encodes it with Base64.
     *
     * @experimental
     */
    static base64sha512(value) {
        return asString(terraformFunction("base64sha512", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/bcrypt.html bcrypt} computes a hash of the given string using the Blowfish cipher, returning a string in the Modular Crypt Format usually expected in the shadow password file on many Unix systems.
     *
     * @experimental
     */
    static bcrypt(value, cost) {
        return asString(terraformFunction("bcrypt", listOf(anyValue))(value, cost));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/filebase64sha256.html filebase64sha256} is a variant of base64sha256 that hashes the contents of a given file rather than a literal string.
     *
     * @experimental
     */
    static filebase64sha256(value) {
        return asString(terraformFunction("filebase64sha256", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/filebase64sha512.html filebase64sha512} is a variant of base64sha512 that hashes the contents of a given file rather than a literal string.
     *
     * @experimental
     */
    static filebase64sha512(value) {
        return asString(terraformFunction("filebase64sha512", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/filemd5.html filemd5} is a variant of md5 that hashes the contents of a given file rather than a literal string.
     *
     * @experimental
     */
    static filemd5(value) {
        return asString(terraformFunction("filemd5", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/filesha1.html filesha1} is a variant of sha1 that hashes the contents of a given file rather than a literal string.
     *
     * @experimental
     */
    static filesha1(value) {
        return asString(terraformFunction("filesha1", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/filesha256.html filesha256} is a variant of sha256 that hashes the contents of a given file rather than a literal string.
     *
     * @experimental
     */
    static filesha256(value) {
        return asString(terraformFunction("filesha256", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/filesha512.html filesha512} is a variant of sha512 that hashes the contents of a given file rather than a literal string.
     *
     * @experimental
     */
    static filesha512(value) {
        return asString(terraformFunction("filesha512", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/md5.html md5} computes the MD5 hash of a given string and encodes it with hexadecimal digits.
     *
     * @experimental
     */
    static md5(value) {
        return asString(terraformFunction("md5", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/rsadecrypt.html rsadecrypt} decrypts an RSA-encrypted ciphertext, returning the corresponding cleartext.
     *
     * @experimental
     */
    static rsadecrypt(ciphertext, privatekey) {
        return asString(terraformFunction("rsadecrypt", [stringValue, stringValue])(ciphertext, privatekey));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/sha1.html sha1} computes the SHA1 hash of a given string and encodes it with hexadecimal digits.
     *
     * @experimental
     */
    static sha1(value) {
        return asString(terraformFunction("sha1", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/sha256.html sha256} computes the SHA256 hash of a given string and encodes it with hexadecimal digits.
     *
     * @experimental
     */
    static sha256(value) {
        return asString(terraformFunction("sha256", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/sha512.html sha512} computes the SHA512 hash of a given string and encodes it with hexadecimal digits.
     *
     * @experimental
     */
    static sha512(value) {
        return asString(terraformFunction("sha512", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/uuid.html uuid} generates a unique identifier string.
     *
     * @experimental
     */
    static uuid() {
        return asString(terraformFunction("uuid", [])());
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/uuidv5.html uuidv5} generates a unique identifier string.
     *
     * @experimental
     */
    static uuidv5(namespace, name) {
        return asString(terraformFunction("uuidv5", [stringValue, stringValue])(namespace, name));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/formatdate.html formatdate} converts a timestamp into a different time format.
     *
     * @experimental
     */
    static formatdate(spec, timestamp) {
        return asString(terraformFunction("formatdate", [stringValue, stringValue])(spec, timestamp));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/timeadd.html timeadd} adds a duration to a timestamp, returning a new timestamp.
     *
     * @experimental
     */
    static timeadd(timestamp, duration) {
        return asString(terraformFunction("timeadd", [stringValue, stringValue])(timestamp, duration));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/timestamp.html timestamp} returns a UTC timestamp string in RFC 3339 format.
     *
     * @experimental
     */
    static timestamp() {
        return asString(terraformFunction("timestamp", [])());
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/base64decode.html base64decode} takes a string containing a Base64 character sequence and returns the original string.
     *
     * @experimental
     */
    static base64decode(value) {
        return asString(terraformFunction("base64decode", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/base64encode.html base64encode} takes a string containing a Base64 character sequence and returns the original string.
     *
     * @experimental
     */
    static base64encode(value) {
        return asString(terraformFunction("base64encode", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/base64gzip.html base64gzip} compresses a string with gzip and then encodes the result in Base64 encoding.
     *
     * @experimental
     */
    static base64gzip(value) {
        return asString(terraformFunction("base64gzip", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/csvdecode.html csvdecode} decodes a string containing CSV-formatted data and produces a list of maps representing that data.
     *
     * @experimental
     */
    static csvdecode(value) {
        return asList(terraformFunction("csvdecode", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/jsondecode.html jsondecode} interprets a given string as JSON, returning a representation of the result of decoding that string.
     *
     * @experimental
     */
    static jsondecode(value) {
        return asAny(terraformFunction("jsondecode", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/jsonencode.html jsonencode} encodes a given value to a string using JSON syntax.
     *
     * @experimental
     */
    static jsonencode(value) {
        return asString(terraformFunction("jsonencode", [anyValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/textdecodebase64.html textdecodebase64} function decodes a string that was previously Base64-encoded, and then interprets the result as characters in a specified character encoding.
     *
     * @experimental
     */
    static textdecodebase64(value, encodingName) {
        return asString(terraformFunction("textdecodebase64", [stringValue, stringValue])(value, encodingName));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/textencodebase64.html textencodebase64}  encodes the unicode characters in a given string using a specified character encoding, returning the result base64 encoded because Terraform language strings are always sequences of unicode characters.
     *
     * @experimental
     */
    static textencodebase64(value, encodingName) {
        return asString(terraformFunction("textencodebase64", [stringValue, stringValue])(value, encodingName));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/urlencode.html urlencode} applies URL encoding to a given string.
     *
     * @experimental
     */
    static urlencode(value) {
        return asString(terraformFunction("urlencode", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/yamldecode.html yamldecode} parses a string as a subset of YAML, and produces a representation of its value.
     *
     * @experimental
     */
    static yamldecode(value) {
        return asAny(terraformFunction("yamldecode", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/yamlencode.html yamlencode} encodes a given value to a string using JSON syntax.
     *
     * @experimental
     */
    static yamlencode(value) {
        return asString(terraformFunction("yamlencode", [anyValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/abspath.html abspath} takes a string containing a filesystem path and converts it to an absolute path.
     *
     * @experimental
     */
    static abspath(value) {
        return asString(terraformFunction("abspath", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/dirname.html dirname} takes a string containing a filesystem path and removes the last portion from it.
     *
     * @experimental
     */
    static dirname(value) {
        return asString(terraformFunction("dirname", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/pathexpand.html pathexpand} takes a string containing a filesystem path and removes the last portion from it.
     *
     * @experimental
     */
    static pathexpand(value) {
        return asString(terraformFunction("pathexpand", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/basename.html basename} takes a string containing a filesystem path and removes all except the last portion from it.
     *
     * @experimental
     */
    static basename(value) {
        return asString(terraformFunction("basename", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/file.html file} takes a string containing a filesystem path and removes all except the last portion from it.
     *
     * @experimental
     */
    static file(value) {
        return asString(terraformFunction("file", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/fileexists.html fileexists} determines whether a file exists at a given path.
     *
     * @experimental
     */
    static fileexists(value) {
        return asBoolean(terraformFunction("fileexists", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/fileset.html fileset} enumerates a set of regular file names given a path and pattern.
     *
     * @experimental
     */
    static fileset(path, pattern) {
        return asList(terraformFunction("fileset", [stringValue, stringValue])(path, pattern));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/filebase64.html filebase64} reads the contents of a file at the given path and returns them as a base64-encoded string.
     *
     * @experimental
     */
    static filebase64(value) {
        return asString(terraformFunction("filebase64", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/templatefile.html templatefile} reads the file at the given path and renders its content as a template using a supplied set of template variables.
     *
     * @experimental
     */
    static templatefile(path, vars) {
        return asString(terraformFunction("templatefile", [stringValue, mapValue])(path, vars));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/cidrhost.html cidrhost} calculates a full host IP address for a given host number within a given IP network address prefix.
     *
     * @experimental
     */
    static cidrhost(prefix, hostnum) {
        return asString(terraformFunction("cidrhost", [stringValue, numericValue])(prefix, hostnum));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/cidrnetmask.html cidrnetmask} converts an IPv4 address prefix given in CIDR notation into a subnet mask address.
     *
     * @experimental
     */
    static cidrnetmask(prefix) {
        return asString(terraformFunction("cidrnetmask", [stringValue])(prefix));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/cidrsubnet.html cidrsubnet} calculates a subnet address within given IP network address prefix.
     *
     * @experimental
     */
    static cidrsubnet(prefix, newbits, netnum) {
        return asString(terraformFunction("cidrsubnet", [
            stringValue,
            numericValue,
            numericValue,
        ])(prefix, newbits, netnum));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/cidrsubnets.html cidrsubnets} calculates a sequence of consecutive IP address ranges within a particular CIDR prefix.
     *
     * @experimental
     */
    static cidrsubnets(prefix, newbits) {
        return asList(terraformFunction("cidrsubnets", listOf(anyValue))(prefix, ...newbits));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/abs.html abs} returns the absolute value of the given number.
     *
     * @experimental
     */
    static abs(value) {
        return asNumber(terraformFunction("abs", [numericValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/ceil.html ceil} returns the closest whole number that is greater than or equal to the given value, which may be a fraction.
     *
     * @experimental
     */
    static ceil(value) {
        return asNumber(terraformFunction("ceil", [numericValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/floor.html floor} returns the closest whole number that is less than or equal to the given value, which may be a fraction.
     *
     * @experimental
     */
    static floor(value) {
        return asNumber(terraformFunction("floor", [numericValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/log.html log} returns the logarithm of a given number in a given base.
     *
     * @experimental
     */
    static log(value, base) {
        return asNumber(terraformFunction("log", [numericValue, numericValue])(value, base));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/max.html max} takes one or more numbers and returns the greatest number from the set.
     *
     * @experimental
     */
    static max(values) {
        return asNumber(terraformFunction("max", listOf(numericValue))(...values));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/min.html min} takes one or more numbers and returns the smallest number from the set.
     *
     * @experimental
     */
    static min(values) {
        return asNumber(terraformFunction("min", listOf(numericValue))(...values));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/parseint.html parseInt} parses the given string as a representation of an integer in the specified base and returns the resulting number. The base must be between 2 and 62 inclusive.
     *
     * @experimental
     */
    static parseInt(value, base) {
        return asNumber(terraformFunction("parseint", [stringValue, numericValue])(value, base));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/pow.html pow} calculates an exponent, by raising its first argument to the power of the second argument.
     *
     * @experimental
     */
    static pow(value, power) {
        return asNumber(terraformFunction("pow", [numericValue, numericValue])(value, power));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/signum.html signum} determines the sign of a number, returning a number between -1 and 1 to represent the sign.
     *
     * @experimental
     */
    static signum(value) {
        return asNumber(terraformFunction("signum", [numericValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/chomp.html chomp} removes newline characters at the end of a string.
     *
     * @experimental
     */
    static chomp(value) {
        return asString(terraformFunction("chomp", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/format.html format} produces a string by formatting a number of other values according to a specification string.
     *
     * @experimental
     */
    static format(spec, values) {
        return asString(terraformFunction("format", listOf(anyValue))(spec, ...values));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/formatlist.html formatlist} produces a list of strings by formatting a number of other values according to a specification string.
     *
     * @experimental
     */
    static formatlist(spec, values) {
        return asList(terraformFunction("formatlist", listOf(anyValue))(spec, ...values));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/indent.html indent} adds a given number of spaces to the beginnings of all but the first line in a given multi-line string.
     *
     * @experimental
     */
    static indent(indentation, value) {
        return asString(terraformFunction("indent", [numericValue, stringValue])(indentation, value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/join.html join} produces a string by concatenating together all elements of a given list of strings with the given delimiter.
     *
     * @experimental
     */
    static join(separator, value) {
        return asString(terraformFunction("join", [stringValue, listOf(anyValue)])(separator, value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/lower.html lower} converts all cased letters in the given string to lowercase.
     *
     * @experimental
     */
    static lower(value) {
        return asString(terraformFunction("lower", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/regexall.html regexall} applies a regular expression to a string and returns a list of all matches.
     *
     * @experimental
     */
    static regexall(pattern, value) {
        return asList(terraformFunction("regexall", [stringValue, stringValue])(pattern, value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/replace.html replace} searches a given string for another given substring, and replaces each occurrence with a given replacement string.
     *
     * @experimental
     */
    static replace(value, substring, replacement) {
        return asString(terraformFunction("replace", [stringValue, stringValue, stringValue])(value, substring, replacement));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/split.html split} produces a list by dividing a given string at all occurrences of a given separator.
     *
     * @experimental
     */
    static split(seperator, value) {
        return asList(terraformFunction("split", [stringValue, stringValue])(seperator, value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/strrev.html strrev} reverses the characters in a string.
     *
     * @experimental
     */
    static strrev(value) {
        return asString(terraformFunction("strrev", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/substr.html substr} extracts a substring from a given string by offset and length.
     *
     * @experimental
     */
    static substr(value, offset, length) {
        return asString(terraformFunction("substr", [stringValue, numericValue, numericValue])(value, offset, length));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/title.html title} converts the first letter of each word in the given string to uppercase.
     *
     * @experimental
     */
    static title(value) {
        return asString(terraformFunction("title", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/trim.html trim} removes the specified characters from the start and end of the given string.
     *
     * @experimental
     */
    static trim(value, replacement) {
        return asString(terraformFunction("trim", [stringValue, stringValue])(value, replacement));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/trimprefix.html trimprefix} removes the specified prefix from the start of the given string.
     *
     * @experimental
     */
    static trimprefix(value, prefix) {
        return asString(terraformFunction("trimprefix", [stringValue, stringValue])(value, prefix));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/trimsuffix.html trimsuffix} removes the specified suffix from the end of the given string.
     *
     * @experimental
     */
    static trimsuffix(value, suffix) {
        return asString(terraformFunction("trimsuffix", [stringValue, stringValue])(value, suffix));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/trimspace.html trimspace} removes any space characters from the start and end of the given string.
     *
     * @experimental
     */
    static trimspace(value) {
        return asString(terraformFunction("trimspace", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/upper.html upper} converts all cased letters in the given string to uppercase.
     *
     * @experimental
     */
    static upper(value) {
        return asString(terraformFunction("upper", [stringValue])(value));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/can.html can} evaluates the given expression and returns a boolean value indicating whether the expression produced a result without any errors.
     *
     * @experimental
     */
    static can(expression) {
        return asBoolean(terraformFunction("can", [anyValue])(expression));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/nonsensitive.html nonsensitive} takes a sensitive value and returns a copy of that value with the sensitive marking removed, thereby exposing the sensitive value.
     *
     * @experimental
     */
    static nonsensitive(expression) {
        return asAny(terraformFunction("nonsensitive", [anyValue])(expression));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/sensitive.html sensitive} takes any value and returns a copy of it marked so that Terraform will treat it as sensitive, with the same meaning and behavior as for sensitive input variables.
     *
     * @experimental
     */
    static sensitive(expression) {
        return asAny(terraformFunction("sensitive", [anyValue])(expression));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/tobool.html tobool} converts its argument to a boolean value.
     *
     * @experimental
     */
    static tobool(expression) {
        return asBoolean(terraformFunction("tobool", [anyValue])(expression));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/tolist.html tolist} converts its argument to a list value.
     *
     * @experimental
     */
    static tolist(expression) {
        return asList(terraformFunction("tolist", [anyValue])(expression));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/tomap.html tomap} converts its argument to a map value.
     *
     * @experimental
     */
    static tomap(expression) {
        return asAny(terraformFunction("tomap", [anyValue])(expression));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/tonumber.html tonumber} converts its argument to a number value.
     *
     * @experimental
     */
    static tonumber(expression) {
        return asNumber(terraformFunction("tonumber", [anyValue])(expression));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/toset.html toset} converts its argument to a set value.
     *
     * @experimental
     */
    static toset(expression) {
        return asAny(terraformFunction("toset", [anyValue])(expression));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/tostring.html tostring} converts its argument to a string value.
     *
     * @experimental
     */
    static tostring(expression) {
        return asString(terraformFunction("tostring", [anyValue])(expression));
    }
    /**
     * (experimental) {@link https://www.terraform.io/docs/language/functions/try.html try} evaluates all of its argument expressions in turn and returns the result of the first one that does not produce any errors.
     *
     * @experimental
     */
    static try(expression) {
        return asAny(terraformFunction("try", listOf(anyValue))(...expression));
    }
    /**
     * (experimental) Use this function to wrap a string and escape it properly for the use in Terraform This is only needed in certain scenarios (e.g., if you have unescaped double quotes in the string).
     *
     * @experimental
     */
    static rawString(str) {
        return asString(_1.rawString(str));
    }
}
exports.Fn = Fn;
_a = JSII_RTTI_SYMBOL_1;
Fn[_a] = { fqn: "cdktf.Fn", version: "0.9.0-pre.6" };
//# sourceMappingURL=data:application/json;base64,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