"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const index_1 = require("../index");
const resource_1 = require("../../../test/helper/resource");
const matchers_1 = require("../matchers");
const data_source_1 = require("../../../test/helper/data-source");
const terraform_stack_1 = require("../../terraform-stack");
const provider_1 = require("../../../test/helper/provider");
const fs = require("fs");
const path = require("path");
function corruptSynthesizedStack(stackPath) {
    const manifest = JSON.parse(fs.readFileSync(path.resolve(stackPath, "manifest.json"), "utf8"));
    const { synthesizedStackPath } = Object.values(manifest.stacks)[0];
    const absoluteStackPath = path.resolve(stackPath, synthesizedStackPath);
    fs.writeFileSync(absoluteStackPath, fs.readFileSync(absoluteStackPath, "utf8") +
        "Weird text that makes the JSON file invalid", "utf8");
}
describe("deepEqualIgnoringObjectCasing", () => {
    it("should compare simple items", () => {
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing(1, 1)).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing("1", "1")).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing(true, true)).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing(false, false)).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing(null, null)).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing(undefined, undefined)).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing([1, 2, 3], [1, 2, 3])).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: 1, b: 2 }, { a: 1, b: 2 })).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing(1, 2)).toBe(false);
    });
    it("should compare arrays", () => {
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing([1, 2, 3], [1, 2, 3])).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing([1, 2, 3], [1, 2, 4])).toBe(false);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing([1, 2, 3], [1, 2, 3, 4])).toBe(false);
    });
    it("should compare objects", () => {
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: 1, b: 2 }, { a: 1, b: 2 })).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: { c: 3 }, b: 2 }, { a: { c: 3 }, b: 2 })).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: 3, b: 2 }, { a: { c: 3 }, b: 2 })).toBe(false);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: { c: 3 }, b: 2 }, { a: 3, b: 2 })).toBe(false);
    });
    it("should ignore case when comparing object keys", () => {
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: 1, fooBar: 2 }, { a: 1, foo_bar: 2 })).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: { fooBar: 2 } }, { a: { foo_bar: 2 } })).toBe(true);
    });
    it("should ignore keys not present in expectation", () => {
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: { fooBar: 2 } }, { a: { foo_bar: 2 }, b: 3 })).toBe(true);
    });
});
describe("matchers", () => {
    const toHaveResourceWithProperties = matchers_1.getToHaveResourceWithProperties();
    describe("toHaveResourceWithProperties", () => {
        let synthesizedStack;
        beforeEach(() => {
            synthesizedStack = index_1.Testing.synthScope((scope) => {
                new resource_1.TestResource(scope, "test", { name: "test", tags: { foo: "bar" } });
            });
        });
        it("should pass with no properties", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, resource_1.TestResource);
            expect(res.pass).toBeTruthy();
            expect(res.message()).toMatchInlineSnapshot(`
        "Expected no test_resource with properties {} to be present in synthesised stack.
        Found 1 test_resource resources instead:
        [
          {
            \\"name\\": \\"test\\",
            \\"tags\\": {
              \\"foo\\": \\"bar\\"
            }
          }
        ]"
      `);
        });
        it("should fail with wrong resouce", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, data_source_1.TestDataSource);
            expect(res.pass).toBeFalsy();
            expect(res.message()).toMatchInlineSnapshot(`
        "Expected test_data_source with properties {} to be present in synthesised stack.
        Found no test_data_source resources instead"
      `);
        });
        it("should work on nested elements", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, resource_1.TestResource, {
                tags: { foo: "bar" },
            });
            expect(res.pass).toBeTruthy();
        });
    });
    describe("toHaveDataSourceWithProperties", () => {
        const toHaveDataSourceWithProperties = matchers_1.getToHaveDataSourceWithProperties();
        let synthesizedStack;
        beforeEach(() => {
            synthesizedStack = index_1.Testing.synthScope((scope) => {
                new resource_1.TestResource(scope, "test", { name: "test" });
                new data_source_1.TestDataSource(scope, "test-data-source", { name: "data" });
            });
        });
        it("should not find resources", () => {
            const res = toHaveDataSourceWithProperties(synthesizedStack, resource_1.TestResource);
            expect(res.pass).toBeFalsy();
        });
        it("should find data sources", () => {
            const res = toHaveDataSourceWithProperties(synthesizedStack, data_source_1.TestDataSource, { name: "data" });
            expect(res.pass).toBeTruthy();
        });
    });
    describe("toBeValidTerraform", () => {
        it("fails if anything but a path is passed", () => {
            const res = matchers_1.toBeValidTerraform("not a path");
            expect(res.pass).toBeFalsy();
            expect(res.message()).toMatchInlineSnapshot(`"Expected subject to be a terraform directory: Error: ENOENT: no such file or directory, stat 'not a path'"`);
        });
        it("succeeds if the terraform config passed is valid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const res = matchers_1.toBeValidTerraform(index_1.Testing.fullSynth(stack));
            expect(res.pass).toBeTruthy();
            expect(res.message()).toMatchInlineSnapshot(`"Expected subject not to be a valid terraform stack"`);
        });
        it("fails if the terraform config passed is invalid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const result = index_1.Testing.fullSynth(stack);
            corruptSynthesizedStack(result);
            const res = matchers_1.toBeValidTerraform(result);
            expect(res.pass).toBeFalsy();
            expect(res.message()).toEqual(expect.stringContaining("Expected subject to be a valid terraform stack"));
        });
    });
    describe("toPlanSuccessfully", () => {
        it("fails if anything but a path is passed", () => {
            const res = matchers_1.toPlanSuccessfully("not a path");
            expect(res.pass).toBeFalsy();
            expect(res.message()).toMatchInlineSnapshot(`"Expected subject to be a terraform directory: Error: ENOENT: no such file or directory, stat 'not a path'"`);
        });
        it("succeeds if the terraform config passed is valid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const res = matchers_1.toPlanSuccessfully(index_1.Testing.fullSynth(stack));
            expect(res.pass).toBeTruthy();
            expect(res.message()).toMatchInlineSnapshot(`"Expected subject not to plan successfully"`);
        });
        it("fails if the terraform config passed is invalid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const result = index_1.Testing.fullSynth(stack);
            corruptSynthesizedStack(result);
            const res = matchers_1.toPlanSuccessfully(result);
            expect(res.pass).toBeFalsy();
            expect(res.message()).toEqual(expect.stringContaining("Expected subject to plan successfully"));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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