"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.toPlanSuccessfully = exports.toBeValidTerraform = exports.getToHaveResourceWithProperties = exports.getToHaveDataSourceWithProperties = exports.asymetricDeepEqualIgnoringObjectCasing = void 0;
const fs = require("fs");
const path = require("path");
const child_process_1 = require("child_process");
const util_1 = require("../util");
const terraformBinaryName = process.env.TERRAFORM_BINARY_NAME || "terraform";
// All expected properties are matched and considered equal if
// There can be more properties in the received object than in the expected object while still returning true
function asymetricDeepEqualIgnoringObjectCasing(expected, received) {
    switch (typeof expected) {
        case "object":
            if (Array.isArray(expected)) {
                return (Array.isArray(received) &&
                    expected.length === received.length &&
                    expected.every((item, index) => asymetricDeepEqualIgnoringObjectCasing(item, received[index]) // recursively compare arrays
                    ));
            }
            if (expected === null && received === null) {
                return true;
            }
            if (expected === undefined && received === undefined) {
                return true;
            }
            if (expected === null || received === null) {
                return false;
            }
            // recursively compare objects and allow snake case as well as camel case
            return Object.keys(expected).every((key) => {
                if (received[key] !== undefined) {
                    return asymetricDeepEqualIgnoringObjectCasing(expected[key], received[key]);
                }
                if (received[util_1.snakeCase(key)] !== undefined) {
                    return asymetricDeepEqualIgnoringObjectCasing(expected[key], received[util_1.snakeCase(key)]);
                }
                return false;
            });
        default:
            return expected === received;
    }
}
exports.asymetricDeepEqualIgnoringObjectCasing = asymetricDeepEqualIgnoringObjectCasing;
const defaultPassEvaluation = (items, assertedProperties) => {
    return Object.values(items).some((item) => asymetricDeepEqualIgnoringObjectCasing(assertedProperties, item));
};
function isAsymmetric(obj) {
    return !!obj && typeof obj === "object" && "asymmetricMatch" in obj;
}
// You can use expect.Anything(), expect.ObjectContaining, etc in jest, this makes it nicer to read
// when we print error mesages
function jestAsymetricMatcherStringifyReplacer(_key, value) {
    return isAsymmetric(value) ? `expect.${value.toString()}` : value;
}
function getAssertElementWithProperties(
// We have the evaluation function configurable so we can make use of the specific testing frameworks capabilities
// This makes the resulting tests more native to the testing framework
customPassEvaluation) {
    const passEvaluation = customPassEvaluation || defaultPassEvaluation;
    return function getAssertElementWithProperties(type, received, itemType, properties = {}) {
        var _a;
        let stack;
        try {
            stack = JSON.parse(received);
        }
        catch (e) {
            throw new Error(`invalid JSON string passed: ${received}`);
        }
        if (!(type in stack)) {
            throw new Error(`Type ${type} not found in stack`);
        }
        const items = Object.values(((_a = Object.entries(stack[type] || {}) // for all data/resource entries
            .find(
        // find the object with a matching name
        ([type, _values]) => type === itemType.tfResourceType)) === null || _a === void 0 ? void 0 : _a[1]) || {} // get all items of that type (encoded as a record of name -> config)
        ) || []; // get a list of all configs of that type
        const pass = passEvaluation(items, properties);
        if (pass) {
            return {
                pass,
                message: () => `Expected no ${itemType.tfResourceType} with properties ${JSON.stringify(properties, jestAsymetricMatcherStringifyReplacer)} to be present in synthesised stack.
Found ${items.length === 0 ? "no" : items.length} ${itemType.tfResourceType} resources instead${items.length > 0 ? ":\n" + JSON.stringify(items, null, 2) : ""}`,
            };
        }
        else {
            return {
                message: () => `Expected ${itemType.tfResourceType} with properties ${JSON.stringify(properties, jestAsymetricMatcherStringifyReplacer)} to be present in synthesised stack.
Found ${items.length === 0 ? "no" : items.length} ${itemType.tfResourceType} resources instead${items.length > 0 ? ":\n" + JSON.stringify(items, null, 2) : ""}`,
                pass,
            };
        }
    };
}
function getToHaveDataSourceWithProperties(customPassEvaluation) {
    return function toHaveDataSourceWithProperties(received, resourceType, properties = {}) {
        return getAssertElementWithProperties(customPassEvaluation)("data", received, resourceType, properties);
    };
}
exports.getToHaveDataSourceWithProperties = getToHaveDataSourceWithProperties;
function getToHaveResourceWithProperties(customPassEvaluation) {
    return function toHaveResourceWithProperties(received, resourceType, properties = {}) {
        return getAssertElementWithProperties(customPassEvaluation)("resource", received, resourceType, properties);
    };
}
exports.getToHaveResourceWithProperties = getToHaveResourceWithProperties;
function toBeValidTerraform(received) {
    try {
        if (!fs.statSync(received).isDirectory()) {
            throw new Error("Path is not a directory");
        }
    }
    catch (e) {
        return {
            message: () => `Expected subject to be a terraform directory: ${e}`,
            pass: false,
        };
    }
    try {
        const manifest = JSON.parse(fs.readFileSync(path.resolve(received, "manifest.json"), "utf8"));
        const stacks = Object.entries(manifest.stacks);
        stacks.forEach(([name, stack]) => {
            const opts = {
                cwd: path.resolve(received, stack.workingDirectory),
                env: process.env,
                stdio: "pipe",
            };
            child_process_1.execSync(`${terraformBinaryName} init`, opts);
            const out = child_process_1.execSync(`${terraformBinaryName} validate -json`, opts);
            const result = JSON.parse(out.toString());
            if (!result.valid) {
                throw new Error(`Found ${result.error_count} Errors in stack ${name}: ${result.diagnostics.join("\n")}`);
            }
        });
        return {
            pass: true,
            message: () => `Expected subject not to be a valid terraform stack`,
        };
    }
    catch (e) {
        return {
            pass: false,
            message: () => `Expected subject to be a valid terraform stack: ${e}`,
        };
    }
}
exports.toBeValidTerraform = toBeValidTerraform;
function toPlanSuccessfully(received) {
    try {
        if (!fs.statSync(received).isDirectory()) {
            throw new Error("Path is not a directory");
        }
    }
    catch (e) {
        return {
            message: () => `Expected subject to be a terraform directory: ${e}`,
            pass: false,
        };
    }
    try {
        const manifest = JSON.parse(fs.readFileSync(path.resolve(received, "manifest.json"), "utf8"));
        const stacks = Object.entries(manifest.stacks);
        stacks.forEach(([, stack]) => {
            const opts = {
                cwd: path.resolve(received, stack.workingDirectory),
                env: process.env,
                stdio: "ignore",
            };
            child_process_1.execSync(`${terraformBinaryName} init`, opts);
            // Throws on a non-zero exit code
            child_process_1.execSync(`${terraformBinaryName} plan -input=false `, opts);
        });
        return {
            pass: true,
            message: () => `Expected subject not to plan successfully`,
        };
    }
    catch (e) {
        return {
            pass: false,
            message: () => `Expected subject to plan successfully: ${e}`,
        };
    }
}
exports.toPlanSuccessfully = toPlanSuccessfully;
//# sourceMappingURL=data:application/json;base64,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