"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.call = exports.orOperation = exports.andOperation = exports.neqOperation = exports.eqOperation = exports.lteOperation = exports.ltOperation = exports.gteOperation = exports.gtOperation = exports.subOperation = exports.addOperation = exports.modOperation = exports.divOperation = exports.mulOperation = exports.negateOperation = exports.notOperation = exports.conditional = exports.propertyAccess = exports.ref = exports.rawString = void 0;
const intrinsic_1 = require("./tokens/private/intrinsic");
const token_1 = require("./tokens/token");
const _1 = require(".");
class TFExpression extends intrinsic_1.Intrinsic {
    constructor() {
        super(...arguments);
        this.isInnerTerraformExpression = false;
    }
    resolveArg(context, arg) {
        const resolvedArg = context.resolve(arg);
        if (token_1.Tokenization.isResolvable(arg)) {
            return resolvedArg;
        }
        if (typeof arg === "string") {
            return this.resolveString(arg, resolvedArg);
        }
        if (Array.isArray(resolvedArg)) {
            return `[${resolvedArg
                .map((_, index) => this.resolveArg(context, arg[index]))
                .join(", ")}]`;
        }
        if (typeof resolvedArg === "object") {
            return `{${Object.keys(resolvedArg)
                .map((key) => `${key} = ${this.resolveArg(context, arg[key])}`)
                .join(", ")}}`;
        }
        return resolvedArg;
    }
    /**
     * Escape string removes characters from the string that are not allowed in Terraform or JSON
     * It must only be used on non-token values
     */
    escapeString(str) {
        return str // Escape double quotes
            .replace(/\n/g, "\\n") // escape newlines
            .replace(/\${/g, "$$${"); // escape ${ to $${
    }
    resolveString(str, resolvedArg) {
        const tokenList = token_1.Tokenization.reverseString(str);
        const numberOfTokens = tokenList.tokens.length + tokenList.intrinsic.length;
        // String literal
        if (numberOfTokens === 0) {
            return resolvedArg !== `"` &&
                resolvedArg.startsWith('"') &&
                resolvedArg.endsWith('"')
                ? this.escapeString(resolvedArg)
                : `"${this.escapeString(resolvedArg)}"`;
        }
        // Only a token reference
        if (tokenList.literals.length === 0 && numberOfTokens === 1) {
            return resolvedArg;
        }
        // String literal + token reference combination
        return `"${tokenList.join({
            join: (left, right) => {
                const leftTokens = token_1.Tokenization.reverse(left);
                const rightTokens = token_1.Tokenization.reverse(right);
                const leftValue = leftTokens.length === 0
                    ? this.escapeString(left)
                    : `\${${leftTokens[0]}}`;
                const rightValue = rightTokens.length === 0
                    ? this.escapeString(right)
                    : `\${${rightTokens[0]}}`;
                return `${leftValue}${rightValue}`;
            },
        })}"`;
    }
}
// A string that represents an input value to be escaped
class RawString extends TFExpression {
    constructor(str) {
        super(str);
        this.str = str;
    }
    resolve() {
        const qts = this.isInnerTerraformExpression ? `"` : ``;
        return `${qts}${this.escapeString(this.str).replace(/\"/g, '\\"')}${qts}`; // eslint-disable-line no-useless-escape
    }
    toString() {
        return this.str;
    }
}
function rawString(str) {
    return new RawString(str);
}
exports.rawString = rawString;
class Reference extends TFExpression {
    constructor(identifier) {
        super(identifier);
        this.identifier = identifier;
    }
    resolve() {
        return this.isInnerTerraformExpression
            ? this.identifier
            : `\${${this.identifier}}`;
    }
}
function ref(identifier) {
    return new Reference(identifier);
}
exports.ref = ref;
function markAsInner(arg) {
    if (arg instanceof TFExpression) {
        arg.isInnerTerraformExpression = true;
        return;
    }
    // reverese tokens here and set inner tf expression flag
    token_1.Tokenization.reverse(arg).map((resolvable) => {
        if (resolvable instanceof TFExpression) {
            resolvable.isInnerTerraformExpression = true;
        }
        else if (resolvable instanceof _1.LazyBase) {
            resolvable.addPostProcessor({
                postProcess: (value) => {
                    markAsInner(value);
                    return value;
                },
            });
        }
    });
    if (typeof arg === "object") {
        if (Array.isArray(arg)) {
            arg.forEach(markAsInner);
        }
        else {
            Object.keys(arg).forEach((key) => markAsInner(arg[key]));
        }
    }
}
class PropertyAccess extends TFExpression {
    constructor(target, args) {
        super({ target, args });
        this.target = target;
        this.args = args;
    }
    resolve(context) {
        markAsInner(this.target);
        this.args.forEach(markAsInner);
        const serializedArgs = this.args
            .map((arg) => this.resolveArg(context, arg))
            .map((a) => `[${a}]`) // property access
            .join("");
        const expr = `${this.target}${serializedArgs}`;
        return this.isInnerTerraformExpression ? expr : `\${${expr}}`;
    }
}
function propertyAccess(target, args) {
    return new PropertyAccess(target, args);
}
exports.propertyAccess = propertyAccess;
class ConditionalExpression extends TFExpression {
    constructor(condition, trueValue, falseValue) {
        super({ condition, trueValue, falseValue });
        this.condition = condition;
        this.trueValue = trueValue;
        this.falseValue = falseValue;
    }
    resolve(context) {
        markAsInner(this.condition);
        markAsInner(this.trueValue);
        markAsInner(this.falseValue);
        const condition = this.resolveArg(context, this.condition);
        const trueValue = this.resolveArg(context, this.trueValue);
        const falseValue = this.resolveArg(context, this.falseValue);
        const expr = `${condition} ? ${trueValue} : ${falseValue}`;
        return this.isInnerTerraformExpression ? expr : `\${${expr}}`;
    }
}
/**
 * creates an expression like `true ? 1 : 0`
 */
function conditional(condition, trueValue, falseValue) {
    return new ConditionalExpression(condition, trueValue, falseValue);
}
exports.conditional = conditional;
class OperatorExpression extends TFExpression {
    constructor(operator, left, right // optional for ! and - operator
    ) {
        super({ operator, leftValue: left, rightValue: right });
        this.operator = operator;
        this.left = left;
        this.right = right;
    }
    resolve(context) {
        markAsInner(this.left);
        if (this.right)
            markAsInner(this.right);
        const left = this.resolveArg(context, this.left);
        const right = this.right ? this.resolveArg(context, this.right) : undefined;
        let expr = "";
        switch (this.operator) {
            case "!": {
                expr = `!${left}`; // no braces needed as ! has highest precedence
                break;
            }
            case "-": {
                if (right) {
                    // subtraction
                    expr = `(${left} - ${right})`;
                }
                else {
                    // negation
                    expr = `-${left}`; // no braces needed as - has highest precedence
                }
                break;
            }
            default: {
                expr = `(${left} ${this.operator} ${right})`;
            }
        }
        return this.isInnerTerraformExpression ? expr : `\${${expr}}`;
    }
}
function notOperation(expression) {
    return new OperatorExpression("!", expression);
}
exports.notOperation = notOperation;
function negateOperation(expression) {
    return new OperatorExpression("-", expression);
}
exports.negateOperation = negateOperation;
function mulOperation(left, right) {
    return new OperatorExpression("*", left, right);
}
exports.mulOperation = mulOperation;
function divOperation(left, right) {
    return new OperatorExpression("/", left, right);
}
exports.divOperation = divOperation;
function modOperation(left, right) {
    return new OperatorExpression("%", left, right);
}
exports.modOperation = modOperation;
function addOperation(left, right) {
    return new OperatorExpression("+", left, right);
}
exports.addOperation = addOperation;
function subOperation(left, right) {
    return new OperatorExpression("-", left, right);
}
exports.subOperation = subOperation;
function gtOperation(left, right) {
    return new OperatorExpression(">", left, right);
}
exports.gtOperation = gtOperation;
function gteOperation(left, right) {
    return new OperatorExpression(">=", left, right);
}
exports.gteOperation = gteOperation;
function ltOperation(left, right) {
    return new OperatorExpression("<", left, right);
}
exports.ltOperation = ltOperation;
function lteOperation(left, right) {
    return new OperatorExpression("<=", left, right);
}
exports.lteOperation = lteOperation;
function eqOperation(left, right) {
    return new OperatorExpression("==", left, right);
}
exports.eqOperation = eqOperation;
function neqOperation(left, right) {
    return new OperatorExpression("!=", left, right);
}
exports.neqOperation = neqOperation;
function andOperation(left, right) {
    return new OperatorExpression("&&", left, right);
}
exports.andOperation = andOperation;
function orOperation(left, right) {
    return new OperatorExpression("||", left, right);
}
exports.orOperation = orOperation;
class FunctionCall extends TFExpression {
    constructor(name, args) {
        super({ name, args });
        this.name = name;
        this.args = args;
    }
    resolve(context) {
        this.args.forEach(markAsInner);
        const serializedArgs = this.args
            .map((arg) => this.resolveArg(context, arg))
            .join(", ");
        const expr = `${this.name}(${serializedArgs})`;
        return this.isInnerTerraformExpression ? expr : `\${${expr}}`;
    }
}
function call(name, args) {
    return new FunctionCall(name, args);
}
exports.call = call;
//# sourceMappingURL=data:application/json;base64,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