"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DependableTrait = exports.ConcreteDependable = void 0;
/**
 * A set of constructs to be used as a dependable
 *
 * This class can be used when a set of constructs which are disjoint in the
 * construct tree needs to be combined to be used as a single dependable.
 *
 * @experimental
 */
class ConcreteDependable {
    constructor() {
        this._dependencyRoots = new Array();
        // eslint-disable-next-line @typescript-eslint/no-this-alias
        const self = this;
        DependableTrait.implement(this, {
            get dependencyRoots() {
                return self._dependencyRoots;
            },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(construct) {
        this._dependencyRoots.push(construct);
    }
}
exports.ConcreteDependable = ConcreteDependable;
const DEPENDABLE_SYMBOL = Symbol.for("@aws-cdk/core.DependableTrait");
/**
 * Trait for IDependable
 *
 * Traits are interfaces that are privately implemented by objects. Instead of
 * showing up in the public interface of a class, they need to be queried
 * explicitly. This is used to implement certain framework features that are
 * not intended to be used by Construct consumers, and so should be hidden
 * from accidental use.
 *
 * @example
 *
 * // Usage
 * const roots = DependableTrait.get(construct).dependencyRoots;
 *
 * // Definition
 * DependableTrait.implement(construct, {
 *   get dependencyRoots() { return []; }
 * });
 *
 * @experimental
 */
class DependableTrait {
    /**
     * Register `instance` to have the given DependableTrait
     *
     * Should be called in the class constructor.
     */
    static implement(instance, trait) {
        // I would also like to reference classes (to cut down on the list of objects
        // we need to manage), but we can't do that either since jsii doesn't have the
        // concept of a class reference.
        instance[DEPENDABLE_SYMBOL] = trait;
    }
    /**
     * Return the matching DependableTrait for the given class instance.
     */
    static get(instance) {
        const ret = instance[DEPENDABLE_SYMBOL];
        if (!ret) {
            throw new Error(`${instance} does not implement DependableTrait`);
        }
        return ret;
    }
}
exports.DependableTrait = DependableTrait;
//# sourceMappingURL=data:application/json;base64,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