"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractTokenDouble = exports.createTokenDouble = exports.unresolved = exports.containsComplexElement = exports.isComplexElement = exports.containsListTokenElement = exports.NullConcat = exports.regexQuote = exports.TokenString = exports.VALID_KEY_CHARS = exports.END_TOKEN_MARKER = exports.BEGIN_LIST_TOKEN_MARKER = exports.BEGIN_STRING_TOKEN_MARKER = void 0;
const string_fragments_1 = require("../string-fragments");
const token_1 = require("../token");
// Details for encoding and decoding Tokens into native types; should not be exported
exports.BEGIN_STRING_TOKEN_MARKER = "${TfToken[";
exports.BEGIN_LIST_TOKEN_MARKER = "#{TfToken[";
exports.END_TOKEN_MARKER = "]}";
exports.VALID_KEY_CHARS = "a-zA-Z0-9:._-";
const QUOTED_BEGIN_STRING_TOKEN_MARKER = regexQuote(exports.BEGIN_STRING_TOKEN_MARKER);
const QUOTED_BEGIN_LIST_TOKEN_MARKER = regexQuote(exports.BEGIN_LIST_TOKEN_MARKER);
const QUOTED_END_TOKEN_MARKER = regexQuote(exports.END_TOKEN_MARKER);
const STRING_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_STRING_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, "g");
const LIST_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_LIST_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, "g");
const NUMBER_TOKEN_REGEX = new RegExp("[-|\\d|.\\+]+(e[-|\\d|.|e|E|\\+]+)", "g");
/**
 * A string with markers in it that can be resolved to external values
 */
class TokenString {
    constructor(str, re, regexMatchIndex = 1) {
        this.str = str;
        this.re = re;
        this.regexMatchIndex = regexMatchIndex;
    }
    /**
     * Returns a `TokenString` for this string.
     */
    static forString(s) {
        return new TokenString(s, STRING_TOKEN_REGEX);
    }
    /**
     * Returns a `TokenString` for this string (must be the first string element of the list)
     */
    static forListToken(s) {
        return new TokenString(s, LIST_TOKEN_REGEX);
    }
    /**
     * Returns a `TokenString` for this string that handles encoded numbers
     */
    static forNumbers(s) {
        return new TokenString(s, NUMBER_TOKEN_REGEX, 0);
    }
    /**
     * Split string on markers, substituting markers with Tokens
     */
    split(lookup) {
        const ret = new string_fragments_1.TokenizedStringFragments();
        let rest = 0;
        this.re.lastIndex = 0; // Reset
        let m = this.re.exec(this.str);
        while (m) {
            if (m.index > rest) {
                ret.addLiteral(this.str.substring(rest, m.index));
            }
            const token = lookup(m[this.regexMatchIndex]);
            if (token) {
                ret.addToken(token);
            }
            else {
                ret.addLiteral(this.str.substring(m.index, this.re.lastIndex));
            }
            rest = this.re.lastIndex;
            m = this.re.exec(this.str);
        }
        if (rest < this.str.length) {
            ret.addLiteral(this.str.substring(rest));
        }
        return ret;
    }
    /**
     * Indicates if this string includes tokens.
     */
    test() {
        this.re.lastIndex = 0; // Reset
        return this.re.test(this.str);
    }
}
exports.TokenString = TokenString;
/**
 * Quote a string for use in a regex
 */
function regexQuote(s) {
    return s.replace(/[.?*+^$[\]\\(){}|-]/g, "\\$&");
}
exports.regexQuote = regexQuote;
/**
 * Concatenator that disregards the input
 *
 * Can be used when traversing the tokens is important, but the
 * result isn't.
 */
class NullConcat {
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    join(_left, _right) {
        return undefined;
    }
}
exports.NullConcat = NullConcat;
function containsListTokenElement(xs) {
    return xs.some((x) => typeof x === "string" && TokenString.forListToken(x).test());
}
exports.containsListTokenElement = containsListTokenElement;
function isComplexElement(xs) {
    return (typeof xs === "object" &&
        xs !== null &&
        "withinArray" in xs &&
        "interpolationAsList" in xs);
}
exports.isComplexElement = isComplexElement;
function containsComplexElement(xs) {
    return xs.length > 0 && isComplexElement(xs[0]);
}
exports.containsComplexElement = containsComplexElement;
/**
 * Returns true if obj is a token (i.e. has the resolve() method or is a string
 * that includes token markers), or it's a listifictaion of a Token string.
 *
 * @param obj The object to test.
 */
function unresolved(obj) {
    if (typeof obj === "string") {
        return TokenString.forString(obj).test();
    }
    else if (typeof obj === "number") {
        return extractTokenDouble(obj) !== undefined;
    }
    else if (Array.isArray(obj) && obj.length === 1) {
        return (typeof obj[0] === "string" && TokenString.forListToken(obj[0]).test());
    }
    else {
        return token_1.Tokenization.isResolvable(obj);
    }
}
exports.unresolved = unresolved;
/**
 * Bit pattern in the top 16 bits of a double to indicate a Token
 *
 * An IEEE double in LE memory order looks like this (grouped
 * into octets, then grouped into 32-bit words):
 *
 * mmmmmmmm.mmmmmmmm.mmmmmmmm.mmmmmmmm | mmmmmmmm.mmmmmmmm.EEEEmmmm.sEEEEEEE
 *
 * - m: mantissa (52 bits)
 * - E: exponent (11 bits)
 * - s: sign (1 bit)
 *
 * We put the following marker into the top 16 bits (exponent and sign), and
 * use the mantissa part to encode the token index. To save some bit twiddling
 * we use all top 16 bits for the tag. That loses us 4 mantissa bits to store
 * information in but we still have 48, which is going to be plenty for any
 * number of tokens to be created during the lifetime of any CDK application.
 *
 * Can't have all bits set because that makes a NaN, so unset the least
 * significant exponent bit.
 *
 * Currently not supporting BE architectures.
 */
// tslint:disable-next-line:no-bitwise
const DOUBLE_TOKEN_MARKER_BITS = 0xfdff << 16;
/**
 * Highest encodable number
 */
const MAX_ENCODABLE_INTEGER = Math.pow(2, 48) - 1;
/**
 * Get 2^32 as a number, so we can do multiplication and div instead of bit shifting
 *
 * Necessary because in JavaScript, bit operations implicitly convert
 * to int32 and we need them to work on "int64"s.
 *
 * So instead of x >> 32, we do Math.floor(x / 2^32), and vice versa.
 */
const BITS32 = Math.pow(2, 32);
/**
 * Return a special Double value that encodes the given nonnegative integer
 *
 * We use this to encode Token ordinals.
 */
function createTokenDouble(x) {
    if (Math.floor(x) !== x || x < 0) {
        throw new Error("Can only encode positive integers");
    }
    if (x > MAX_ENCODABLE_INTEGER) {
        throw new Error(`Got an index too large to encode: ${x}`);
    }
    const buf = new ArrayBuffer(8);
    const ints = new Uint32Array(buf);
    // tslint:disable:no-bitwise
    ints[0] = x & 0x0000ffffffff; // Bottom 32 bits of number
    // This needs an "x >> 32" but that will make it a 32-bit number instead
    // of a 64-bit number.
    ints[1] = (shr32(x) & 0xffff) | DOUBLE_TOKEN_MARKER_BITS; // Top 16 bits of number and the mask
    // tslint:enable:no-bitwise
    return new Float64Array(buf)[0];
}
exports.createTokenDouble = createTokenDouble;
/**
 * Shift a 64-bit int right 32 bits
 */
function shr32(x) {
    return Math.floor(x / BITS32);
}
/**
 * Shift a 64-bit left 32 bits
 */
function shl32(x) {
    return x * BITS32;
}
/**
 * Extract the encoded integer out of the special Double value
 *
 * Returns undefined if the float is a not an encoded token.
 */
function extractTokenDouble(encoded) {
    const buf = new ArrayBuffer(8);
    new Float64Array(buf)[0] = encoded;
    const ints = new Uint32Array(buf);
    // tslint:disable:no-bitwise
    if ((ints[1] & 0xffff0000) !== DOUBLE_TOKEN_MARKER_BITS) {
        return undefined;
    }
    // Must use + instead of | here (bitwise operations
    // will force 32-bits integer arithmetic, + will not).
    return ints[0] + shl32(ints[1] & 0xffff);
    // tslint:enable:no-bitwise
}
exports.extractTokenDouble = extractTokenDouble;
//# sourceMappingURL=data:application/json;base64,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