"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RememberingTokenResolver = exports.findTokens = exports.resolve = void 0;
const resolvable_1 = require("../resolvable");
const encoding_1 = require("./encoding");
const token_map_1 = require("./token-map");
// This file should not be exported to consumers, resolving should happen through Construct.resolve()
const tokenMap = token_map_1.TokenMap.instance();
/**
 * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
 * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
 *
 * @param obj The object to resolve.
 * @param prefix Prefix key path components for diagnostics.
 */
function resolve(obj, options) {
    const prefix = options.prefix || [];
    const pathName = "/" + prefix.join("/");
    /**
     * Make a new resolution context
     */
    function makeContext(appendPath) {
        const newPrefix = appendPath !== undefined ? prefix.concat([appendPath]) : options.prefix;
        let postProcessor;
        const context = {
            preparing: options.preparing,
            scope: options.scope,
            registerPostProcessor(pp) {
                postProcessor = pp;
            },
            resolve(x) {
                return resolve(x, { ...options, prefix: newPrefix });
            },
        };
        return [
            context,
            {
                postProcess(x) {
                    return postProcessor ? postProcessor.postProcess(x, context) : x;
                },
            },
        ];
    }
    // protect against cyclic references by limiting depth.
    if (prefix.length > 200) {
        throw new Error("Unable to resolve object tree with circular reference. Path: " + pathName);
    }
    //
    // undefined
    //
    if (typeof obj === "undefined") {
        return undefined;
    }
    //
    // null
    //
    if (obj === null) {
        return null;
    }
    //
    // functions - not supported (only tokens are supported)
    //
    if (typeof obj === "function") {
        throw new Error(`Trying to resolve a non-data object. Only token are supported for lazy evaluation. Path: ${pathName}. Object: ${obj}`);
    }
    //
    // string - potentially replace all stringified Tokens
    //
    if (typeof obj === "string") {
        // If this is a "list element" Token, it should never occur by itself in string context
        if (encoding_1.TokenString.forListToken(obj).test()) {
            throw new Error("Found an encoded list token string in a scalar string context. Use 'Fn.element(list, 0)' (not 'list[0]') to extract elements from token lists.");
        }
        let str = obj;
        const tokenStr = encoding_1.TokenString.forString(str);
        if (tokenStr.test()) {
            const fragments = tokenStr.split(tokenMap.lookupToken.bind(tokenMap));
            str = options.resolver.resolveString(fragments, makeContext()[0]);
        }
        // replace concatenated token numbers
        const tokenNumberStr = encoding_1.TokenString.forNumbers(str);
        if (tokenNumberStr.test()) {
            const fragments = tokenNumberStr.split((id) => {
                return token_map_1.TokenMap.instance().lookupNumberToken(parseFloat(id));
            });
            str = fragments
                .mapTokens({
                mapToken: (resolvable) => makeContext()[0].resolve(resolvable),
            })
                .join(new resolvable_1.StringConcat());
        }
        return str;
    }
    //
    // number - potentially decode Tokenized number
    //
    if (typeof obj === "number") {
        return resolveNumberToken(obj, makeContext()[0]);
    }
    //
    // primitives - as-is
    //
    if (typeof obj !== "object" || obj instanceof Date) {
        return obj;
    }
    //
    // arrays - resolve all values, remove undefined and remove empty arrays
    //
    if (Array.isArray(obj)) {
        if (encoding_1.containsListTokenElement(obj)) {
            return options.resolver.resolveList(obj, makeContext()[0]);
        }
        const arr = obj
            .map((x, i) => makeContext(`${i}`)[0].resolve(x))
            .filter((x) => typeof x !== "undefined");
        return arr;
    }
    //
    // tokens - invoke 'resolve' and continue to resolve recursively
    //
    if (encoding_1.unresolved(obj)) {
        const [context, postProcessor] = makeContext();
        return options.resolver.resolveToken(obj, context, postProcessor);
    }
    //
    // objects - deep-resolve all values
    //
    // Must not be a Construct at this point, otherwise you probably made a typo
    // mistake somewhere and resolve will get into an infinite loop recursing into
    // child.parent <---> parent.children
    if (isConstruct(obj)) {
        throw new Error("Trying to resolve() a Construct at " + pathName);
    }
    const result = {};
    for (const key of Object.keys(obj)) {
        const resolvedKey = makeContext()[0].resolve(key);
        if (typeof resolvedKey !== "string") {
            throw new Error(`"${key}" is used as the key in a map so must resolve to a string, but it resolves to: ${JSON.stringify(resolvedKey)}`);
        }
        const value = makeContext(key)[0].resolve(obj[key]);
        // skip undefined
        if (typeof value === "undefined") {
            continue;
        }
        result[resolvedKey] = value;
    }
    return result;
}
exports.resolve = resolve;
/**
 * Find all Tokens that are used in the given structure
 */
function findTokens(scope, fn) {
    const resolver = new RememberingTokenResolver(new resolvable_1.StringConcat());
    resolve(fn(), { scope, prefix: [], resolver, preparing: true });
    return resolver.tokens;
}
exports.findTokens = findTokens;
/**
 * Remember all Tokens encountered while resolving
 */
class RememberingTokenResolver extends resolvable_1.DefaultTokenResolver {
    constructor() {
        super(...arguments);
        this.tokensSeen = new Set();
    }
    resolveToken(t, context, postProcessor) {
        this.tokensSeen.add(t);
        return super.resolveToken(t, context, postProcessor);
    }
    resolveString(s, context) {
        const ret = super.resolveString(s, context);
        return ret;
    }
    get tokens() {
        return Array.from(this.tokensSeen);
    }
}
exports.RememberingTokenResolver = RememberingTokenResolver;
/**
 * Determine whether an object is a Construct
 *
 * Not in 'construct.ts' because that would lead to a dependency cycle via 'uniqueid.ts',
 * and this is a best-effort protection against a common programming mistake anyway.
 */
function isConstruct(x) {
    return x._children !== undefined && x._metadata !== undefined;
}
function resolveNumberToken(x, context) {
    const token = token_map_1.TokenMap.instance().lookupNumberToken(x);
    if (token === undefined) {
        return x;
    }
    return context.resolve(token);
}
//# sourceMappingURL=data:application/json;base64,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