"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeUniqueId = void 0;
// copied from https://github.com/aws/constructs/blob/e01e47f78ef1e9b600efcd23ff7705aa8d384017/lib/private/uniqueid.ts
// tslint:disable-next-line:no-var-requires
const crypto = require("crypto");
const encoding_1 = require("./encoding");
/**
 * Resources with this ID are hidden from humans
 *
 * They do not appear in the human-readable part of the logical ID,
 * but they are included in the hash calculation.
 */
const HIDDEN_FROM_HUMAN_ID = "Resource";
/**
 * Resources with this ID are complete hidden from the logical ID calculation.
 */
const HIDDEN_ID = "Default";
const PATH_SEP = "/";
const HASH_LEN = 8;
const MAX_HUMAN_LEN = 240; // max ID len is 255
const MAX_ID_LEN = 255;
/**
 * Calculates a unique ID for a set of textual components.
 *
 * This is done by calculating a hash on the full path and using it as a suffix
 * of a length-limited "human" rendition of the path components.
 *
 * @param components The path components
 * @returns a unique alpha-numeric identifier with a maximum length of 255
 */
function makeUniqueId(components) {
    components = components.filter((x) => x !== HIDDEN_ID);
    if (components.length === 0) {
        throw new Error("Unable to calculate a unique id for an empty set of components");
    }
    // Lazy require in order to break a module dependency cycle
    const unresolvedTokens = components.filter((c) => encoding_1.unresolved(c));
    if (unresolvedTokens.length > 0) {
        throw new Error(`ID components may not include unresolved tokens: ${unresolvedTokens.join(",")}`);
    }
    // top-level resources will simply use the `name` as-is in order to support
    // transparent migration of cloudformation templates to the CDK without the
    // need to rename all resources.
    if (components.length === 1) {
        // we filter out non-alpha characters but that is actually a bad idea
        // because it could create conflicts ("A-B" and "AB" will render the same
        // logical ID). sadly, changing it in the 1.x version line is impossible
        // because it will be a breaking change. we should consider for v2.0.
        // https://github.com/aws/aws-cdk/issues/6421
        const candidate = removeNonAlphanumeric(components[0]);
        // if our candidate is short enough, use it as is. otherwise, fall back to
        // the normal mode.
        if (candidate.length <= MAX_ID_LEN) {
            return candidate;
        }
    }
    const hash = pathHash(components);
    const human = removeDupes(components)
        .filter((x) => x !== HIDDEN_FROM_HUMAN_ID)
        .map(removeNonAlphanumeric)
        .join("")
        .slice(0, MAX_HUMAN_LEN);
    return human + hash;
}
exports.makeUniqueId = makeUniqueId;
/**
 * Take a hash of the given path.
 *
 * The hash is limited in size.
 */
function pathHash(path) {
    const md5 = crypto
        .createHash("md5")
        .update(path.join(PATH_SEP))
        .digest("hex");
    return md5.slice(0, HASH_LEN).toUpperCase();
}
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, "");
}
/**
 * Remove duplicate "terms" from the path list
 *
 * If the previous path component name ends with this component name, skip the
 * current component.
 */
function removeDupes(path) {
    const ret = new Array();
    for (const component of path) {
        if (ret.length === 0 || !ret[ret.length - 1].endsWith(component)) {
            ret.push(component);
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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