"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const actions_1 = require("./actions");
const construct_internals_1 = require("./private/construct-internals");
const stage_1 = require("./stage");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * (experimental) A Pipeline to deploy CDK apps.
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 *
 * @experimental
 */
class CdkPipeline extends core_2.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        const pipelineStack = core_1.Stack.of(this);
        if (props.codePipeline) {
            if (props.pipelineName) {
                throw new Error('Cannot set \'pipelineName\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (props.crossAccountKeys !== undefined) {
                throw new Error('Cannot set \'crossAccountKeys\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            this._pipeline = props.codePipeline;
        }
        else {
            this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: props.pipelineName,
                crossAccountKeys: props.crossAccountKeys,
                restartExecutionOnUpdate: true,
            });
        }
        if (props.sourceAction && !props.synthAction) {
            // Because of ordering limitations, you can: bring your own Source, bring your own
            // Both, or bring your own Nothing. You cannot bring your own Build (which because of the
            // current CodePipeline API must go BEFORE what we're adding) and then having us add a
            // Source after it. That doesn't make any sense.
            throw new Error('When passing a \'sourceAction\' you must also pass a \'synthAction\' (or a \'codePipeline\' that already has both)');
        }
        if (!props.sourceAction && (!props.codePipeline || props.codePipeline.stages.length < 1)) {
            throw new Error('You must pass a \'sourceAction\' (or a \'codePipeline\' that already has a Source stage)');
        }
        if (props.sourceAction) {
            this._pipeline.addStage({
                stageName: 'Source',
                actions: [props.sourceAction],
            });
        }
        if (props.synthAction) {
            this._pipeline.addStage({
                stageName: 'Build',
                actions: [props.synthAction],
            });
        }
        if ((_b = props.selfMutating) !== null && _b !== void 0 ? _b : true) {
            this._pipeline.addStage({
                stageName: 'UpdatePipeline',
                actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                        cloudAssemblyInput: this._cloudAssemblyArtifact,
                        pipelineStackName: pipelineStack.stackName,
                        cdkCliVersion: props.cdkCliVersion,
                        projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                    })],
            });
        }
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: props.cdkCliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
        });
    }
    /**
     * (experimental) The underlying CodePipeline object.
     *
     * You can use this to add more Stages to the pipeline, or Actions
     * to Stages.
     *
     * @experimental
     */
    get codePipeline() {
        return this._pipeline;
    }
    /**
     * (experimental) Access one of the pipeline's stages by stage name.
     *
     * You can use this to add more Actions to a stage.
     *
     * @experimental
     */
    stage(stageName) {
        return this._pipeline.stage(stageName);
    }
    /**
     * (experimental) Add pipeline stage that will deploy the given application stage.
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     *
     * @experimental
     */
    addApplicationStage(appStage, options = {}) {
        const stage = this.addStage(appStage.stageName);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * (experimental) Add a new, empty stage to the pipeline.
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     *
     * @experimental
     */
    addStage(stageName) {
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * (experimental) Get the StackOutput object that holds this CfnOutput's value in this pipeline.
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     *
     * @experimental
     */
    stackOutput(cfnOutput) {
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(`Artifact_${stack.artifactId}_Outputs`);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * (experimental) Validate that we don't have any stacks violating dependency order in the pipeline.
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     *
     * @experimental
     */
    validate() {
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    core_1.Annotations.of(this).addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
exports.CdkPipeline = CdkPipeline;
_a = JSII_RTTI_SYMBOL_1;
CdkPipeline[_a] = { fqn: "@aws-cdk/pipelines.CdkPipeline", version: "1.101.0" };
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        // CodePipelines has a hard limit of 50 actions per stage. See https://github.com/aws/aws-cdk/issues/9353
        this.MAX_PUBLISHERS_PER_STAGE = 50;
        this.publishers = {};
        this.assetRoles = {};
        this.stages = [];
        this._fileAssetCtr = 0;
        this._dockerAssetCtr = 0;
        this.myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        this.pipeline = this.props.pipeline;
        // Hacks to get access to the innards of Pipeline
        const stages = this.props.pipeline._stages;
        // Any asset publishing stages will be added directly after the last stage that currently exists.
        this.lastStageBeforePublishing = stages.slice(-1)[0];
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        var _b;
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        // The path cannot be outside the asm root. I don't really understand how this could ever
        // come to pass, but apparently it has (see https://github.com/aws/aws-cdk/issues/9766).
        // Add a sanity check here so we can catch it more quickly next time.
        if (relativePath.startsWith(`..${path.sep}`)) {
            throw new Error(`The asset manifest (${command.assetManifestPath}) cannot be outside the Cloud Assembly directory (${this.myCxAsmRoot}). Please report this error at https://github.com/aws/aws-cdk/issues to help us debug why this is happening.`);
        }
        // Late-binding here (rather than in the constructor) to prevent creating the role in cases where no asset actions are created.
        if (!this.assetRoles[command.assetType]) {
            this.generateAssetRole(command.assetType);
        }
        let action = this.publishers[command.assetId];
        if (!action) {
            // Dynamically create new stages as needed, with `MAX_PUBLISHERS_PER_STAGE` assets per stage.
            const stageIndex = Math.floor((this._fileAssetCtr + this._dockerAssetCtr) / this.MAX_PUBLISHERS_PER_STAGE);
            if (stageIndex >= this.stages.length) {
                const previousStage = (_b = this.stages.slice(-1)[0]) !== null && _b !== void 0 ? _b : this.lastStageBeforePublishing;
                this.stages.push(this.pipeline.addStage({
                    stageName: `Assets${stageIndex > 0 ? stageIndex + 1 : ''}`,
                    placement: { justAfter: previousStage },
                }));
            }
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = command.assetType === actions_1.AssetType.FILE ? `FileAsset${++this._fileAssetCtr}` : `DockerAsset${++this._dockerAssetCtr}`;
            // NOTE: It's important that asset changes don't force a pipeline self-mutation.
            // This can cause an infinite loop of updates (see https://github.com/aws/aws-cdk/issues/9080).
            // For that reason, we use the id as the actionName below, rather than the asset hash.
            action = this.publishers[command.assetId] = new actions_1.PublishAssetsAction(this, id, {
                actionName: id,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
                role: this.assetRoles[command.assetType],
                vpc: this.props.vpc,
                subnetSelection: this.props.subnetSelection,
            });
            this.stages[stageIndex].addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    generateAssetRole(assetType) {
        if (this.assetRoles[assetType]) {
            return this.assetRoles[assetType];
        }
        const rolePrefix = assetType === actions_1.AssetType.DOCKER_IMAGE ? 'Docker' : 'File';
        const assetRole = new iam.Role(this, `${rolePrefix}Role`, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(core_1.Stack.of(this).account)),
        });
        // Logging permissions
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: '/aws/codebuild/*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: [logGroupArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        }));
        // CodeBuild report groups
        const codeBuildArn = core_1.Stack.of(this).formatArn({
            service: 'codebuild',
            resource: 'report-group',
            resourceName: '*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'codebuild:CreateReportGroup',
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
                'codebuild:BatchPutCodeCoverages',
            ],
            resources: [codeBuildArn],
        }));
        // CodeBuild start/stop
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // Publishing role access
        const rolePattern = assetType === actions_1.AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        // Artifact access
        this.pipeline.artifactBucket.grantRead(assetRole);
        // VPC permissions required for CodeBuild
        // Normally CodeBuild itself takes care of this but we're creating a singleton role so now
        // we need to do this.
        if (this.props.vpc) {
            assetRole.attachInlinePolicy(new iam.Policy(assetRole, 'VpcPolicy', {
                statements: [
                    new iam.PolicyStatement({
                        resources: [`arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:network-interface/*`],
                        actions: ['ec2:CreateNetworkInterfacePermission'],
                        conditions: {
                            StringEquals: {
                                'ec2:Subnet': this.props.vpc
                                    .selectSubnets(this.props.subnetSelection).subnetIds
                                    .map(si => `arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:subnet/${si}`),
                                'ec2:AuthorizedService': 'codebuild.amazonaws.com',
                            },
                        },
                    }),
                    new iam.PolicyStatement({
                        resources: ['*'],
                        actions: [
                            'ec2:CreateNetworkInterface',
                            'ec2:DescribeNetworkInterfaces',
                            'ec2:DeleteNetworkInterface',
                            'ec2:DescribeSubnets',
                            'ec2:DescribeSecurityGroups',
                            'ec2:DescribeDhcpOptions',
                            'ec2:DescribeVpcs',
                        ],
                    }),
                ],
            }));
        }
        this.assetRoles[assetType] = assetRole.withoutPolicyUpdates();
        return this.assetRoles[assetType];
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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