"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DestinationPattern = exports.DestinationIdentifier = exports.DockerImageManifestEntry = exports.FileManifestEntry = exports.AssetManifestReader = void 0;
// FIXME: copied from `ckd-assets`, because this tool needs to read the asset manifest aswell.
const fs = require("fs");
const path = require("path");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
/**
 * A manifest of assets
 */
class AssetManifestReader {
    constructor(directory, manifest) {
        this.manifest = manifest;
        this.directory = directory;
    }
    /**
     * Load an asset manifest from the given file
     */
    static fromFile(fileName) {
        try {
            const obj = cloud_assembly_schema_1.Manifest.loadAssetManifest(fileName);
            return new AssetManifestReader(path.dirname(fileName), obj);
        }
        catch (e) {
            throw new Error(`Canot read asset manifest '${fileName}': ${e.message}`);
        }
    }
    /**
     * Load an asset manifest from the given file or directory
     *
     * If the argument given is a directoy, the default asset file name will be used.
     */
    static fromPath(filePath) {
        let st;
        try {
            st = fs.statSync(filePath);
        }
        catch (e) {
            throw new Error(`Cannot read asset manifest at '${filePath}': ${e.message}`);
        }
        if (st.isDirectory()) {
            return AssetManifestReader.fromFile(path.join(filePath, AssetManifestReader.DEFAULT_FILENAME));
        }
        return AssetManifestReader.fromFile(filePath);
    }
    /**
     * Select a subset of assets and destinations from this manifest.
     *
     * Only assets with at least 1 selected destination are retained.
     *
     * If selection is not given, everything is returned.
     */
    select(selection) {
        if (selection === undefined) {
            return this;
        }
        const ret = { version: this.manifest.version, dockerImages: {}, files: {} };
        for (const assetType of ASSET_TYPES) {
            for (const [assetId, asset] of Object.entries(this.manifest[assetType] || {})) {
                const filteredDestinations = filterDict(asset.destinations, (_, destId) => selection.some(sel => sel.matches(new DestinationIdentifier(assetId, destId))));
                if (Object.keys(filteredDestinations).length > 0) {
                    ret[assetType][assetId] = {
                        ...asset,
                        destinations: filteredDestinations,
                    };
                }
            }
        }
        return new AssetManifestReader(this.directory, ret);
    }
    /**
     * Describe the asset manifest as a list of strings
     */
    list() {
        return [
            ...describeAssets('file', this.manifest.files || {}),
            ...describeAssets('docker-image', this.manifest.dockerImages || {}),
        ];
        function describeAssets(type, assets) {
            const ret = new Array();
            for (const [assetId, asset] of Object.entries(assets || {})) {
                ret.push(`${assetId} ${type} ${JSON.stringify(asset.source)}`);
                const destStrings = Object.entries(asset.destinations).map(([destId, dest]) => ` ${assetId}:${destId} ${JSON.stringify(dest)}`);
                ret.push(...prefixTreeChars(destStrings, '  '));
            }
            return ret;
        }
    }
    /**
     * List of assets, splat out to destinations
     */
    get entries() {
        return [
            ...makeEntries(this.manifest.files || {}, FileManifestEntry),
            ...makeEntries(this.manifest.dockerImages || {}, DockerImageManifestEntry),
        ];
        function makeEntries(assets, ctor) {
            const ret = new Array();
            for (const [assetId, asset] of Object.entries(assets)) {
                for (const [destId, destination] of Object.entries(asset.destinations)) {
                    ret.push(new ctor(new DestinationIdentifier(assetId, destId), asset.source, destination));
                }
            }
            return ret;
        }
    }
}
exports.AssetManifestReader = AssetManifestReader;
/**
 * The default name of the asset manifest in a cdk.out directory
 */
AssetManifestReader.DEFAULT_FILENAME = 'assets.json';
const ASSET_TYPES = ['files', 'dockerImages'];
/**
 * A manifest entry for a file asset
 */
class FileManifestEntry {
    constructor(
    /** Identifier for this asset */
    id, 
    /** Source of the file asset */
    source, 
    /** Destination for the file asset */
    destination) {
        this.id = id;
        this.source = source;
        this.destination = destination;
        this.type = 'file';
        this.genericSource = source;
        this.genericDestination = destination;
    }
}
exports.FileManifestEntry = FileManifestEntry;
/**
 * A manifest entry for a docker image asset
 */
class DockerImageManifestEntry {
    constructor(
    /** Identifier for this asset */
    id, 
    /** Source of the file asset */
    source, 
    /** Destination for the file asset */
    destination) {
        this.id = id;
        this.source = source;
        this.destination = destination;
        this.type = 'docker-image';
        this.genericSource = source;
        this.genericDestination = destination;
    }
}
exports.DockerImageManifestEntry = DockerImageManifestEntry;
/**
 * Identify an asset destination in an asset manifest
 */
class DestinationIdentifier {
    constructor(assetId, destinationId) {
        this.assetId = assetId;
        this.destinationId = destinationId;
    }
    /**
     * Return a string representation for this asset identifier
     */
    toString() {
        return this.destinationId ? `${this.assetId}:${this.destinationId}` : this.assetId;
    }
}
exports.DestinationIdentifier = DestinationIdentifier;
function filterDict(xs, pred) {
    const ret = {};
    for (const [key, value] of Object.entries(xs)) {
        if (pred(value, key)) {
            ret[key] = value;
        }
    }
    return ret;
}
/**
 * A filter pattern for an destination identifier
 */
class DestinationPattern {
    constructor(assetId, destinationId) {
        this.assetId = assetId;
        this.destinationId = destinationId;
    }
    /**
     * Parse a ':'-separated string into an asset/destination identifier
     */
    static parse(s) {
        if (!s) {
            throw new Error('Empty string is not a valid destination identifier');
        }
        const parts = s.split(':').map(x => x !== '*' ? x : undefined);
        if (parts.length === 1) {
            return new DestinationPattern(parts[0]);
        }
        if (parts.length === 2) {
            return new DestinationPattern(parts[0] || undefined, parts[1] || undefined);
        }
        throw new Error(`Asset identifier must contain at most 2 ':'-separated parts, got '${s}'`);
    }
    /**
     * Whether or not this pattern matches the given identifier
     */
    matches(id) {
        return (this.assetId === undefined || this.assetId === id.assetId)
            && (this.destinationId === undefined || this.destinationId === id.destinationId);
    }
    /**
     * Return a string representation for this asset identifier
     */
    toString() {
        var _a, _b;
        return `${(_a = this.assetId) !== null && _a !== void 0 ? _a : '*'}:${(_b = this.destinationId) !== null && _b !== void 0 ? _b : '*'}`;
    }
}
exports.DestinationPattern = DestinationPattern;
/**
 * Prefix box-drawing characters to make lines look like a hanging tree
 */
function prefixTreeChars(xs, prefix = '') {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        const isLast = i === xs.length - 1;
        const boxChar = isLast ? '└' : '├';
        ret.push(`${prefix}${boxChar}${xs[i]}`);
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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