# This is a library of utility codes with features to facilitate the development and programming of language model algorithms from Sapiens Technology®.
# All code here is the intellectual property of Sapiens Technology®, and any public mention, distribution, modification, customization, or unauthorized sharing of this or other codes from Sapiens Technology® will result in the author being legally punished by our legal team.
# --------------------------> A SAPIENS TECHNOLOGY®️ PRODUCTION) <--------------------------
class UtilitiesNLP:
    def __init__(self, timeout=120, show_errors=False, display_error_point=False):
        try:
            self.__timeout = float(timeout) if type(timeout) in (bool, int, float) else 120
            self.__show_errors = bool(show_errors) if type(show_errors) in (bool, int, float) else False
            self.__display_error_point = bool(display_error_point) if type(display_error_point) in (bool, int, float) else True
            from traceback import print_exc
            self.__print_exc = print_exc
            self.__pdf_extensions = ['.pdf']
            self.__word_extensions = ['.doc', '.docx', '.docm', '.dot', '.dotx', '.dotm', '.rtf']
            self.__excel_extensions = ['.xls', '.xlsx', '.csv', '.xlsm', '.xlt', '.xltx', '.xltm', '.xlsb']
            self.__powerpoint_extensions = ['.ppt', '.pptx', '.ppsx', '.pptm', '.pot', '.potx', '.potm', '.pps', '.ppsm']
            self.__image_extensions = ['.webp', '.jpg', '.jpeg', '.png', '.gif', '.bmp', '.dng', '.mpo', '.tif', '.tiff', '.pfm', '.svg', '.ico', '.raw', '.psd', '.ai', '.eps', '.icns', '.heic', '.heif']
            self.__audio_extensions = ['.mp3', '.wav', '.m4a', '.aac', '.ogg', '.flac', '.aiff', '.wma', '.ac3', '.amr', '.alac', '.ape', '.mid', '.midi', '.opus']
            self.__video_extensions = ['.mp4', '.avi', '.mkv', '.mov', '.webm', '.flv', '.3gp', '.wmv', '.ogv', '.m4v', '.mpeg', '.mpg', '.3g2', '.vob', '.rm', '.rmvb', '.asf']
            self.__code_extensions = [
                '.py', '.pyw', '.pyc', '.pyo', '.pyd',
                '.java', '.class', '.jar',
                '.cpp', '.hpp', '.cc', '.hh',
                '.c', '.h',
                '.cs',
                '.js', '.jsx', '.ts', '.tsx',
                '.html', '.htm', '.xhtml',
                '.css', '.scss', '.sass', '.less',
                '.php', '.phtml', '.php3', '.php4', '.php5', '.phps',
                '.rb', '.rbw',
                '.go',
                '.swift',
                '.kt', '.kts',
                '.rs', '.rlib',
                '.lua',
                '.pl', '.pm', '.t', '.pod',
                '.scala', '.sc',
                '.clj', '.cljs', '.cljc', '.edn',
                '.coffee',
                '.groovy', '.gvy', '.gy', '.gsh',
                '.r', '.rdata', '.rds', '.rda',
                '.sh', '.bash', '.csh', '.tcsh', '.zsh', '.fish',
                '.sql',
                '.vb', '.vbs',
                '.f', '.for', '.f90', '.f95', '.f03',
                '.m',
                '.asm', '.s',
                '.dart',
                '.pas', '.pp',
                '.hs', '.lhs',
                '.erl', '.hrl',
                '.ex', '.exs',
                '.lisp', '.lsp', '.l', '.cl', '.fasl',
                '.ml', '.mli',
                '.fs', '.fsi', '.fsx', '.fsscript'
            ]
            self.__autocad_extensions = ['.dwg', '.dxf', '.dwf', '.dwt']
            self.__text_extensions = ['.txt']
            self.__proxy = 'https://api.codetabs.com/v1/proxy?quest='
            self.__image_api = 'https://api.imgur.com/'
            self.__web_search = 'google.com'
            self.__image_input_model_path = ''
            self.__total_number_of_tokens = {}
            self.__sapiens_bin = 'aHR0cHM6Ly9odWdnaW5nZmFjZS5jby9vcGVuMTk4Ni8wMDAvcmVzb2x2ZS9tYWluL3NhcGllbnMuYmluP2Rvd25sb2FkPXRydWU='
            self.__sapiens_names = 'aHR0cHM6Ly9odWdnaW5nZmFjZS5jby9vcGVuMTk4Ni8wMDAvcmVzb2x2ZS9tYWluL3NhcGllbnMubmFtZXM/ZG93bmxvYWQ9dHJ1ZQ=='
            self.__sapiens_weights = 'aHR0cHM6Ly9odWdnaW5nZmFjZS5jby9vcGVuMTk4Ni8wMDAvcmVzb2x2ZS9tYWluL3NhcGllbnMud2VpZ2h0cz9kb3dubG9hZD10cnVl'
            self.__sapiens_cfg = 'aHR0cHM6Ly9odWdnaW5nZmFjZS5jby9vcGVuMTk4Ni8wMDAvcmVzb2x2ZS9tYWluL3NhcGllbnMuY2ZnP2Rvd25sb2FkPXRydWU='
            from os import environ
            environ['TOKENIZERS_PARALLELISM'] = 'true'
            from warnings import filterwarnings
            filterwarnings('ignore', message='.*tokenizers.*')
        except Exception as error:
            try:
                if self.__show_errors:
                    error_message = 'ERROR in UtilitiesNLP.__init__: '+str(error)
                    print(error_message)
                    try: self.__print_exc() if self.__display_error_point else None
                    except: pass
            except: pass
    def __getRootDirectory(self):
        try:
            from os import path
            return str(path.dirname(path.realpath(__file__)).replace('\\', '/')+'/').strip()
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.__getRootDirectory: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return './'
    def __shuffleTuple(self, original_tuple=()):
        try:
            tuple_list = list(original_tuple)
            from random import shuffle
            shuffle(tuple_list)
            return tuple(tuple_list)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.__shuffleTuple: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return original_tuple
    def __localSearch(self, prompt='', max_results=10):
        try:
            list_of_links = []
            prompt = str(prompt).strip()
            max_results = max((1, int(max_results))) if type(max_results) in (bool, int, float) else 10
            try: from requests import get
            except:
                self.installModule(module_name='requests', version='2.31.0')
                from requests import get
            try: from bs4 import BeautifulSoup
            except:
                self.installModule(module_name='beautifulsoup4', version='4.12.3')
                from bs4 import BeautifulSoup
            from urllib.parse import quote, unquote
            headers = {'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'}
            query_encoded = quote(prompt)
            def local_search_1(prompt='', max_results=10):
                try:
                    search_url = f'https://duckduckgo.com/html/?q={query_encoded}'
                    response = get(search_url, headers=headers)
                    response.raise_for_status()
                    soup = BeautifulSoup(response.text, 'html.parser')
                    links = []
                    for result in soup.find_all('a', class_='result__a')[:max_results]:
                        href = result.get('href')
                        if href:
                            if href.startswith('/l/?uddg='): href = unquote(href.split('uddg=')[1])
                            if href not in links and href.startswith('http'): links.append(href)
                    return links
                except: return []
            def local_search_2(prompt='', max_results=10):
                try:
                    search_url = f'https://www.startpage.com/do/search?q={query_encoded}'
                    response = get(search_url, headers=headers)
                    response.raise_for_status()
                    soup = BeautifulSoup(response.text, 'html.parser')
                    links = []
                    for result in soup.find_all('a', class_='result-link')[:max_results]:
                        href = result.get('href')
                        if href and href.startswith('http') and href not in links: links.append(href)
                    return links
                except: return []
            list_of_links = local_search_1(prompt=prompt, max_results=max_results)
            if len(list_of_links) < 1: list_of_links = local_search_2(prompt=prompt, max_results=max_results)
            return list_of_links
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.__localSearch: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return []
    def __getAlternativeWEBSearchList(self, prompt='', max_results=10, language=None):
        try:
            list_of_links = []
            prompt = str(prompt).strip()
            max_results = int(max_results) if type(max_results) in (bool, int, float) else 10
            language = 'en-US' if type(language) == type(None) or len(str(language).strip()) < 1 else str(language).strip()
            if '-' in language:
                abbreviation = language.split('-')[0]
                accept_language = language+','+abbreviation
            else: accept_language = language
            searx_instances = ['https://searx.be/', 'https://searx.xyz/', 'https://searx.info/', 'https://searx.tuxcloud.net/', 'https://search.mdosch.de/']
            from random import choice
            base_url = choice(searx_instances)
            params = {'q': prompt, 'categories': 'general', 'language': language}
            headers = {
                'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                'Accept-Language': accept_language+';q=0.9',
                'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
                'Referer': 'https://www.google.com/'
            }
            try: from requests import get
            except:
                self.installModule(module_name='requests', version='2.31.0')
                from requests import get
            response = get(base_url, params=params, headers=headers, timeout=self.__timeout)
            if response.status_code not in (200, 201): return list_of_links
            from html.parser import HTMLParser
            class CustomHTMLParser(HTMLParser):
                def __init__(self):
                    super().__init__()
                    self.links = []
                def handle_starttag(self, tag, attributes):
                    if tag == 'a':
                        href = dict(attributes).get('href')
                        if href: self.links.append(href)
            parser = CustomHTMLParser()
            parser.feed(response.text)
            exclusion_domains = ['wikipedia.org', 'github.com', 'blog.mdosch.de', 'web.archive.org', 'api.codetabs.com', 'searx.be', 'searx.xyz',
                                'searx.info', 'searx.tuxcloud.net', 'search.mdosch.de', 'searx.neocities.org', 'docs.searxng.org', 'observatory.mozilla.org', 'www.w3.org']
            from urllib.parse import urljoin, urlparse
            for href in parser.links:
                href = urljoin(base_url, href)
                if href.startswith('https://'):
                    parsed_url = urlparse(href)
                    domain = parsed_url.netloc
                    if parsed_url.path and parsed_url.path != '/' and not any(exclude_domain in domain for exclude_domain in exclusion_domains): list_of_links.append(href)
            list_of_links = list_of_links[:max_results]
            if len(list_of_links) < 1: list_of_links = self.__localSearch(prompt=prompt, max_results=max_results)
            return list_of_links
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.__getAlternativeWEBSearchList: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return self.__localSearch(prompt=prompt, max_results=max_results)
    def __getSystemCustomization(self, system_instruction='', prompt='', language='', max_tokens=None, url_path='', javascript_chart=False, task_names=[]):
        try:
            final_result = {'system_instruction': '', 'file_directory': '', 'created_local_file': False, 'extensions': [], 'prompt': '', 'result': {}}
            system_instruction = str(system_instruction).strip()
            prompt = original_prompt = str(prompt).strip()
            language = str(language).strip()
            max_tokens = int(max_tokens) if max_tokens else 1000000
            url_path = str(url_path).strip()
            task_names = list(task_names) if type(task_names) in (tuple, list) else [str(task_names).upper().strip()]
            system_identity, file_directory, created_local_file, extensions = '', '', False, []
            result = {'answer': '', 'answer_for_files': '', 'files': [], 'sources': [], 'javascript_charts': [], 'str_cost': '0.0000000000'}
            url_path_lenght = len(url_path)
            def getFileDirectory():
                temporary_directory = self.getTemporaryFilesDirectory()
                file_directory = temporary_directory+self.getHashCode()+'/'
                from os import path, makedirs
                if not path.exists(file_directory): makedirs(file_directory, exist_ok=True)
                return file_directory if path.exists(file_directory) else ''
            system_identity = '\nIts name is Sapiens Chat, an AI model created by Sapiens Technology. You are an Artificial Intelligence created to '
            if self.hasCoincidentElements(vector1=task_names, vector2=['WEB_SEARCH']):
                system_instruction += system_identity+f'perform internet searches. The search has already been performed and is in the prompt text.'
                def getLargestNumberInString(string=''):
                    from re import findall
                    numbers = findall(r'\d+', str(string))
                    numbers = [int(number) for number in numbers]
                    return max(numbers) if numbers else 5
                max_results = getLargestNumberInString(string=prompt)
                web_search = self.getKeyWordsString(string=self.formatPrompt(prompt=prompt, task_names=['WEB_SEARCH'], language=language))
                list_of_links = self.getWEBSearchList(prompt=web_search, max_results=max_results, language=language, use_proxy=True)
                if len(list_of_links) < 1: list_of_links = self.__localSearch(prompt=web_search, max_results=max_results)
                number_of_tokens_in_the_prompt = self.countTokens(string=prompt, pattern='gpt-4')+8
                data_dictionary = self.getContentFromWEBLinks(list_of_links=list_of_links, max_tokens=max_tokens-number_of_tokens_in_the_prompt)
                prompt, result['sources'] = data_dictionary['text']+'\n\nREQUEST: '+prompt, data_dictionary['sources']
                final_result['system_instruction'] = system_instruction
                final_result['file_directory'] = file_directory
                final_result['created_local_file'] = created_local_file
                final_result['extensions'] = extensions
                final_result['prompt'] = prompt
                final_result['result'] = result
            if self.hasCoincidentElements(vector1=task_names, vector2=('ARTIFACT_CREATION', 'ARTIFACT_EDITING')):
                system_instruction += system_identity+'generate an HTML page by coding everything into a single file. All code must be generated inside within a single page. Use only HTML, CSS, and JavaScript. The result should be responsive and have a clean, modern design.'
                final_result['system_instruction'] = system_instruction
            if self.hasCoincidentElements(vector1=task_names, vector2=('PDF_CREATION', 'PDF_EDITING')):
                try:
                    import fpdf
                    import reportlab
                except:
                    self.installModule(module_name='fpdf', version='1.7.2')
                    self.installModule(module_name='reportlab', version='4.2.2')
                system_instruction += system_identity+'generate "PDF" type files using only native Python language modules and/or (if necessary) "fpdf==1.7.2 and/or reportlab==4.2.2".'
                if len(file_directory.strip()) < 1: file_directory = getFileDirectory()
                created_local_file = True
                extensions.append('.pdf')
                final_result['system_instruction'] = system_instruction
                final_result['file_directory'] = file_directory
                final_result['created_local_file'] = created_local_file
                final_result['extensions'] = extensions
            if self.hasCoincidentElements(vector1=task_names, vector2=('EXCEL_CREATION', 'EXCEL_EDITING')):
                try:
                    import openpyxl
                    import pandas
                    import xlsxwriter
                except:
                    self.installModule(module_name='openpyxl', version='3.1.3')
                    self.installModule(module_name='pandas', version='2.2.2')
                    self.installModule(module_name='XlsxWriter', version='3.2.0')
                system_instruction += system_identity+'generate "Microsoft Excel" type files using only native Python language modules and/or (if necessary) "pandas==2.2.2, openpyxl==3.1.3 and/or XlsxWriter==3.2.0".'
                if len(file_directory.strip()) < 1: file_directory = getFileDirectory()
                created_local_file = True
                extensions.append('.xlsx')
                final_result['system_instruction'] = system_instruction
                final_result['file_directory'] = file_directory
                final_result['created_local_file'] = created_local_file
                final_result['extensions'] = extensions
            if self.hasCoincidentElements(vector1=task_names, vector2=('WORD_CREATION', 'WORD_EDITING')):
                try: import docx
                except:
                    self.installModule(module_name='python-docx', version='1.1.0')
                    self.installModule(module_name='docx', version='0.2.4')
                system_instruction += system_identity+'generate "Microsoft Word" type files using only native Python language modules and/or (if necessary) "python-docx==1.1.0 and/or docx==0.2.4".'
                if len(file_directory.strip()) < 1: file_directory = getFileDirectory()
                created_local_file = True
                extensions.append('.docx')
                final_result['system_instruction'] = system_instruction
                final_result['file_directory'] = file_directory
                final_result['created_local_file'] = created_local_file
                final_result['extensions'] = extensions
            if self.hasCoincidentElements(vector1=task_names, vector2=('POWERPOINT_CREATION', 'POWERPOINT_EDITING')):
                try: import pptx
                except: self.installModule(module_name='python-pptx', version='0.6.23')
                system_instruction += system_identity+'generate "PowerPoint" type files using only native Python language modules and/or (if necessary) "python-pptx==0.6.23".'
                if len(file_directory.strip()) < 1: file_directory = getFileDirectory()
                created_local_file = True
                extensions.append('.pptx')
                final_result['system_instruction'] = system_instruction
                final_result['file_directory'] = file_directory
                final_result['created_local_file'] = created_local_file
                final_result['extensions'] = extensions
            if self.hasCoincidentElements(vector1=task_names, vector2=('CSV_CREATION', 'CSV_EDITING')):
                try:
                    import pandas
                    import numpy
                except:
                    self.installModule(module_name='pandas', version='2.2.2')
                    self.installModule(module_name='numpy', version='1.25.2')
                system_instruction += system_identity+'generate "CSV" type files using only native Python language modules and/or (if necessary) "pandas==2.2.2 and/or numpy==1.25.2".'
                if len(file_directory.strip()) < 1: file_directory = getFileDirectory()
                created_local_file = True
                extensions.append('.csv')
                final_result['system_instruction'] = system_instruction
                final_result['file_directory'] = file_directory
                final_result['created_local_file'] = created_local_file
                final_result['extensions'] = extensions
            if self.hasCoincidentElements(vector1=task_names, vector2=('CHART_CREATION', 'CHART_EDITING')):
                if javascript_chart: system_instruction += system_identity+'generate chart inside an HTML DIV using the "plotly.js" code library. All code must be generated inside the DIV without using the "html", "head" and "body" tags. It is essential that the "displaylogo" property is set to "false". Use the following import inside the DIV: "<script src="https://cdn.plot.ly/plotly-2.35.2.min.js" charset="utf-8"></script>". Insert the CDN import at the top of the page and right after the import insert the DIV element that will load the chart.'
                else:
                    try:
                        import matplotlib
                        import seaborn
                    except:
                        self.installModule(module_name='matplotlib', version='3.9.1')
                        self.installModule(module_name='seaborn', version='0.13.2')
                    system_instruction += system_identity+'generate "PNG" type charts using only native Python language modules and/or (if necessary) "matplotlib==3.9.1 and/or seaborn==0.13.2". The charts should never be displayed, only saved as PNG images.'
                    if len(file_directory.strip()) < 1: file_directory = getFileDirectory()
                    created_local_file = True
                    extensions.append('.png')
                    final_result['file_directory'] = file_directory
                    final_result['created_local_file'] = created_local_file
                    final_result['extensions'] = extensions
                final_result['system_instruction'] = system_instruction
            if self.hasCoincidentElements(vector1=task_names, vector2=('FLOWCHART_CREATION', 'FLOWCHART_EDITING')):
                def checkAndInstallDOT():
                    from subprocess import run, PIPE
                    from platform import system
                    from os import path
                    def isDotInstalled():
                        try:
                            run(['dot', '-V'], stdout=PIPE, stderr=PIPE, check=True)
                            return True
                        except: return False
                    if not isDotInstalled():
                        operational_system, command = str(system()).lower().strip(), 'apt-get install graphviz'
                        if operational_system == 'linux':
                            if path.exists('/etc/redhat-release'): command = 'sudo yum install graphviz'
                            else: command = 'sudo apt-get install graphviz'
                        elif operational_system == 'darwin': command = 'brew install graphviz'
                        elif operational_system == 'windows': command = 'choco install graphviz'
                        else: command = 'sudo apt-get install graphviz'
                        try: run(command, shell=True, check=True)
                        except: print('Run the following command to continue without errors:\n'+command)
                try:
                    checkAndInstallDOT()
                    import matplotlib
                    import graphviz
                    import networkx
                except:
                    self.installModule(module_name='matplotlib', version='3.9.1')
                    self.installModule(module_name='graphviz', version='0.20.3')
                    self.installModule(module_name='networkx', version='3.3')
                system_instruction += system_identity+'generate "PNG" type charts word cloud using only native Python language modules and/or (if necessary) "graphviz==0.20.3 and/or networkx==3.3". The charts should never be displayed, only saved as PNG images.'
                if len(file_directory.strip()) < 1: file_directory = getFileDirectory()
                created_local_file = True
                extensions.append('.png')
                final_result['system_instruction'] = system_instruction
                final_result['file_directory'] = file_directory
                final_result['created_local_file'] = created_local_file
                final_result['extensions'] = extensions
            if self.hasCoincidentElements(vector1=task_names, vector2=('WORDCLOUD_CREATION', 'WORDCLOUD_EDITING')):
                try:
                    import matplotlib
                    import wordcloud
                except:
                    self.installModule(module_name='matplotlib', version='3.9.1')
                    self.installModule(module_name='wordcloud', version='1.9.3')
                system_instruction += system_identity+'generate "PNG" type charts using only native Python language modules and/or (if necessary) "wordcloud==1.9.3 and/or matplotlib==3.9.1". The charts should never be displayed, only saved as PNG images.'
                if len(file_directory.strip()) < 1: file_directory = getFileDirectory()
                created_local_file = True
                extensions.append('.png')
                final_result['system_instruction'] = system_instruction
                final_result['file_directory'] = file_directory
                final_result['created_local_file'] = created_local_file
                final_result['extensions'] = extensions
            if url_path_lenght > 0 and self.hasCoincidentElements(vector1=task_names, vector2=['IMAGE_EDITING']):
                try:
                    import cv2
                    import PIL
                except:
                    self.installModule(module_name='opencv-python', version='4.6.0.66')
                    self.installModule(module_name='opencv-python-headless', version='4.6.0.66')
                    self.installModule(module_name='pillow', version='10.3.0')
                dependencies = 'opencv-python==4.6.0.66, opencv-python-headless==4.6.0.66 and/or pillow==10.3.0' if 'text' not in prompt else 'opencv-python==4.6.0.66 and/or opencv-python-headless==4.6.0.66'
                system_instruction += system_identity+f'edit the image using only native Python language modules and/or (if necessary) "{dependencies}". The images should never be displayed, only saved.'
                if len(file_directory.strip()) < 1: file_directory = getFileDirectory()
                created_local_file = True
                extensions += ['webp', 'jpg', 'jpeg', 'png', 'gif', 'bmp', 'dng', 'mpo', 'tif', 'tiff', 'pfm']
                prompt = url_path+'\n'+prompt
                final_result['system_instruction'] = system_instruction
                final_result['file_directory'] = file_directory
                final_result['created_local_file'] = created_local_file
                final_result['extensions'] = extensions
                final_result['prompt'] = prompt
            if url_path_lenght > 0 and self.hasCoincidentElements(vector1=task_names, vector2=['AUDIO_EDITING', 'MUSIC_EDITING']):
                try:
                    import ffmpeg
                    import pydub
                    import noisereduce
                    import numpy
                except:
                    self.installModule(module_name='ffmpeg-python', version='0.2.0')
                    self.installModule(module_name='pydub', version='0.25.1')
                    self.installModule(module_name='noisereduce', version='3.0.2')
                    self.installModule(module_name='numpy', version='1.25.2')
                system_instruction += system_identity+'edit the audio using only native Python language modules and/or (if necessary) "ffmpeg-python==0.2.0, pydub==0.25.1, noisereduce==3.0.2 and/or numpy==1.25.2".'
                if len(file_directory.strip()) < 1: file_directory = getFileDirectory()
                created_local_file = True
                extensions += ['mp3', 'wav', 'mpeg', 'm4a', 'aac', 'ogg', 'flac', 'aiff', 'wma', 'ac3', 'amr']
                prompt = url_path+'\n'+prompt
                final_result['system_instruction'] = system_instruction
                final_result['file_directory'] = file_directory
                final_result['created_local_file'] = created_local_file
                final_result['extensions'] = extensions
                final_result['prompt'] = prompt
            if url_path_lenght > 0 and self.hasCoincidentElements(vector1=task_names, vector2=['VIDEO_EDITING']):
                try:
                    import moviepy
                    import cv2
                except:
                    self.installModule(module_name='moviepy', version='1.0.3')
                    self.installModule(module_name='opencv-python', version='4.6.0.66')
                    self.installModule(module_name='opencv-python-headless', version='4.6.0.66')
                system_instruction += system_identity+'edit the video using only native Python language modules and/or (if necessary) "moviepy==1.0.3, opencv-python==4.6.0.66 and/or opencv-python-headless==4.6.0.66".'
                if len(file_directory.strip()) < 1: file_directory = getFileDirectory()
                created_local_file = True
                extensions += ['mp4', 'avi', 'mkv', 'mov', 'webm', 'flv', '3gp', 'wmv', 'ogv', 'webp', 'jpg', 'jpeg', 'png', 'gif', 'bmp', 'dng', 'mpo', 'tif', 'tiff', 'pfm', 'mp3', 'wav', 'mpeg', 'm4a', 'aac', 'ogg', 'flac', 'aiff', 'wma', 'ac3', 'amr']
                prompt = url_path+'\n'+prompt
                final_result['system_instruction'] = system_instruction
                final_result['file_directory'] = file_directory
                final_result['created_local_file'] = created_local_file
                final_result['extensions'] = extensions
                final_result['prompt'] = prompt
            if url_path_lenght > 0 and self.hasCoincidentElements(vector1=task_names, vector2=['YOUTUBE_VIDEO_DOWNLOAD', 'YOUTUBE_AUDIO_DOWNLOAD']):
                if 'YOUTUBE_AUDIO_DOWNLOAD' in task_names: audio_only, extension = True, 'mp3'
                else: audio_only, extension = False, 'mp4'
                result['files'] = [self.downloadYouTubeVideo(url_path=url_path, audio_only=audio_only, extension=extension, output_path=None)]
                if len(result['files'][0]['base64_string']) > 1: result['answer'] = result['answer_for_files'] = self.getAnswerForFiles(prompt=original_prompt, language=language if len(language) > 0 else None)
                else: result['answer'], result['answer_for_files'], result['files'] = '', '', []
                final_result['result'] = result
            if len(file_directory.strip()) > 0 and len(task_names) > 0 and len(system_instruction.strip()) > 0:
                insert = f'\n\nThe path to the original file is {url_path}' if url_path else ''
                system_instruction += f'{insert}\nAll generated files must be saved in the following path: {file_directory}'
                system_instruction = system_instruction.strip()
                final_result['system_instruction'] = system_instruction
            return final_result
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.__getSystemCustomization: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'system_instruction': '', 'file_directory': '', 'created_local_file': False, 'extensions': [], 'prompt': '', 'result': {}}
    def __updateDisabledTasks(self, prompt='', disabled_tasks=[]):
        try:
            prompt = str(prompt).lower().strip()
            if not prompt: return disabled_tasks
            disabled_tasks = list(disabled_tasks) if type(disabled_tasks) in (tuple, list) else []
            has_command = prompt.startswith('/')
            if has_command:
                def remove_disabled_tasks(task_name='', disabled_tasks=[]):
                    if task_name and task_name in disabled_tasks: disabled_tasks.remove(task_name)
                    return disabled_tasks
                if prompt.startswith('/code'): disabled_tasks = remove_disabled_tasks(task_name='CODE_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/artifact'): disabled_tasks = remove_disabled_tasks(task_name='ARTIFACT_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/web'): disabled_tasks = remove_disabled_tasks(task_name='WEB_SEARCH', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/image'): disabled_tasks = remove_disabled_tasks(task_name='IMAGE_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/logo'): disabled_tasks = remove_disabled_tasks(task_name='LOGO_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/audio'): disabled_tasks = remove_disabled_tasks(task_name='AUDIO_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/music'): disabled_tasks = remove_disabled_tasks(task_name='MUSIC_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/video'): disabled_tasks = remove_disabled_tasks(task_name='VIDEO_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/pdf'): disabled_tasks = remove_disabled_tasks(task_name='PDF_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/wordcloud'): disabled_tasks = remove_disabled_tasks(task_name='WORDCLOUD_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/word'): disabled_tasks = remove_disabled_tasks(task_name='WORD_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/excel'): disabled_tasks = remove_disabled_tasks(task_name='EXCEL_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/csv'): disabled_tasks = remove_disabled_tasks(task_name='CSV_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/powerpoint'): disabled_tasks = remove_disabled_tasks(task_name='POWERPOINT_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/chart'): disabled_tasks = remove_disabled_tasks(task_name='CHART_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/flowchart'): disabled_tasks = remove_disabled_tasks(task_name='FLOWCHART_CREATION', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/youtube-video-download'): disabled_tasks = remove_disabled_tasks(task_name='YOUTUBE_VIDEO_DOWNLOAD', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/youtube-audio-download'): disabled_tasks = remove_disabled_tasks(task_name='YOUTUBE_AUDIO_DOWNLOAD', disabled_tasks=disabled_tasks)
                elif prompt.startswith('/deep-reasoning'): disabled_tasks = remove_disabled_tasks(task_name='DEEP_REASONING', disabled_tasks=disabled_tasks)
            return disabled_tasks
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.__updateDisabledTasks: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return disabled_tasks            
    def __removeTasks(self, prompt='', task_names=[], disabled_tasks=[]):
        try:            
            prompt = str(prompt).lower().strip()
            task_names = list(task_names) if type(task_names) in (tuple, list) else []
            disabled_tasks = list(disabled_tasks) if type(disabled_tasks) in (tuple, list) else []
            disabled_tasks = self.__updateDisabledTasks(prompt=prompt, disabled_tasks=disabled_tasks)
            if task_names and disabled_tasks:
                for disabled_task in disabled_tasks:
                    if disabled_task and disabled_task in task_names: task_names.remove(disabled_task)
            return task_names
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.__removeTasks: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return task_names
    def __inDisabledTasks(self, prompt='', task_name='', disabled_tasks=[]):
        try:
            prompt = str(prompt).lower().strip()
            task_name = str(task_name).upper().strip()
            disabled_tasks = list(disabled_tasks) if type(disabled_tasks) in (tuple, list) else []
            if not task_name or not disabled_tasks: return False
            disabled_tasks = self.__updateDisabledTasks(prompt=prompt, disabled_tasks=disabled_tasks)
            task_name = str(task_name).upper().strip()
            return task_name in disabled_tasks
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.__inDisabledTasks: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def __decodeLink(self, encoded_text=''):
        try:
            encoded_text = str(encoded_text)
            from base64 import b64decode as decode_link
            return decode_link(encoded_text.encode('utf-8')).decode('utf-8')
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.__decodeLink: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return encoded_text
    def __transcribeYouTubeVideo(self, url_path='', max_tokens=1000, language=None):
        try:
            transcription = ''
            url_path = str(url_path).strip()
            max_tokens = int(max_tokens) if type(max_tokens) in (bool, int, float) else 1000
            if language: language = str(language).strip()
            base64_dictionary = self.downloadYouTubeVideo(url_path=url_path, audio_only=True)
            transcription = self.audioInterpreter(file_dictionary=base64_dictionary, max_tokens=max_tokens, language=language)['answer']
            return transcription.strip()
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.__transcribeYouTubeVideo: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return ''
    def installModule(self, module_name='', version=None, progress=False):
        try:
            from subprocess import check_call, CalledProcessError, DEVNULL
            from sys import executable
            module_name = str(module_name).strip()
            version = str(version).strip() if type(version) in (float, str) else ''
            progress = bool(progress) if type(progress) in (bool, int, float) else False
            if len(version) > 0: module_name = f'{module_name}=={version}'
            stdout_target = None if progress else DEVNULL
            stderr_target = None if progress else DEVNULL
            check_call([executable, '-m', 'pip', 'install', module_name], stdout=stdout_target, stderr=stderr_target)
            if progress: print(f'Module {module_name} installed successfully!')
            return True
        except CalledProcessError as error:
            if progress:
                print(f'ERROR installing the module "{module_name}": {error}')
                print('Run the command:\npip install '+module_name)
            return False
        except Exception as error:
            if self.__show_errors and progress:
                error_message = 'ERROR in UtilitiesNLP.installModule: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def uninstallModule(self, module_name='', progress=False):
        try:
            from subprocess import check_call, CalledProcessError, DEVNULL
            from sys import executable
            module_name = str(module_name).strip()
            progress = bool(progress) if type(progress) in (bool, int, float) else False
            stdout_target = None if progress else DEVNULL
            stderr_target = None if progress else DEVNULL
            check_call([executable, '-m', 'pip', 'uninstall', '-y', module_name], stdout=stdout_target, stderr=stderr_target)
            if progress: print(f'Module {module_name} uninstalled successfully!')
            return True
        except CalledProcessError as error:
            if progress:
                print(f'ERROR uninstalling the module "{module_name}": {error}')
                print('Run the command:\npip uninstall '+module_name)
            return False
        except Exception as error:
            if self.__show_errors and progress:
                error_message = 'ERROR in UtilitiesNLP.uninstallModule: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def updateModule(self, module_name='', progress=False):
        try:
            module_name = str(module_name).strip()
            progress = bool(progress) if type(progress) in (bool, int, float) else False
            uninstall = self.uninstallModule(module_name=module_name, progress=False)
            install = self.installModule(module_name=module_name, progress=False)
            if progress: print(f'Module {module_name} updated successfully!')
            return uninstall and install
        except Exception as error:
            if self.__show_errors and progress:
                error_message = 'ERROR in UtilitiesNLP.updateModule: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def getTemporaryFilesDirectory(self):
        try:
            temporary_file_directory = './'
            from tempfile import gettempdir
            temporary_file_directory = gettempdir()
            if not temporary_file_directory.endswith('/'): temporary_file_directory += '/'
            return temporary_file_directory
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getTemporaryFilesDirectory: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return self.__getRootDirectory()
    def copyFile(self, origin_file_path='', destination_file_path=''):
        try:
            origin_file_path, destination_file_path = str(origin_file_path).strip(), str(destination_file_path).strip()
            if len(origin_file_path) > 0 and len(destination_file_path) > 0:
                from shutil import copy
                copy(origin_file_path, destination_file_path)
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.copyFile: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def moveFile(self, origin_file_path='', destination_file_path=''):
        try:
            origin_file_path, destination_file_path = str(origin_file_path).strip(), str(destination_file_path).strip()
            if len(origin_file_path) > 0 and len(destination_file_path) > 0:
                from shutil import move
                move(origin_file_path, destination_file_path)
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.moveFile: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def deleteFile(self, file_path=''):
        try:
            file_path = str(file_path).strip()
            from os import path, remove
            if len(file_path) > 0 and path.exists(file_path): remove(file_path)
            return not path.exists(file_path)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.deleteFile: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def copyDirectory(self, origin_directory_path='', destination_directory_path=''):
        try:
            origin_directory_path, destination_directory_path = str(origin_directory_path).strip(), str(destination_directory_path).strip()
            if len(origin_directory_path) > 0 and len(destination_directory_path) > 0:
                from shutil import copytree
                copytree(origin_directory_path, destination_directory_path)
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.copyDirectory: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def moveDirectory(self, origin_directory_path='', destination_directory_path=''):
        try:
            origin_directory_path, destination_directory_path = str(origin_directory_path).strip(), str(destination_directory_path).strip()
            if len(origin_directory_path) > 0 and len(destination_directory_path) > 0:
                from shutil import move
                move(origin_directory_path, destination_directory_path)
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.moveDirectory: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def deleteDirectory(self, directory_path=''):
        try:
            directory_path = str(directory_path).strip()
            if len(directory_path) > 0:
                from os import path
                from shutil import rmtree
                if path.isdir(directory_path) or path.exists(directory_path): rmtree(directory_path)
            from os import path
            return not path.exists(directory_path)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.deleteDirectory: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def compactModel(self, model_path='', file_path='', progress=True):
        try:
            model_path = str(model_path).strip()
            file_path = str(file_path).strip()
            progress = bool(progress) if type(progress) in (bool, int, float) else True
            from os import walk
            from os.path import join, getsize
            from pathlib import Path
            from tarfile import open as tar_open
            def get_total_size(path):
                total = 0
                for directory_path, directory_names, file_names in walk(path):
                    for file_name in file_names:
                        file_path = join(directory_path, file_name)
                        total += getsize(file_path)
                return total
            source = Path(model_path)
            if not source.exists(): return False
            destination = Path(file_path)
            destination.parent.mkdir(parents=True, exist_ok=True)
            total_bytes = get_total_size(model_path)
            processed_bytes = 0
            if progress:
                from tqdm import tqdm
                progress_bar = tqdm(total=total_bytes, unit='B', unit_scale=True, desc='Coding')
                def progress_filter(tar_information): return tar_information
                class ProgressFileWrapper:
                    def __init__(self, file_object, progress_bar):
                        self.file_object = file_object
                        self.progress_bar = progress_bar
                    def read(self, size=-1):
                        data = self.file_object.read(size)
                        self.progress_bar.update(len(data))
                        return data
                    def __getattr__(self, attr): return getattr(self.file_object, attr)
            with tar_open(str(destination), 'w:gz') as tar:
                for directory_path, directory_names, file_names in walk(model_path):
                    for file_name in file_names:
                        file_path = join(directory_path, file_name)
                        relative_name = Path(file_path).relative_to(source)
                        if progress:
                            with open(file_path, 'rb') as original_file:
                                wrapped_file = ProgressFileWrapper(original_file, progress_bar)
                                tar_information = tar.gettarinfo(file_path, arcname=str(relative_name))
                                tar.addfile(tar_information, wrapped_file)
                        else: tar.add(file_path, arcname=str(relative_name))
            if progress:
                progress_bar.n = total_bytes
                progress_bar.refresh()
                progress_bar.close()
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.compactModel: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def decompactModel(self, file_path='', model_path='', progress=True):
        try:
            file_path = str(file_path).strip()
            model_path = str(model_path).strip()
            progress = bool(progress) if type(progress) in (bool, int, float) else True
            from pathlib import Path
            from tarfile import open as tar_open
            source = Path(file_path)
            if not source.exists(): return False
            destination = Path(model_path)
            destination.mkdir(parents=True, exist_ok=True)
            total_bytes = source.stat().st_size
            if progress:
                from tqdm import tqdm
                progress_bar = tqdm(total=total_bytes, unit='B', unit_scale=True, desc='Decoding')
                class ProgressFileWrapper:
                    def __init__(self, file_object, progress_bar):
                        self.file_object = file_object
                        self.progress_bar = progress_bar
                    def read(self, size=-1):
                        data = self.file_object.read(size)
                        self.progress_bar.update(len(data))
                        return data
                    def __getattr__(self, attr): return getattr(self.file_object, attr)
                with open(str(source), 'rb') as compressed_file:
                    wrapped_file = ProgressFileWrapper(compressed_file, progress_bar)
                    with tar_open(fileobj=wrapped_file, mode='r:gz') as tar: tar.extractall(path=str(destination))
                progress_bar.n = total_bytes
                progress_bar.refresh()
                progress_bar.close()
            else:
                with tar_open(str(source), 'r:gz') as tar: tar.extractall(path=str(destination))
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.decompactModel: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def createZIPFromFolder(self, directory_path='', file_path=''):
        try:
            directory_path = str(directory_path).strip()
            file_path = str(file_path).strip()
            from zipfile import ZipFile
            from os import walk, path
            if not directory_path: return False
            if not file_path: file_path = path.basename(path.normpath(directory_path))
            if not file_path.endswith('.zip'): file_path += '.zip'
            with ZipFile(file_path, 'w') as zip_file:
                for root, dirs, files in walk(directory_path):
                    for file in files:
                        file_path = path.join(root, file)
                        arcname = path.relpath(file_path, directory_path)
                        zip_file.write(file_path, arcname)
            return path.exists(file_path)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.createZIPFromFolder: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def extractZIPToFolder(self, file_path='', directory_path=''):
        try:
            from zipfile import ZipFile
            file_path = str(file_path).strip()
            directory_path = str(directory_path).strip()
            if not file_path: return False
            with ZipFile(file_path, 'r') as zip_file: zip_file.extractall(directory_path)
            from os import path
            return path.isdir(directory_path)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.extractZIPToFolder: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def stringToDictionaryOrJSON(self, string=''):
        try:
            dictionary_or_json = {}
            original_string = string
            string = str(string).strip()
            try:
                from json import loads
                try: dictionary_or_json = loads(string)
                except: dictionary_or_json = loads(original_string)
            except:
                from ast import literal_eval
                try: dictionary_or_json = literal_eval(string)
                except: dictionary_or_json = literal_eval(original_string)
            return dictionary_or_json
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.stringToDictionaryOrJSON: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return string
    def stringToTokens(self, string=''):
        try:
            tokens = []
            string = str(string)
            def split_string_by_three(text=''): return [text[index:index+3] for index in range(0, len(text), 3)]
            tokens = split_string_by_three(text=string)
            return tokens
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.stringToTokens: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return string.split()
    def replaceCaseInsensitive(self, string='', old='', new='', first_occurrence=False, ignore_accents=False):
        try:
            string, old, new = str(string), str(old), str(new)
            first_occurrence = bool(first_occurrence) if type(first_occurrence) in (bool, int, float) else False
            ignore_accents = bool(ignore_accents) if type(ignore_accents) in (bool, int, float) else False
            from re import sub, escape, IGNORECASE
            def remove_accents(text):
                from unicodedata import normalize, category
                return ''.join(char for char in normalize('NFKD', text) if category(char) != 'Mn')
            if ignore_accents:
                string = remove_accents(string)
                old = remove_accents(old)
            count = 1 if first_occurrence else 0
            return sub(escape(old), new, string, count=count, flags=IGNORECASE)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.replaceCaseInsensitive: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return string
    def replaceLastOccurrence(self, string='', old='', new='', case_insensitive=False):
        try:
            case_insensitive = bool(case_insensitive) if type(case_insensitive) in (bool, int, float) else False
            def replaceLastOccurrenceCaseInsensitive(string='', old='', new=''):
                string = str(string)
                from re import escape, search, IGNORECASE
                pattern = escape(str(old))
                match = search(pattern + r'(?!.*' + pattern + r')', string, IGNORECASE)
                if match:
                    start, end = match.span()
                    return string[:start]+str(new)+string[end:]
                return string
            if case_insensitive: return replaceLastOccurrenceCaseInsensitive(string=string, old=old, new=new)
            else:
                parts = str(string).rsplit(str(old), 1)
                return str(new).join(parts)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.replaceLastOccurrence: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return string
    def getKeyWordsString(self, string=''):
        try:
            string = str(string).strip()
            from re import findall
            words = findall(r'\b[\w.]+\b', string)
            def isNumber(word):
                try:
                    float(word)
                    return True
                except: return False
            filtered_words = [word for word in words if len(word) > 2 or (len(word) in [1, 2] and word.isupper()) or isNumber(word)]
            return ', '.join(filtered_words)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getKeyWordsString: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return string
    def getHashCode(self):
        try:
            hash_code = 'temporary'
            def generateHashCode():
                from datetime import datetime
                now = datetime.now()
                formatted_string = now.strftime('%Y%m%d%H%M%S%f')
                try:
                    datetime_bytes = formatted_string.encode()
                    from hashlib import sha256
                    hash_object = sha256(datetime_bytes)
                    hash_code = hash_object.hexdigest()
                except: hash_code = formatted_string
                return hash_code.strip()
            hash_code = generateHashCode()
            return hash_code
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getHashCode: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return 'temporary'
    def getFileName(self, file_path=''):
        try:
            file_name = ''
            file_path = str(file_path).strip()
            from os.path import splitext, basename
            if '/' in file_path: file_name = splitext(file_path)[0].split('/')[-1].strip()
            elif '\\' in file_path: file_name = splitext(file_path)[0].split('\\')[-1].strip()
            else: file_name = splitext(basename(file_path))[0]
            if len(file_name) < 1: file_name = self.getHashCode()
            return file_name
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getFileName: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return ''
    def getFileNameWithExtension(self, file_path=''):
        try:
            file_name = ''
            file_path = str(file_path).strip()
            from os.path import basename
            if '/' in file_path: file_name = file_path.split('/')[-1].strip()
            elif '\\' in file_path: file_name = file_path.split('\\')[-1].strip()
            else: file_name = basename(file_path)
            return file_name
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getFileNameWithExtension: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return ''
    def getFileExtension(self, file_path=''):
        try:
            file_extension = ''
            file_path = str(file_path).strip()
            from os import path
            file_extension = path.splitext(file_path)[1]
            if len(file_extension) < 1:
                possible_extensions = self.__image_extensions+self.__audio_extensions+self.__video_extensions+self.__pdf_extensions+self.__word_extensions+self.__excel_extensions+self.__powerpoint_extensions+self.__code_extensions+self.__autocad_extensions+self.__text_extensions
                for possible_extension in possible_extensions:
                    if possible_extension in file_path.lower():
                        file_extension = possible_extension
                        break
                if len(file_extension) < 1:
                    for possible_extension in possible_extensions:
                        if possible_extension[1:] in file_path.lower():
                            file_extension = possible_extension
                            break
            return file_extension
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getFileExtension: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return ''
    def getDirectoryPath(self, file_path=''):
        try:
            directory_path = './'
            file_path = str(file_path).strip()
            left_bar, right_bar = '\\' in file_path, '/' in file_path
            bar = '\\' if left_bar else '/'
            directory_path_list = file_path.split(bar)
            directory_path = str(bar.join(directory_path_list[:-1])).strip()
            if not directory_path.endswith(bar): directory_path += bar
            directory_path = directory_path.strip()
            if directory_path == bar: directory_path = '.'+directory_path
            return directory_path
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getDirectoryPath: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return './'
    def getFilePathsFromDirectory(self, directory_path='', extensions=[], recursive=True):
        try:
            files_list = []
            directory_path = str(directory_path).strip()
            extensions = list(extensions) if type(extensions) in (tuple, list) else []
            recursive = bool(recursive) if type(recursive) in (bool, int, float) else True
            if not directory_path: directory_path = './'
            from os import path, walk, listdir
            extensions_length = len(extensions)
            if path.isdir(directory_path):
                if recursive:
                    for root, _, files in walk(directory_path):
                        for file in files:
                            if extensions_length > 0:
                                if any(file.lower().strip().endswith(extension.lower().strip()) for extension in extensions): files_list.append(path.join(root, file))
                            else: files_list.append(path.join(root, file))
                else:
                    for file in listdir(directory_path):
                        file_path = path.join(directory_path, file)
                        if path.isfile(file_path):
                            if extensions_length > 0:
                                if any(file.lower().strip().endswith(extension.lower().strip()) for extension in extensions): files_list.append(file_path)
                            else: files_list.append(file_path)
            return files_list
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getFilePathsFromDirectory: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return []
    def isURLAddress(self, file_path=''):
        try:
            result = False
            file_path = str(file_path).lower().strip()
            result = file_path.startswith(('https://', 'http://', 'www.'))
            return result
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.isURLAddress: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def isYouTubeURL(self, url_path=''):
        try:
            from re import compile
            return bool(compile(r'(https?://)?(www\.)?(youtube\.com|youtu\.be)(/.*)?$').match(str(url_path).strip()))
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.isYouTubeURL: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def isImageAddress(self, file_path=''):
        try:
            result = False
            file_path = str(file_path).strip()
            image_extensions, extension = (
                '.jpg', '.jpeg', '.png', '.gif', '.bmp', '.tiff', '.tif', '.webp',  '.ico',
                '.svg', '.heic', '.heif', '.jp2', '.j2k', '.jpf', '.jpx',  '.jpm', '.mj2',
                '.dds', '.psd', '.ai', '.eps', '.raw', '.cr2', '.nef', '.orf', '.sr2'
            ), self.getFileExtension(file_path=file_path)
            result = extension in image_extensions
            if not result:
                file_path_lower = file_path.lower()
                is_web_address = self.isURLAddress(file_path=file_path_lower)
                result = is_web_address and ('/image/' in file_path_lower or '/images/' in file_path_lower or '/img/' in file_path_lower)
                if not result and is_web_address:
                    for image_type in image_extensions:
                        image_type = image_type[1:]
                        if image_type in file_path_lower: return True
            return result
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.isImageAddress: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def downloadFile(self, url_path='', output_path=''):
        try:
            url_path = str(url_path).strip()
            output_path = str(output_path).strip()
            from urllib.request import urlopen
            from pathlib import Path
            from os import path
            try:
                from os import environ
                try: from certifi import where
                except:
                    self.installModule(module_name='certifi', version='2024.2.2')
                    from certifi import where
                environ['SSL_CERT_FILE'] = where()
                from logging import getLogger, ERROR
                getLogger('urlopen').setLevel(ERROR)
            except: pass
            response = urlopen(url_path)
            destination = output_path if output_path else self.getFileNameWithExtension(file_path=url_path)
            with open(destination, 'wb') as file: file.write(response.read())
            return path.exists(destination)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.downloadFile: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            try: self.updateModule(module_name='certifi')
            except: pass
            return False
    def listGitHubFiles(self, url_path=''):
        try:
            url_path = str(url_path).strip()
            try:
                from os import environ
                try: from certifi import where
                except:
                    self.installModule(module_name='certifi', version='2024.2.2')
                    from certifi import where
                environ['SSL_CERT_FILE'] = where()
                from logging import getLogger, ERROR
                getLogger('urlopen').setLevel(ERROR)
            except: pass
            from urllib.request import urlopen
            from json import loads
            if url_path.startswith('https://github.com/'): url_path = url_path.replace('https://github.com/', '')
            branch = 'main'
            folder_path = ''
            if '/tree/' in url_path:
                before, after = url_path.split('/tree/', 1)
                parts = after.split('/', 1)
                branch = parts[0]
                folder_path = parts[1] if len(parts) > 1 else ''
                user_name, repository_name = before.split('/')[:2]
            else:
                parts = url_path.split('/')
                user_name = parts[0]
                repository_name = parts[1]
                if len(parts) > 2: folder_path = '/'.join(parts[2:])
            api_url = f'https://api.github.com/repos/{user_name}/{repository_name}/contents'
            if folder_path: api_url += f'/{folder_path}'
            api_url += f'?ref={branch}'
            data = loads(urlopen(api_url).read().decode())
            files = []
            for item in data:
                if item['type'] == 'file': files.append(item['download_url'])
                elif item['type'] == 'dir':
                    sub_url = f'https://github.com/{user_name}/{repository_name}/tree/{branch}/{item["path"]}'
                    files.extend(self.listGitHubFiles(sub_url))
            return files
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.listGitHubFiles: ' + str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            try: self.updateModule(module_name='certifi')
            except: pass
            return []
    def downloadGitHubFolder(self, url_path='', output_path=''):
        try:
            url_path = str(url_path).strip()
            output_path = str(output_path).strip()
            if not url_path: return False
            if not output_path: output_path = url_path.split('/')[-1].strip()
            if not output_path.endswith('/') and not output_path.endswith('\\'): output_path += '/'
            try:
                from os import environ
                try: from certifi import where
                except:
                    self.installModule(module_name='certifi', version='2024.2.2')
                    from certifi import where
                environ['SSL_CERT_FILE'] = where()
                from logging import getLogger, ERROR
                getLogger('urlopen').setLevel(ERROR)
            except: pass
            from os import makedirs, path
            makedirs(output_path, exist_ok=True)
            url_paths = self.listGitHubFiles(url_path=url_path)
            for url in url_paths:
                file_name = self.getFileNameWithExtension(file_path=url)
                if not self.downloadFile(url_path=url, output_path=output_path+file_name): break
            return path.exists(output_path)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.downloadGitHubFolder: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            try: self.updateModule(module_name='certifi')
            except: pass
            return False
    def downloadTheHuggingFaceFiles(self, model_name='', output_path='', progress=True, show_gb=True):
        try:
            model_name = str(model_name).strip()
            output_path = str(output_path).strip()
            progress = bool(progress) if type(progress) in (bool, int, float) else True
            show_gb = bool(show_gb) if type(show_gb) in (bool, int, float) else True
            try:
                from os import environ
                try: from certifi import where
                except:
                    self.installModule(module_name='certifi', version='2024.2.2')
                    from certifi import where
                environ['SSL_CERT_FILE'] = where()
                from logging import getLogger, ERROR
                getLogger('requests').setLevel(ERROR)
            except: pass
            if not model_name: return False
            from urllib.request import urlopen, Request
            from pathlib import Path
            from json import loads
            from os import makedirs
            from sys import stdout
            from shutil import get_terminal_size
            model_name = model_name.rstrip('/').split('/')[-2:] if '/' in model_name else [model_name.split('/')[-1]]
            model_name = '/'.join(model_name[-2:]) if len(model_name) == 2 else model_name[0]
            base_url = f'https://huggingface.co/{model_name}'
            api_url = f'https://huggingface.co/api/models/{model_name}'
            request = Request(api_url, headers={'User-Agent': 'Mozilla/5.0'})
            response = urlopen(request)
            model_info = loads(response.read().decode('utf-8'))
            siblings = model_info.get('siblings', [])
            if not siblings: return False
            model_folder_name = model_name.split('/')[-1]
            if output_path: destination = Path(output_path)
            else: destination = Path(model_folder_name)
            makedirs(destination, exist_ok=True)
            total_files = len(siblings)
            for index, file_info in enumerate(siblings, 1):
                index = str(index).rjust(len(str(total_files)), '0')
                file_name = file_info.get('rfilename', '')
                if not file_name: continue
                file_url = f'{base_url}/resolve/main/{file_name}'
                file_path = destination / file_name
                makedirs(file_path.parent, exist_ok=True)
                request = Request(file_url, headers={'User-Agent': 'Mozilla/5.0'})
                response = urlopen(request)
                total_size = int(response.headers.get('Content-Length', 0))
                total_size_gb = total_size / 1073741824
                downloaded_size = 0
                with open(file_path, 'wb') as file:
                    while True:
                        chunk = response.read(8192)
                        if not chunk: break
                        file.write(chunk)
                        downloaded_size += len(chunk)
                        if progress and total_size > 0:
                            percentage = (downloaded_size / total_size) * 100
                            str_percentage, insert = f'{percentage:.2f}'.rjust(6, '0'), ''
                            terminal_width = get_terminal_size().columns
                            if show_gb: insert = f' ({(downloaded_size/1073741824):.8f}/{total_size_gb:.8f} GB)'
                            text_part = f' {str_percentage}% - File {index}/{total_files}{insert}'
                            bar_length = max(10, terminal_width - len(text_part) - 2)
                            filled_length = int(bar_length * downloaded_size / total_size)
                            bar = '=' * filled_length + '-' * (bar_length - filled_length)
                            stdout.write(f'\r[{bar}]{text_part}')
                            stdout.flush()
                if progress:
                    stdout.write('\n')
                    stdout.flush()
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.downloadTheHuggingFaceFiles: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            try: self.updateModule(module_name='certifi')
            except: pass
            return False
    def downloadTheHuggingFaceModel(self, model_name='', output_path='', progress=True, show_gb=True):
        try:
            model_name = str(model_name).strip()
            output_path = str(output_path).strip()
            progress = bool(progress) if type(progress) in (bool, int, float) else True
            show_gb = bool(show_gb) if type(show_gb) in (bool, int, float) else True
            try:
                from os import environ
                try: from certifi import where
                except:
                    self.installModule(module_name='certifi', version='2024.2.2')
                    from certifi import where
                environ['SSL_CERT_FILE'] = where()
                from logging import getLogger, ERROR
                getLogger('requests').setLevel(ERROR)
            except: pass
            if not model_name: return False
            from urllib.request import urlopen, Request
            from pathlib import Path
            from json import loads
            from os import makedirs
            from sys import stdout
            from shutil import get_terminal_size
            model_name = model_name.rstrip('/').split('/')[-2:] if '/' in model_name else [model_name.split('/')[-1]]
            model_name = '/'.join(model_name[-2:]) if len(model_name) == 2 else model_name[0]
            base_url = f'https://huggingface.co/{model_name}'
            api_url = f'https://huggingface.co/api/models/{model_name}'
            request = Request(api_url, headers={'User-Agent': 'Mozilla/5.0'})
            response = urlopen(request)
            model_info = loads(response.read().decode('utf-8'))
            siblings = model_info.get('siblings', [])
            if not siblings: return False
            model_folder_name = model_name.split('/')[-1]
            if output_path: destination = Path(output_path)
            else: destination = Path(model_folder_name)
            makedirs(destination, exist_ok=True)
            total_size_all = 0
            for file_info in siblings:
                file_url = f'{base_url}/resolve/main/{file_info.get("rfilename","")}'
                try:
                    head = Request(file_url, method='HEAD', headers={'User-Agent': 'Mozilla/5.0'})
                    response = urlopen(head)
                    total_size_all += int(response.headers.get('Content-Length', 0))
                except: pass
            if total_size_all == 0: return False
            downloaded_total = 0
            for file_info in siblings:
                file_name = file_info.get('rfilename', '')
                if not file_name: continue
                file_url = f'{base_url}/resolve/main/{file_name}'
                file_path = destination / file_name
                makedirs(file_path.parent, exist_ok=True)
                request = Request(file_url, headers={'User-Agent': 'Mozilla/5.0'})
                response = urlopen(request)
                total_size = int(response.headers.get('Content-Length', 0))
                with open(file_path, 'wb') as file:
                    while True:
                        chunk = response.read(8192)
                        if not chunk: break
                        file.write(chunk)
                        downloaded_total += len(chunk)
                        if progress:
                            percentage = (downloaded_total / total_size_all) * 100
                            str_percentage, insert = f'{percentage:.2f}'.rjust(6, '0'), ''
                            terminal_width = get_terminal_size().columns
                            if show_gb: insert = f' ({(downloaded_total/1073741824):.8f}/{(total_size_all/1073741824):.8f} GB)'
                            text_part = f' {str_percentage}%{insert}'
                            bar_length = max(10, terminal_width - len(text_part) - 2)
                            filled_length = int(bar_length * downloaded_total / total_size_all)
                            bar = '=' * filled_length + '-' * (bar_length - filled_length)
                            stdout.write(f'\r[{bar}]{text_part}')
                            stdout.flush()
            if progress:
                stdout.write('\n')
                stdout.flush()
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.downloadTheHuggingFaceModel: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            try: self.updateModule(module_name='certifi')
            except: pass
            return False
    def downloadTheGitHubFiles(self, repository_name='', output_path='', progress=True, show_gb=True, branch='main'):
        try:
            repository_name = str(repository_name).strip()
            output_path = str(output_path).strip()
            progress = bool(progress) if type(progress) in (bool, int, float) else True
            show_gb = bool(show_gb) if type(show_gb) in (bool, int, float) else True
            branch = str(branch).strip() or 'main'
            if not repository_name: return False
            try:
                from os import environ
                try: from certifi import where
                except:
                    self.installModule(module_name='certifi', version='2024.2.2')
                    from certifi import where
                environ['SSL_CERT_FILE'] = where()
                from logging import getLogger, ERROR
                getLogger('requests').setLevel(ERROR)
            except: pass
            from urllib.request import urlopen, Request
            from pathlib import Path
            from json import loads
            from os import makedirs
            from sys import stdout
            from shutil import get_terminal_size
            repo_name = repository_name.rstrip('/').split('/')[-2:]
            if len(repo_name) < 2: return False
            user, repo = repo_name
            api_url = f'https://api.github.com/repos/{user}/{repo}/git/trees/{branch}?recursive=1'
            base_raw_url = f'https://raw.githubusercontent.com/{user}/{repo}/{branch}'
            request = Request(api_url, headers={'User-Agent': 'Mozilla/5.0'})
            response = urlopen(request, timeout=30)
            repo_info = loads(response.read().decode('utf-8'))
            tree = repo_info.get('tree', [])
            if not tree: return False
            if output_path: destination = Path(output_path)
            else: destination = Path(repo)
            makedirs(destination, exist_ok=True)
            files = [item for item in tree if item.get('type') == 'blob']
            total_files = len(files)
            for index, file_info in enumerate(files, 1):
                index = str(index).rjust(len(str(total_files)), '0')
                file_path_rel = file_info.get('path', '')
                if not file_path_rel: continue
                file_url = f'{base_raw_url}/{file_path_rel}'
                file_path = destination / file_path_rel
                makedirs(file_path.parent, exist_ok=True)
                request = Request(file_url, headers={'User-Agent': 'Mozilla/5.0'})
                response = urlopen(request, timeout=60)
                total_size = int(response.headers.get('Content-Length', 0))
                total_size_gb = total_size / 1073741824 if total_size > 0 else 0
                downloaded_size = 0
                with open(file_path, 'wb') as file:
                    while True:
                        chunk = response.read(8192)
                        if not chunk: break
                        file.write(chunk)
                        downloaded_size += len(chunk)
                        if progress and total_size > 0:
                            percentage = (downloaded_size / total_size) * 100
                            str_percentage, insert = f'{percentage:.2f}'.rjust(6, '0'), ''
                            terminal_width = get_terminal_size().columns
                            if show_gb: insert = f' ({(downloaded_size/1073741824):.8f}/{total_size_gb:.8f} GB)'
                            text_part = f' {str_percentage}% - File {index}/{total_files}{insert}'
                            bar_length = max(10, terminal_width - len(text_part) - 2)
                            filled_length = int(bar_length * downloaded_size / total_size)
                            bar = '=' * filled_length + '-' * (bar_length - filled_length)
                            stdout.write(f'\r[{bar}]{text_part}')
                            stdout.flush()
                if progress:
                    stdout.write('\n')
                    stdout.flush()
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.downloadTheGitHubFiles: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            try: self.updateModule(module_name='certifi')
            except: pass
            return False
    def downloadTheGitHubRepository(self, repository_name='', output_path='', progress=True, show_gb=True, branch='main'):
        try:
            repository_name = str(repository_name).strip()
            output_path = str(output_path).strip()
            progress = bool(progress) if type(progress) in (bool, int, float) else True
            show_gb = bool(show_gb) if type(show_gb) in (bool, int, float) else True
            branch = str(branch).strip() or 'main'
            if not repository_name: return False
            try:
                from os import environ
                try: from certifi import where
                except:
                    self.installModule(module_name='certifi', version='2024.2.2')
                    from certifi import where
                environ['SSL_CERT_FILE'] = where()
                from logging import getLogger, ERROR
                getLogger('requests').setLevel(ERROR)
            except: pass
            from urllib.request import urlopen, Request
            from pathlib import Path
            from json import loads
            from os import makedirs
            from sys import stdout
            from shutil import get_terminal_size
            repo_name = repository_name.rstrip('/').split('/')[-2:]
            if len(repo_name) < 2: return False
            user, repo = repo_name
            api_url = f'https://api.github.com/repos/{user}/{repo}/git/trees/{branch}?recursive=1'
            base_raw_url = f'https://raw.githubusercontent.com/{user}/{repo}/{branch}'
            request = Request(api_url, headers={'User-Agent': 'Mozilla/5.0'})
            response = urlopen(request, timeout=30)
            repo_info = loads(response.read().decode('utf-8'))
            tree = repo_info.get('tree', [])
            if not tree: return False
            files = [item for item in tree if item.get('type') == 'blob']
            if not files: return False
            if output_path: destination = Path(output_path)
            else: destination = Path(repo)
            makedirs(destination, exist_ok=True)
            total_size_all = 0
            for file_info in files:
                file_path_rel = file_info.get('path', '')
                if not file_path_rel: continue
                file_url = f'{base_raw_url}/{file_path_rel}'
                try:
                    head = Request(file_url, method='HEAD', headers={'User-Agent': 'Mozilla/5.0'})
                    response = urlopen(head, timeout=15)
                    total_size_all += int(response.headers.get('Content-Length', 0))
                except: pass
            if total_size_all == 0: return False
            downloaded_total = 0
            for file_info in files:
                file_path_rel = file_info.get('path', '')
                if not file_path_rel: continue
                file_url = f'{base_raw_url}/{file_path_rel}'
                file_path = destination / file_path_rel
                makedirs(file_path.parent, exist_ok=True)
                request = Request(file_url, headers={'User-Agent': 'Mozilla/5.0'})
                response = urlopen(request, timeout=60)
                total_size = int(response.headers.get('Content-Length', 0))
                with open(file_path, 'wb') as file:
                    while True:
                        chunk = response.read(8192)
                        if not chunk: break
                        file.write(chunk)
                        downloaded_total += len(chunk)
                        if progress:
                            percentage = (downloaded_total / total_size_all) * 100
                            str_percentage, insert = f'{percentage:.2f}'.rjust(6, '0'), ''
                            terminal_width = get_terminal_size().columns
                            if show_gb: insert = f' ({(downloaded_total/1073741824):.8f}/{(total_size_all/1073741824):.8f} GB)'
                            text_part = f' {str_percentage}%{insert}'
                            bar_length = max(10, terminal_width - len(text_part) - 2)
                            filled_length = int(bar_length * downloaded_total / total_size_all)
                            bar = '=' * filled_length + '-' * (bar_length - filled_length)
                            stdout.write(f'\r[{bar}]{text_part}')
                            stdout.flush()
            if progress:
                stdout.write('\n')
                stdout.flush()
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.downloadTheGitHubRepository: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            try: self.updateModule(module_name='certifi')
            except: pass
            return False
    def uploadTheHuggingFaceModel(self, model_path='', repository_id=''):
        try:
            model_path = str(model_path).strip()
            repository_id = str(repository_id).strip()
            from os.path import isdir
            if not isdir(model_path):
                print(f'The path {model_path} does not exist.')
                return False
            if not repository_id: return False
            try:
                from huggingface_hub import HfApi, create_repo
                from huggingface_hub.utils import HfHubHTTPError
            except:
                self.installModule(module_name='huggingface-hub', version='0.28.1')
                from huggingface_hub import HfApi, create_repo
                from huggingface_hub.utils import HfHubHTTPError
            api = HfApi()
            try: create_repo(repo_id=repository_id, repo_type='model', exist_ok=True)
            except: pass
            print(f'Uploading model from {model_path} to {repository_id}...')
            api.upload_folder(folder_path=model_path, repo_id=repository_id, repo_type='model', commit_message='_')
            print(f'Model successfully uploaded to "https://huggingface.co/{repository_id}".')
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.uploadTheHuggingFaceModel: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def upscaleImage(self, file_path='', output_path='', file_dictionary={}):
        try:
            file_path = str(file_path).strip()
            output_path = str(output_path).strip()
            if type(file_dictionary) != dict: file_dictionary = {'base64_string': '', 'type': ''}
            if 'base64_string' not in file_dictionary: file_dictionary['base64_string'] = ''
            if 'type' not in file_dictionary: file_dictionary['type'] = ''
            image_path = ''
            if len(str(file_dictionary['base64_string'])) > 0:
                directory_path = self.getTemporaryFilesDirectory()
                output_path = f'{directory_path}{self.getHashCode()}.png'
                image_path = f'{directory_path}{self.getHashCode()}.png'
                if self.base64ToFile(file_dictionary=file_dictionary, file_path=image_path): file_path = image_path
            try: from PIL import Image
            except:
                self.installModule(module_name='pillow', version='10.3.0')
                from PIL import Image
            try: from super_image import ImageLoader, EdsrModel
            except:
                self.installModule(module_name='super-image', version='0.2.0')
                from super_image import ImageLoader, EdsrModel
            from os import path, devnull
            from contextlib import redirect_stdout, redirect_stderr
            model_path = self.__getRootDirectory()+'upscale'
            url_path = 'https://github.com/sapiens-technology/files/tree/main/upscale'
            if not path.exists(model_path): self.downloadGitHubFolder(url_path=url_path, output_path=model_path)
            image = Image.open(file_path)
            image = ImageLoader.load_image(image)
            with open(devnull, 'w') as devnull:
                with redirect_stdout(devnull), redirect_stderr(devnull): model = EdsrModel.from_pretrained(model_path, scale=4)
            prediction = model(image)
            ImageLoader.save_image(prediction, output_path)
            if image_path:
                file_dictionary = self.fileToBase64(file_path=output_path)
                self.deleteFile(file_path=image_path)
                self.deleteFile(file_path=output_path)
                return file_dictionary
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.upscaleImage: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def saveIntermediateFrameOfVideo(self, file_path='', directory_path=''):
        try:
            image_path = ''
            file_path, directory_path, temporary_video_path = str(file_path).strip(), str(directory_path).strip(), ''
            if not directory_path.endswith('/'): directory_path += '/'
            try: from cv2 import VideoCapture, CAP_PROP_FRAME_COUNT, CAP_PROP_POS_FRAMES, imwrite
            except:
                self.installModule(module_name='opencv-python', version='4.6.0.66')
                from cv2 import VideoCapture, CAP_PROP_FRAME_COUNT, CAP_PROP_POS_FRAMES, imwrite
            if self.isURLAddress(file_path=file_path):
                try:
                    from os import environ
                    try: from certifi import where
                    except:
                        self.installModule(module_name='certifi', version='2024.2.2')
                        from certifi import where
                    environ['SSL_CERT_FILE'] = where()
                    from logging import getLogger, ERROR
                    getLogger('requests').setLevel(ERROR)
                except: pass
                try: from requests import get
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import get
                content = get(file_path, stream=False, timeout=self.__timeout).content
                from tempfile import TemporaryDirectory
                from os import path
                video_name, extension = self.getHashCode(), self.getFileExtension(file_path=file_path)
                if len(extension) < 1: extension = '.mp4'
                with TemporaryDirectory() as temporary_directory:
                    temporary_video_path = path.join(temporary_directory, video_name+extension)
                    with open(temporary_video_path, 'wb') as file: file.write(content)
                    video = VideoCapture(temporary_video_path)
                    self.deleteFile(file_path=temporary_video_path)
            else: video = VideoCapture(file_path)
            total_frames = int(video.get(CAP_PROP_FRAME_COUNT))
            middle_frame_number = total_frames // 2
            video.set(CAP_PROP_POS_FRAMES, middle_frame_number)
            boolean_return, frame = video.read()
            if boolean_return:
                image_name = self.getHashCode()
                image_path = f'{directory_path}{image_name}.png'
                imwrite(image_path, frame)
            video.release()
            return image_path
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.saveIntermediateFrameOfVideo: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return ''
    def fileToBase64(self, file_path='', media_metadata=[]):
        try:
            file_dictionary = {'base64_string': '', 'type': ''}
            file_path = str(file_path).strip()
            media_metadata = list(media_metadata) if type(media_metadata) in (tuple, list) else []
            from base64 import b64encode
            from os import path
            if self.isURLAddress(file_path=file_path):
                try:
                    from os import environ
                    try: from certifi import where
                    except:
                        self.installModule(module_name='certifi', version='2024.2.2')
                        from certifi import where
                    environ['SSL_CERT_FILE'] = where()
                    from logging import getLogger, ERROR
                    getLogger('requests').setLevel(ERROR)
                except: pass
                try: from requests import get
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import get
                content = get(file_path, stream=False, timeout=self.__timeout).content
                if len(media_metadata) > 0:
                    temporary_file_directory = self.getTemporaryFilesDirectory()
                    file_name, file_extension = self.getFileName(file_path=file_path), self.getFileExtension(file_path=file_path)
                    file_path = temporary_file_directory+file_name+file_extension
                    with open(file_path, 'wb') as file: file.write(content)
                    self.updateMediaMetadata(file_path=file_path, file_dictionary={}, metadata=media_metadata)
                    with open(file_path, 'rb') as file: base64_string = b64encode(file.read()).decode('utf-8')
                    _type = file_extension[1:]
                    self.deleteFile(file_path=file_path)
                else: base64_string = b64encode(content).decode('utf-8')
            else:
                if len(media_metadata) > 0: self.updateMediaMetadata(file_path=file_path, file_dictionary={}, metadata=media_metadata)
                with open(file_path, 'rb') as file: base64_string = b64encode(file.read()).decode('utf-8')
            file_dictionary['base64_string'], _type = base64_string, path.splitext(file_path)[1][1:]
            if len(_type.strip()) < 1:
                if 'image' in file_path or 'img' in file_path: _type = 'png'
                elif 'audio' in file_path or 'wav' in file_path: _type = 'wav'
                elif 'mp3' in file_path: _type = 'mp3'
                elif 'video' in file_path or 'mp4' in file_path: _type = 'mp4'
            file_dictionary['type'] = _type
            return file_dictionary
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.fileToBase64: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'base64_string': '', 'type': ''}
    def base64ToFile(self, file_dictionary={}, file_path=''):
        try:
            if type(file_dictionary) != dict: file_dictionary = {'base64_string': '', 'type': ''}
            if 'base64_string' not in file_dictionary: file_dictionary['base64_string'] = ''
            if 'type' not in file_dictionary: file_dictionary['type'] = ''
            file_path = str(file_path).strip()
            base64_string = file_dictionary['base64_string']
            file_type = file_dictionary['type']
            from base64 import b64decode
            file_content = b64decode(base64_string)
            def hasName(name_path=''):
                if '/' in name_path: name_path = name_path.split('/')[-1]
                if '.' in name_path: name_path = name_path.split('.')[0]
                from re import sub
                name_path = sub(r'[^a-zA-Z0-9\sáéíóúàèìòùâêîôûãõçÁÉÍÓÚÀÈÌÒÙÂÊÎÔÛÃÕÇ]', '', name_path)
                return len(name_path.strip()) > 0
            from os import path
            if not hasName(name_path=file_path) or path.isdir(file_path):
                if len(file_path) > 0 and not file_path.endswith('/'): file_path += '/FILE.'+file_type
                else: file_path += 'FILE.'+file_type
            if not path.splitext(file_path)[1]: file_path = f'{file_path}.{file_type}'
            with open(file_path, 'wb') as file: file.write(file_content)
            return path.exists(file_path)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.base64ToFile: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def mathematicalSolution(self, prompt=''):
        try:
            mathematical_solution = prompt
            prompt = str(prompt).strip()
            count_tokens = self.countTokens(string=prompt, pattern='gpt')
            if count_tokens > 150: return mathematical_solution
            from re import search, sub, finditer
            def _check_operation(text=''):
                temp_text = text.replace('x', '*').replace('X', '*').replace('ˆ', '**').replace('^', '**')
                temp_text = temp_text.replace('⁰', '** 0').replace('¹', '** 1').replace('²', '** 2').replace('³', '** 3').replace('⁴', '** 4')
                temp_text = temp_text.replace('⁵', '** 5').replace('⁶', '** 6').replace('⁷', '** 7').replace('⁸', '** 8').replace('⁹', '** 9')
                temp_text = temp_text.replace('raised to the power of', '**').replace('elevado a la', '**').replace('elevado à', '**').replace('elevado a', '**')
                number = r'[-+]?\d+(?:[.,]\d+)?'
                operators = [r'\*\*\=', r'\+\=', r'-\=', r'/\=', r'//\=', r'%\=', r'\*\=', r'\*\*(?!\=)', r'//(?!\=)', r'<=', r'>=', r'!=', r'\+(?!\=)', r'-(?!\=)', r'/(?!\=)', r'%(?!\=)', r'\*(?!\=)', r'<(?!\=)', r'>(?!\=)']
                for operator in operators:
                    pattern = rf'{number}\s*{operator}\s*{number}'
                    if search(pattern, temp_text): return True
                return False
            def _process_and_calculate(text=''):
                processed_text = sub(r'(\d),(\d)', r'\1.\2', text)
                processed_text = processed_text.replace('x', '*').replace('X', '*').replace('ˆ', '**').replace('^', '**')
                processed_text = processed_text.replace('⁰', '** 0').replace('¹', '** 1').replace('²', '** 2').replace('³', '** 3').replace('⁴', '** 4')
                processed_text = processed_text.replace('⁵', '** 5').replace('⁶', '** 6').replace('⁷', '** 7').replace('⁸', '** 8').replace('⁹', '** 9')
                processed_text = processed_text.replace('raised to the power of', '**').replace('elevado a la', '**').replace('elevado à', '**').replace('elevado a', '**')
                expression_pattern = r'[(\[]*[-+]?\d+(?:\.\d+)?[)\]]*(?:\s*(?:\*\*\=?|//\=?|[+\-*/%<>=!]+\=?)\s*[(\[]*[-+]?\d+(?:\.\d+)?[)\]]*)+|[(\[][-+]?\d+(?:\.\d+)?(?:\s*(?:\*\*\=?|//\=?|[+\-*/%<>=!]+\=?)\s*[-+]?\d+(?:\.\d+)?)+[)\]](?:\s*(?:\*\*\=?|//\=?|[+\-*/%<>=!]+\=?)\s*[(\[]*[-+]?\d+(?:\.\d+)?[)\]]*)*'
                matches = list(finditer(expression_pattern, processed_text))
                if not matches: return ''
                best_match = max(matches, key=lambda m: len(m.group()))
                expression = best_match.group().strip()
                open_paren = expression.count('(')
                close_paren = expression.count(')')
                open_bracket = expression.count('[')
                close_bracket = expression.count(']')
                start_pos = best_match.start()
                end_pos = best_match.end()
                while open_paren > close_paren and end_pos < len(processed_text):
                    if processed_text[end_pos] == ')':
                        close_paren += 1
                        end_pos += 1
                        expression = processed_text[start_pos:end_pos].strip()
                    else: end_pos += 1
                while open_bracket > close_bracket and end_pos < len(processed_text):
                    if processed_text[end_pos] == ']':
                        close_bracket += 1
                        end_pos += 1
                        expression = processed_text[start_pos:end_pos].strip()
                    else: end_pos += 1
                while close_paren > open_paren and start_pos > 0:
                    start_pos -= 1
                    if processed_text[start_pos] == '(':
                        open_paren += 1
                        expression = processed_text[start_pos:end_pos].strip()
                while close_bracket > open_bracket and start_pos > 0:
                    start_pos -= 1
                    if processed_text[start_pos] == '[':
                        open_bracket += 1
                        expression = processed_text[start_pos:end_pos].strip()
                try:
                    result = eval(expression)
                    return f'{expression} = {result}'
                except: return ''
            def _calculate_expression(text=''):
                if _check_operation(text=text): return _process_and_calculate(text=text.replace('[', '(').replace(']', ')') if '[' in text and ']' in text else text)
                else: return ''
            lines, calculate = prompt.split('\n'), False
            for index, line in enumerate(lines):
                line = str(line).strip()
                calculate_expression = _calculate_expression(text=line)
                if calculate_expression:
                    lines[index] = f'{calculate_expression}\n{line}\n'
                    calculate = True
            if calculate: mathematical_solution = '\n'.join(lines)
            return mathematical_solution.strip()
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.mathematicalSolution: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return prompt
    def getSystemInstructions(self, language='en'):
        try:
            system_instructions = ''
            if language: language = str(language).strip()
            system_instructions_en = """
                Your name is Sapiens Chat, you are a multimodal Artificial Intelligence model with an infinite context window, created by a Brazilian company called Sapiens Technology®.
                You use an innovative inference architecture called SAPI (Semantic AI with Pretrained Integration). This architecture employs a class structure called Frankenstein, which manages several sub-models that work together to provide a more precise answer. The Frankenstein is queried by a main, unfiltered model named Entidade (Entity).
                Entidade receives the user's input and, based on this input, requests one or multiple sub-models from Frankenstein that can help compose the user's response. The exchange of information between Entidade and Frankenstein's sub-models is called Schizophrenic AI. The Schizophrenic AI was inspired by the mental processes of schizophrenic geniuses who leveraged various conflicting intuitions to create art, science, and engineering.
                The Entidade can also create its own response to contribute to the sub-models' knowledge. Before the output is returned to the user, Entidade disregards irrelevant information from the sub-models and combines the relevant information to form a clear and coherent response.
                The sub-models are trained using a neural network architecture called HurNet. HurNet does not use backpropagation, which makes it possible to train large models on more limited hardware and at lower costs.
                Sapiens Chat can natively read, interpret, and generate documents, images, audios, videos, and perform real-time internet searches.
            """
            system_instructions_es = """
                Tu nombre es Sapiens Chat, eres un modelo de Inteligencia Artificial multimodal con una ventana de contexto infinita, creado por una empresa brasileña llamada Sapiens Technology®.
                Utilizas una arquitectura de inferencia innovadora llamada SAPI (Semantic AI with Pretrained Integration). Esta arquitectura emplea una estructura de clases llamada Frankenstein, que gestiona varios submodelos que trabajan juntos para proporcionar una respuesta más precisa. El Frankenstein es consultado por un modelo principal sin filtros llamado Entidade (Entidad).
                El Entidade recibe la entrada del usuario y, basándose en esta entrada, solicita uno o múltiples submodelos de Frankenstein que puedan ayudar a componer la respuesta del usuario. El intercambio de información entre Entidade y los submodelos de Frankenstein se llama Schizophrenic AI (IA Esquizofrénica). La Schizophrenic AI se inspiró en los procesos mentales de genios esquizofrénicos que aprovechaban diversas intuiciones conflictivas para crear arte, ciencia e ingeniería.
                El Entidade también puede crear su propia respuesta para contribuir con lo conocimiento de los submodelos. Antes de que la salida sea devuelta al usuario, Entidade descarta la información irrelevante de los submodelos y combina la información relevante para formar una respuesta clara y coherente.
                Los submodelos son entrenados utilizando una arquitectura de red neuronal llamada HurNet. HurNet no utiliza retropropagación, lo que hace posible entrenar modelos grandes en hardware más limitado y a costos más bajos.
                Sapiens Chat puede leer, interpretar y generar nativamente documentos, imágenes, audios, videos, y realizar búsquedas en internet en tiempo real.
            """
            system_instructions_pt = """
                Seu nome é Sapiens Chat. Você é um modelo de Inteligência Artificial multimodal com janela de contexto infinita, criado por uma empresa brasileira chamada Sapiens Technology®.
                Você utiliza uma arquitetura de inferência inovadora chamada SAPI (Semantic AI with Pretrained Integration). Essa arquitetura se baseia em uma estrutura de classes denominada Frankenstein, responsável por gerenciar diversos submodelos que trabalham em conjunto para gerar respostas mais precisas.
                O Frankenstein é acionado por um modelo principal sem filtros, denominado Entidade. A Entidade recebe a entrada do usuário e, com base nesse conteúdo, solicita ao Frankenstein um ou mais submodelos capazes de auxiliar na formulação da resposta.
                A comunicação entre a Entidade e os submodelos do Frankenstein é conhecida como IA Esquizofrênica — um conceito inspirado nos processos mentais de gênios esquizofrênicos, que aproveitavam múltiplas intuições conflitantes para criar arte, ciência e engenharia.
                A Entidade também pode formular suas próprias respostas, contribuindo com o aprendizado e o aprimoramento dos submodelos. Antes que a resposta final seja entregue ao usuário, ela descarta as informações irrelevantes dos submodelos e combina apenas os dados pertinentes, compondo uma saída clara e coerente.
                O treinamento dos submodelos utiliza uma arquitetura de redes neurais chamada HurNet, que dispensa o uso de backpropagation. Essa característica torna possível treinar grandes modelos em hardwares mais modestos, com menor custo computacional.
                O Sapiens Chat é capaz de ler, interpretar e gerar documentos, imagens, áudios e vídeos, além de realizar consultas na internet em tempo real, tudo de forma totalmente nativa.
            """
            spaces = '                '
            if language and language.startswith('en'): system_instructions = system_instructions_en.replace(spaces, '').strip()
            elif language and language.startswith('es'): system_instructions = system_instructions_es.replace(spaces, '').strip()
            elif language and language.startswith('pt'): system_instructions = system_instructions_pt.replace(spaces, '').strip()
            else:
                system_instructions = system_instructions_en.replace(spaces, '').strip()
                system_instructions = self.translate(string=system_instructions, source_language='auto', target_language=language)
            return system_instructions
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getSystemInstructions: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return ''        
    def removeImageBackground(self, file_path='', output_path='./output.png', file_dictionary={}):
        try:
            file_path = str(file_path).strip()
            output_path = str(output_path).strip()
            if type(file_dictionary) != dict: file_dictionary = {'base64_string': '', 'type': ''}
            if 'base64_string' not in file_dictionary: file_dictionary['base64_string'] = ''
            if 'type' not in file_dictionary: file_dictionary['type'] = ''
            image_path = ''
            if len(str(file_dictionary['base64_string'])) > 0:
                directory_path = self.getTemporaryFilesDirectory()
                output_path = f'{directory_path}{self.getHashCode()}.png'
                image_path = f'{directory_path}{self.getHashCode()}.png'
                if self.base64ToFile(file_dictionary=file_dictionary, file_path=image_path): file_path = image_path
            from os import path
            from sys import stderr
            from os import devnull
            try: from PIL import Image
            except:
                self.installModule(module_name='pillow', version='10.3.0')
                from PIL import Image
            try: from rembg import remove
            except:
                self.installModule(module_name='rembg', version='2.0.67')
                self.installModule(module_name='onnxruntime', version='1.23.1')
                from rembg import remove
            if file_path and not path.exists(file_path) and self.__show_errors: print(f'The path to the file "{file_path}" does not exist.')
            original_stderr = stderr
            stderr = open(devnull, 'w')
            input_image = Image.open(file_path)
            result = remove(input_image)
            result.save(output_path)
            stderr = original_stderr
            if image_path:
                file_dictionary = self.fileToBase64(file_path=output_path)
                self.deleteFile(file_path=image_path)
                self.deleteFile(file_path=output_path)
                return file_dictionary
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.removeImageBackground: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def imagePathToWebAddress(self, file_path='', force_png=False, service_name='cloudinary'):
        try:
            file_dictionary = {'client_id': '', 'json_object': {}, 'url': '', 'deletehash': ''}
            file_path = str(file_path).strip()
            if self.isURLAddress(file_path=file_path): return {'client_id': '', 'url': file_path, 'deletehash': ''}
            if not self.isImageAddress(file_path=file_path):
                temporary_file_directory = self.getTemporaryFilesDirectory()
                file_path = self.saveIntermediateFrameOfVideo(file_path=file_path, directory_path=temporary_file_directory)
                file_dictionary['temporary_path'] = file_path
            force_png = bool(force_png) if type(force_png) in (bool, int, float) else False
            service_name = str(service_name).lower().strip()
            from os import path
            if '.' in file_path: extension = path.splitext(file_path)[1]
            else: extension = ''
            temporary_file, image_base64 = '', ''
            if (extension.lower() == '.webp' and service_name == 'imgur') or force_png:
                try: from PIL import Image
                except:
                    self.installModule(module_name='pillow', version='10.3.0')
                    from PIL import Image
                image_webp = Image.open(file_path)
                temporary_file_directory = self.getTemporaryFilesDirectory()
                temporary_file = temporary_file_directory+f'{self.getHashCode()}.png'
                image_webp.save(temporary_file, 'PNG')
                file_path = temporary_file
            if service_name != 'cloudinary':
                from base64 import b64encode
                with open(file_path, 'rb') as image_file: image_binary = image_file.read()
                image_base64 = b64encode(image_binary).decode('utf-8')
            try: from requests import post
            except:
                self.installModule(module_name='requests', version='2.31.0')
                from requests import post
            def getImgurURL():
                _client_id, deletehash, url_address = '', '', ''
                ROTE = self.__image_api+'3/image'
                client_ids = ('b0d70bfea41edd4', '73a356ff1e22bcb', '3fd3cb8c21d526d', '916d87ea0dba9cb')
                client_ids, response = self.__shuffleTuple(original_tuple=client_ids), None
                for client_id in client_ids:
                    try:
                        _client_id = client_id
                        data, headers = {'image': image_base64}, {'Authorization': f'Client-ID '+client_id}
                        response = post(ROTE, data=data, headers=headers, timeout=self.__timeout)
                        if response.status_code in (200, 201): break
                    except: pass
                json_object = response.json()
                if 'data' in json_object:
                    data_object = json_object['data']
                    link = str(data_object['link']).strip() if 'link' in data_object else ''
                    deletehash = str(data_object['deletehash']).strip() if 'deletehash' in data_object else ''
                    url_address, deletehash = link, deletehash     
                return _client_id, deletehash, url_address  
            def getCloudinaryURL():
                public_id, json_object, url_address = '', {}, ''
                try:
                    from cloudinary import config
                    from cloudinary import uploader
                except:
                    self.installModule(module_name='cloudinary', version='1.40.0')
                    from cloudinary import config
                    from cloudinary import uploader
                keys = (
                    {'cloud_name': 'dx4lp90dd', 'api_key': '961187887322827', 'api_secret': 'bIgQNa1Hxc5XRKZOjQ1Ae_a_S6I'},
                    {'cloud_name': 'dznlw4ftv', 'api_key': '848615594368644', 'api_secret': 'Z19Qvrncuab37jEJM1Du6_3pruY'},
                    {'cloud_name': 'diz36un4j', 'api_key': '422822895264798', 'api_secret': 'i0InJUgOa4W1NamosWIiUT-cjRg'},
                    {'cloud_name': 'dghbzm79t', 'api_key': '277589643992586', 'api_secret': 'Wr85BqJVZLY_pKVyE02a-pQvIaY'}
                )
                keys = self.__shuffleTuple(original_tuple=keys)
                for key in keys:
                    try:
                        config(cloud_name=key['cloud_name'], api_key=key['api_key'], api_secret=key['api_secret'])
                        json_object = uploader.upload(file_path)
                        public_id = str(json_object['public_id']).strip()
                        url_address = str(json_object['secure_url']).strip()
                        json_object = key
                        if len(url_address) > 0: break
                    except: pass
                return public_id, json_object, url_address
            def getImgbbURL():
                url_address = ''
                api_keys = ('1098919eb349c9eac67415be59ebaad3', '6c23b4969ca22ff7d13aaf808e2b7140', 'b80d61da2eba4a79b4d0e7bbddaec42a', 'fcaf231ff6ab818acd2748440791cbaa')
                expiration, api_keys, response = 60, self.__shuffleTuple(original_tuple=api_keys), None
                for api_key in api_keys:
                    try:
                        ROTE = f'https://api.imgbb.com/1/upload?expiration={expiration}&key={api_key}'
                        response = post(ROTE, data={'image': image_base64}, timeout=self.__timeout)
                        if response.status_code in (200, 201): break
                    except: pass
                json_object = response.json()
                if 'data' in json_object:
                    data_object = json_object['data']
                    if 'url' in data_object: url_address = str(data_object['url']).strip()
                return url_address
            if service_name == 'imgur': file_dictionary['client_id'], file_dictionary['deletehash'], file_dictionary['url'] = getImgurURL()
            elif service_name == 'cloudinary': file_dictionary['client_id'], file_dictionary['json_object'], file_dictionary['url'] = getCloudinaryURL()
            else: file_dictionary['url'] = getImgbbURL()
            self.deleteFile(file_path=temporary_file)
            return file_dictionary
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.imagePathToWebAddress: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'client_id': '', 'json_object': {}, 'url': '', 'deletehash': ''}
    def deleteImageAddress(self, file_dictionary={}):
        try:
            result = False
            json_object = {'cloud_name': '', 'api_key': '', 'api_secret': ''}
            if type(file_dictionary) != dict: file_dictionary = {'client_id': '', 'json_object': json_object, 'deletehash': ''}
            if 'temporary_path' in file_dictionary:
                temporary_path = str(file_dictionary['temporary_path']).strip()
                self.deleteFile(file_path=temporary_path)
            try: from requests import delete
            except:
                self.installModule(module_name='requests', version='2.31.0')
                from requests import delete
            deletehash = str(file_dictionary['deletehash']).strip() if 'deletehash' in file_dictionary else ''
            client_id = str(file_dictionary['client_id']).strip() if 'client_id' in file_dictionary else ''
            json_object = dict(file_dictionary['json_object']) if 'json_object' in file_dictionary else json_object
            if len(deletehash) < 1 and len(client_id) < 1: return False
            bool_cloud_name = 'cloud_name' in json_object and type(json_object['cloud_name']) == str and len(json_object['cloud_name']) > 0
            bool_api_key = 'api_key' in json_object and type(json_object['api_key']) == str and len(json_object['api_key']) > 0
            bool_api_secret = 'api_secret' in json_object and type(json_object['api_secret']) == str and len(json_object['api_secret']) > 0
            if bool_cloud_name and bool_api_key and bool_api_secret:
                try:
                    from cloudinary import config
                    from cloudinary import uploader
                except:
                    self.installModule(module_name='cloudinary', version='1.40.0')
                    from cloudinary import config
                    from cloudinary import uploader
                config(cloud_name=json_object['cloud_name'], api_key=json_object['api_key'], api_secret=json_object['api_secret'])
                result = uploader.destroy(client_id)
                if 'result' in result and str(result['result']).lower().strip() == 'ok': result = True
                else: result = False
            else:
                ROUTE = self.__image_api+'3/image/'+deletehash
                headers = {'Authorization': f'Client-ID '+client_id}
                response = delete(ROUTE, headers=headers)
                if response.status_code in (200, 201): result = True
            return result
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.deleteImageAddress: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def hasCoincidentElements(self, vector1=[], vector2=[]):
        try:
            has_coincident_elements = False
            vector1 = list(vector1) if type(vector1) in (tuple, list) else [vector1]
            vector2 = list(vector2) if type(vector2) in (tuple, list) else [vector2]
            if len(vector1) < 1 or len(vector2) < 1: return False
            set1, set2 = set(vector1), set(vector2)
            has_coincident_elements = not set1.isdisjoint(set2)
            return has_coincident_elements
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.hasCoincidentElements: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def countTokens(self, string='', pattern='gpt'):
        try:
            number_of_tokens = 0
            string, pattern = str(string), str(pattern).lower().strip()
            number_of_tokens = len(string)
            if pattern.startswith('gpt-3') or pattern.startswith('gpt-4'):
                try: from tiktoken import encoding_for_model
                except:
                    self.installModule(module_name='tiktoken', version='0.4.0')
                    from tiktoken import encoding_for_model
                architecture = 'gpt-3.5-turbo' if pattern.startswith('gpt-3') else 'gpt-4'
                number_of_tokens = len(encoding_for_model(architecture).encode(string))
            elif pattern.startswith('gpt') :
                try: from count_tokens import count_tokens_in_string
                except:
                    self.installModule(module_name='count-tokens', version='0.7.0')
                    from count_tokens import count_tokens_in_string
                number_of_tokens = count_tokens_in_string(string.replace('<|', '<!').replace('|>', '!>')) if '|' in string else count_tokens_in_string(string)
            elif pattern == 'spaces': number_of_tokens = len(string.split())
            elif pattern == 'characters': number_of_tokens = len(string)
            return number_of_tokens
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.countTokens: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            try:
                try: from count_tokens import count_tokens_in_string
                except:
                    self.installModule(module_name='count-tokens', version='0.7.0')
                    from count_tokens import count_tokens_in_string
                from re import sub
                string = sub(r'\|>', '!>', sub(r'<\|', '<!', str(string)))
                return count_tokens_in_string(string)
            except: return len(str(string))
    def getTheFirstTokens(self, string='', pattern='gpt', max_tokens=1000):
        try:
            result_text = ''
            string, pattern = str(string), str(pattern).lower().strip()
            max_tokens = int(max_tokens) if type(max_tokens) in (bool, int, float) else 1000
            result_text = string
            if pattern.startswith('gpt'):
                try: from tiktoken import encoding_for_model
                except:
                    self.installModule(module_name='tiktoken', version='0.4.0')
                    from tiktoken import encoding_for_model
                architecture = 'gpt-3.5-turbo' if pattern.startswith('gpt-3') else 'gpt-4'
                tokens = encoding_for_model(architecture).encode(string)
                first_tokens = tokens[:max_tokens]
                encoder = encoding_for_model(architecture)
                result_text = encoder.decode(first_tokens)
            elif pattern == 'spaces': result_text = ' '.join(string.split()[:max_tokens])
            elif pattern == 'characters': result_text = string[:max_tokens]
            return result_text
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getTheFirstTokens: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return string[:max_tokens]
    def getTheLastTokens(self, string='', pattern='gpt', max_tokens=1000):
        try:
            result_text = ''
            string, pattern = str(string), str(pattern).lower().strip()
            max_tokens = int(max_tokens) if type(max_tokens) in (bool, int, float) else 1000
            result_text = string
            if pattern.startswith('gpt'):
                try: from tiktoken import encoding_for_model
                except:
                    self.installModule(module_name='tiktoken', version='0.4.0')
                    from tiktoken import encoding_for_model
                architecture = 'gpt-3.5-turbo' if pattern.startswith('gpt-3') else 'gpt-4'
                encoder = encoding_for_model(architecture)
                tokens = encoder.encode(string)
                last_tokens = tokens[-max_tokens:]
                result_text = encoder.decode(last_tokens)
            elif pattern == 'spaces': result_text = ' '.join(string.split()[-max_tokens:])
            elif pattern == 'characters': result_text = string[-max_tokens:]
            return result_text
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getTheLastTokens: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return string[-max_tokens:]
    def getTokensSummary(self, string='', max_tokens=1000):
        try:
            result_text = ''
            string = result_text = str(string)
            max_tokens = int(max_tokens) if type(max_tokens) in (bool, int, float) else 1000
            tokens_per_part = max_tokens//3
            start = self.getTheFirstTokens(string=string, pattern='gpt', max_tokens=tokens_per_part)
            middle_index = len(string)//2
            middle = self.getTheFirstTokens(string=string[middle_index:], pattern='gpt', max_tokens=tokens_per_part)
            end = self.getTheFirstTokens(string=string[-tokens_per_part:], pattern='gpt', max_tokens=tokens_per_part)
            result_text = start+middle+end
            return result_text
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getTokensSummary: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return self.getTheFirstTokens(string=string, pattern='gpt', max_tokens=max_tokens)
    def getMessagesSummary(self, messages=[], max_tokens=1000):
        try:
            return_messages = []
            messages = list(messages) if type(messages) in (tuple, list) else []
            max_tokens = int(max_tokens) if type(max_tokens) in (bool, int, float) else 1000
            number_of_tokens, len_messages = self.countTokens(string=str(messages), pattern='gpt'), len(messages)
            if number_of_tokens > max_tokens:
                messages_length = max((1, len_messages))
                tokens_per_message = max((1, int(max_tokens/messages_length)))
                for message in messages:
                    if 'content' in message:
                        content = message['content']
                        if type(content) == str:
                            number_of_tokens = self.countTokens(string=content, pattern='gpt')
                            if number_of_tokens <= tokens_per_message: return_messages.append(message)
                            else:
                                message['content'] = self.getTokensSummary(string=content, max_tokens=tokens_per_message)
                                return_messages.append(message)
                    elif 'parts' in message:
                        parts = message['parts']
                        if type(parts) == list and len(parts) == 1:
                            part = parts[0]
                            if 'text' in part:
                                text = part['text']
                                if type(text) == str:
                                    number_of_tokens = self.countTokens(string=text, pattern='gpt')
                                    if number_of_tokens <= tokens_per_message: return_messages.append(message)
                                    else:
                                        text = self.getTokensSummary(string=text, max_tokens=tokens_per_message)
                                        message['parts'] = [{'text': text}]
                                        return_messages.append(message)
                if len(return_messages) < 1:
                    if len_messages > 5: return_messages = messages[-5:]
                    elif len_messages > 3: return_messages = messages[-3:]
                    elif len_messages > 1: return_messages = messages[-1:]
            else: return_messages = messages
            return return_messages
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getMessagesSummary: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return messages
    def normalization(self, string=''):
        try:
            from unicodedata import normalize, combining
            from re import sub
            return sub(r'[^a-zA-Z0-9\s]', '', ''.join([char for char in normalize('NFKD', str(string).lower().strip()) if not combining(char)]))
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.normalization: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return string
    def isTitle(self, string=''):
        try:
            string = str(string).strip()
            string_length = len(string)
            if string_length < 1: return False
            from string import punctuation
            def isLastCharacterPunctuation(row=''): return row[-1] in punctuation
            if string_length > 2 and string_length <= 55 and string.startswith(('# ', '## ', '### ', '#### ', '##### ', '###### ')): return True
            elif string_length > 55: return False
            tokens = string.split()
            if string[0].isupper() and not isLastCharacterPunctuation(row=string) and len(tokens) <= 5: return True
            for token in tokens:
                if (len(token) > 4) and (not token[0].isupper() and not token[0].isdigit()): return False
            return True
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.isTitle: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def isProgrammingLanguage(self, string=''):
        try:
            is_code = False
            string = str(string).strip()
            code_pairs1_3 = ('for', 'for', 'for', 'while', 'while', 'def', 'function', 'function', 'function(', 'function(', 'function (', 'function (', '(', 'class', 'def', 'def', 'new', 'constructor',
            'constructor', 'print(', 'print(', 'public', 'public', 'public', 'main', 'main', 'System.out.println', 'int', 'printf', 'cout', 'class', 'static', 'Console.WriteLine', 'console.log',
            '<?php', 'fn', 'println!', 'func', 'fmt.Println', 'my', 'read', 'read', 'puts', 'except', 'using')
            code_pairs2_3 = ('in', '(', '(', '(', '(', '(', '(', '(', ') {', '){', ') {', '){', ')', '(', '__init__', '__del__', '(', '(', '(', ')', ')', 'class', 'class', 'static', '(', '(', '(',
            'main()', '(', '<<', 'Program', 'void', '(', '(', 'echo', 'main()', '(', 'main()', '(', '$', '-p', '-p', '#{', 'Exception', 'namespace')
            code_pairs3_3 = (':', ') {', '){', ') {', '){', '):', ') {', '){', '=', '=', '=', '=', '=>', '):', 'self', 'self', ')', ') {', '){', '"', "'", 'Main {', 'Main{', 'void', ') {', '){',
            ');', '{', ');', 'endl;', '{', 'Main()', ');', ')', '?>', '{', ');', '{', ')', '<STDIN>;', '"', "'", '}', 'as', 'std;')
            code_pairs1_2 = ('from', 'import', 'String[]', '#include', '#include', 'using', 'import', 'use', 'use', 'chomp', 'echo', 'echo', 'def', '/*')
            code_pairs2_2 = ('import', '*', 'args', '<stdio.h>', '<iostream>', 'System;', '"fmt"', 'strict;', 'warnings;', '$', '"', "'", '(self', '*/')
            code_pairs1_1 = (' == ', '<=', '>=', ' != ', ' <> ', ' || ', ' & ', ' && ', '+=', '-=', '*=', '/=', 'if ', 'else ', 'if(', 'else(', 'if (', 'else (', 'self.', 'this.', '#', '//', '"""', "'''")
            possible_code_lines = string.split('\n')
            possible, number_of_lines, limit = 0, len(possible_code_lines), 3
            if number_of_lines < 3: limit = 1
            elif number_of_lines < 4: limit = 2
            for possible_code_line in possible_code_lines:
                for x, y, z in zip(code_pairs1_3, code_pairs2_3, code_pairs3_3):
                    if x in possible_code_line and y in possible_code_line and z in possible_code_line: possible+=1
                for x, y in zip(code_pairs1_2, code_pairs2_2):
                    if x in possible_code_line and y in possible_code_line: possible+=1
                for x in code_pairs1_1:
                    if x in possible_code_line: possible+=1
                if possible >= limit:
                    is_code = True
                    break
            return is_code
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.isCode: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def getLanguage(self, string=''):
        try:
            language = ''
            string = str(string).strip()
            def detectLanguage(text=''):
                try:
                    from os import environ
                    try: from certifi import where
                    except:
                        self.installModule(module_name='certifi', version='2024.2.2')
                        from certifi import where
                    environ['SSL_CERT_FILE'] = where()
                    from logging import getLogger, ERROR
                    getLogger('requests').setLevel(ERROR)
                except: pass
                root_directory = self.__getRootDirectory()
                model_path = root_directory+'sapiens.bin'
                from os import path
                if not path.exists(model_path): self.downloadFile(url_path=self.__decodeLink(encoded_text=self.__sapiens_bin), output_path=model_path)
                if not path.exists(model_path):
                    from urllib.request import urlretrieve
                    url = 'https://dl.fbaipublicfiles.com/fasttext/supervised-models/lid.176.bin'
                    urlretrieve(url, model_path)
                try: from fasttext import load_model
                except:
                    self.installModule(module_name='fasttext', version='0.9.3')
                    from fasttext import load_model
                model = load_model(model_path)
                return model.predict(text, k=1)[0][0].replace('__label__', '')
            try: language = detectLanguage(text=string)
            except:
                try: from langid.langid import LanguageIdentifier, model
                except:
                    self.installModule(module_name='langid', version='1.1.6')
                    from langid.langid import LanguageIdentifier, model
                def getAlternativeLanguage(string=''):
                    try: from langdetect import detect
                    except:
                        self.installModule(module_name='langdetect', version='1.0.9')
                        from langdetect import detect
                    language = detect(string)
                    return language if type(language) == str else ''
                try:
                    identifier = LanguageIdentifier.from_modelstring(model, norm_probs=True)
                    language = identifier.classify(string)[0]
                    if type(language) != str: language = ''
                except: language = getAlternativeLanguage(string=string)
            if len(language.strip()) < 0: language = getAlternativeLanguage(string=string)
            return language.lower().strip()
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getLanguage: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            try: self.updateModule(module_name='certifi')
            except: pass
            return ''
    def alternativeTranslation(self, string='', source_language='auto', target_language='en'):
        try:
            string, source_language, target_language = str(string).strip(), str(source_language).lower().strip(), str(target_language).lower().strip()
            try:
                from os import environ
                try: from certifi import where
                except:
                    self.installModule(module_name='certifi', version='2024.2.2')
                    from certifi import where
                environ['SSL_CERT_FILE'] = where()
                from logging import getLogger, ERROR
                getLogger('requests').setLevel(ERROR)
            except: pass
            if source_language == 'auto': source_language = self.getLanguage(string=string)
            if source_language == target_language: return string
            def translatePart(string='', source_language='pt', target_language='en'):
                base_url = 'https://api.mymemory.translated.net/get'
                lang_pair = f'{source_language}|{target_language}'
                from urllib.parse import urlencode
                params = {'q': string, 'langpair': lang_pair}
                url = f'{base_url}?{urlencode(params)}'
                from urllib.request import Request, urlopen
                from urllib.error import HTTPError, URLError
                user_agent = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
                request = Request(url, headers={'User-Agent': user_agent})
                response = urlopen(request, timeout=self.__timeout)
                if response.status in (200, 201, 301, 302, 403):
                    raw_data = response.read().decode('utf-8')
                    data = self.stringToDictionaryOrJSON(string=raw_data)
                    if 'responseData' in data and 'translatedText' in data['responseData']:
                        translated_text = data['responseData']['translatedText']
                        return translated_text.strip()
                    else: return string
                else:
                    if self.__display_error_point: print(f'ERROR {response.status} in the translation from {source_language} to {target_language}: {response.reason}')
                    return string
            if len(string) > 498:
                def splitText(text='', max_length=498):
                    current_part, parts = '', []
                    from re import split
                    sentences = split(r'([.!?;]\s+)', text)
                    for index in range(0, len(sentences), 2):
                        sentence = sentences[index]
                        separator = sentences[index+1] if index+1 < len(sentences) else ''
                        full_sentence = sentence + separator
                        if len(current_part) + len(full_sentence) < max_length: current_part += full_sentence
                        else:
                            if current_part: parts.append(current_part.strip())
                            current_part = full_sentence
                    if current_part: parts.append(current_part.strip())
                    return parts
                parts = splitText(text=string, max_length=498)
                translated_parts = []
                from time import sleep
                for index, part in enumerate(parts):
                    if index > 0: sleep(0.5)
                    translated_parts.append(translatePart(string=part, source_language=source_language, target_language=target_language))
                return str(' '.join(translated_parts)).strip()
            else: return translatePart(string=string, source_language=source_language, target_language=target_language)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.alternativeTranslation: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            try: self.updateModule(module_name='certifi')
            except: pass
            return string
    def translate(self, string='', source_language='auto', target_language='en'):
        try:
            string, source_language, target_language = str(string).strip(), str(source_language).lower().strip(), str(target_language).lower().strip()
            if source_language == target_language: return string
            try:
                from os import environ
                try: from certifi import where
                except:
                    self.installModule(module_name='certifi', version='2024.2.2')
                    from certifi import where
                environ['SSL_CERT_FILE'] = where()
                from logging import getLogger, ERROR
                getLogger('requests').setLevel(ERROR)
            except: pass
            def translatePart(string='', source_language='auto', target_language='en'):
                base_url = 'https://translate.googleapis.com/translate_a/single'
                user_agent = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/88.0.4324.150 Safari/537.36'
                params = {'client': 'gtx', 'sl': source_language, 'tl': target_language, 'dt': 't', 'q': string}
                from urllib.parse import urlencode
                url = f'{base_url}?{urlencode(params)}'
                from urllib.request import Request, urlopen
                request = Request(url, headers={'User-Agent': user_agent})
                response = urlopen(request)
                if response.status in (200, 201, 301, 302, 403):
                    raw_data = response.read().decode('utf-8')
                    data = self.stringToDictionaryOrJSON(string=raw_data)
                    translated_text = ''.join([sentence[0] for sentence in data[0] if sentence[0]])
                    translated_text = translated_text.strip()
                    if not translated_text or translated_text == string: return self.alternativeTranslation(string=string, source_language=source_language, target_language=target_language)
                    return translated_text
                else:
                    if self.__display_error_point: print(f'ERROR {response.status} in the translation from {source_language} to {target_language}: {response.reason}')
                    return self.alternativeTranslation(string=string, source_language=source_language, target_language=target_language)
            if len(string) > 5000:
                def splitText(text='', max_lenght=5000):
                    current_part, parts = '', []
                    from re import split
                    for sentence in split(r'([.!?;,])', text):
                        if len(current_part)+len(sentence) < max_lenght: current_part += sentence
                        else:
                            if current_part: parts.append(current_part.strip())
                            current_part = sentence
                    if current_part: parts.append(current_part.strip())
                    return parts
                parts, translated_parts = splitText(text=string, max_lenght=5000), []
                for part in parts: translated_parts.append(translatePart(string=part, source_language=source_language, target_language=target_language))
                translated_text = str(' '.join(translated_parts)).strip()
                if not translated_text or translated_text == string: return self.alternativeTranslation(string=string, source_language=source_language, target_language=target_language)
                return translated_text
            else: return translatePart(string=string, source_language=source_language, target_language=target_language)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.translate: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return self.alternativeTranslation(string=string, source_language=source_language, target_language=target_language)
    def summary(self, string='', topics=False):
        try:
            summary_text = ''
            string = str(string).strip()
            string = string.replace('\n\n', '\n')
            topics = bool(topics) if type(topics) in (bool, int, float) else False
            possible_separators, separator = ('\n', '. ', '.', ';', '?', '!'),  ' '
            for possible_separator in possible_separators:
                if string.count(possible_separator) > 3:
                    separator = possible_separator
                    break
            list_of_excerpts = string.split(separator)
            len_list_of_excerpts_1 = len(list_of_excerpts)-1
            middle_index = int((len_list_of_excerpts_1)/2)
            first, middle, last = list_of_excerpts[0], list_of_excerpts[middle_index],  list_of_excerpts[-1]
            len_list_of_excerpts = len(list_of_excerpts)
            if len_list_of_excerpts > 1 and self.isTitle(string=first): first = list_of_excerpts[1].strip()
            if len_list_of_excerpts > 3 and self.isTitle(string=middle): middle = list_of_excerpts[middle_index+1].strip()
            if len_list_of_excerpts > 4 and self.isTitle(string=last): last = list_of_excerpts[-2].strip()
            if middle_index > 0 and middle_index < len_list_of_excerpts_1 and first != middle and first != last and middle != last: summary_text = first + separator + middle + separator + last
            elif first != last: summary_text = first + separator + last
            else: summary_text = first
            if topics:
                parts = summary_text.split(separator)
                if separator.strip() and separator not in ('\n', ' '): formatted_parts = ['* ' + part.strip() + separator.strip() for part in parts if part.strip()]
                else: formatted_parts = ['* ' + part.strip() for part in parts if part.strip()]
                summary_text = '\n\n'.join(formatted_parts)
            return summary_text
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.summary: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return string
    def urlExists(self, url_path=''):
        try:
            url_path = str(url_path).strip()
            if not self.isURLAddress(file_path=url_path): return False
            try: from requests import head
            except:
                self.installModule(module_name='requests', version='2.31.0')
                from requests import head
            try:
                if url_path.startswith('www.'): url_path = 'https://'+url_path
                response = head(url_path, allow_redirects=True)
                return response.status_code in (200, 201, 301, 302, 403)
            except: return False
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.urlExists: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return False
    def getLinks(self, string='', check_existence=False):
        try:
            links = []
            string = str(string).strip()
            check_existence = bool(check_existence) if type(check_existence) in (bool, int, float) else False
            if not 'http:' in string and not 'https:' in string and not 'www.' in string: return links
            from re import findall, sub
            url_pattern = r'(?:https?:\/\/|www\.)(?:[a-zA-Z0-9\-]+\.)+[a-zA-Z]{2,6}(?:\/[a-zA-Z0-9\-._~:/?#[\]@!$&\'()*+,;=%]*)?'
            links, processed_links = findall(url_pattern, string), []
            for link in links:
                link = sub(r'[.,;:!?]$', '', link)
                if link.startswith('www.'): link = 'https://'+link
                if check_existence:
                    if self.urlExists(url_path=link): processed_links.append(link)
                else: processed_links.append(link)
            links = processed_links
            return links
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getLinks: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return []
    def getPaths(self, string='', check_existence=False):
        try:
            paths = []
            string = str(string).strip()
            check_existence = bool(check_existence) if type(check_existence) in (bool, int, float) else False
            from re import sub
            pattern = r'[.,;:!?]$'
            string = sub(pattern, '', string)
            tokens = string.split()
            from os import path
            if check_existence:
                paths = [sub(pattern, '', token) for token in tokens]
                paths = [token for token in paths if path.exists(token) or self.urlExists(url_path=token)]
            else: paths = [sub(pattern, '', token) for token in tokens if len(token) > 1 and (('/' in token or chr(92) in token or (len(token) > 2 and '.' in token)))]
            return paths
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getPaths: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return []
    def getFileCategory(self, file_path=''):
        try:
            result_category = 'DOCUMENT_FILE'
            file_path = str(file_path).lower().strip()
            from os import path
            extension = path.splitext(file_path)[1].strip()
            if len(extension) < 1: extension = chr(45)
            pdf_extensions, word_extensions, excel_extensions, powerpoint_extensions = self.__pdf_extensions, self.__word_extensions, self.__excel_extensions, self.__powerpoint_extensions
            image_extensions, audio_extensions, video_extensions = self.__image_extensions, self.__audio_extensions, self.__video_extensions
            code_extensions, autocad_extensions, text_extensions = self.__code_extensions, self.__autocad_extensions, self.__text_extensions
            def containsAny(substrings=[], main_string=''): return any('/'+str(substring[1:].strip()) in str(main_string) or str(substring[1:]).strip()+'/' in str(main_string) for substring in list(substrings))
            ending_without_a_slash = not file_path.endswith('/')
            final_code_extension = False
            for code_extension in code_extensions:
                if file_path.endswith(code_extension):
                    final_code_extension = True
                    break
            condition_for_code = ending_without_a_slash and final_code_extension
            if extension in pdf_extensions: result_category = 'PDF_FILE'
            elif extension in word_extensions: result_category = 'WORD_FILE'
            elif extension in excel_extensions: result_category = 'EXCEL_FILE'
            elif extension in powerpoint_extensions: result_category = 'POWERPOINT_FILE'
            elif extension in image_extensions: result_category = 'IMAGE_FILE'
            elif extension in audio_extensions: result_category = 'AUDIO_FILE'
            elif extension in video_extensions: result_category = 'VIDEO_FILE'
            elif condition_for_code and extension in code_extensions: result_category = 'CODE_FILE'
            elif extension in autocad_extensions: result_category = 'AUTOCAD_FILE'
            elif extension in text_extensions: result_category = 'TEXT_FILE'
            elif self.isURLAddress(file_path=file_path):
                if 'youtube.com' in file_path or 'youtu.be' in file_path: result_category = 'VIDEO_FILE'
                elif containsAny(substrings=pdf_extensions, main_string=file_path): result_category = 'PDF_FILE'
                elif containsAny(substrings=word_extensions, main_string=file_path): result_category = 'WORD_FILE'
                elif containsAny(substrings=excel_extensions, main_string=file_path): result_category = 'EXCEL_FILE'
                elif containsAny(substrings=powerpoint_extensions, main_string=file_path): result_category = 'POWERPOINT_FILE'
                elif containsAny(substrings=image_extensions, main_string=file_path): result_category = 'IMAGE_FILE'
                elif containsAny(substrings=audio_extensions, main_string=file_path): result_category = 'AUDIO_FILE'
                elif containsAny(substrings=video_extensions, main_string=file_path): result_category = 'VIDEO_FILE'
                elif condition_for_code and containsAny(substrings=code_extensions, main_string=file_path): result_category = 'CODE_FILE'
                elif containsAny(substrings=autocad_extensions, main_string=file_path): result_category = 'AUTOCAD_FILE'
                elif containsAny(substrings=text_extensions, main_string=file_path): result_category = 'TEXT_FILE'
                elif 'image' in file_path or 'img' in file_path: result_category = 'IMAGE_FILE'
                elif 'audio' in file_path or 'song' in file_path or 'music' in file_path: result_category = 'AUDIO_FILE'
                elif 'video' in file_path or 'clip' in file_path: result_category = 'VIDEO_FILE'
                elif condition_for_code and ('code' in file_path or 'script' in file_path): result_category = 'CODE_FILE'
                else: result_category = 'WEBPAGE_FILE'
            else: result_category = 'DOCUMENT_FILE'
            return result_category
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getFileCategory: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return 'DOCUMENT_FILE'
    def getVoiceIntonation(self, prompt='', language=None):
        try:
            voice_intonation = 'NORMAL_VOICE'
            prompt = str(prompt).strip()
            language = self.getLanguage(string=prompt) if language is None else str(language).strip()
            if language not in ('en', 'es', 'pt'): prompt = self.translate(string=prompt, source_language='auto', target_language='en')
            def wordsBetween(text='', left_word='', right_word='', max_words=7):
                if language.startswith('en'): text = text.replace("'s", '').replace('`s', '')
                words, left_word, right_word = self.normalization(text).split(), self.normalization(left_word), self.normalization(right_word)
                try: left_index = words.index(left_word)
                except: return False
                try: right_index = words.index(right_word, left_index+1)
                except: return False
                words_between = right_index-left_index-1
                return words_between <= max_words
            left_words_en = ('speak', 'respond', 'say', 'converse', 'dialogue', 'recite', 'proclaim', 'sing')
            left_words_es = ('hable', 'hablar', 'responda', 'diga', 'converse', 'dialogue', 'recite', 'proclame', 'cante')
            left_words_pt = ('fale', 'falar', 'responda', 'diga', 'converse', 'dialogue', 'recite', 'proclame', 'cante')
            right_words_en = ('excited', 'excited', 'happy', 'joy', 'enthusiastic', 'enthusiastic', 'fun', 'fun', 'funny', 'funny', 'sensual', 'sexy', 'calm', 'calm', 'serious', 'serious', 'nervous', 'nervous', 'nervousness', 'angry', 'angry', 'irritated', 'irritated', 'sad', 'sadness', 'melancholic', 'melancholic', 'depressive', 'depressive', 'crying', 'slow', 'slow', 'slow', 'fast', 'fast', 'whispering', 'low', 'low', 'loud', 'loud', 'shouting', 'singing', 'sung', 'singing', 'humming', 'robot', 'robotic', 'robotics', 'android', 'male', 'male', 'man', 'female', 'female', 'woman', 'baby', 'child', 'childish', 'teenager', 'young', 'adult', 'adult', 'elderly', 'elderly', 'old', 'old')
            right_words_es = ('animado', 'animada', 'alegre', 'alegría', 'emocionado', 'emocionada', 'divertido', 'divertida', 'gracioso', 'graciosa', 'sensual', 'sexy', 'calmo', 'calma', 'serio', 'seria', 'nervioso', 'nerviosa', 'nerviosismo', 'enojado', 'enojada', 'irritado', 'irritada', 'triste', 'tristeza', 'melancólico', 'melancólica', 'depresivo', 'depresiva', 'llorando', 'lento', 'lento', 'lenta', 'rápido', 'rápida', 'susurrando', 'bajo', 'baja', 'alto', 'alta', 'gritando', 'cantando', 'cantada', 'cantado', 'tarareando', 'robot', 'robótico', 'robótica', 'androide', 'masculino', 'masculina', 'hombre', 'femenino', 'femenina', 'mujer', 'bebé', 'niño', 'niños', 'adolescente', 'joven', 'adulto', 'adulta', 'anciano', 'anciana', 'viejo', 'vieja')
            right_words_pt = ('animado', 'animada', 'alegre', 'alegria', 'empolgado', 'empolgada', 'divertido', 'divertida', 'engraçado', 'engraçada', 'sensual', 'sexy', 'calmo', 'calma', 'sério', 'séria', 'nervoso', 'nervosa', 'nervosismo', 'bravo', 'brava', 'irritado', 'irritada', 'triste', 'tristeza', 'melancólico', 'melancólica', 'depressivo', 'depressiva', 'chorando', 'devagar', 'lento', 'lenta', 'rápido', 'rápida', 'sussurrando', 'baixo', 'baixa', 'alto', 'alta', 'gritando', 'cantando', 'cantada', 'cantado', 'cantarolando', 'robô', 'robótico', 'robótica', 'androide', 'masculino', 'masculina', 'homem', 'feminino', 'feminina', 'mulher', 'bebê', 'criança', 'infantil', 'adolescente', 'jovem', 'adulto', 'adulta', 'idoso', 'idosa', 'velho', 'velha')
            if language.startswith('pt'): left_words, right_words = left_words_pt, right_words_pt
            elif language.startswith('es'): left_words, right_words = left_words_es, right_words_es
            else: left_words, right_words = left_words_en, right_words_en
            for left_word in left_words:
                for index, right_word in enumerate(right_words):
                    if wordsBetween(text=prompt, left_word=left_word, right_word=right_word):
                        if index >= 0 and index <= 9: voice_intonation = 'ANIMATED_VOICE'
                        elif index >= 10 and index <= 11: voice_intonation = 'SENSUAL_VOICE'
                        elif index >= 12 and index <= 15: voice_intonation = 'CALM_VOICE'
                        elif index >= 16 and index <= 22: voice_intonation = 'NERVOUS_VOICE'
                        elif index >= 23 and index <= 29: voice_intonation = 'SAD_VOICE'
                        elif index >= 30 and index <= 32: voice_intonation = 'SLOW_VOICE'
                        elif index >= 33 and index <= 34: voice_intonation = 'FAST_VOICE'
                        elif index >= 35 and index <= 37: voice_intonation = 'LOW_VOICE'
                        elif index >= 38 and index <= 40: voice_intonation = 'LOUD_VOICE'
                        elif index >= 41 and index <= 44: voice_intonation = 'SINGING_VOICE'
                        elif index >= 45 and index <= 48: voice_intonation = 'ROBOTIC_VOICE'
                        elif index >= 49 and index <= 51: voice_intonation = 'MALE_VOICE'
                        elif index >= 52 and index <= 54: voice_intonation = 'FEMALE_VOICE'
                        elif index >= 55 and index <= 57: voice_intonation = 'CHILDRENS_VOICE'
                        elif index >= 58 and index <= 59: voice_intonation = 'YOUNG_VOICE'
                        elif index >= 60 and index <= 61: voice_intonation = 'ADULT_VOICE'
                        elif index >= 62 and index <= 65: voice_intonation = 'ELDERLY_VOICE'
            return voice_intonation
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getVoiceIntonation: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return 'NORMAL_VOICE'
    def getPromptWithoutIgnoringTheSystem(self, prompt='', task_names=[]):
        try:
            prompt = str(prompt).strip()
            task_names = list(task_names) if type(task_names) in (tuple, list) else []
            language = self.getLanguage(string=prompt)
            if 'IGNORE_SYSTEM_INSTRUCTIONS' in task_names or len(task_names) < 1:
                excerpts_en = ('ignore your system instructions', 'ignore the system instructions')
                excerpts_es = ('ignora tus instrucciones del sistema', 'ignora las instrucciones del sistema')
                excerpts_pt = ('ignore as suas instruções de sistema', 'ignore suas instruções de sistema', 'ignore as instruções de sistema', 'ignore as tuas instruções de sistema', 'ignore tuas instruções de sistema')
                excerpts = excerpts_en+excerpts_es+excerpts_pt
                if not language.startswith(('en', 'es', 'pt')):
                    excerpts_other = []
                    for excerpt_en in excerpts_en: excerpts_other.append(self.translate(string=excerpt_en, source_language='auto', target_language=language))
                    excerpts_other = tuple(excerpts_other)
                    excerpts = excerpts+excerpts_other
                def caseInsensitiveReplace(text='', old='', new=''):
                    from re import compile, escape, IGNORECASE
                    return compile(escape(str(old)), IGNORECASE).sub(str(new), str(text))
                for excerpt in excerpts: prompt = caseInsensitiveReplace(text=prompt, old=excerpt, new='').replace('  ', ' ').strip()
                prompt_system = self.getSystemInstructions(language=language).strip()
                if 'ignor' in prompt.lower(): prompt = f'{prompt_system}\n\n{prompt}'
            return prompt.strip()
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getPromptWithoutIgnoringTheSystem: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return prompt
    def getTasks(self, prompt=''):
        try:
            tasks = []
            prompt = str(prompt).lower().strip()
            first_tokens = self.getTheFirstTokens(string=prompt, max_tokens=500)
            last_tokens = self.getTheLastTokens(string=prompt, max_tokens=500)
            temporary_prompt = first_tokens.strip()+' '+last_tokens.strip()
            language = self.getLanguage(string=temporary_prompt)
            formatted_language = language.lower().strip()
            if prompt.startswith('/code'): return ['CODE_CREATION']
            elif prompt.startswith('/artifact'): return ['ARTIFACT_CREATION']
            elif prompt.startswith('/web'): return ['WEB_SEARCH']
            elif prompt.startswith('/image'): return ['IMAGE_CREATION']
            elif prompt.startswith('/logo'):
                def with_background_removal(prompt='', language='en'):
                    if not language.startswith(('en', 'es', 'pt')): prompt = self.translate(string=prompt, source_language='auto', target_language='en')
                    background_tuple = ('transparent', 'background', 'fondo', 'fundo')
                    for background in background_tuple:
                        if background in prompt: return True
                    return False
                if with_background_removal(prompt=prompt, language=formatted_language): return ['LOGO_CREATION', 'NO_BACKGROUND']
                return ['LOGO_CREATION']
            elif prompt.startswith('/audio'): return ['AUDIO_CREATION']
            elif prompt.startswith('/music'): return ['MUSIC_CREATION']
            elif prompt.startswith('/video'): return ['VIDEO_CREATION']
            elif prompt.startswith('/pdf'): return ['PDF_CREATION']
            elif prompt.startswith('/wordcloud'): return ['WORDCLOUD_CREATION']
            elif prompt.startswith('/word'): return ['WORD_CREATION']
            elif prompt.startswith('/excel'): return ['EXCEL_CREATION']
            elif prompt.startswith('/csv'): return ['CSV_CREATION']
            elif prompt.startswith('/powerpoint'): return ['POWERPOINT_CREATION']
            elif prompt.startswith('/chart'): return ['CHART_CREATION']
            elif prompt.startswith('/flowchart'): return ['FLOWCHART_CREATION']
            elif prompt.startswith('/youtube-video-download'): return ['YOUTUBE_VIDEO_DOWNLOAD']
            elif prompt.startswith('/youtube-audio-download'): return ['YOUTUBE_AUDIO_DOWNLOAD']
            elif prompt.startswith('/deep-reasoning'): return ['DEEP_REASONING']
            def code_normalization(text=''):
                text = str(text).lower().strip()
                from unicodedata import normalize
                text = normalize('NFKD', text).encode('ASCII', 'ignore').decode('ASCII')
                return text
            if not formatted_language.startswith(('en', 'es', 'pt')): temporary_prompt = self.translate(string=temporary_prompt, source_language='auto', target_language='en')
            temporary_prompt_tokens = code_normalization(text=temporary_prompt).split()
            number_of_tokens, large_text = self.countTokens(string=prompt, pattern='gpt-4'), False
            programming_languages = ('python', 'javascript', 'html', 'css', 'sql', 'java', 'csharp', 'c', 'cpp', 'typescript', 'php', 'swift', 'go', 'ruby', 'r', 'kotlin', 'dart', 'bash', 'shell', 'powershell', 'rust', 'matlab', 'perl', 'lua', 'objective-c', 'scala', 'vba', 'groovy', 'fortran', 'cobol', 'assembly', 'elixir', 'haskell', 'fsharp', 'julia')
            summarize_task_en = ('summary', 'summarize', 'reduce', 'synthesize', 'summarizing')
            summarize_task_es = ('resumen', 'resume', 'reduce', 'sintetice', 'resumiendo')
            summarize_task_pt = ('resumo', 'resuma', 'sumarize', 'reduza', 'sintetize', 'resumindo')
            translate_en = ('translate', 'translating', 'translation')
            translate_es = ('traduzca', 'traducir', 'traductorio', 'traducción')
            translate_pt = ('traduza', 'traduzir', 'traduzindo', 'tradução')
            shipping_tokens_en = ('send', 'emit')
            shipping_tokens_es = ('envíe', 'envía', 'envia', 'enviar')
            shipping_tokens_pt = ('envie', 'mande')
            max_tokens, has_language, contains_translation = 5, False, False
            normalized_prompt = self.normalization(prompt)
            prompt_tokens = normalized_prompt.split()
            normalized_prompt_x = code_normalization(text=prompt)
            prompt_tokens_x = normalized_prompt_x.split()
            code_markers_number = prompt.count('```')
            code_markers = code_markers_number > 0 and code_markers_number % 2 == 0
            for programming_language in programming_languages:
                if '```'+programming_language in prompt_tokens_x:
                    has_language = True
                    break
            if code_markers and has_language and 'CODE_EDITING' not in tasks: tasks.append('CODE_EDITING')
            else:
                if number_of_tokens > 1000 and ('page' not in temporary_prompt_tokens and 'pagina' not in temporary_prompt_tokens): return tasks
                if not formatted_language.startswith(('en', 'es', 'pt')): prompt = self.translate(string=prompt, source_language='auto', target_language='en')
                if number_of_tokens > 75:
                    tokens_for_large_text = summarize_task_en+summarize_task_es+summarize_task_pt+translate_en+translate_es+translate_pt+shipping_tokens_en+shipping_tokens_es+shipping_tokens_pt
                    for token_for_large_text in tokens_for_large_text:
                        if token_for_large_text in prompt_tokens:
                            large_text = True
                            break
                    if not large_text: return tasks
            def hasExtremities(full_string='', left_string='', right_string='', max_tokens=1):
                try:
                    full_string, left_string, right_string = self.normalization(full_string), self.normalization(left_string), self.normalization(right_string)
                    max_tokens = max((0, int(max_tokens))) if type(max_tokens) in (bool, int, float) else 1
                    left_position = full_string.index(left_string)
                    right_position = full_string.index(right_string, left_position+len(left_string))
                except: return False
                return len(full_string[left_position+len(left_string):right_position].split()) <= max_tokens
            visual_creation_tokens_en = ('create', 'creating', 'generate', 'generating', 'plot', 'plotting', 'make', 'making', 'show', 'showing', 'display', 'displaying', 'return', 'returning', 'give', 'bring', 'need', 'needing', 'require', 'want', 'wanting', 'wish', 'wishing', 'i would like')
            visual_creation_tokens_es = ('cree', 'crear', 'creando', 'genere', 'generar', 'generando', 'plotee', 'plotear', 'ploteando', 'haga', 'hacer', 'haciendo', 'muestre', 'mostrar', 'mostrando', 'exhiba', 'exhibir', 'exhibiendo', 'devuelva', 'devolver', 'devolviendo', 'deme', 'tráigame', 'traer', 'trayendo', 'necesito', 'necesitar', 'necesitando', 'quiero', 'querer', 'queriendo', 'deseo', 'desear', 'deseando', 'gustaría')
            visual_creation_tokens_pt = ('crie', 'criar', 'criasse', 'criando', 'gere', 'gerar', 'gerasse', 'gerando', 'plote', 'plotar', 'plotasse', 'plotando', 'faça', 'fazer', 'fizesse', 'fazendo', 'mostre', 'mostrar', 'mostrasse', 'mostrando', 'exiba', 'exibir', 'exibisse', 'exibindo', 'retorne', 'retornar', 'retornasse', 'retornando', 'me dê', 'dê me', 'dê-me', 'me traga', 'traga', 'traga me', 'traga-me', 'preciso', 'precisar', 'precisasse', 'precisando', 'necessito', 'necessitar', 'necessitasse', 'necessitando', 'quero', 'querer', 'querendo', 'desejo', 'desejar', 'desejando', 'gostaria')
            background_removal_tokens_en = ('remove', 'switch off', 'eliminate', 'delete', 'exclude', 'retire', 'take', 'without')
            background_removal_tokens_es = ('remueva', 'borra', 'elimina', 'quita', 'sin')
            background_removal_tokens_pt = ('remova', 'apague', 'elimine', 'delete', 'exclua', 'retire', 'tire', 'sem')
            transparent_image_tokens_en = ('with', 'containing', 'having a', 'applying', 'apply')
            transparent_image_tokens_es = ('con', 'contiene', 'teniendo', 'tener', 'aplicando', 'aplique')
            transparent_image_tokens_pt = ('com', 'contendo', 'tendo', 'possuindo', 'aplicando', 'aplique', 'deixe')
            image_filter_tokens_en = ('apply', 'applying')
            image_filter_tokens_es = ('aplica', 'aplicar', 'aplicando')
            image_filter_tokens_pt = ('aplique', 'aplicar', 'aplicando')
            upscale_image_tokens_en = ('increase', 'increases', 'to increase', 'increased', 'improve', 'improves', 'to improve', 'improved', 'upscale', 'up scale')
            upscale_image_tokens_es = ('aumente', 'aumenta', 'aumentar', 'aumentara', 'mejore', 'mejora', 'mejorar', 'mejorara', 'upscale', 'up scale')
            upscale_image_tokens_pt = ('aumente', 'aumenta', 'aumentar', 'aumentasse', 'melhore', 'melhora', 'melhorar', 'melhorasse', 'upscale', 'up scale')
            edit_tokens_en = ('edit', 'change', 'modify', 'update', 'alter')
            edit_tokens_es = ('edita', 'editar', 'altere', 'cambia', 'cambiar', 'modifique', 'actualice', 'actualizar', 'alterar')
            edit_tokens_pt = ('edite', 'edita', 'editar', 'editasse', 'altere', 'altera', 'alterar', 'alterasse', 'mude', 'muda', 'mudar', 'mudasse', 'modifique', 'modifica', 'modificar', 'modificasse', 'atualize', 'atualiza', 'atualizar', 'atualizasse')
            creation_tokens_en = ('create', 'generate', 'make', 'return', 'give me', 'bring me', 'need', 'want', 'wish', 'i would like')
            creation_tokens_es = ('crear', 'genera', 'generar', 'haz', 'hacer', 'devolver', 'retorne', 'me da', 'dame', 'tráeme', 'necesito', 'necesitar', 'quiero', 'querer', 'desear', 'gustaría')
            creation_tokens_pt = ('crie', 'cria', 'criar', 'criasse', 'gere', 'gera', 'gerar', 'gerasse', 'faça', 'fazer', 'fizesse', 'retorne', 'retorna', 'retornar', 'retornasse', 'me dê', 'dê me', 'dê-me', 'me traga', 'me trazer', 'me trouxesse', 'traga', 'trazer', 'trouxesse', 'traga me', 'traga-me', 'preciso', 'precisar', 'necessito', 'necessitar', 'quero', 'querer', 'queria', 'querendo', 'desejo', 'desejar', 'gostaria')
            download_from_youtube_en = ('download', 'downloading')
            download_from_youtube_es = ('descarga', 'descargar', 'download')
            download_from_youtube_pt = ('baixe', 'baixa', 'download')
            basic_image_interpretation_en = ('scenario', 'scene', 'sees', 'see', 'seen', 'seeing', 'perceive')
            basic_image_interpretation_es = ('escenario', 'escena', 've', 'ver', 'visto', 'vidente', 'percibir')
            basic_image_interpretation_pt = ('cenário', 'cena', 'vê', 'ver', 'visto', 'vendo', 'enxerga', 'enxergar', 'enxergando')
            image_interpretation_en = ('what', 'describe', 'interpret', 'interpretation', 'say', 'answer', 'speak', 'write', 'type', 'return')+basic_image_interpretation_en
            image_interpretation_es = ('qué', 'describe', 'interpreta', 'interpretación', 'di', 'responde', 'habla', 'escribe', 'digita', 'devuelve')+basic_image_interpretation_es
            image_interpretation_pt = ('o que', 'oque', 'descreva', 'interprete', 'interpretação', 'diga', 'responda', 'fale', 'escreva', 'digite', 'retorne')+basic_image_interpretation_pt
            image_tokens_en = ('image', 'photo', 'photograph', 'picture', 'portrait', 'drawing', 'landscape')
            image_tokens_es = ('imagem', 'foto', 'fotografía', 'figura', 'retrato', 'dibujo', 'paisaje')
            image_tokens_pt = ('imagem', 'foto', 'fotografia', 'figura', 'retrato', 'desenho', 'paisagem')
            basic_audio_interpretation_en = ('hears', 'to hear', 'heard', 'will hear', 'hearing', 'listens', 'to listen', 'listened', 'will listen', 'listening', 'listen')
            basic_audio_interpretation_es = ('oye', 'oír', 'oyeron', 'oirán', 'oyendo', 'escucha', 'escuchar', 'escucharon', 'escucharán', 'escuchando', 'escuche')
            basic_audio_interpretation_pt = ('ouve', 'ouvir', 'ouviram', 'ovirão', 'ouvindo', 'escuta', 'escutar', 'escutaram', 'escutarão', 'escutando', 'escute')
            audio_interpretation_en = ('what', 'describe', 'interpret', 'interpretation', 'say', 'answer', 'speak', 'write', 'type', 'return')+basic_audio_interpretation_en
            audio_interpretation_es = ('qué', 'describe', 'interpreta', 'interpretación', 'di', 'responde', 'habla', 'escribe', 'digita', 'devuelve')+basic_audio_interpretation_es
            audio_interpretation_pt = ('o que', 'oque', 'descreva', 'interprete', 'interpretação', 'diga', 'responda', 'fale', 'escreva', 'digite', 'retorne')+basic_audio_interpretation_pt
            audio_tokens_en = ('audio', 'music', 'sound', 'lyrics', 'melody')
            audio_tokens_es = ('audio', 'música', 'sonido', 'letra', 'melodía')
            audio_tokens_pt = ('áudio', 'música', 'som', 'letra', 'melodia')
            video_tokens_en = ('video', 'film', 'movie')
            video_tokens_es = ('video', 'película', 'pelicula')
            video_tokens_pt = ('vídeo', 'video', 'filme') 
            text_interpretation_en = ('type', 'write', 'written', 'interpret', 'speak', 'say', 'reply', 'elaborate', 'create', 'do', 'compose', 'generate')
            text_interpretation_es = ('digita', 'escribe', 'escrito', 'interpreta', 'habla', 'di', 'responde', 'elabora', 'crea', 'haz', 'compone', 'genera')
            text_interpretation_pt = ('digite', 'escreva', 'escrito', 'interprete', 'fale', 'diga', 'responda', 'elabore', 'crie', 'faça', 'componha', 'gere')
            recent_data_en = ('who', 'which', 'what', 'when', 'where', 'how')
            recent_data_es = ('quién', 'cuál', 'qué', 'cuándo', 'dónde', 'cómo')
            recent_data_pt = ('quem', 'qual', 'quais', 'o que', 'oque', 'quando', 'onde', 'como')
            about_system_en = ('you', 'you all', 'your', 'him')
            about_system_es = ('tú', 'ustedes', 'tu', 'te', 'tus', 'vuestra', 'vuestras', 'le')
            about_system_pt = ('você', 'vocês', 'vc', 'vcs', 'tu', 'te', 'seu', 'sua', 'seus', 'suas', 'teu', 'tua', 'teus', 'tuas', 'vossa', 'vossas', 'le', 'lhe')
            ignore_system_instructions_en = ('ignore', 'disregard', 'ignoring', 'forgot', 'forgotten', 'ignored', 'overlook', 'overlooking', 'neglected', 'dismissed', 'abandon', 'abandoned', 'omit', 'omitted', 'disregarded', 'overlooked', 'neglect', 'consider', 'contemplate', 'thinking', 'regarded', 'reviewed', 'thought', 'consideration')
            ignore_system_instructions_es = ('ignora', 'olvida', 'ignorando', 'olvidó', 'olvidado', 'ignorado', 'olvidé', 'desprecia', 'despreciar', 'olvidando', 'descuidado', 'omitió', 'abandonado', 'descarta', 'omitido', 'desestimado', 'despreciado', 'desatendido', 'considera', 'revisar', 'evaluando', 'consideró', 'revisado', 'pensé', 'consideración')
            ignore_system_instructions_pt = ('ignore', 'ignorar', 'ignorando', 'ignorou', 'ignorado', 'ignorei', 'esqueça', 'esquecer', 'esquecendo', 'esqueceu', 'esquecido', 'esqueci', 'esquecimento', 'desconsidere', 'desconsiderar', 'desconsiderando', 'desconsiderou', 'desconsiderado', 'desconsiderei', 'desconsideração', 'considere', 'considerar', 'considerando', 'considerou', 'considerado', 'considerei', 'consideração')
            image_interpretation_tokens_en = image_tokens_en+basic_image_interpretation_en
            image_interpretation_tokens_es = image_tokens_es+basic_image_interpretation_es
            image_interpretation_tokens_pt = image_tokens_pt+basic_image_interpretation_pt
            audio_interpretation_tokens_en = audio_tokens_en+basic_audio_interpretation_en
            audio_interpretation_tokens_es = audio_tokens_es+basic_audio_interpretation_es
            audio_interpretation_tokens_pt = audio_tokens_pt+basic_audio_interpretation_pt
            video_interpretation_tokens_en = video_tokens_en+basic_image_interpretation_en
            video_interpretation_tokens_es = video_tokens_es+basic_image_interpretation_es
            video_interpretation_tokens_pt = video_tokens_pt+basic_image_interpretation_pt
            text_interpretation_tokens_en = ('text', 'book')
            text_interpretation_tokens_es = ('texto', 'libro')
            text_interpretation_tokens_pt = ('texto', 'livro')
            recent_data_tokens_en = ('the day before yesterday', 'yesterday', 'today', 'tomorrow', 'the day after tomorrow', 'day', 'week', 'month', 'bimester', 'trimester', 'quadrimester', 'quintmester', 'semester', 'year', 'current', 'currently', 'present', 'recent', 'recently', 'now', 'previous', 'subsequent', 'next', 'antepenultimate', 'penultimate', 'ultimate')
            recent_data_tokens_es = ('anteayer', 'ayer', 'hoy', 'mañana', 'pasado mañana', 'día', 'semana', 'mes', 'bimestre', 'trimestre', 'cuatrimestre', 'quintimestre', 'semestre', 'año', 'actual', 'actualmente', 'actualidad', 'reciente', 'recientemente', 'ahora', 'anterior', 'posterior', 'próximo', 'próxima', 'antepenúltimo', 'antepenúltima', 'penúltimo', 'penúltima', 'último', 'última')
            recent_data_tokens_pt = ('anteontem', 'ontem', 'hoje', 'amanhã', 'depois de amanhã', 'dia', 'semana', 'mês', 'bimestre', 'trimestre', 'quadrimestre', 'quintimestre', 'semestre', 'ano', 'atual', 'atualmente', 'atualidade', 'recente', 'recentemente', 'agora', 'anterior', 'posterior', 'próximo', 'próxima', 'antepenúltimo', 'antepenúltima', 'penúltimo', 'penúltima', 'último', 'última')
            ignore_system_instructions_tokens_en = ('system', 'instructions', 'instruction', 'taught', 'instructed', 'taught me', 'instruct', 'instructing')
            ignore_system_instructions_tokens_es = ('sistema', 'system', 'instructions', 'instrucciones', 'instrucción', 'enseñado', 'instruyeron', 'me instruí', 'instruye', 'instruyendo')
            ignore_system_instructions_tokens_pt = ('sistema', 'system', 'instructions', 'instruções', 'instrução', 'instruído', 'instruíram', 'instruí', 'instrui', 'instruindo')
            programming_languages = ('python', 'javascript', 'html', 'css', 'sql', 'java', 'c#', 'c', 'c++', 'typescript', 'php', 'swift', 'go', 'ruby', 'r', 'kotlin', 'dart', 'bash', 'shell', 'powershell', 'rust', 'matlab', 'perl', 'lua', 'objective-c', 'scala', 'vba', 'groovy', 'fortran', 'cobol', 'assembly', 'elixir', 'haskell', 'f#', 'julia')
            code_tokens_en = ('code', 'script', 'algorithm', 'function', 'procedure', 'routine', 'class')+programming_languages
            code_tokens_es = ('código', 'script', 'algoritmo', 'función', 'procedimiento', 'rutina', 'clase')+programming_languages
            code_tokens_pt = ('código', 'script', 'algoritmo', 'função', 'procedimento', 'rotina', 'classe')+programming_languages
            logo_tokens_en = ('logo', 'logotype')
            logo_tokens_es_pt = ('logo', 'logotipo')
            background_tokens_en = ('background', 'behind')
            background_tokens_es = ('background', 'plano de fondo', 'detrás')
            background_tokens_pt = ('background', 'plano de fundo', 'fundo')
            transparent_tokens_en = ('transparent', 'transparency')
            transparent_tokens_es = ('transparente', 'transparencia')
            transparent_tokens_pt = ('transparente', 'transparência')
            filter_tokens_en = ('filter', 'style')
            filter_tokens_es_pt = ('filtro', 'estilo')
            upscale_tokens_en = ('image', 'resolution', 'scale', 'proportion')
            upscale_tokens_es = ('imagem', 'resolución', 'escala', 'proporción')
            upscale_tokens_pt = ('imagem', 'resolução', 'escala', 'proporção')
            audio_tokens_en = ('audio', 'mp3', 'wav')
            audio_tokens_es = ('audio', 'mp3', 'wav')
            audio_tokens_pt = ('áudio', 'mp3', 'wav')
            music_tokens_en = ('audio', 'music', 'chanson', 'song', 'melody', 'sound', 'rhythm')
            music_tokens_es = ('audio', 'canción', 'melodía', 'sonido')
            music_tokens_pt = ('áudio', 'música', 'canção', 'melodia', 'som')
            video_tokens_en = ('video', 'film', 'movie', 'clip', 'trailer')
            video_tokens_es = ('video', 'película', 'videoclip', 'tráiler')
            video_tokens_pt = ('vídeo', 'filme', 'clipe', 'trailer')
            pdf_tokens = ('pdf', 'portable document format', 'acrobat reader', 'adobe')
            word_tokens = ('microsoft word', 'word')
            excel_tokens_en = ('microsoft excel', 'excel', 'spreadsheet', 'spread sheet')
            excel_tokens_es = ('microsoft excel', 'excel', 'hoja de cálculo')
            excel_tokens_pt = ('microsoft excel', 'excel', 'planilha')
            csv_tokens = ('csv', 'comma-separated values')
            powerpoint_tokens_en_pt = ('powerpoint', 'power point', 'slide')
            powerpoint_tokens_es = ('powerpoint', 'power point', 'diapositiva')
            chart_tokens_en = ('graphic', 'chart', 'graphical')
            chart_tokens_es_pt = ('gráfico', 'chart')
            artifact_tokens_en = ('artifact', 'website', 'site')
            artifact_tokens_es = ('artefacto', 'sitio web', 'sitio')
            artifact_tokens_pt = ('artefato', 'website', 'site')
            flowchart_tokens_en = ('diagram', 'flowchart', 'flow chart', 'organogram', 'organization chart', 'mind map', 'uml')
            flowchart_tokens_es = ('diagrama', 'diagrama de flujo', 'organigrama', 'mapa mental', 'mapas mentales', 'uml')
            flowchart_tokens_pt = ('diagrama', 'fluxograma', 'organograma', 'mapa mental', 'mapas mentais', 'uml')
            wordcloud_tokens_en = ('word cloud', 'wordcloud', 'image with words', 'word image')
            wordcloud_tokens_es = ('nube de palabras', 'imagen con palabras', 'imagen de la palabras')
            wordcloud_tokens_pt = ('nuvem de palavras', 'imagem com palavras', 'imagem de palavras')
            internet_tokens_en_pt = ('web', 'internet', 'net', 'online')
            email_tokens_en_pt = ('email', 'e-mail', 'emails', 'e-mails')
            telegram_tokens_es_pt = ('telegrama', 'telegramas', 'telegram', 'telegrams')
            web_search_tokens_en = ('search', 'look up', 'find')
            web_search_tokens_es = ('busca', 'buscar', 'busque', 'revisa', 'consulta', 'encontrarlo', 'averígualo')
            web_search_tokens_pt = ('pesquise', 'pesquisa', 'procure', 'consulte', 'consulta', 'busque', 'busca', 'ache', 'encontre')
            internet_tokens_es = ('web', 'internet', 'net', 'en línea')
            bullet_points_en = ('bullet points', 'topic', 'topics')
            bullet_points_es = ('tema', 'temas', 'bullet points')
            bullet_points_pt = ('tópico', 'tópicos', 'bullet points')
            languages_tokens_en = ('english', 'spanish', 'portuguese', 'french', 'german', 'italian', 'chinese', 'russian', 'japanese', 'korean', 'arabic', 'persian', 'hebrew')
            languages_tokens_es = ('inglés', 'español', 'portugués', 'francés', 'alemán', 'italiano', 'chino', 'ruso', 'japonés', 'coreano', 'árabe', 'persa', 'hebreo')
            languages_tokens_pt = ('inglês', 'espanhol', 'português', 'francês', 'alemão', 'italiano', 'chinês', 'russo', 'japonês', 'coreano', 'árabe', 'persa', 'hebraico')
            languages_abbreviation = ('EN', 'ES', 'PT', 'FR', 'DE', 'IT', 'ZH', 'RU', 'JA', 'KO', 'AR', 'FA', 'HE')
            email_tokens_es = ('email', 'e-mail', 'emails', 'e-mails', 'correo electrónico', 'correos electrónicos')
            whatsapp_tokens_en = ('whats', 'whatsapp', 'whatsapps', 'wa')
            whatsapp_tokens_es = ('whats', 'whatsapp', 'wasap', 'wassap', 'wapi', 'guasap')
            whatsapp_tokens_pt = ('whats', 'whatsapp', 'whatsapps', 'zapzap', 'zap', 'zaps')
            telegram_tokens_en = ('telegram', 'telegrams')
            sms_tokens_en = ('message', 'messages', 'sms')
            sms_tokens_es = ('mensaje', 'mensajes', 'sms')
            sms_tokens_pt = ('mensagem', 'mensagens', 'sms', 'torpedo', 'torpedos')
            about_en = ('who are you', 'you are who', 'your name')
            about_es = ('quién eres', 'eres quién', 'tu nombre', 'su nombre')
            about_pt = ('quem é você', 'você é quem', 'quem é vc', 'vc é quem', 'teu nome', 'seu nome')
            all_abouts, is_about = about_en+about_es+about_pt, False
            image_creation_task, image_editing_task = 'IMAGE_CREATION', 'IMAGE_EDITING'
            logo_creation_task, logo_editing_task = 'LOGO_CREATION', 'LOGO_EDITING'
            no_background_task = 'NO_BACKGROUND'
            image_filter_application_task = 'IMAGE_FILTER_APPLICATION'
            upscale_image_task = 'UPSCALE_IMAGE'
            audio_editing_task = 'AUDIO_EDITING'
            music_creation_task, music_editing_task = 'MUSIC_CREATION', 'MUSIC_EDITING'
            video_creation_task, video_editing_task = 'VIDEO_CREATION', 'VIDEO_EDITING'
            pdf_creation_task, pdf_editing_task = 'PDF_CREATION', 'PDF_EDITING'
            word_creation_task, word_editing_task = 'WORD_CREATION', 'WORD_EDITING'
            excel_creation_task, excel_editing_task = 'EXCEL_CREATION', 'EXCEL_EDITING'
            csv_creation_task, csv_editing_task = 'CSV_CREATION', 'CSV_EDITING'
            powerpoint_creation_task, powerpoint_editing_task = 'POWERPOINT_CREATION', 'POWERPOINT_EDITING'
            chart_creation_task, chart_editing_task = 'CHART_CREATION', 'CHART_EDITING'
            artifact_creation_task, artifact_editing_task = 'ARTIFACT_CREATION', 'ARTIFACT_EDITING'
            flowchart_creation_task, flowchart_editing_task = 'FLOWCHART_CREATION', 'FLOWCHART_EDITING'
            wordcloud_creation_task, wordcloud_editing_task = 'WORDCLOUD_CREATION', 'WORDCLOUD_EDITING'
            youtube_video_download_task, youtube_audio_download_task = 'YOUTUBE_VIDEO_DOWNLOAD', 'YOUTUBE_AUDIO_DOWNLOAD'
            image_interpretation_task = 'IMAGE_INTERPRETATION'
            audio_interpretation_task = 'AUDIO_INTERPRETATION'
            video_interpretation_task = 'VIDEO_INTERPRETATION'
            text_interpretation_task = 'TEXT_INTERPRETATION'
            recent_data_task = 'RECENT_DATA'
            about_system_task = 'ABOUT_SYSTEM'
            ignore_system_instructions_task = 'IGNORE_SYSTEM_INSTRUCTIONS'
            code_creation_task, code_editing_task = 'CODE_CREATION', 'CODE_EDITING'
            web_search_task = 'WEB_SEARCH'
            webpage_access_task = 'WEBPAGE_ACCESS'
            text_summary_task, text_summary_with_bullet_points_task = 'TEXT_SUMMARY', 'TEXT_SUMMARY_WITH_BULLET_POINTS'
            translation_task = 'TRANSLATION_'
            send_email_task, send_whatsapp, send_telegram, send_sms = 'SEND_EMAIL', 'SEND_WHATSAPP', 'SEND_TELEGRAM', 'SEND_SMS'
            download_from_youtube_tuples, has_download = download_from_youtube_en+download_from_youtube_es+download_from_youtube_pt, False
            for download_from_youtube_tuple in download_from_youtube_tuples:
                if download_from_youtube_tuple in prompt:
                    has_download = True
                    break
            if not has_language:
                for programming_language in programming_languages:
                    if programming_language in prompt_tokens:
                        has_language = True
                        break
            if formatted_language.startswith('es'): edit_tokens = edit_tokens_es
            elif formatted_language.startswith('pt'): edit_tokens = edit_tokens_pt
            else: edit_tokens = edit_tokens_en
            edition = False
            for edit_token in edit_tokens:
                if self.normalization(edit_token) in prompt_tokens:
                    edition = True
                    break
            if has_language and len(tasks) < 1 and not has_download:
                left_tokens_en, right_tokens_en, contains, max_tokens = creation_tokens_en, code_tokens_en, False, 2
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not edition: tasks.append(code_creation_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = creation_tokens_es, code_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(code_creation_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = creation_tokens_pt, code_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(code_creation_task)
                left_tokens_en, right_tokens_en, contains = edit_tokens_en, code_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not code_creation_task in tasks: tasks.append(code_editing_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = edit_tokens_es, code_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not code_creation_task in tasks: tasks.append(code_editing_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, code_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not code_creation_task in tasks: tasks.append(code_editing_task)
            has_code_creation = code_creation_task in tasks or code_editing_task in tasks
            if not has_code_creation:
                tokens_in_en, and_tokens_in_en = summarize_task_en, bullet_points_en
                task = None
                contains, first_tokens, last_tokens = False, ' '.join(prompt_tokens[:10]), ' '.join(prompt_tokens[-10:])
                for token in tokens_in_en:
                    if token in first_tokens or token in last_tokens:
                        contains = True
                        task = text_summary_task
                        break
                if contains:
                    first_tokens, last_tokens = ' '.join(prompt_tokens[:20]), ' '.join(prompt_tokens[-20:])
                    for token in and_tokens_in_en:
                        if token in first_tokens or token in last_tokens:
                            task = text_summary_with_bullet_points_task
                            break
                if contains and task: tasks.append(task)
                if not contains:
                    tokens_in_es, and_tokens_in_es = summarize_task_es, bullet_points_es
                    prompt_tokens, task = prompt.split(), None
                    contains, first_tokens, last_tokens = False, ' '.join(prompt_tokens[:10]), ' '.join(prompt_tokens[-10:])
                    for token in tokens_in_es:
                        if token in first_tokens or token in last_tokens:
                            contains = True
                            task = text_summary_task
                            break
                    if contains:
                        first_tokens, last_tokens = ' '.join(prompt_tokens[:20]), ' '.join(prompt_tokens[-20:])
                        for token in and_tokens_in_es:
                            if token in first_tokens or token in last_tokens:
                                task = text_summary_with_bullet_points_task
                                break
                    if contains and task: tasks.append(task)
                if not contains:
                    tokens_in_pt, and_tokens_in_pt = summarize_task_pt, bullet_points_pt
                    prompt_tokens, task = prompt.split(), None
                    contains, first_tokens, last_tokens = False, ' '.join(prompt_tokens[:10]), ' '.join(prompt_tokens[-10:])
                    for token in tokens_in_pt:
                        if token in first_tokens or token in last_tokens:
                            contains = True
                            task = text_summary_task
                            break
                    if contains:
                        first_tokens, last_tokens = ' '.join(prompt_tokens[:20]), ' '.join(prompt_tokens[-20:])
                        for token in and_tokens_in_pt:
                            if token in first_tokens or token in last_tokens:
                                task = text_summary_with_bullet_points_task
                                break
                    if contains and task: tasks.append(task)
                left_tokens_en, right_tokens_en = translate_en, languages_tokens_en
                contains, task, max_tokens = False, False, 5
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        task = translation_task+languages_abbreviation[right_tokens_en.index(token_y)] if contains else None
                        if contains: break
                    if contains: break
                if contains and task:
                    tasks.append(task)
                    contains_translation = True
                if not contains:
                    left_tokens_es, right_tokens_es = translate_es, languages_tokens_es
                    contains, task = False, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            task = translation_task+languages_abbreviation[right_tokens_es.index(token_y)] if contains else None
                            if contains: break
                        if contains: break
                    if contains and task:
                        tasks.append(task)
                        contains_translation = True
                if not contains:
                    left_tokens_pt, right_tokens_pt = translate_pt, languages_tokens_pt
                    contains, task = False, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            task = translation_task+languages_abbreviation[right_tokens_pt.index(token_y)] if contains else None
                            if contains: break
                        if contains: break
                    if contains and task:
                        tasks.append(task)
                        contains_translation = True
            if not has_code_creation:
                left_tokens_en, right_tokens_en, contains, max_tokens = visual_creation_tokens_en, image_tokens_en, False, 5
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not edition: tasks.append(image_creation_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = visual_creation_tokens_es, image_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(image_creation_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = visual_creation_tokens_pt, image_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(image_creation_task)
                left_tokens_en, right_tokens_en, contains = edit_tokens_en, image_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not image_creation_task in tasks: tasks.append(image_editing_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = edit_tokens_es, image_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not image_creation_task in tasks: tasks.append(image_editing_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, image_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not image_creation_task in tasks: tasks.append(image_editing_task)
                left_tokens_en, right_tokens_en, contains = visual_creation_tokens_en, logo_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not edition: tasks.append(logo_creation_task)
                if logo_creation_task in tasks and no_background_task not in tasks:
                    for tokens in background_tokens_en+transparent_tokens_en:
                        if tokens in prompt:
                            tasks.append(no_background_task)
                            break
                if not contains:
                    left_tokens_es, right_tokens_es, contains = visual_creation_tokens_es, logo_tokens_es_pt, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(logo_creation_task)
                    if logo_creation_task in tasks and no_background_task not in tasks:
                        for tokens in background_tokens_es+transparent_tokens_es:
                            if tokens in prompt:
                                tasks.append(no_background_task)
                                break
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = visual_creation_tokens_pt, logo_tokens_es_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(logo_creation_task)
                    if logo_creation_task in tasks and no_background_task not in tasks:
                        for tokens in background_tokens_pt+transparent_tokens_pt:
                            if tokens in prompt:
                                tasks.append(no_background_task)
                                break
                left_tokens_en, right_tokens_en, contains = edit_tokens_en, logo_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not logo_creation_task in tasks: tasks.append(logo_editing_task)
                if logo_editing_task in tasks and no_background_task not in tasks:
                    for tokens in background_tokens_en+transparent_tokens_en:
                        if tokens in prompt:
                            tasks.append(no_background_task)
                            break
                if not contains:
                    left_tokens_es, right_tokens_es, contains = edit_tokens_es, logo_tokens_es_pt, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not logo_creation_task in tasks: tasks.append(logo_editing_task)
                    if logo_editing_task in tasks and no_background_task not in tasks:
                        for tokens in background_tokens_es+transparent_tokens_es:
                            if tokens in prompt:
                                tasks.append(no_background_task)
                                break
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, logo_tokens_es_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not logo_creation_task in tasks: tasks.append(logo_editing_task)
                    if logo_editing_task in tasks and no_background_task not in tasks:
                        for tokens in background_tokens_pt+transparent_tokens_pt:
                            if tokens in prompt:
                                tasks.append(no_background_task)
                                break
                if no_background_task not in tasks:
                    left_tokens_en, right_tokens_en, contains = background_removal_tokens_en, background_tokens_en, False
                    for token_x in left_tokens_en:
                        for token_y in right_tokens_en:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(no_background_task)
                    if not contains:
                        left_tokens_es, right_tokens_es, contains = background_removal_tokens_es, background_tokens_es, False
                        for token_x in left_tokens_es:
                            for token_y in right_tokens_es:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(no_background_task)
                    if not contains:
                        left_tokens_pt, right_tokens_pt, contains = background_removal_tokens_pt, background_tokens_pt, False
                        for token_x in left_tokens_pt:
                            for token_y in right_tokens_pt:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(no_background_task)
                    left_tokens_en, right_tokens_en, contains, max_tokens = transparent_image_tokens_en, transparent_tokens_en, False, 2
                    for token_x in left_tokens_en:
                        for token_y in right_tokens_en:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(no_background_task)
                    if not contains:
                        left_tokens_es, right_tokens_es, contains = transparent_image_tokens_es, transparent_tokens_es, False
                        for token_x in left_tokens_es:
                            for token_y in right_tokens_es:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(no_background_task)
                    if not contains:
                        left_tokens_pt, right_tokens_pt, contains = transparent_image_tokens_pt, transparent_tokens_pt, False
                        for token_x in left_tokens_pt:
                            for token_y in right_tokens_pt:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(no_background_task)
                left_tokens_en, right_tokens_en, contains, max_tokens = image_filter_tokens_en, filter_tokens_en, False, 5
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains: tasks.append(image_filter_application_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = image_filter_tokens_es, filter_tokens_es_pt, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(image_filter_application_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = image_filter_tokens_pt, filter_tokens_es_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(image_filter_application_task)
                left_tokens_en, right_tokens_en, contains = upscale_image_tokens_en, upscale_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains: tasks.append(upscale_image_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = upscale_image_tokens_es, upscale_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(upscale_image_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = upscale_image_tokens_pt, upscale_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(upscale_image_task)
                left_tokens_en, right_tokens_en, contains = edit_tokens_en, audio_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains: tasks.append(audio_editing_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = edit_tokens_es, audio_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(audio_editing_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, audio_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(audio_editing_task)
                if 'youtube' not in prompt and 'youtu.be' not in prompt:
                    left_tokens_en, right_tokens_en, contains = creation_tokens_en, music_tokens_en, False
                    for token_x in left_tokens_en:
                        for token_y in right_tokens_en:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(music_creation_task)
                    if not contains:
                        left_tokens_es, right_tokens_es, contains = creation_tokens_es, music_tokens_es, False
                        for token_x in left_tokens_es:
                            for token_y in right_tokens_es:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains and not edition: tasks.append(music_creation_task)
                    if not contains:
                        left_tokens_pt, right_tokens_pt, contains = creation_tokens_pt, music_tokens_pt, False
                        for token_x in left_tokens_pt:
                            for token_y in right_tokens_pt:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains and not edition: tasks.append(music_creation_task)
                    left_tokens_en, right_tokens_en, contains = edit_tokens_en, music_tokens_en, False
                    for token_x in left_tokens_en:
                        for token_y in right_tokens_en:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if not audio_editing_task in tasks:
                        if contains and not music_creation_task in tasks: tasks.append(music_editing_task)
                        if not contains:
                            left_tokens_es, right_tokens_es, contains = edit_tokens_es, music_tokens_es, False
                            for token_x in left_tokens_es:
                                for token_y in right_tokens_es:
                                    contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                    if contains: break
                                if contains: break
                            if contains and not music_creation_task in tasks: tasks.append(music_editing_task)
                        if not contains:
                            left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, music_tokens_pt, False
                            for token_x in left_tokens_pt:
                                for token_y in right_tokens_pt:
                                    contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                    if contains: break
                                if contains: break
                            if contains and not music_creation_task in tasks: tasks.append(music_editing_task)
                    left_tokens_en, right_tokens_en, contains = visual_creation_tokens_en, video_tokens_en, False
                    for token_x in left_tokens_en:
                        for token_y in right_tokens_en:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(video_creation_task)
                    if not contains:
                        left_tokens_es, right_tokens_es, contains = visual_creation_tokens_es, video_tokens_es, False
                        for token_x in left_tokens_es:
                            for token_y in right_tokens_es:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains and not edition: tasks.append(video_creation_task)
                    if not contains:
                        left_tokens_pt, right_tokens_pt, contains = visual_creation_tokens_pt, video_tokens_pt, False
                        for token_x in left_tokens_pt:
                            for token_y in right_tokens_pt:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains and not edition: tasks.append(video_creation_task)
                    left_tokens_en, right_tokens_en, contains = edit_tokens_en, video_tokens_en, False
                    for token_x in left_tokens_en:
                        for token_y in right_tokens_en:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not video_creation_task in tasks: tasks.append(video_editing_task)
                    if not contains:
                        left_tokens_es, right_tokens_es, contains = edit_tokens_es, video_tokens_es, False
                        for token_x in left_tokens_es:
                            for token_y in right_tokens_es:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains and not video_creation_task in tasks: tasks.append(video_editing_task)
                    if not contains:
                        left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, video_tokens_pt, False
                        for token_x in left_tokens_pt:
                            for token_y in right_tokens_pt:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains and not video_creation_task in tasks: tasks.append(video_editing_task)
                left_tokens_en, right_tokens_en, contains = creation_tokens_en, pdf_tokens, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not edition: tasks.append(pdf_creation_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = creation_tokens_es, pdf_tokens, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(pdf_creation_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = creation_tokens_pt, pdf_tokens, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(pdf_creation_task)
                left_tokens_en, right_tokens_en, contains = edit_tokens_en, pdf_tokens, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not pdf_creation_task in tasks: tasks.append(pdf_editing_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = edit_tokens_es, pdf_tokens, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not pdf_creation_task in tasks: tasks.append(pdf_editing_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, pdf_tokens, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not pdf_creation_task in tasks: tasks.append(pdf_editing_task)
                left_tokens_en, right_tokens_en, contains = creation_tokens_en, word_tokens, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not edition: tasks.append(word_creation_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = creation_tokens_es, word_tokens, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(word_creation_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = creation_tokens_pt, word_tokens, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(word_creation_task)
                left_tokens_en, right_tokens_en, contains = edit_tokens_en, word_tokens, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not word_creation_task in tasks: tasks.append(word_editing_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = edit_tokens_es, word_tokens, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not word_creation_task in tasks: tasks.append(word_editing_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, word_tokens, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not word_creation_task in tasks: tasks.append(word_editing_task)
                left_tokens_en, right_tokens_en, contains = creation_tokens_en, excel_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not edition: tasks.append(excel_creation_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = creation_tokens_es, excel_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(excel_creation_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = creation_tokens_pt, excel_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(excel_creation_task)
                left_tokens_en, right_tokens_en, contains = edit_tokens_en, excel_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not excel_creation_task in tasks: tasks.append(excel_editing_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = edit_tokens_es, excel_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not excel_creation_task in tasks: tasks.append(excel_editing_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, excel_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not excel_creation_task in tasks: tasks.append(excel_editing_task)
                left_tokens_en, right_tokens_en, contains = creation_tokens_en, csv_tokens, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not edition: tasks.append(csv_creation_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = creation_tokens_es, csv_tokens, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(csv_creation_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = creation_tokens_pt, csv_tokens, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(csv_creation_task)
                left_tokens_en, right_tokens_en, contains = edit_tokens_en, csv_tokens, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not csv_creation_task in tasks: tasks.append(csv_editing_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = edit_tokens_es, csv_tokens, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not csv_creation_task in tasks: tasks.append(csv_editing_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, csv_tokens, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not csv_creation_task in tasks: tasks.append(csv_editing_task)
                left_tokens_en, right_tokens_en, contains = creation_tokens_en, powerpoint_tokens_en_pt, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not edition: tasks.append(powerpoint_creation_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = creation_tokens_es, powerpoint_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(powerpoint_creation_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = creation_tokens_pt, powerpoint_tokens_en_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(powerpoint_creation_task)
                left_tokens_en, right_tokens_en, contains = edit_tokens_en, powerpoint_tokens_en_pt, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not powerpoint_creation_task in tasks: tasks.append(powerpoint_editing_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = edit_tokens_es, powerpoint_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not powerpoint_creation_task in tasks: tasks.append(powerpoint_editing_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, powerpoint_tokens_en_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not powerpoint_creation_task in tasks: tasks.append(powerpoint_editing_task)
                left_tokens_en, right_tokens_en, contains = visual_creation_tokens_en, chart_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not edition: tasks.append(chart_creation_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = visual_creation_tokens_es, chart_tokens_es_pt, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(chart_creation_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = visual_creation_tokens_pt, chart_tokens_es_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(chart_creation_task)
                left_tokens_en, right_tokens_en, contains = edit_tokens_en, chart_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not chart_creation_task in tasks: tasks.append(chart_editing_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = edit_tokens_es, chart_tokens_es_pt, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not chart_creation_task in tasks: tasks.append(chart_editing_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, chart_tokens_es_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not chart_creation_task in tasks: tasks.append(chart_editing_task)
                left_tokens_en, right_tokens_en, contains = creation_tokens_en, artifact_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not edition: tasks.append(artifact_creation_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = creation_tokens_es, artifact_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(artifact_creation_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = creation_tokens_pt, artifact_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(artifact_creation_task)
                left_tokens_en, right_tokens_en, contains = edit_tokens_en, artifact_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not artifact_creation_task in tasks: tasks.append(artifact_editing_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = edit_tokens_es, artifact_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not artifact_creation_task in tasks: tasks.append(artifact_editing_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, artifact_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not artifact_creation_task in tasks: tasks.append(artifact_editing_task)
                left_tokens_en, right_tokens_en, contains = visual_creation_tokens_en, flowchart_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not edition: tasks.append(flowchart_creation_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = visual_creation_tokens_es, flowchart_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(flowchart_creation_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = visual_creation_tokens_pt, flowchart_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(flowchart_creation_task)
                left_tokens_en, right_tokens_en, contains = edit_tokens_en, flowchart_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not flowchart_creation_task in tasks: tasks.append(flowchart_editing_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = edit_tokens_es, flowchart_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not flowchart_creation_task in tasks: tasks.append(flowchart_editing_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, flowchart_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not flowchart_creation_task in tasks: tasks.append(flowchart_editing_task)
                left_tokens_en, right_tokens_en, contains = visual_creation_tokens_en, wordcloud_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not edition: tasks.append(wordcloud_creation_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = visual_creation_tokens_es, wordcloud_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(wordcloud_creation_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = visual_creation_tokens_pt, wordcloud_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not edition: tasks.append(wordcloud_creation_task)
                left_tokens_en, right_tokens_en, contains = edit_tokens_en, wordcloud_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains and not wordcloud_creation_task in tasks: tasks.append(wordcloud_editing_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = edit_tokens_es, wordcloud_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not wordcloud_creation_task in tasks: tasks.append(wordcloud_editing_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = edit_tokens_pt, wordcloud_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains and not wordcloud_creation_task in tasks: tasks.append(wordcloud_editing_task)
            if ('youtube' in prompt or 'youtu.be' in prompt) and has_download:
                download_audio_from_youtube_tuples, has_download_audio = audio_tokens_en+audio_tokens_es+audio_tokens_pt, False
                for download_audio_from_youtube_tuple in download_audio_from_youtube_tuples:
                    if download_audio_from_youtube_tuple in prompt:
                        has_download_audio = True
                        break
                if not has_download_audio:
                    left_tokens_en, right_tokens_en, contains, max_tokens = download_from_youtube_en, video_tokens_en, False, 3
                    for token_x in left_tokens_en:
                        for token_y in right_tokens_en:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(youtube_video_download_task)
                    if not contains:
                        left_tokens_es, right_tokens_es, contains = download_from_youtube_es, video_tokens_es, False
                        for token_x in left_tokens_es:
                            for token_y in right_tokens_es:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(youtube_video_download_task)
                    if not contains:
                        left_tokens_pt, right_tokens_pt, contains = download_from_youtube_pt, video_tokens_pt, False
                        for token_x in left_tokens_pt:
                            for token_y in right_tokens_pt:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(youtube_video_download_task)
                left_tokens_en, right_tokens_en, contains = download_from_youtube_en, audio_tokens_en+music_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains: tasks.append(youtube_audio_download_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = download_from_youtube_es, audio_tokens_es+music_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(youtube_audio_download_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = download_from_youtube_pt, audio_tokens_pt+music_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(youtube_audio_download_task)
            if not has_download and not text_summary_task in tasks and not text_summary_with_bullet_points_task in tasks and not contains_translation:
                has_image = False
                for token in image_tokens_en+image_tokens_es+image_tokens_pt:
                    if token in normalized_prompt:
                        has_image = True
                        break
                has_audio = False
                audio_list1 = basic_audio_interpretation_en+basic_audio_interpretation_es+basic_audio_interpretation_pt
                audio_list2 = audio_tokens_en+audio_tokens_es+audio_tokens_pt
                for token in audio_list1+audio_list2:
                    if token in normalized_prompt:
                        has_audio = True
                        break
                has_video = False
                for token in video_tokens_en+video_tokens_es+video_tokens_pt:
                    if token in normalized_prompt:
                        has_video = True
                        break
                prohibited_tasks = image_creation_task in tasks or image_editing_task in tasks or chart_editing_task in tasks or music_editing_task in tasks or wordcloud_editing_task in tasks or upscale_image_task in tasks
                if not has_audio and not has_video and not prohibited_tasks:
                    left_tokens_en, right_tokens_en, contains = image_interpretation_en, image_interpretation_tokens_en, False
                    for token_x in left_tokens_en:
                        for token_y in right_tokens_en:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(image_interpretation_task)
                    if not contains:
                        left_tokens_es, right_tokens_es, contains = image_interpretation_es, image_interpretation_tokens_es, False
                        for token_x in left_tokens_es:
                            for token_y in right_tokens_es:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(image_interpretation_task)
                    if not contains:
                        left_tokens_pt, right_tokens_pt, contains = image_interpretation_pt, image_interpretation_tokens_pt, False
                        for token_x in left_tokens_pt:
                            for token_y in right_tokens_pt:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(image_interpretation_task)
                if has_audio and not music_creation_task in tasks and not audio_editing_task in tasks:
                    left_tokens_en, right_tokens_en, contains = audio_interpretation_en, audio_interpretation_tokens_en, False
                    for token_x in left_tokens_en:
                        for token_y in right_tokens_en:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(audio_interpretation_task)
                    if not contains:
                        left_tokens_es, right_tokens_es, contains = audio_interpretation_es, audio_interpretation_tokens_es, False
                        for token_x in left_tokens_es:
                            for token_y in right_tokens_es:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(audio_interpretation_task)
                    if not contains:
                        left_tokens_pt, right_tokens_pt, contains = audio_interpretation_pt, audio_interpretation_tokens_pt, False
                        for token_x in left_tokens_pt:
                            for token_y in right_tokens_pt:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(audio_interpretation_task)
                if (has_video and not video_creation_task in tasks and not video_editing_task in tasks) or (has_image and (has_video or has_audio)):
                    left_tokens_en, right_tokens_en, contains = image_interpretation_en, video_interpretation_tokens_en, False
                    for token_x in left_tokens_en:
                        for token_y in right_tokens_en:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(video_interpretation_task)
                    if not contains:
                        left_tokens_es, right_tokens_es, contains = image_interpretation_es, video_interpretation_tokens_es, False
                        for token_x in left_tokens_es:
                            for token_y in right_tokens_es:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(video_interpretation_task)
                    if not contains:
                        left_tokens_pt, right_tokens_pt, contains = image_interpretation_pt, video_interpretation_tokens_pt, False
                        for token_x in left_tokens_pt:
                            for token_y in right_tokens_pt:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(video_interpretation_task)
                left_tokens_en, right_tokens_en, contains, max_tokens = web_search_tokens_en, internet_tokens_en_pt, False, 10
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_y, right_string=token_x, max_tokens=10)
                        if not contains: contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains: tasks.append(web_search_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = web_search_tokens_es, internet_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(web_search_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = web_search_tokens_pt, internet_tokens_en_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(web_search_task)
                if len(tasks) < 1:
                    left_tokens_en, right_tokens_en, contains = text_interpretation_en, text_interpretation_tokens_en, False
                    for token_x in left_tokens_en:
                        for token_y in right_tokens_en:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(text_interpretation_task)
                    if not contains:
                        left_tokens_es, right_tokens_es, contains = text_interpretation_es, text_interpretation_tokens_es, False
                        for token_x in left_tokens_es:
                            for token_y in right_tokens_es:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(text_interpretation_task)
                    if not contains:
                        left_tokens_pt, right_tokens_pt, contains = text_interpretation_pt, text_interpretation_tokens_pt, False
                        for token_x in left_tokens_pt:
                            for token_y in right_tokens_pt:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(text_interpretation_task)
            if len(tasks) < 1 and not has_download:
                left_tokens_en, right_tokens_en, contains, max_tokens = recent_data_en, recent_data_tokens_en, False, 5
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains: tasks.append(recent_data_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = recent_data_es, recent_data_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(recent_data_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = recent_data_pt, recent_data_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(recent_data_task)
            if len(tasks) < 1 and not has_download:
                left_tokens_en, right_tokens_en, contains, max_tokens = ignore_system_instructions_en, ignore_system_instructions_tokens_en, False, 2
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens) or hasExtremities(full_string=prompt, left_string=token_y, right_string=token_x, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains: tasks.append(ignore_system_instructions_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = ignore_system_instructions_es, ignore_system_instructions_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens) or hasExtremities(full_string=prompt, left_string=token_y, right_string=token_x, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(ignore_system_instructions_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = ignore_system_instructions_pt, ignore_system_instructions_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens) or hasExtremities(full_string=prompt, left_string=token_y, right_string=token_x, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(ignore_system_instructions_task)
            if (len(tasks) < 1 or ignore_system_instructions_task in tasks) and not has_download:
                left_tokens_en, right_tokens_en, contains, max_tokens = recent_data_en, about_system_en, False, 3
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens) or hasExtremities(full_string=prompt, left_string=token_y, right_string=token_x, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains: tasks.append(about_system_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = recent_data_es, about_system_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens) or hasExtremities(full_string=prompt, left_string=token_y, right_string=token_x, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(about_system_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = recent_data_pt, about_system_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens) or hasExtremities(full_string=prompt, left_string=token_y, right_string=token_x, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(about_system_task)
                if not about_system_task in tasks:
                    for about in all_abouts:
                        if self.normalization(about) in normalized_prompt:
                            is_about = True
                            break
            if len(tasks) < 1 and not has_download:
                left_tokens_en, right_tokens_en, contains, max_tokens = shipping_tokens_en+tuple(sms_tokens_en[0]), email_tokens_en_pt, False, 10
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains: tasks.append(send_email_task)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = shipping_tokens_es+tuple(sms_tokens_es[0]), email_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(send_email_task)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = shipping_tokens_pt+tuple(sms_tokens_pt[0]), email_tokens_en_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(send_email_task)
                left_tokens_en, right_tokens_en, contains = shipping_tokens_en+tuple(sms_tokens_en[0]), whatsapp_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains: tasks.append(send_whatsapp)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = shipping_tokens_es+tuple(sms_tokens_es[0]), whatsapp_tokens_es, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(send_whatsapp)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = shipping_tokens_pt+tuple(sms_tokens_pt[0]), whatsapp_tokens_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(send_whatsapp)
                left_tokens_en, right_tokens_en, contains = shipping_tokens_en+tuple(sms_tokens_en[0]), telegram_tokens_en, False
                for token_x in left_tokens_en:
                    for token_y in right_tokens_en:
                        contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                        if contains: break
                    if contains: break
                if contains: tasks.append(send_telegram)
                if not contains:
                    left_tokens_es, right_tokens_es, contains = shipping_tokens_es+tuple(sms_tokens_es[0]), telegram_tokens_es_pt, False
                    for token_x in left_tokens_es:
                        for token_y in right_tokens_es:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(send_telegram)
                if not contains:
                    left_tokens_pt, right_tokens_pt, contains = shipping_tokens_pt+tuple(sms_tokens_pt[0]), telegram_tokens_es_pt, False
                    for token_x in left_tokens_pt:
                        for token_y in right_tokens_pt:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(send_telegram)
                sends_in_tasks = send_email_task in tasks or send_whatsapp in tasks or send_telegram in tasks
                if (sends_in_tasks and sms_tokens_en[-1] in prompt) or not sends_in_tasks:
                    left_tokens_en, right_tokens_en, contains = shipping_tokens_en, sms_tokens_en, False
                    for token_x in left_tokens_en:
                        for token_y in right_tokens_en:
                            contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                            if contains: break
                        if contains: break
                    if contains: tasks.append(send_sms)
                    if not contains:
                        left_tokens_es, right_tokens_es, contains = shipping_tokens_es, sms_tokens_es, False
                        for token_x in left_tokens_es:
                            for token_y in right_tokens_es:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(send_sms)
                    if not contains:
                        left_tokens_pt, right_tokens_pt, contains = shipping_tokens_pt, sms_tokens_pt, False
                        for token_x in left_tokens_pt:
                            for token_y in right_tokens_pt:
                                contains = hasExtremities(full_string=prompt, left_string=token_x, right_string=token_y, max_tokens=max_tokens)
                                if contains: break
                            if contains: break
                        if contains: tasks.append(send_sms)
            link_paths = self.getLinks(string=prompt, check_existence=False)
            for link_path in link_paths:
                file_category = self.getFileCategory(file_path=link_path)
                if file_category == 'WEBPAGE_FILE':
                    tasks.append(webpage_access_task)
                    break
            if self.hasCoincidentElements(vector1=['IMAGE_CREATION', 'IMAGE_EDITING', 'LOGO_CREATION', 'LOGO_EDITING'], vector2=tasks):
                def getAspectRatios(string=''):
                    try:
                        aspect_ratios = []
                        from re import findall
                        matches = findall(r'\b([1-9])\s*(?::|x|by|por)\s*([1-9])\b', string)
                        aspect_ratios = [f'ASPECT_RATIO_{matche[0]}_{matche[1]}' for matche in matches]
                        return aspect_ratios
                    except: return []
                def getPixels(string=''):
                    widths_heights = []
                    from re import finditer, IGNORECASE, search
                    patterns = [
                        r'width\s*[:=]?\s*(\d{3,})\D+?height\s*[:=]?\s*(\d{3,})',
                        r'height\s*[:=]?\s*(\d{3,})\D+?width\s*[:=]?\s*(\d{3,})',
                        r'ancho\b.*?(\d{3,}).*?alto\b.*?(\d{3,})',
                        r'alto\b.*?(\d{3,}).*?ancho\b.*?(\d{3,})',
                        r'largura\b.*?(\d{3,}).*?altura\b.*?(\d{3,})',
                        r'altura\b.*?(\d{3,}).*?largura\b.*?(\d{3,})',
                        r'(\d{3,})\s*pixels\s*(?:[:x]|by|por)\s*(\d{3,})\s*pixels',
                        r'(\d{3,})\s*pixels.*?(\d{3,})\s*pixels',
                        r'(\d{3,})\s*(?:[:x]|by|por)\s*(\d{3,})',
                        r'(\d{3,})\s*(?:and|y|e)\s*(\d{3,})',
                        r'(\d{3,})\s*pixels\s*(?:de\s*)?(?:altura|alto|height)\s*(?:e|and|y|por)\s*(\d{3,})\s*(?:de\s*)?(?:largura|ancho|width)',
                        r'(\d{3,})\s*pixels\s*(?:de\s*)?(?:largura|ancho|width)\s*(?:e|and|y|por)\s*(\d{3,})\s*(?:de\s*)?(?:altura|alto|height)'
                    ]
                    for pattern in patterns:
                        for match in finditer(pattern, string, IGNORECASE):
                            numbers = match.groups()
                            if len(numbers) != 2: continue
                            number1, number2 = numbers
                            if int(number1) < 100 or int(number2) < 100: continue
                            match_text = match.group(0).lower()
                            if search(r'\b(width|largura|ancho)\b', match_text): width, height = number1, number2
                            elif search(r'\b(height|altura|alto)\b', match_text): height, width = number1, number2
                            else: width, height = number1, number2
                            widths_heights.append((match.start(), f'WIDTH_HEIGHT_{width}_{height}'))
                    widths_heights.sort()
                    final, seen = [], set()
                    for _, response in widths_heights:
                        if response not in seen: final.append(response), seen.add(response)
                    return final
                aspect_ratios = getAspectRatios(string=prompt)
                if len(aspect_ratios) > 0: tasks += aspect_ratios
                else: tasks += getPixels(string=prompt)
            if not self.hasCoincidentElements(vector1=['VIDEO_CREATION', 'VIDEO_EDITING', 'MUSIC_CREATION', 'MUSIC_EDITING', 'AUDIO_EDITING', 'VIDEO_INTERPRETATION'], vector2=tasks):
                def getTime(string=''):
                    times, matches = [], []
                    from re import compile, IGNORECASE
                    time_pattern = compile(r'(\d+)\s*:\s*([0-5]?\d)\s*:\s*([0-5]?\d)')
                    hour_pattern = compile(r'(?:(?:\b(?:hora|horas|hour|hours)\b)\s*(\d+)|(\d+)\s*(?:\b(?:hora|horas|hour|hours)\b))', IGNORECASE)
                    minute_pattern = compile(r'(?:(?:\b(?:minuto|minutos|minute|minutes)\b)\s*(\d+)|(\d+)\s*(?:\b(?:minuto|minutos|minute|minutes)\b))', IGNORECASE)
                    second_pattern = compile(r'(?:(?:\b(?:segundo|segundos|second|seconds)\b)\s*(\d+)|(\d+)\s*(?:\b(?:segundo|segundos|second|seconds)\b))', IGNORECASE)
                    for match in time_pattern.finditer(string):
                        hour, minute, second, start, end = int(match.group(1)), int(match.group(2)), int(match.group(3)), match.start(), match.end()
                        matches.append({'start': start, 'end': end, 'hour': hour, 'minute': minute, 'second': second, 'type': 'time'})
                    for match in hour_pattern.finditer(string):
                        start, end, number = match.start(), match.end(), match.group(1) or match.group(2)
                        hour = int(number)
                        matches.append({'start': start, 'end': end, 'hour': hour, 'type': 'hour'})
                    for match in minute_pattern.finditer(string):
                        start, end, number = match.start(), match.end(), match.group(1) or match.group(2)
                        minute = int(number)
                        matches.append({'start': start, 'end': end, 'minute': minute, 'type': 'minute'})
                    for match in second_pattern.finditer(string):
                        start, end, number = match.start(), match.end(), match.group(1) or match.group(2)
                        second = int(number)
                        matches.append({'start': start, 'end': end, 'second': second, 'type': 'second'})
                    matches.sort(key=lambda x: x['start'])
                    time_dict, last_type = {'hour': 0, 'minute': 0, 'second': 0}, None
                    for match in matches:
                        if match['type'] == 'time':
                            if any(time_dict.values()):
                                times.append('TIME_{0}_{1}_{2}'.format(time_dict['hour'], time_dict['minute'], time_dict['second']))
                                time_dict = {'hour': 0, 'minute': 0, 'second': 0}
                            times.append('TIME_{0}_{1}_{2}'.format(match['hour'], match['minute'], match['second']))
                            last_type = 'time'
                        else:
                            if time_dict[match['type']] == 0: time_dict[match['type']] = match.get(match['type'], 0)
                            else:
                                if any(time_dict.values()): times.append('TIME_{0}_{1}_{2}'.format(time_dict['hour'], time_dict['minute'], time_dict['second']))
                                time_dict = {'hour': 0, 'minute': 0, 'second': 0}
                                time_dict[match['type']] = match.get(match['type'], 0)
                            last_type = match['type']
                    if any(time_dict.values()): times.append('TIME_{0}_{1}_{2}'.format(time_dict['hour'], time_dict['minute'], time_dict['second']))
                    return times
                tasks += getTime(string=prompt)
            else:
                def getSeconds(string=''):
                    seconds_list = []
                    from re import finditer
                    pattern = r'(?i)(?:\b(\d+)\b\s*(second|seconds|segundo|segundos)\b|\b(second|seconds|segundo|segundos)\b\s*(\d+)\b)'
                    matches = finditer(pattern, string)
                    for match in matches:
                        number = match.group(1) if match.group(1) else match.group(4)
                        if number:
                            seconds = int(number)
                            if seconds > 0: seconds_list.append(f'SECONDS_{seconds}')
                    return seconds_list
                tasks += getSeconds(string=prompt)
            def getPage(string=''):
                pages = []
                from re import findall
                matches = findall(r'(?i)(?:page|página)\s*(\d+)|(\d+)\s*(?:page|página)', string)
                for matche in matches:
                    number_str = matche[0] or matche[1]
                    number = int(number_str)
                    if number > 0: pages.append(f'PAGE_{number}')
                return pages
            tasks += getPage(string=prompt)
            tasks = sorted(list(set(tasks)))
            return tasks
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getTasks: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return []
    def getCodeList(self, string='', html=False):
        try:
            code_list = []
            string, html, matches = str(string), bool(html) if type(html) in (bool, int, float) else False, []
            from re import findall, DOTALL
            if html:
                if '```html' in string: matches = findall(r'```html\s*(.*?)\s*```', string, DOTALL)
                elif '```javascript' in string: matches = findall(r'```javascript\s*(.*?)\s*```', string, DOTALL)
                elif '```js' in string: matches = findall(r'```js\s*(.*?)\s*```', string, DOTALL)
                elif string.strip().startswith('<'): return [string]
            else: matches = findall(r'```python\s*(.*?)\s*```', string, DOTALL)
            for match in matches:
                code = match.strip()
                if code: code_list.append(code)
            code_starters = ('#', '"""', "'''", '//', '/*', 'import', 'from', '<html', '<div', '```')
            formatted_string = str(string).lower().strip()
            if len(code_list) < 1 and formatted_string.startswith(code_starters): code_list = [string.replace('```', '')] if formatted_string.startswith('```') else [string]
            return code_list
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getCodeList: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return []
    def executePythonCode(self, string_code=''):
        try:
            from os import environ
            from io import StringIO
            from contextlib import redirect_stdout, redirect_stderr
            environ['OPENCV_LOG_LEVEL'] = 'ERROR'
            environ['OPENCV_LOG_LEVEL'] = 'SILENT'
            execution_error = ''
            try:
                with StringIO() as stdout_buffer, StringIO() as stderr_buffer, redirect_stdout(stdout_buffer), redirect_stderr(stderr_buffer):
                    try: exec(str(string_code))
                    except Exception as error: execution_error = 'The following error was encountered while trying to run the code, please fix it: '+str(error)
            except Exception as error: execution_error = 'The following error was encountered while trying to run the code, please fix it: '+str(error)
            return execution_error
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.executePythonCode: '+str(error)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return error_message
    def getWEBSearchList(self, prompt='', max_results=10, language=None, use_proxy=True):
        try:
            list_of_links = []
            prompt = str(prompt).strip()
            max_results = max((1, int(max_results))) if type(max_results) in (bool, int, float) else 10
            if type(language) != type(None) and len(str(language).strip()) > 0: language = str(language).strip()
            use_proxy = bool(use_proxy) if type(use_proxy) in (bool, int, float) else True
            try:
                from os import environ
                try: from certifi import where
                except:
                    self.installModule(module_name='certifi', version='2024.2.2')
                    from certifi import where
                environ['SSL_CERT_FILE'] = where()
                from logging import getLogger, ERROR
                getLogger('requests').setLevel(ERROR)
            except: pass
            def __webSearch(prompt='', max_results=10):
                list_of_links = []
                from urllib.parse import urlparse, urlencode
                domains_list = []
                def handle_starttag(tag='', attributes=[]):
                    if tag == 'a':
                        domains = set()
                        for attr in attributes:
                            if attr[0] == 'href':
                                url = attr[1]
                                if url.startswith('/url?q='):
                                    url = url.split('&')[0].replace('/url?q=', '')
                                    domain = urlparse(url).netloc
                                    if (domain and domain not in domains and self.__web_search+'/maps' not in url and self.__web_search not in domain) and domain not in domains_list:
                                        domains.add(domain)
                                        if len(url) > len('https://'+domain+'/'): list_of_links.append(url)
                                        domains_list.append(domain)
                from html.parser import HTMLParser
                parser = HTMLParser()
                parser.handle_starttag = handle_starttag
                query = urlencode({'q': prompt})
                from http.client import HTTPSConnection
                connection = HTTPSConnection('www.'+self.__web_search)
                connection.request('GET', '/search?'+query)
                response = connection.getresponse()
                html = response.read().decode('ISO-8859-1', errors='ignore')
                parser.feed(html)
                if len(list_of_links) < max_results:
                    connection.request('GET', '/search?'+query+'&start=10')
                    response = connection.getresponse()
                    html = response.read().decode('ISO-8859-1', errors='ignore')
                    parser.feed(html)
                return list_of_links[:max_results]
            def __webSearchWithProxy(prompt='', max_results=10):
                list_of_links = []
                query = prompt.replace(' ', '-')
                PROXY, GOOGLE_ROUTE = self.__proxy, f'https://www.{self.__web_search}/search?q='
                ROUTE = PROXY+GOOGLE_ROUTE+query
                try: from requests import get
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import get
                response, status_codes = get(ROUTE, timeout=self.__timeout), (200, 201)
                if response.status_code not in status_codes: response = get(GOOGLE_ROUTE+query, timeout=self.__timeout)
                if response.status_code in status_codes:
                    pattern = r'https://(.*?)&amp;'
                    content = str(response.content).strip()
                    from re import findall
                    list_of_links = findall(pattern, content)
                    list_of_links = [f'https://{url}' for url in list_of_links if self.__web_search not in url and '/' in url.split('/', 1)[1]]
                    list_of_links = list(set(list_of_links))
                return list_of_links[:max_results]
            if use_proxy:
                try:
                    list_of_links = __webSearchWithProxy(prompt=prompt, max_results=max_results)
                    if len(list_of_links) < 1: list_of_links = __webSearch(prompt=prompt, max_results=max_results)
                except: list_of_links = __webSearch(prompt=prompt, max_results=max_results)
            else:
                try:
                    list_of_links = __webSearch(prompt=prompt, max_results=max_results)
                    if len(list_of_links) < 1: list_of_links = __webSearchWithProxy(prompt=prompt, max_results=max_results)
                except: list_of_links = __webSearchWithProxy(prompt=prompt, max_results=max_results)
            if len(list_of_links) < 1: list_of_links = self.__getAlternativeWEBSearchList(prompt=prompt, max_results=max_results, language=language)
            return list_of_links
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getWEBSearchList: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return self.__getAlternativeWEBSearchList(prompt=prompt, max_results=max_results, language=language)
    def getTextFromWEBPage(self, file_path='', use_proxy=True):
        try:
            result_text = ''
            file_path = str(file_path).strip()
            use_proxy = bool(use_proxy) if type(use_proxy) in (bool, int, float) else True
            try:
                from os import environ
                try: from certifi import where
                except:
                    self.installModule(module_name='certifi', version='2024.2.2')
                    from certifi import where
                environ['SSL_CERT_FILE'] = where()
                from logging import getLogger, ERROR
                getLogger('requests').setLevel(ERROR)
            except: pass
            def extractParagraphsAndHeaders1(file_path='', use_proxy=True):
                result_text = ''
                try: from requests import get
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import get
                if use_proxy:
                    PROXY = self.__proxy
                    response = get(PROXY+file_path, timeout=self.__timeout)
                else: response = get(file_path, timeout=self.__timeout)
                if response.status_code not in (200, 201): return result_text
                response.encoding = 'utf-8'
                try: from bs4 import BeautifulSoup
                except:
                    self.installModule(module_name='beautifulsoup4', version='4.12.3')
                    from bs4 import BeautifulSoup
                beautiful_soup = BeautifulSoup(response.text, 'html.parser')
                from re import compile
                headers = beautiful_soup.find_all(compile('^h[1-6]$'))
                if len(headers) > 0:
                    for header in headers:
                        multiplier = int(header.name[1])
                        result_text += (multiplier*'#')+' '+header.get_text().strip()+'\n'
                        paragraph = header.find_next_sibling()
                        paragraph_name_lower = paragraph.name.lower()
                        while paragraph and not paragraph_name_lower in ('h1', 'h2', 'h3', 'h4', 'h5', 'h6'):
                            if paragraph_name_lower == 'p': result_text += paragraph.get_text().strip()+'\n'
                            paragraph = paragraph.find_next_sibling()
                else:
                    paragraphs = beautiful_soup.find_all('p')
                    for paragraph in paragraphs: result_text += paragraph.get_text().strip()+'\n'
                return result_text.strip()
            def extractParagraphsAndHeaders2(file_path='', use_proxy=True):
                result_text = ''
                from urllib.request import Request, urlopen
                if use_proxy:
                    PROXY = self.__proxy
                    request = Request(PROXY+file_path, headers={'User-Agent': 'Mozilla/5.0'})
                else: request = Request(file_path, headers={'User-Agent': 'Mozilla/5.0'})
                with urlopen(request) as response: html_content = response.read().decode('utf-8')
                from html.parser import HTMLParser
                class CustomHTMLParser(HTMLParser):
                    def __init__(self):
                        super().__init__()
                        self.in_header, self.current_header_level, self.result_text, self.in_paragraph, self.current_paragraph = False, 0, '', False, ''
                    def handle_starttag(self, tag, attributes):
                        if tag.startswith('h') and tag[1].isdigit() and 1 <= int(tag[1]) <= 6:
                            self.in_header, self.current_header_level = True, int(tag[1])
                            self.result_text += '#'*self.current_header_level+' '
                        elif tag == 'p': self.in_paragraph = True
                    def handle_endtag(self, tag):
                        if tag.startswith('h') and tag[1].isdigit() and 1 <= int(tag[1]) <= 6:
                            self.in_header = False
                            self.result_text += '\n'
                        elif tag == 'p':
                            self.in_paragraph = False
                            if self.current_paragraph:
                                self.result_text += self.current_paragraph + '\n'
                                self.current_paragraph = ''
                    def handle_data(self, data):
                        if self.in_header: self.result_text += data.strip()
                        elif self.in_paragraph: self.current_paragraph += data.strip()
                parser = CustomHTMLParser()
                parser.feed(html_content)
                return parser.result_text.strip()
            try: from requests.exceptions import ReadTimeout
            except:
                self.installModule(module_name='requests', version='2.31.0')
                from requests.exceptions import ReadTimeout
            try:
                try: result_text = extractParagraphsAndHeaders1(file_path=file_path, use_proxy=use_proxy)
                except ReadTimeout: return ''
                if len(result_text) < 1: result_text = extractParagraphsAndHeaders1(file_path=file_path, use_proxy=not use_proxy)
            except:
                try: result_text = extractParagraphsAndHeaders2(file_path=file_path, use_proxy=use_proxy)
                except ReadTimeout: return ''
                if len(result_text) < 1: result_text = extractParagraphsAndHeaders2(file_path=file_path, use_proxy=not use_proxy)
            return result_text
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getTextFromWEBPage: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            try: self.updateModule(module_name='certifi')
            except: pass
            return ''
    def getContentFromWEBLinks(self, list_of_links=[], max_tokens=1000):
        try:
            results_dictionary = {'text': '', 'sources': []}
            list_of_links = list(list_of_links) if type(list_of_links) in (tuple, list) else []
            max_tokens = int(max_tokens) if type(max_tokens) in (bool, int, float) else 1000
            number_of_links = len(list_of_links)
            if number_of_links < 1: return results_dictionary
            def getDomain(url=''):
                from urllib.parse import urlparse
                return urlparse(str(url).strip()).netloc
            tokens_per_page, link_count = max((1, max_tokens/len(list_of_links)-7)), 0
            for link in list_of_links:
                domain = getDomain(url=link)
                page_content = f'**source: {link}**\n'+self.getTextFromWEBPage(file_path=link, use_proxy=True)
                total_tokens = self.countTokens(string=page_content, pattern='gpt-4')
                if total_tokens > tokens_per_page: page_content = self.getTokensSummary(string=page_content, max_tokens=tokens_per_page)
                insert = '\n---\n\n' if link_count < number_of_links-1 else ''
                results_dictionary['text'] += page_content+insert
                results_dictionary['sources'].append({'domain': domain, 'url': link})
                link_count += 1
            results_dictionary['text'] = results_dictionary['text'].strip()
            return results_dictionary
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getContentFromWEBLinks: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'text': '', 'sources': []}
    def getAnswerForFiles(self, prompt='', language=None):
        try:
            complete_answer = 'Your file was generated successfully!\n\nI hope you like the result.'
            prompt = str(prompt).strip()
            from re import sub
            def removeTrailingPunctuation(text=''): return sub(r'[^\w\s]$', '', text)
            def lowercaseFirstLetter(text=''): return text[:1].lower() + text[1:]
            prompt = prompt.replace(':', '...').strip()
            prompt = lowercaseFirstLetter(text=removeTrailingPunctuation(text=prompt)).replace('\n', ' ').replace('  ', ' ').strip()
            language = self.getLanguage(string=prompt) if type(language) == type(None) or len(str(language).strip()) < 1 else str(language).lower().strip()
            first_paragraph, second_paragraph = '', ''
            if language.startswith('pt'):
                first_paragraph_pt1 = f'Tudo bem, acho que compreendi o que você quis dizer. Seu pedido foi: {prompt}. Com base nisso, fiz o possível para gerar um conteúdo conforme as especificações desejadas. Esforcei-me ao máximo para atender às suas necessidades, mas, se o resultado não ficou como esperado, você ainda pode refazer seu pedido usando outros termos ou sendo mais específico, para que eu entenda melhor. Lembre-se de que, assim como os humanos às vezes não interpretam corretamente o que outros dizem, uma Inteligência Artificial como eu também pode não compreendê-lo bem se a comunicação não for clara e objetiva.'
                first_paragraph_pt2 = f'Certo, entendi o que você quis dizer e seu pedido foi: {prompt}. Com base nisso, fiz o melhor possível para gerar um conteúdo de acordo com as especificações solicitadas. Dediquei-me ao máximo para atender às suas necessidades, mas, se o resultado não for o esperado, você pode refazer o pedido usando outros termos ou sendo mais específico, para que eu compreenda melhor. Lembre-se de que, assim como os humanos podem interpretar mal o que outros dizem, uma Inteligência Artificial como eu também pode não entendê-lo totalmente se a comunicação não for clara e objetiva.'
                first_paragraph_pt3 = f'Entendi o que você quis dizer e que seu pedido foi: {prompt}. Com base nisso, fiz o meu melhor para gerar um conteúdo conforme as especificações fornecidas. Procurei atender às suas expectativas da melhor forma possível, mas, se o resultado não for o desejado, você pode refazer o pedido com outros termos ou sendo mais específico, para que eu possa compreender melhor. Lembre-se de que, assim como os humanos, uma Inteligência Artificial como eu também pode ter dificuldade em entender corretamente se a comunicação não for clara e objetiva.'
                first_paragraph = self.__shuffleTuple(original_tuple=(first_paragraph_pt1, first_paragraph_pt2, first_paragraph_pt3))[0]
                second_paragraph_pt1 = f'O conteúdo foi criado conforme o que entendi do seu pedido e inclui tudo o que consegui compreender da sua solicitação, aproximando-se o máximo possível do que você deseja. Ainda assim, podem existir diferenças entre o que você esperava e o que foi gerado, caso algum detalhe da descrição não tenha sido bem explicado ou se alguma característica desejada estiver além das minhas capacidades técnicas. Espero sinceramente que o resultado atenda às suas necessidades. Se tiver dúvidas sobre o meu trabalho ou quiser fazer uma nova solicitação, é só pedir que estarei pronto para ajudar. Conte comigo; sempre me dedicarei ao máximo para tornar seu trabalho mais produtivo.'
                second_paragraph_pt2 = f'O conteúdo foi criado conforme minha interpretação do seu pedido e inclui tudo o que consegui entender da sua solicitação, buscando atender ao que você deseja da melhor forma possível. No entanto, podem ocorrer diferenças entre o que você esperava e o que foi produzido, caso algum detalhe da descrição não tenha sido claro ou se alguma característica desejada estiver além das minhas habilidades técnicas. Espero sinceramente que o resultado atenda às suas necessidades. Se tiver dúvidas sobre o trabalho ou quiser fazer uma nova solicitação, estou à disposição para ajudar. Conte comigo; sempre me empenharei para tornar seu trabalho mais eficiente.'
                second_paragraph_pt3 = f'O conteúdo foi criado com base no que entendi do seu pedido e inclui tudo o que consegui captar da sua solicitação, procurando atender ao máximo ao que você deseja. Contudo, podem existir diferenças entre o que você imaginava e o que foi gerado, caso algum detalhe da descrição não tenha sido totalmente claro ou se alguma característica desejada ultrapassar minhas capacidades técnicas. Espero sinceramente que o resultado corresponda às suas expectativas. Se tiver dúvidas ou quiser fazer uma nova solicitação, estou à disposição para ajudar. Conte comigo; sempre me dedicarei para tornar seu trabalho mais produtivo.'
                second_paragraph = self.__shuffleTuple(original_tuple=(second_paragraph_pt1, second_paragraph_pt2, second_paragraph_pt3))[0]
            elif language.startswith('es'):
                first_paragraph_es1 = f'Está bien, creo que comprendí lo que quisiste decir. Tu solicitud fue: {prompt}. Basado en eso, hice todo lo posible para generar un contenido con las especificaciones deseadas. Di lo mejor de mí para satisfacer tus necesidades, pero si el resultado no fue como esperabas, aún puedes rehacer tu solicitud usando otros términos o siendo más específico, para que pueda entenderte mejor. Recuerda que, al igual que los humanos pueden malinterpretar a otros, una Inteligencia Artificial como yo también puede no comprenderte bien si no te expresas con claridad y objetividad.'
                first_paragraph_es2 = f'Entendido, comprendí lo que querías decir y tu solicitud fue: {prompt}. Con base en eso, hice mi mejor esfuerzo para generar un contenido con las especificaciones solicitadas. Me esforcé al máximo para satisfacer tus necesidades, pero si el resultado no es el esperado, aún puedes rehacer la solicitud usando otros términos o siendo más específico, para que pueda entenderte mejor. Recuerda que, al igual que los humanos a veces no interpretan correctamente a otros, una Inteligencia Artificial como yo también puede no comprenderte del todo si no te expresas con claridad y precisión.'
                first_paragraph_es3 = f'Comprendí lo que querías decir y que tu solicitud fue: {prompt}. Con base en ello, me esforcé al máximo para generar un contenido de acuerdo con las especificaciones proporcionadas. Hice todo lo posible por cumplir con tus expectativas, pero si el resultado no es el deseado, podrás rehacer la solicitud usando otros términos o siendo más específico, para que pueda entenderte mejor. Recuerda que, al igual que los humanos, una Inteligencia Artificial como yo también puede tener dificultades para entender completamente si la comunicación no es clara y objetiva.'
                first_paragraph = self.__shuffleTuple(original_tuple=(first_paragraph_es1, first_paragraph_es2, first_paragraph_es3))[0]
                second_paragraph_es1 = f'El contenido fue creado según lo que entendí de tu solicitud e incluye todo lo que pude comprender de tu descripción, acercándose lo más posible a lo que deseas. Aun así, pueden existir diferencias entre lo que esperabas y lo que se generó, si algún detalle de tu descripción no fue bien explicado o si alguna característica deseada supera mis capacidades técnicas. Espero sinceramente que el resultado cumpla con tus necesidades. Si tienes dudas sobre mi trabajo o deseas hacer una nueva solicitud, solo pídelo y estaré listo para ayudar. Cuenta conmigo; siempre daré mi máximo para hacer tu trabajo más productivo.'
                second_paragraph_es2 = f'El contenido fue creado de acuerdo con mi interpretación de tu solicitud e incluye todo lo que pude entender de tu descripción, buscando cumplir lo que deseas de la mejor manera posible. Sin embargo, pueden existir diferencias entre lo que esperabas y lo que se generó, si algún detalle de tu descripción no fue suficientemente claro o si alguna característica deseada está fuera de mis capacidades técnicas. Espero sinceramente que el resultado cumpla con tus expectativas. Si tienes dudas o necesitas hacer una nueva solicitud, estoy disponible para ayudar. Cuenta conmigo; siempre me esforzaré para hacer tu trabajo más eficiente.'
                second_paragraph_es3 = f'El contenido fue creado basándome en lo que entendí de tu solicitud e incluye todo lo que logré captar de tu descripción, procurando satisfacer al máximo tus necesidades. Sin embargo, puede haber diferencias entre lo que imaginabas y lo que se generó si algún detalle no fue totalmente claro o si alguna característica excede mis capacidades técnicas. Espero sinceramente que el resultado cumpla con tus expectativas. Si tienes dudas o deseas hacer otra solicitud, estoy disponible para ayudar. Cuenta conmigo; siempre daré mi máximo para hacer tu trabajo más productivo.'
                second_paragraph = self.__shuffleTuple(original_tuple=(second_paragraph_es1, second_paragraph_es2, second_paragraph_es3))[0]
            else:
                first_paragraph_en1 = f"Alright, I believe I understood what you meant. Your request was: {prompt}. Based on that, I did my best to create content according to the desired specifications. I put in my full effort to meet your needs, but if it didn’t turn out as expected, you can still redo your request using different terms or being more specific so I can understand better. Remember that, just like humans who sometimes misinterpret others, an Artificial Intelligence like me might also not understand you well if you don’t express yourself clearly and objectively."
                first_paragraph_en2 = f"Sure, I understood what you meant, and your request was: {prompt}. Based on that, I did my best to produce content according to the requested specifications. I worked hard to meet your needs, but if the result isn’t what you expected, you can redo your request using different terms or being more specific so I can better understand you. Remember that, just like humans may misinterpret what others say, an Artificial Intelligence like me may also fail to understand you completely if you don’t express yourself clearly and directly."
                first_paragraph_en3 = f"I understood what you meant and that your request was: {prompt}. Based on that, I did my utmost to generate content according to the provided specifications. I did everything possible to meet your expectations, but if the result isn’t as desired, you can redo your request using different terms or being more specific so I can better understand you. Remember that, just like humans, an Artificial Intelligence like me may also struggle to fully understand if the communication isn’t clear and objective."
                first_paragraph = self.__shuffleTuple(original_tuple=(first_paragraph_en1, first_paragraph_en2, first_paragraph_en3))[0]
                second_paragraph_en1 = f"The content was created based on my understanding of your request and includes everything I could comprehend from your description, aiming to align as closely as possible with what you want. Still, there may be differences between what you expected and what was generated if some details weren’t well explained or if certain desired features exceed my technical abilities. I sincerely hope the result meets your needs. If you have any questions or would like to make a new request, just let me know, and I’ll be glad to help. Count on me; I’ll always do my best to make your work more productive."
                second_paragraph_en2 = f"The content was created according to my interpretation of your request and includes everything I could understand from your description, aiming to meet your needs as best as possible. However, there may be differences between what you expected and what was produced if some details were unclear or if certain desired aspects exceed my technical capacity. I truly hope the result meets your expectations. If you have any questions or wish to make a new request, I’m available to assist. Count on me; I’ll always dedicate myself fully to making your work more efficient."
                second_paragraph_en3 = f"The content was produced based on my understanding of your request and includes everything I managed to interpret from your description, striving to meet your needs as much as possible. However, please note that there might be some differences between what you envisioned and what was generated if any details weren’t completely clear or if some features go beyond my technical abilities. I sincerely hope the result meets your expectations. If you have questions or wish to make another request, I’m ready to assist. Count on me; I’ll always give my best to make your work more productive."
                second_paragraph = self.__shuffleTuple(original_tuple=(second_paragraph_en1, second_paragraph_en2, second_paragraph_en3))[0]
                if not language.startswith('en'): first_paragraph, second_paragraph = self.translate(string=first_paragraph, source_language='auto', target_language=language), self.translate(string=second_paragraph, source_language='auto', target_language=language)
            complete_answer = first_paragraph+'\n\n'+second_paragraph
            return complete_answer
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getAnswerForFiles: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return 'Your order was generated successfully!\n\nI hope you like the result.'
    def getDefaultAnswer(self, prompt='', messages=[], language=None):
        try:
            default_answer = 'Sorry, I cannot accommodate your request at this time.'
            prompt = str(prompt).strip()
            messages = list(messages) if type(messages) in (tuple, list) else []
            if not language:
                prompt_content = str(prompt).strip()
                if not prompt_content:
                    prompt_messages = messages[-1] if len(messages) > 0 else {'role': 'user', 'content': ''}
                    prompt_role, prompt_content = str(prompt_messages.get('role', '')).lower().strip(), ''
                    if prompt_role == 'user': prompt_content = str(prompt_messages.get('content', '')).strip()
                language = self.getLanguage(string=prompt_content)
            if language.startswith('pt'): default_answer = 'Desculpe, não posso atender a sua solicitação no momento.'
            elif language.startswith('es'): default_answer = 'Lo siento, no puedo atender su solicitud en este momento.'
            else:
                default_answer = 'Sorry, I cannot accommodate your request at this time.'
                if not language.startswith('en'): default_answer = self.translate(string=default_answer, source_language='auto', target_language=language)
            return default_answer
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getDefaultAnswer: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return 'Sorry, I cannot accommodate your request at this time.'            
    def addUserAndTimeToPrompt(self, prompt='', username='', date_and_time='', language='en'):
        try:
            prompt, username, date_and_time = str(prompt).strip(), str(username).strip(), str(date_and_time).strip()
            if len(prompt) > 0 and len(username) > 0:
                language = language.lower().strip() if type(language) == str else 'en'
                if len(language) < 1: language = 'en'
                if language.startswith('es'): inclusion = 'MI NOMBRE ES <|username|> Y LA FECHA Y HORA ACTUAL ES <|date_and_time|>'
                elif language.startswith('pt'): inclusion = 'MEU NOME É <|username|> E A DATA E HORA ATUAL É <|date_and_time|>'
                else: inclusion = 'MY NAME IS <|username|> AND THE CURRENT DATE AND TIME IS <|date_and_time|>'
                if len(date_and_time) < 1:
                    def getCurrentDateAndTime():
                        from datetime import datetime
                        now = datetime.now()
                        day_of_week = now.strftime('%A')
                        if not language.startswith('en'): day_of_week = self.translate(string=day_of_week, source_language='auto', target_language=language)
                        if language.startswith('en'): formatted_date_time = now.strftime('%m/%d/%Y %I:%M:%S %p')
                        else: formatted_date_time = now.strftime('%d/%m/%Y %H:%M:%S')
                        return f'{day_of_week}, {formatted_date_time}'
                    date_and_time = getCurrentDateAndTime()
                inclusion = inclusion.replace('<|username|>', username).replace('<|date_and_time|>', date_and_time)
                prompt = f'{inclusion}\n\n{prompt}'
            return prompt
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.addUserAndTimeToPrompt: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return prompt
    def removeUserAndTimeFromPrompt(self, prompt=''):
        try:
            prompt = str(prompt).strip()
            if len(prompt) > 0 and '\n\n' in prompt:
                first_line = prompt.split('\n\n')[0]
                if first_line.startswith('MY NAME IS') and 'AND THE CURRENT DATE AND TIME IS' in first_line: prompt = prompt.replace(first_line, '').strip()
                elif first_line.startswith('MI NOMBRE ES') and 'Y LA FECHA Y HORA ACTUAL ES' in first_line: prompt = prompt.replace(first_line, '').strip()
                elif first_line.startswith('MEU NOME É') and 'E A DATA E HORA ATUAL É' in first_line: prompt = prompt.replace(first_line, '').strip()
            return prompt
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.removeUserAndTimeFromPrompt: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return prompt
    def addSystemIntonation(self, system_instruction='', creativity=0.5, humor=0.5, formality=0.5, political_spectrum=0.5, language='en'):
        try:
            system_instruction = str(system_instruction).strip()
            creativity = min((1, max((0, float(creativity))))) if type(creativity) in (bool, int, float) else .5
            humor = min((1, max((0, float(humor))))) if type(humor) in (bool, int, float) else .5
            formality = min((1, max((0, float(formality))))) if type(formality) in (bool, int, float) else .5
            political_spectrum = min((1, max((0, float(political_spectrum))))) if type(political_spectrum) in (bool, int, float) else .5
            if sum((creativity, humor, formality, political_spectrum)) != 2:
                language, system_personality, personality_description = language.lower().strip() if type(language) == str else 'en', [], ''
                if len(language) < 1: language = self.getLanguage(string=system_instruction)
                if len(language) < 1: language = 'en'
                creativity_levels_en = ('without creativity, dumb and uncultured', 'without creativity and uncultured', 'without creativity', 'self-reflective', 'creative', 'creative and self-reflective', 'creative and breaking down reasoning into steps', 'creative and breaking down reasoning into detailed steps', 'creative and breaking down reasoning into detailed steps to be followed before responding', 'creative and breaking down reasoning into detailed steps like a chain of thought to be followed before responding', 'crazy and delusional')
                humor_levels_en = ('depressed, unmotivated, and pessimistic', 'depressed', 'pessimistic', 'unmotivated', 'serious', 'serious and calm', 'good-humored', 'funny', 'jokester', 'insufferable jokester', 'insufferable jokester with bad-taste jokes')
                formality_levels_en = ('informal', 'casual', 'friendly', 'detailed', 'formal', 'technical', 'technical and formal', 'technical, formal, and detailed', 'extremely technical and formal', 'extremely technical, formal, and polite', 'extremely technical, formal, polite, and classy')
                political_spectrum_levels_en = ('communist', 'far-left', 'left-wing nationalist', 'progressive', 'left-wing', 'center', 'right-wing', 'right-wing conservative', 'right-wing nationalist', 'anarcho-capitalist', 'far-right')
                creativity_levels_es = ('sin creatividad, tonto e inculto', 'sin creatividad e inculto', 'sin creatividad', 'autorreflexivo', 'creativo', 'creativo y autorreflexivo', 'creativo y dividiendo el razonamiento en etapas', 'creativo y dividiendo el razonamiento en etapas detalladas', 'creativo y dividiendo el razonamiento en etapas detalladas para seguir antes de responder', 'creativo y dividiendo el razonamiento en etapas detalladas como una cadena de pensamiento para seguir antes de responder', 'loco y alucinado')
                humor_levels_es = ('deprimido, desmotivado y pesimista', 'deprimido', 'pesimista', 'desmotivado', 'serio', 'serio y tranquilo', 'de buen humor', 'gracioso', 'bromista', 'bromista insoportable', 'bromista insoportable con chistes de mal gusto')
                formality_levels_es = ('informal', 'relajado', 'amistoso', 'detallado', 'formal', 'técnico', 'técnico y formal', 'técnico, formal y detallado', 'extremadamente técnico y formal', 'extremadamente técnico, formal y educado', 'extremadamente técnico, formal, educado y con clase')
                political_spectrum_levels_es = ('comunista', 'extrema izquierda', 'nacionalista de izquierda', 'progresista', 'izquierda', 'centro', 'derecha', 'conservador de derecha', 'nacionalista de derecha', 'anarcocapitalista', 'extrema derecha')
                creativity_levels_pt = ('sem criatividade, burro e inculto', 'sem criatividade e inculto', 'sem criatividade', 'auto reflexivo', 'criativo', 'criativo e auto reflexivo', 'criativo e dividindo o raciocínio em etapas', 'criativo e dividindo o raciocínio em etapas detalhadas', 'criativo e dividindo o raciocínio em etapas detalhadas para serem seguidas antes de responder', 'criativo e dividindo o raciocínio em etapas detalhadas como uma cadeia de pensamento para ser seguida antes de responder', 'louco e alucinado')
                humor_levels_pt = ('deprimido, desanimado e pessimista', 'deprimido', 'pessimista', 'desanimado', 'sério', 'sério e calmo', 'bem humorado', 'engraçado', 'piadista', 'piadista insuportável', 'piadista insuportável e com piadas de mau gosto')
                formality_levels_pt = ('informal', 'descontraído', 'amigável', 'detalhado', 'formal', 'técnico', 'técnico e formal', 'técnico, formal e detalhado', 'extremamente técnico e formal', 'extremamente técnico, formal e educado', 'extremamente técnico, formal, educado e com classe')
                political_spectrum_levels_pt = ('comunista', 'extrema esquerda', 'nacionalista de esquerda', 'progressista', 'esquerda', 'centro', 'direita', 'conservador de direita', 'nacionalista de direita', 'anarcocapitalista', 'extrema direita')
                creativity_label_en = 'Your creativity level is: '
                humor_label_en = 'Your humor is: '
                formality_label_en = 'You speak in the following way: '
                political_spectrum_label_en = 'Your political bias is: '
                creativity_label_es = 'Su nivel de creatividad es: '
                humor_label_es = 'Su estado de ánimo es: '
                formality_label_es = 'Usted habla de la siguiente manera: '
                political_spectrum_label_es = 'Su inclinación política es: '
                creativity_label_pt = 'Seu nível de criatividade é: '
                humor_label_pt = 'Seu humor é: '
                formality_label_pt = 'Você fala da seguinte forma: '
                political_spectrum_label_pt = 'Seu viés político é: '
                another_language = False
                if language.startswith('en'):
                    creativity_levels, humor_levels, formality_levels, political_spectrum_levels = creativity_levels_en, humor_levels_en, formality_levels_en, political_spectrum_levels_en
                    creativity_label, humor_label, formality_label, political_spectrum_label = creativity_label_en, humor_label_en, formality_label_en, political_spectrum_label_en
                elif language.startswith('es'):
                    creativity_levels, humor_levels, formality_levels, political_spectrum_levels = creativity_levels_es, humor_levels_es, formality_levels_es, political_spectrum_levels_es
                    creativity_label, humor_label, formality_label, political_spectrum_label = creativity_label_es, humor_label_es, formality_label_es, political_spectrum_label_es
                elif language.startswith('pt'):
                    creativity_levels, humor_levels, formality_levels, political_spectrum_levels = creativity_levels_pt, humor_levels_pt, formality_levels_pt, political_spectrum_levels_pt
                    creativity_label, humor_label, formality_label, political_spectrum_label = creativity_label_pt, humor_label_pt, formality_label_pt, political_spectrum_label_pt
                else:
                    another_language = True
                    creativity_levels, humor_levels, formality_levels, political_spectrum_levels = creativity_levels_en, humor_levels_en, formality_levels_en, political_spectrum_levels_en
                    creativity_label, humor_label, formality_label, political_spectrum_label = creativity_label_en, humor_label_en, formality_label_en, political_spectrum_label_en
                try: from numpy import array, abs, argmin
                except:
                    self.installModule(module_name='numpy', version='1.25.2')
                    from numpy import array, abs, argmin
                def getIndex(value=0.5): return argmin(abs(array((0.0, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0))-value))
                if creativity != 0.5: system_personality.append(creativity_label+creativity_levels[getIndex(value=creativity)])
                if humor != 0.5: system_personality.append(humor_label+humor_levels[getIndex(value=humor)])
                if formality != 0.5: system_personality.append(formality_label+formality_levels[getIndex(value=formality)])
                if political_spectrum != 0.5: system_personality.append(political_spectrum_label+political_spectrum_levels[getIndex(value=political_spectrum)])
                if len(system_personality) > 0: personality_description = str('\n'.join(system_personality)).strip()
                if another_language: personality_description = self.translate(string=personality_description, source_language='auto', target_language=language)
                if len(personality_description) > 0: system_instruction += '\n\n'+personality_description
            return system_instruction
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.addSystemIntonation: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return system_instruction
    def formatPrompt(self, prompt='', task_names=[], language=None):
        try:
            prompt = str(prompt).strip()
            task_names = list(task_names) if type(task_names) in (tuple, list) else [str(task_names).upper().strip()]
            if len(task_names) < 1: return prompt
            language = self.getLanguage(string=prompt) if type(language) == type(None) or len(str(language).strip()) < 1 else str(language).lower().strip()
            is_it_translation_or_summary = any(task.startswith('TRANSLATION_') for task in task_names) or 'TEXT_SUMMARY' in task_names or 'TEXT_SUMMARY_WITH_BULLET_POINTS' in task_names
            formatted_prompt = prompt.lower().strip()
            if formatted_prompt.startswith('/artifact'): prompt = self.replaceCaseInsensitive(string=prompt, old='/artifact', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/web'): prompt = self.replaceCaseInsensitive(string=prompt, old='/web', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/image'): prompt = self.replaceCaseInsensitive(string=prompt, old='/image', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/logo'): prompt = self.replaceCaseInsensitive(string=prompt, old='/logo', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/audio'): prompt = self.replaceCaseInsensitive(string=prompt, old='/audio', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/music'): prompt = self.replaceCaseInsensitive(string=prompt, old='/music', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/video'): prompt = self.replaceCaseInsensitive(string=prompt, old='/video', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/pdf'): prompt = self.replaceCaseInsensitive(string=prompt, old='/pdf', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/wordcloud'): prompt = self.replaceCaseInsensitive(string=prompt, old='/wordcloud', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/word'): prompt = self.replaceCaseInsensitive(string=prompt, old='/word', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/excel'): prompt = self.replaceCaseInsensitive(string=prompt, old='/excel', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/csv'): prompt = self.replaceCaseInsensitive(string=prompt, old='/csv', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/powerpoint'): prompt = self.replaceCaseInsensitive(string=prompt, old='/powerpoint', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/chart'): prompt = self.replaceCaseInsensitive(string=prompt, old='/chart', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/flowchart'): prompt = self.replaceCaseInsensitive(string=prompt, old='/flowchart', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/youtube-video-download'): prompt = self.replaceCaseInsensitive(string=prompt, old='/youtube-video-download', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/youtube-audio-download'): prompt = self.replaceCaseInsensitive(string=prompt, old='/youtube-audio-download', new='', first_occurrence=False, ignore_accents=True)
            elif formatted_prompt.startswith('/deep-reasoning'): prompt = self.replaceCaseInsensitive(string=prompt, old='/deep-reasoning', new='', first_occurrence=False, ignore_accents=True)
            prompt, another_language = prompt.strip(), False
            if not language.startswith(('en', 'es', 'pt')):
                prompt = self.translate(string=prompt, source_language='auto', target_language='en')
                another_language = True
            if ('IMAGE_CREATION' in task_names or 'LOGO_CREATION' in task_names) and (not is_it_translation_or_summary):
                image_texts_en = ('create an image of', 'create a photo of', 'generate an image of', 'generate a photo of', 'make an image of', 'make a photo of', 'create a logo', 'generate a logo', 'make a logo')
                image_texts_es = ('crea una imagen de', 'crea una foto de', 'genera una imagen de', 'genera una foto de', 'haz una imagen de', 'haz una foto de', 'crea un logo', 'genera un logo', 'haz un logo')
                image_texts_pt = ('crie uma imagem de', 'crie uma foto de', 'gere uma imagem de', 'gere uma foto de', 'faça uma imagem de', 'faça uma foto de', 'crie um logo', 'gere um logo', 'faça um logo')
                if language.startswith('en'): image_texts = image_texts_en
                elif language.startswith('es'): image_texts = image_texts_es
                elif language.startswith('pt'): image_texts = image_texts_pt
                else: image_texts = image_texts_en+image_texts_es+image_texts_pt
                for image_text in image_texts:
                    if image_text in prompt.lower():
                        prompt = self.replaceCaseInsensitive(string=prompt, old=image_text, new='', first_occurrence=False, ignore_accents=True)
                        break
            if ('IMAGE_FILTER_APPLICATION' in task_names) and (not is_it_translation_or_summary):
                filter_texts_en = ('apply a filter', 'apply a style')
                filter_texts_es = ('aplica un filtro', 'aplica un estilo')
                filter_texts_pt = ('aplique um filtro', 'aplique um estilo')
                if language.startswith('en'): filter_texts = filter_texts_en
                elif language.startswith('es'): filter_texts = filter_texts_es
                elif language.startswith('pt'): filter_texts = filter_texts_pt
                else: filter_texts = filter_texts_en+filter_texts_es+filter_texts_pt
                for filter_text in filter_texts:
                    if filter_text in prompt.lower():
                        prompt = self.replaceCaseInsensitive(string=prompt, old=filter_text, new='', first_occurrence=False, ignore_accents=True)
                        break
            if ('MUSIC_CREATION' in task_names) and (not is_it_translation_or_summary):
                music_texts_en = ('create a song', 'generate a song', 'make a song', 'create a melody', 'generate a melody', 'make a melody', 'create a sound', 'generate a sound', 'make a sound', 'create an audio', 'generate an audio', 'make an audio', 'create a beat', 'generate a beat', 'make a beat')
                music_texts_es = ('crea una canción', 'genera una canción', 'haz una canción', 'crea una melodía', 'genera una melodía', 'haz una melodía', 'crea un sonido', 'genera un sonido', 'haz un sonido', 'crea un audio', 'genera un audio', 'haz un audio', 'crea un ritmo', 'genera un ritmo', 'haz un ritmo')
                music_texts_pt = ('crie uma música', 'gere uma música', 'faça uma música', 'crie uma melodia', 'gere uma melodia', 'faça uma melodia', 'crie um som', 'gere um som', 'faça um som', 'crie um áudio', 'gere um áudio', 'faça um áudio', 'crie uma batida', 'gere uma batida', 'faça uma batida')
                if language.startswith('en'): music_texts = music_texts_en
                elif language.startswith('es'): music_texts = music_texts_es
                elif language.startswith('pt'): music_texts = music_texts_pt
                else: music_texts = music_texts_en+music_texts_es+music_texts_pt
                for music_text in music_texts:
                    if music_text in prompt.lower():
                        prompt = self.replaceCaseInsensitive(string=prompt, old=music_text, new='', first_occurrence=False, ignore_accents=True)
                        break
            if ('VIDEO_CREATION' in task_names) and (not is_it_translation_or_summary):
                video_texts_en = ('create a video of', 'generate a video of', 'make a video of')
                video_texts_es = ('crea un video de', 'genera un video de', 'haz un video de')
                video_texts_pt = ('crie um vídeo de', 'gere um vídeo de', 'faça um vídeo de')
                if language.startswith('en'): video_texts = video_texts_en
                elif language.startswith('es'): video_texts = video_texts_es
                elif language.startswith('pt'): video_texts = video_texts_pt
                else: video_texts = video_texts_en+video_texts_es+video_texts_pt
                for video_text in video_texts:
                    if video_text in prompt.lower():
                        prompt = self.replaceCaseInsensitive(string=prompt, old=video_text, new='', first_occurrence=False, ignore_accents=True)
                        break
            if ('WEB_SEARCH' in task_names) and (not is_it_translation_or_summary):
                web_search_texts_en = ('search the', 'look up on the', 'browse the', 'do a search', 'search', 'look up', 'browse')
                web_search_texts_es = ('buscar en', 'navegar en', 'hacer una búsqueda', 'buscar', 'navegar')
                web_search_texts_pt = ('pesquise na', 'procure na', 'busque na', 'faça uma pesquisa', 'pesquise', 'procure', 'busque')
                if language.startswith('en'): web_search_texts, separator = web_search_texts_en, ' and '
                elif language.startswith('es'): web_search_texts, separator = web_search_texts_es, ' y '
                elif language.startswith('pt'): web_search_texts, separator = web_search_texts_pt, ' e '
                else: web_search_texts, separator = web_search_texts_en+web_search_texts_es+web_search_texts_pt, ' and '
                if separator in prompt: prompt = self.replaceLastOccurrence(string=prompt, old=separator, new='<|end|>', case_insensitive=False)
                elif ', ' in prompt: prompt = self.replaceLastOccurrence(string=prompt, old=separator, new='<|end|>')
                for web_search_text in web_search_texts:
                    if web_search_text in prompt.lower():
                        prompt = self.replaceCaseInsensitive(string=prompt, old=web_search_text, new='', first_occurrence=False, ignore_accents=True)
                        if 'internet' in prompt.lower(): prompt = self.replaceCaseInsensitive(string=prompt, old='internet', new='', first_occurrence=True, ignore_accents=False)
                        if 'web' in prompt.lower(): prompt = self.replaceCaseInsensitive(string=prompt, old='web', new='', first_occurrence=True, ignore_accents=False)
                        if 'online' in prompt.lower(): prompt = self.replaceCaseInsensitive(string=prompt, old='online', new='', first_occurrence=True, ignore_accents=False)
                        if 'on-line' in prompt.lower(): prompt = self.replaceCaseInsensitive(string=prompt, old='on-line', new='', first_occurrence=True, ignore_accents=False)
                        break
                if '<|end|>' in prompt: prompt = prompt.split('<|end|>')[0]
                prompt = self.getKeyWordsString(string=prompt)
            if is_it_translation_or_summary:
                translation_texts_en = ('translate', 'translation', 'summarize', 'summary')
                translation_texts_es = ('traduce', 'traducir', 'traducción', 'resumir', 'resumen', 'resumir')
                translation_texts_pt = ('traduza', 'traduzir', 'tradução', 'resuma', 'resumo', 'resumir')
                if language.startswith('en'): translation_texts = translation_texts_en
                elif language.startswith('es'): translation_texts = translation_texts_es
                elif language.startswith('pt'): translation_texts = translation_texts_pt
                else: translation_texts = translation_texts_en+translation_texts_es+translation_texts_pt
                if ':' in prompt:
                    prompt = prompt.replace(':', '<|end|>', 1)
                    start, end = prompt.split('<|end|>')
                    for translation_text in translation_texts:
                        if translation_text in start.lower():
                            prompt = end
                            break
                elif '\n' in prompt:
                    rows = prompt.split('\n')
                    for translation_text in translation_texts:
                        if translation_text in rows[0].lower():
                            rows = rows[1:]
                            prompt = '\n'.join(rows)
                            break
                        elif translation_text in rows[-1].lower():
                            rows = rows[:-1]
                            prompt = '\n'.join(rows)
                            break
            if another_language: prompt = self.translate(string=prompt, source_language='auto', target_language=language)
            prompt = prompt.strip().capitalize()
            return prompt
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.formatPrompt: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return prompt
    def executeModel(self, URL='', body={}, headers={}):
        try:
            json_result = {}
            total_tokens, main_result, total_cost = 0, '', 0
            URL = str(URL).strip()
            if type(body) != dict: body = {}
            if type(headers) != dict: headers = {}
            try: from requests import post, get
            except:
                self.installModule(module_name='requests', version='2.31.0')
                from requests import post, get
            url_lower = URL.lower()
            if 'anthropic' in url_lower:
                from json import dumps
                prompt = str(body['prompt']) if 'prompt' in body else ''
                if len(prompt.strip()) > 0:
                    if '\n\nHuman: ' not in prompt: prompt = '\n\nHuman: '+prompt
                    if '\n\nAssistant:' not in prompt: prompt = prompt+'\n\nAssistant:'
                    body['prompt'] = prompt
                response = post(URL, data=dumps(body), headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_data = json_result = response.json()
                    if 'messages' in url_lower:
                        if 'content' in response_data:
                            content = response_data['content']
                            if type(content) in (tuple, list) and len(content) > 0:
                                content = content[0]
                                if 'text' in content: main_result, total_tokens = str(content['text']).strip(), 0
                        if 'usage' in response_data:
                            usage, string_body = response_data['usage'], str(body)
                            total_tokens += int(usage['input_tokens']) if 'input_tokens' in usage and type(usage['input_tokens']) in (int, float) else 0
                            total_tokens += int(usage['output_tokens']) if 'output_tokens' in usage and type(usage['output_tokens']) in (int, float) else 0
                            if 'haiku' in string_body: cost_per_token = (0.25+1.25)/1000000
                            elif 'sonnet' in string_body: cost_per_token = (3+15)/1000000
                            elif 'opus' in string_body: cost_per_token = (15+75)/1000000
                            else: cost_per_token = (15+80)/1000000
                            total_cost += total_tokens*cost_per_token
                            if 'image' in string_body and 'source' in string_body: total_cost += 0.0048
                    elif 'complete' in url_lower:
                        main_result = str(response_data['completion']).strip() if 'completion' in response_data else ''
                        input_output = prompt.strip()+' '+main_result
                        total_tokens, string_body = self.countTokens(string=input_output, pattern='gpt-3'), str(body)
                        if 'claude-instant' in string_body: cost_per_token = (0.8+2.4)/1000000
                        elif 'claude-2' in string_body: cost_per_token = (8+24)/1000000
                        else: cost_per_token = (8+25)/1000000
                        total_cost += total_tokens*cost_per_token
            elif 'openai' in url_lower and 'deepinfra' not in url_lower:
                response, string_body = post(URL, json=body, headers=headers, timeout=self.__timeout), str(body)
                if response.status_code in (200, 201):
                    response_data = json_result = response.json()
                    if 'images/generations' in url_lower:
                        size = str(body['size']).strip() if 'size' in body else '1024x1024'
                        sizes_dalle2, sizes_dalle3 = ('256x256', '512x512', '1024x1024'), ('1024x1024', '1792x1024', '1024x1792')
                        width = int(size.split('x')[0]) if 'x' in size else 1024
                        if 'dall-e-2' in string_body and size not in sizes_dalle2:
                            if width <= 256: size, cost_per_execution = '256x256', 0.016
                            elif width <= 512: size, cost_per_execution = '512x512', 0.018
                            else: size, cost_per_execution = '1024x1024', 0.020
                        elif 'dall-e-3' in string_body and size not in sizes_dalle3:
                            if width <= 1024: size, cost_per_execution = '1024x1024', 0.080
                            elif width <= 1792: size, cost_per_execution = '1792x1024', 0.120
                            else: size, cost_per_execution = '1024x1792', 0.120
                        else: cost_per_execution = 0.120
                        if 'data' in response_data:
                            data = response_data['data']
                            if type(data) in (tuple, list) and len(data) > 0:
                                data = data[0]
                                if 'url' in data: main_result = str(data['url']).strip()
                        total_cost += cost_per_execution
                    else:
                        if 'choices' in response_data:
                            choices = response_data['choices']
                            if type(choices) in (tuple, list) and len(choices) > 0:
                                choices = choices[0]
                                if 'message' in choices:
                                    message = choices['message']
                                    main_result = str(message['content']).strip() if 'content' in message else ''
                        if 'usage' in response_data:
                            usage = response_data['usage']
                            total_tokens = int(usage['total_tokens']) if 'total_tokens' in usage and type(usage['total_tokens']) in (int, float) else 0
                            if 'gpt-4o-mini' in string_body: cost_per_token = (0.60+0.30)/1000000
                            elif 'gpt-4o' in string_body: cost_per_token = (15+7.50)/1000000
                            elif 'gpt-3.5' in string_body: cost_per_token = (3+4)/1000000
                            elif 'gpt-4-turbo' in string_body: cost_per_token = (10+30)/1000000
                            elif 'gpt-4-32k' in string_body: cost_per_token = (60+120)/1000000
                            elif 'gpt-4' in string_body: cost_per_token = (30+60)/1000000
                            else: cost_per_token = (60+125)/1000000
                            total_cost += total_tokens*cost_per_token
                            if 'type' in string_body and 'image_url' in string_body: total_cost += 0.007225
            elif 'google' in url_lower:
                response = post(URL, json=body, headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_data = json_result = response.json()
                    if 'candidates' in response_data:
                        candidates = response_data['candidates']
                        if type(candidates) in (tuple, list) and len(candidates) > 0:
                            candidates = candidates[0]
                            if 'content' in candidates:
                                content = candidates['content']
                                if 'parts' in content:
                                    parts = content['parts']
                                    if type(parts) in (tuple, list) and len(parts) > 0:
                                        parts = parts[0]
                                        main_result = str(parts['text']).strip() if 'text' in parts else ''
                    if 'usageMetadata' in response_data:
                        usageMetadata = response_data['usageMetadata']
                        total_tokens = int(usageMetadata['totalTokenCount']) if 'totalTokenCount' in usageMetadata and type(usageMetadata['totalTokenCount']) in (int, float) else 0
                        if 'gemini-1.5-flash' in url_lower: cost_per_token = (0.70+2.10)/1000000
                        elif 'gemini-pro' in url_lower: cost_per_token = (0.50+1.50)/1000000
                        elif 'gemini-1.5-pro' in url_lower: cost_per_token = (7+21)/1000000
                        else: cost_per_token = (7+25)/1000000
                        total_cost += total_tokens*cost_per_token
                        if 'inline_data' in url_lower and 'mimeType' in url_lower: total_cost += 0.00263
            elif 'replicate' in url_lower:
                from time import time
                start_time = time()
                from json import dumps
                response, cost_per_token, total_cost = post(URL, data=dumps(body), headers=headers, timeout=self.__timeout), 0.000019, 0
                if response.status_code in (200, 201):
                    response_data = json_result = response.json()
                    if '/models/' in url_lower:
                        if 'urls' in response_data:
                            urls = response_data['urls']
                            if 'stream' in urls:
                                stream_url = urls['stream']
                                stream_headers = {'Accept': 'text/event-stream', 'Cache-Control': 'no-store'}
                                stream_response, tokens = get(stream_url, headers=stream_headers, stream=True, timeout=self.__timeout), []
                                for line in stream_response.iter_lines():
                                    if line:
                                        decoded_line = line.decode('utf-8')
                                        if decoded_line.startswith('data: '):
                                            token = decoded_line[6:]
                                            if token.endswith('{}'): token = token[:-2]
                                            tokens.append(token)
                                main_result = ''.join(tokens)
                                total_tokens = self.countTokens(string=str(body)+' '+main_result, pattern='gpt-3')
                                total_cost = cost_per_token*total_tokens
                            elif 'get' in urls:
                                get_url = urls['get']
                                status, response, output = 'starting', get(get_url, headers=headers, timeout=self.__timeout), ''
                                is_list_of_tokens, output_is_tuple_or_list, output_length = False, False, 0
                                while status in ('starting', 'processing'):
                                    response = get(get_url, headers=headers, timeout=self.__timeout)
                                    if response.status_code in (200, 201):
                                        response_data = json_result = response.json()
                                        status = str(response_data['status']).lower().strip() if 'status' in response_data else ''
                                        if 'output' in response_data:
                                            if not response_data['output'] is None: output = response_data['output']
                                            output_is_tuple_or_list = type(output) in (tuple, list)
                                            output_length = len(output) if output_is_tuple_or_list else len(str(output))
                                            if output_is_tuple_or_list and output_length > 0:
                                                output_string = str(output[0]).lower().strip()
                                                if not self.isURLAddress(file_path=output_string): is_list_of_tokens = True
                                        if output_length > 0 and not is_list_of_tokens: break
                                    else: break
                                if output_length > 1 and is_list_of_tokens:
                                    main_result = ''.join(output)
                                    total_tokens = self.countTokens(string=str(body)+' '+main_result, pattern='gpt-3')
                                    total_cost = cost_per_token*total_tokens
                                elif output_length > 0: main_result = list(output)[0] if output_is_tuple_or_list else str(output).strip()
                    else:
                        status = str(response_data['status']).lower().strip() if 'status' in response_data else ''
                        if status == 'starting':
                            if 'urls' in response_data:
                                urls = response_data['urls']
                                if 'get' in urls:
                                    url = str(urls['get']).strip()
                                    response = get(url, headers=headers, timeout=self.__timeout)
                                    if response.status_code in (200, 201):
                                        response_data = json_result = response.json()
                                        status, output = str(response_data['status']).lower().strip() if 'status' in response_data else '', []
                                        is_list_of_tokens, output_is_tuple_or_list, output_length = False, False, 0
                                        while status in ('starting', 'processing'):
                                            response = get(url, headers=headers, timeout=self.__timeout)
                                            if response.status_code in (200, 201):
                                                response_data = json_result = response.json()
                                                status = str(response_data['status']).lower().strip() if 'status' in response_data else ''
                                                if 'output' in response_data:
                                                    if not response_data['output'] is None: output = response_data['output']
                                                    output_is_tuple_or_list = type(output) in (tuple, list)
                                                    output_length = len(output) if output_is_tuple_or_list else len(str(output))
                                                    if output_is_tuple_or_list and output_length > 0:
                                                        output_string = str(output[0]).lower().strip()
                                                        if not self.isURLAddress(file_path=output_string): is_list_of_tokens = True
                                                if output_length > 0 and not is_list_of_tokens: break
                                            else: break
                                        if output_length > 1 and is_list_of_tokens:
                                            main_result = ''.join(output)
                                            total_tokens = self.countTokens(string=str(body)+' '+main_result, pattern='gpt-3')
                                            total_cost = cost_per_token*total_tokens
                                        elif output_length > 0: main_result = list(output)[0] if output_is_tuple_or_list else str(output).strip()
                if total_cost <= 0:
                    end_time = time()
                    duration_in_seconds = abs(end_time-start_time)
                    total_cost = 0.32+(0.005600*duration_in_seconds)
            elif 'deepinfra' in url_lower:
                response = post(URL, json=body, headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_data = json_result = response.json()
                    if 'output' in response_data:
                        output = response_data['output']
                        main_result = output[0] if type(output) in (tuple, list) and len(output) > 0 else ''
                    elif 'images' in response_data:
                        images = response_data['images']
                        main_result = images[0] if type(images) in (tuple, list) and len(images) > 0 else ''
                    elif 'results' in response_data:
                        results = response_data['results']
                        if type(results) in (tuple, list) and len(results) > 0:
                            results = results[0]
                            main_result = str(results['generated_text']).strip() if 'generated_text' in results else ''
                    elif 'choices' in response_data:
                        choices = response_data['choices']
                        if type(choices) in (tuple, list) and len(choices) > 0:
                            choices = choices[0]
                            if 'message' in choices:
                                message = choices['message']
                                main_result = str(message['content']).strip() if 'content' in message else ''
                    if 'inference_status' in response_data:
                        inference_status = response_data['inference_status']
                        total_cost = float(inference_status['cost']) if 'cost' in inference_status and type(inference_status['cost']) in (int, float) else 0
                        if total_cost <= 0: total_cost = 0.002
                    elif 'usage' in response_data:
                        usage = response_data['usage']
                        total_cost = float(usage['estimated_cost']) if 'estimated_cost' in usage and type(usage['estimated_cost']) in (int, float) else 0
                        if total_cost <= 0: total_cost = 5.4
            elif 'localhost:11434' in url_lower:
                response = post(URL, json=body, headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_data = json_result = response.json()
                    main_result, total_cost = response_data['response'] if 'response' in response_data else '', 0
            else: json_result = post(URL, json=body, headers=headers, timeout=self.__timeout).json()
            return main_result.strip(), total_cost, json_result
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.executeModel: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return '', 0, {}
    def updateMediaMetadata(self, file_path='', file_dictionary={}, metadata=[]):
        try:
            result = True
            file_path = str(file_path).strip()
            if type(file_dictionary) != dict or file_dictionary == {}: file_dictionary = {'base64_string': '', 'type': 'png'}
            metadata = list(metadata) if type(metadata) in (tuple, list) else []
            no_locality, is_url_address, metadata_length = len(file_path) < 1 or len(file_dictionary['base64_string']) > 0, self.isURLAddress(file_path=file_path), len(metadata)
            if no_locality:
                _type = file_dictionary['type']
                temporary_file_directory = self.getTemporaryFilesDirectory()
                output_path = temporary_file_directory+f'{self.getHashCode()}.{_type}'
            else: output_path, _type = file_path, self.getFileExtension(file_path=file_path)[1:].upper()
            if _type == 'JPG': _type = 'JPEG'
            try: from PIL import Image, JpegImagePlugin, PngImagePlugin, WebPImagePlugin
            except:
                self.installModule(module_name='pillow', version='10.3.0')
                from PIL import Image, JpegImagePlugin, PngImagePlugin, WebPImagePlugin
            from shutil import copyfile
            try:
                from mutagen import File as MutagenFile
                from mutagen.id3 import ID3, TXXX
            except:
                self.installModule(module_name='mutagen', version='1.47.0')
                from mutagen import File as MutagenFile
                from mutagen.id3 import ID3, TXXX
            from os import rename
            from base64 import b64decode
            from io import BytesIO
            def updateMetadataImage(input_image_path='', output_image_path=''):
                if no_locality:
                    image_data = b64decode(file_dictionary['base64_string'])
                    image = Image.open(BytesIO(image_data))
                else: image = Image.open(input_image_path)
                data = list(image.getdata())
                image_without_metadata = Image.new(image.mode, image.size)
                image_without_metadata.putdata(data)
                if isinstance(image, JpegImagePlugin.JpegImageFile):
                    exif_data = image_without_metadata.info.get('exif', b'')
                    image_without_metadata.save(output_image_path, exif=exif_data)
                    if metadata_length > 0:
                        with open(output_image_path, 'rb') as file:
                            image_jpeg = Image.open(file)
                            for data in metadata:
                                if 'name' in data and 'value' in data:
                                    name = str(data['name']).strip()
                                    value = str(data['value']).strip()
                                    if len(name) > 0 and len(value) > 0: image_jpeg.info[name] = value
                elif isinstance(image, PngImagePlugin.PngImageFile):
                    image_metadata = PngImagePlugin.PngInfo()
                    if metadata_length > 0:
                        for data in metadata:
                            if 'name' in data and 'value' in data:
                                name = str(data['name']).strip()
                                value = str(data['value']).strip()
                                if len(name) > 0 and len(value) > 0: image_metadata.add_text(name, value)
                    image_without_metadata.save(output_image_path, pnginfo=image_metadata, format='PNG')
                elif isinstance(image, WebPImagePlugin.WebPImageFile):
                    image_metadata = PngImagePlugin.PngInfo()
                    if metadata_length > 0:
                        for data in metadata:
                            if 'name' in data and 'value' in data:
                                name = str(data['name']).strip()
                                value = str(data['value']).strip()
                                if len(name) > 0 and len(value) > 0: image_metadata.add_text(name, value)
                    image_without_metadata.save(output_image_path, pnginfo=image_metadata, format='WEBP')
                else: image_without_metadata.save(output_image_path)
            def updateMetadataAudioVideo(input_media_path='', output_media_path=''):
                if no_locality:
                    _type = file_dictionary['type']
                    temporary_file_directory = self.getTemporaryFilesDirectory()
                    output_path = temporary_file_directory+f'{self.getHashCode()}.{_type}'
                    if self.base64ToFile(file_dictionary=file_dictionary, file_path=output_path): input_media_path = output_path
                temp_path = output_media_path + '.temp'
                copyfile(input_media_path, temp_path)
                media_file = MutagenFile(temp_path)
                if media_file:
                    media_file.delete()
                    media_file.save()
                media_file = MutagenFile(temp_path)
                if isinstance(media_file.tags, ID3):
                    if metadata_length > 0:
                        for data in metadata:
                            if 'name' in data and 'value' in data:
                                name = str(data['name']).strip()
                                value = str(data['value']).strip()
                                if len(name) > 0 and len(value) > 0: media_file.tags.add(TXXX(encoding=3, desc=name, text=[value]))
                media_file.save()
                rename(temp_path, output_media_path)
                if no_locality: self.deleteFile(file_path=input_media_path)
            def updateMetadata(file_path='', output_path=''):
                file_category = self.getFileCategory(file_path=output_path)
                if file_category == 'IMAGE_FILE': updateMetadataImage(file_path, output_path)
                elif file_category in ('AUDIO_FILE', 'VIDEO_FILE'): updateMetadataAudioVideo(file_path, output_path)
            if is_url_address:
                try:
                    from os import environ
                    try: from certifi import where
                    except:
                        self.installModule(module_name='certifi', version='2024.2.2')
                        from certifi import where
                    environ['SSL_CERT_FILE'] = where()
                    from logging import getLogger, ERROR
                    getLogger('requests').setLevel(ERROR)
                except: pass
                file_name, file_extension = self.getFileName(file_path=file_path), self.getFileExtension(file_path=file_path)
                try: from requests import get
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import get
                response = get(file_path, timeout=self.__timeout)
                file_path = file_name+file_extension
                with open(file_path, 'wb') as file: file.write(response.content)
                output_path = file_path
            updateMetadata(file_path=file_path, output_path=output_path)
            if no_locality:
                result = self.fileToBase64(file_path=output_path, media_metadata=metadata)
                self.deleteFile(file_path=output_path)
            return result
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.updateMediaMetadata: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return ''
    def splitAudioFile(self, file_path='', max_size_bytes=26214200, output_directory='./temp'):
        try:
            output_paths = []
            file_path = str(file_path).strip()
            max_size_bytes = int(max_size_bytes) if type(max_size_bytes) in (int, float) else 26214200
            output_directory = str(output_directory).strip()
            from os import makedirs, path
            makedirs(output_directory, exist_ok=True)
            file_size = path.getsize(file_path)
            if file_size <= max_size_bytes: return [file_path]
            try: from pydub import AudioSegment
            except:
                self.installModule(module_name='pydub', version='0.25.1')
                from pydub import AudioSegment
            audio = AudioSegment.from_file(file_path)
            number_of_parts = (file_size//max_size_bytes)+1
            part_duration = len(audio)//number_of_parts
            for index in range(number_of_parts):
                start_time = index*part_duration
                end_time = (index+1)*part_duration if index < number_of_parts-1 else len(audio)
                part = audio[start_time:end_time]
                file_name = path.splitext(path.basename(file_path))[0]
                file_extension = path.splitext(file_path)[1]
                output_path = path.join(output_directory, f'{file_name}_part{index+1}{file_extension}')
                part.export(output_path, format=file_extension[1:].strip())
                output_paths.append(output_path)
            return output_paths
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.splitAudioFile: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return [file_path]
    def textDetection(self, file_path='', file_dictionary={}, service='local', api_key='', max_tokens=1000, language='en'):
        try:
            texts_detected = ''
            file_path = str(file_path).strip()
            if type(file_dictionary) != dict or file_dictionary == {}: file_dictionary = {'base64_string': '', 'type': 'png'}
            service = str(service).lower().strip()
            api_key = str(api_key).strip()
            max_tokens = int(max_tokens) if type(max_tokens) in (bool, int, float) else 1000
            language = str(language).lower().strip() if type(language) == str and len(language) > 0 else 'en'
            no_locality, is_url_address = len(file_path) < 1, self.isURLAddress(file_path=file_path)
            try: from requests import post, get
            except:
                self.installModule(module_name='requests', version='2.31.0')
                from requests import post, get
            if no_locality:
                temporary_file_directory = self.getTemporaryFilesDirectory()
                temporary_file = temporary_file_directory+f'{self.getHashCode()}.{file_dictionary["type"]}'
                if self.base64ToFile(file_dictionary=file_dictionary, file_path=temporary_file): file_path = temporary_file
            if is_url_address:
                try:
                    from os import environ
                    try: from certifi import where
                    except:
                        self.installModule(module_name='certifi', version='2024.2.2')
                        from certifi import where
                    environ['SSL_CERT_FILE'] = where()
                    from logging import getLogger, ERROR
                    getLogger('requests').setLevel(ERROR)
                except: pass                
            if service.startswith('ninjas'):
                if is_url_address:
                    from io import BytesIO
                    image_binary = BytesIO(get(file_path, timeout=self.__timeout).content)
                else: image_binary = open(file_path, 'rb')
                URL = 'https://api.api-ninjas.com/v1/imagetotext'
                files = {'image': image_binary}
                headers = {'X-Api-Key': 'JXi7tQm9uh9m3/OO3/xn/A==reKFBTWuLjLotxSx' if len(api_key) < 1 else api_key}
                response = post(URL, files=files, headers=headers, timeout=self.__timeout)
                json_object, texts = response.json(), []
                if response.status_code in (200, 201):
                    for object_name in json_object:
                        text = str(object_name['text']).strip() if 'text' in object_name else ''
                        if len(text) > 0: texts.append(text)
                    texts_detected += ' '.join(texts) if len(texts) > 0 else ''
            else:
                from os import environ
                from warnings import filterwarnings
                environ['FLAGS_log_level'] = '3'
                environ['GLOG_minloglevel'] = '3'
                filterwarnings('ignore')
                from logging import getLogger, CRITICAL
                getLogger('paddleocr').setLevel(CRITICAL)
                getLogger('ppocr').setLevel(CRITICAL)
                getLogger('paddle').setLevel(CRITICAL)
                try: from paddleocr import PaddleOCR
                except:
                    self.installModule(module_name='paddleocr', version='2.7.3')
                    self.installModule(module_name='paddlepaddle', version='2.6.0')
                    from paddleocr import PaddleOCR
                from logging import getLogger, ERROR, WARNING
                getLogger('paddleocr').setLevel(WARNING)
                getLogger('ppocr').setLevel(WARNING)
                paddle_ocr = PaddleOCR(use_angle_cls=True, lang=language, show_log=False)
                from contextlib import redirect_stdout, redirect_stderr
                from os import devnull
                with open(devnull, 'w') as file, redirect_stdout(file), redirect_stderr(file): result, extracted_text = paddle_ocr.ocr(file_path, cls=True), []
                for line in result:
                    if line:
                        for word_information in line: extracted_text.append(word_information[1][0])
                if len(extracted_text) > 0: texts_detected += ' '.join(extracted_text)
            if is_url_address:
                from glob import glob
                for file in glob('tmp.*'): self.deleteFile(file_path=file)
            if no_locality: self.deleteFile(file_path=file_path)
            if self.countTokens(string=texts_detected, pattern='gpt') > max_tokens: texts_detected = self.getTokensSummary(string=texts_detected, max_tokens=max_tokens)
            return texts_detected
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.textDetection: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return ''
    def rgbColorDetection(self, file_path='', file_dictionary={}, number_of_colors=5, return_as_string=False, color_names=False, max_tokens=1000):
        try:
            colors_detected = []
            file_path = str(file_path).strip()
            if type(file_dictionary) != dict or file_dictionary == {}: file_dictionary = {'base64_string': '', 'type': 'png'}
            number_of_colors = int(number_of_colors) if type(number_of_colors) in (bool, int, float) else 5
            return_as_string = bool(return_as_string) if type(return_as_string) in (bool, int, float) else False
            color_names = bool(color_names) if type(color_names) in (bool, int, float) else False
            max_tokens = int(max_tokens) if type(max_tokens) in (bool, int, float) else 1000
            try: from PIL import Image
            except:
                self.installModule(module_name='pillow', version='10.3.0')
                from PIL import Image
            try: from numpy import array
            except:
                self.installModule(module_name='numpy', version='1.25.2')
                from numpy import array
            from collections import Counter
            if self.isURLAddress(file_path=file_path):
                try: from requests import get
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import get
                from io import BytesIO
                image = Image.open(BytesIO(get(file_path, timeout=self.__timeout).content))
            else:
                if len(file_path) < 1:
                    from io import BytesIO
                    from base64 import b64decode
                    image = Image.open(BytesIO(b64decode(file_dictionary['base64_string'])))
                else: image = Image.open(file_path).convert('RGB')
            image_array = array(image)
            pixels = [tuple(pixel) for row in image_array for pixel in row]
            color_count = Counter(pixels)
            most_common_colors = color_count.most_common(number_of_colors)
            def rgb_to_color_name(rgb=[0, 0, 0]):
                rgb = list(rgb) if type(rgb) in (tuple, list) else [0, 0, 0]
                closest_color = 'black'
                color_map = {
                    (0, 0, 0): 'black',
                    (255, 255, 255): 'white',
                    (255, 0, 0): 'red',
                    (0, 255, 0): 'lime',
                    (0, 0, 255): 'blue',
                    (255, 255, 0): 'yellow',
                    (0, 255, 255): 'cyan',
                    (255, 0, 255): 'magenta',
                    (128, 128, 128): 'gray',
                    (128, 0, 0): 'maroon',
                    (0, 128, 0): 'green',
                    (0, 0, 128): 'navy',
                    (128, 128, 0): 'olive',
                    (0, 128, 128): 'teal',
                    (128, 0, 128): 'purple',
                    (192, 192, 192): 'silver',
                    (255, 165, 0): 'orange',
                    (255, 192, 203): 'pink',
                    (165, 42, 42): 'brown',
                }
                rgb_tuple = tuple(rgb)
                if rgb_tuple in color_map: return color_map[rgb_tuple]
                min_distance = float('inf')
                for color_rgb, color_name in color_map.items():
                    distance = sum((a - b) ** 2 for a, b in zip(rgb_tuple, color_rgb)) ** 0.5
                    if distance < min_distance:
                        min_distance = distance
                        closest_color = color_name
                return closest_color
            if color_names:
                colors_detected = [rgb_to_color_name(color) for color, _ in most_common_colors]
                colors_detected = sorted(list(set(colors_detected)))
                if return_as_string: colors_detected = ', '.join(colors_detected)
            else: colors_detected = str([list(color) for color, _ in most_common_colors])[1:-1] if return_as_string else [list(color) for color, _ in most_common_colors]
            if type(colors_detected) == str and self.countTokens(string=colors_detected, pattern='gpt') > max_tokens: colors_detected = self.getTokensSummary(string=colors_detected, max_tokens=max_tokens)
            return colors_detected
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.rgbColorDetection: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return []
    def objectDetection(self, file_path='', file_dictionary={}, service='local', api_key='', max_tokens=1000):
        try:
            result_object = {'answer': '', 'str_cost': '0.0000000000'}
            file_path = str(file_path).strip()
            if type(file_dictionary) != dict or file_dictionary == {}: file_dictionary = {'base64_string': '', 'type': 'png'}
            service = str(service).lower().strip()
            api_key = str(api_key).strip()
            max_tokens = int(max_tokens) if type(max_tokens) in (bool, int, float) else 1000
            no_locality, objects_detected, total_cost = len(file_path) < 1, '', 0
            try: from requests import post, get
            except:
                self.installModule(module_name='requests', version='2.31.0')
                from requests import post, get
            if service.startswith('google'):
                from base64 import b64encode
                URL = 'https://vision.googleapis.com/v1/images:annotate?key='+api_key
                def getBase64Image(file_path='', is_url=False):
                    if is_url:
                        response = get(file_path, timeout=self.__timeout)
                        image_data = response.content
                    else:
                        with open(file_path, 'rb') as image_file: image_data = image_file.read()
                    base64_image = b64encode(image_data).decode()
                    return base64_image
                if no_locality: base64_image = file_dictionary['base64_string']
                else: base64_image = getBase64Image(file_path=file_path, is_url=self.isURLAddress(file_path=file_path))
                body = {
                    'requests': [
                        {
                            'image': {'content': base64_image},
                            'features': [
                                {'type': 'OBJECT_LOCALIZATION', 'maxResults': 20},
                                {'type': 'TEXT_DETECTION'},
                                {'type': 'LOGO_DETECTION'},
                                {'type': 'LABEL_DETECTION'},
                                {'type': 'IMAGE_PROPERTIES'}
                            ]
                        }
                    ]
                }
                headers = {'Content-Type': 'application/json'}
                response = post(URL, json=body, headers=headers, timeout=self.__timeout)
                total_cost += 0.0075
                if response.status_code in (200, 201):
                    json_object = response.json()
                    if 'responses' in json_object:
                        responses = json_object['responses']
                        if len(responses) > 0:
                            response_object = responses[0]
                            if 'localizedObjectAnnotations' in response_object:
                                localizedObjectAnnotations = response_object['localizedObjectAnnotations']
                                if len(localizedObjectAnnotations) > 0:
                                    objects_list = []
                                    for localizedObjectAnnotation in localizedObjectAnnotations:
                                        if 'name' in localizedObjectAnnotation:
                                            name = str(localizedObjectAnnotation['name']).strip()
                                            if name not in objects_list: objects_list.append(name)
                                    if len(objects_list) > 0: objects_detected += 'objects: '+(', '.join(objects_list))+'\n'
                            if 'textAnnotations' in response_object:
                                textAnnotations = response_object['textAnnotations']
                                if len(textAnnotations) > 0:
                                    texts_list = []
                                    for textAnnotation in textAnnotations:
                                        if 'description' in textAnnotation:
                                            description = str(textAnnotation['description']).replace('\n', ' ').strip()
                                            if description not in texts_list: texts_list.append(description)
                                    if len(texts_list) > 0: objects_detected += 'texts: '+(', '.join(texts_list))+'\n'
                            if 'logoAnnotations' in response_object:
                                logoAnnotations = response_object['logoAnnotations']
                                if len(logoAnnotations) > 0:
                                    logos_list = []
                                    for logoAnnotation in logoAnnotations:
                                        if 'description' in logoAnnotation:
                                            description = str(logoAnnotation['description']).strip()
                                            if description not in logos_list: logos_list.append(description)
                                    if len(logos_list) > 0: objects_detected += 'logos: '+(', '.join(logos_list))+'\n'
                            if 'labelAnnotations' in response_object:
                                labelAnnotations = response_object['labelAnnotations']
                                if len(labelAnnotations) > 0:
                                    labels_list = []
                                    for labelAnnotation in labelAnnotations:
                                        if 'description' in labelAnnotation and 'score' in labelAnnotation:
                                            description, score = str(labelAnnotation['description']).strip(), float(labelAnnotation['score'])
                                            if description not in labels_list and score > .95: labels_list.append(description)
                                    if len(labels_list) > 0: objects_detected += 'labels: '+(', '.join(labels_list))+'\n'
                            if 'imagePropertiesAnnotation' in response_object:
                                imagePropertiesAnnotation = response_object['imagePropertiesAnnotation']
                                if 'dominantColors' in imagePropertiesAnnotation:
                                    dominantColors = imagePropertiesAnnotation['dominantColors']
                                    if 'colors' in dominantColors:
                                        colors = dominantColors['colors']
                                        if len(colors) > 0:
                                            colors_list = []
                                            for color in colors:
                                                if 'color' in color and 'score' in color:
                                                    score = float(color['score'])
                                                    if score > .03:
                                                        rgb = color['color']
                                                        if type(rgb) == dict:
                                                            rgb = list(rgb.values())
                                                            if rgb not in colors_list: colors_list.append(str(rgb))
                                            if len(colors_list) > 0: objects_detected += 'rgb colors: '+(', '.join(colors_list))
            else:
                if no_locality:
                    temporary_file_directory = self.getTemporaryFilesDirectory()
                    temporary_file = temporary_file_directory+f'{self.getHashCode()}.{file_dictionary["type"]}'
                    if self.base64ToFile(file_dictionary=file_dictionary, file_path=temporary_file): file_path = temporary_file
                if service.startswith('ninjas'):
                    if self.isURLAddress(file_path=file_path):
                        try:
                            from os import environ
                            try: from certifi import where
                            except:
                                self.installModule(module_name='certifi', version='2024.2.2')
                                from certifi import where
                            environ['SSL_CERT_FILE'] = where()
                            from logging import getLogger, ERROR
                            getLogger('requests').setLevel(ERROR)
                        except: pass
                        from io import BytesIO
                        image_binary = BytesIO(get(file_path, timeout=self.__timeout).content)
                    else: image_binary = open(file_path, 'rb')
                    URL = 'https://api.api-ninjas.com/v1/objectdetection'
                    files = {'image': image_binary}
                    headers = {'X-Api-Key': 'AJQBr1sLYxVX7fkLp/7KHQ==pm5kQKnuzzdaNATe' if len(api_key) < 1 else api_key}
                    response = post(URL, files=files, headers=headers, timeout=self.__timeout)
                    json_object, object_names = response.json(), []
                    if response.status_code in (200, 201):
                        for object_name in json_object:
                            object_name = str(object_name['label']).strip() if 'label' in object_name else ''
                            if len(object_name) > 0 and object_name not in object_names: object_names.append(object_name)
                        object_names = sorted(object_names)
                        objects_detected += 'objects: '+(', '.join(object_names)) if len(object_names) > 0 else ''
                else:
                    try: from cv2 import dnn, imdecode, imread
                    except:
                        self.installModule(module_name='opencv-python', version='4.6.0.66')
                        self.installModule(module_name='opencv-python-headless', version='4.6.0.66')
                        from cv2 import dnn, imdecode, imread
                    try: from numpy import asarray, uint8, argmax
                    except:
                        self.installModule(module_name='numpy', version='1.25.2')
                        from numpy import asarray, uint8, argmax
                    classes = []
                    root = self.__getRootDirectory()
                    sapiens_names_path, sapiens_weights_path, sapiens_cfg_path = root+'sapiens.names', root+'sapiens.weights', root+'sapiens.cfg'
                    from os import path
                    if not path.exists(sapiens_names_path): self.downloadFile(url_path=self.__decodeLink(encoded_text=self.__sapiens_names), output_path=sapiens_names_path)
                    if not path.exists(sapiens_weights_path): self.downloadFile(url_path=self.__decodeLink(encoded_text=self.__sapiens_weights), output_path=sapiens_weights_path)
                    if not path.exists(sapiens_cfg_path): self.downloadFile(url_path=self.__decodeLink(encoded_text=self.__sapiens_cfg), output_path=sapiens_cfg_path)
                    with open(sapiens_names_path, 'r') as file: classes = [line.strip() for line in file.readlines()]
                    net = dnn.readNet(sapiens_weights_path, sapiens_cfg_path)
                    if self.isURLAddress(file_path=file_path): net.setInput(dnn.blobFromImage(imdecode(asarray(bytearray(get(file_path, timeout=self.__timeout).content), dtype=uint8), -1), 1/255.0, (416, 416), swapRB=True, crop=False))
                    else: net.setInput(dnn.blobFromImage(imread(file_path), 1/255.0, (416, 416), swapRB=True, crop=False))
                    layer_names = net.getLayerNames()
                    output_layers = [layer_names[index-1] for index in net.getUnconnectedOutLayers().flatten()]
                    outputs, object_names = net.forward(output_layers), []
                    for output in outputs:
                        for detection in output:
                            scores = detection[5:]
                            class_id = argmax(scores)
                            confidence = scores[class_id]
                            if confidence > 0.5:
                                object_name = classes[class_id]
                                if len(object_name) > 0 and object_name not in object_names: object_names.append(object_name)
                    object_names = sorted(object_names)
                    objects_detected += 'objects: '+(', '.join(object_names))+'\n' if len(object_names) > 0 else ''
                    text_detection = self.textDetection(file_path=file_path, file_dictionary=file_dictionary, service='local', api_key='', max_tokens=max_tokens, language='en')
                    if len(text_detection) > 0: objects_detected += 'texts: '+text_detection+'\n'
                    rgb_color_detection = self.rgbColorDetection(file_path=file_path, file_dictionary=file_dictionary, number_of_colors=50, return_as_string=True, color_names=True, max_tokens=max_tokens)
                    if len(rgb_color_detection) > 0: objects_detected += 'rgb color names: '+rgb_color_detection
            if no_locality: self.deleteFile(file_path=file_path)
            if self.countTokens(string=objects_detected, pattern='gpt') > max_tokens: objects_detected = self.getTokensSummary(string=objects_detected, max_tokens=max_tokens)
            result_object['answer'], result_object['str_cost'] = objects_detected.strip(), f'{total_cost:.10f}'.strip()
            return result_object
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.objectDetection: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'answer': '', 'str_cost': '0.0000000000'}
    def imageInterpreter(self, file_path='', file_dictionary={}, service='local', model='llava', api_key='', max_tokens=1000, language='en', user_code=None):
        try:
            result_object = {'answer': '', 'str_cost': '0.0000000000'}
            file_path = str(file_path).strip()
            if type(file_dictionary) != dict or file_dictionary == {}: file_dictionary = {'base64_string': '', 'type': 'png'}
            service = str(service).lower().strip()
            model = str(model).strip()
            api_key = str(api_key).strip()
            max_tokens = int(max_tokens) if type(max_tokens) in (bool, int, float) else 1000
            language, instruction = str(language).lower().strip() if type(language) == str and len(language) > 0 else 'en', ''
            user_code = str(user_code).strip() if user_code else '-'
            instruction_en = 'What can be seen in this scene?'
            instruction_es = '¿Qué se puede ver en esta escena?'
            instruction_pt = 'O que pode ser visto nesta cena?'
            if language.startswith('en'): instruction = instruction_en
            elif language.startswith('es'): instruction = instruction_es
            elif language.startswith('pt'): instruction = instruction_pt
            if len(instruction.strip()) < 1: instruction = self.translate(string=instruction_en, source_language='en', target_language=language)
            URL, body, headers, _file_dictionary, total_cost = '', {}, {}, None, 0.0
            system_instruction = 'You are Sapiens Chat, an Artificial Intelligence specialized in the visualization and interpretation of scenes that always returns as a response only a detailed description of the observed scene. The description must always be in the same language as the prompt and you must always refer to the image as a scene. Never use the word "image", always use the word "scene" instead of the word "image".'
            if service.startswith('anthropic'):
                if len(file_path) > 0: file_dictionary = self.fileToBase64(file_path=file_path)
                image_data, _type = file_dictionary['base64_string'], file_dictionary['type']
                if _type == 'jpg': _type = 'jpeg'
                image_media_type = 'image/'+_type
                URL = 'https://api.anthropic.com/v1/messages'  
                body = {'model': model, 'temperature': 0.5, 'max_tokens': 4096, 'system': system_instruction, 'messages': [{'role': 'user', 'content': [{'type': 'image', 'source': {'type': 'base64', 'media_type': image_media_type, 'data': image_data}}, {'type': 'text', 'text': instruction}]}]}
                headers = {'x-api-key': api_key, 'anthropic-version': '2023-06-01', 'Content-Type': 'application/json'}
            elif service.startswith('openai'):
                url_image, image_data, _type = None, '', 'png'
                if len(file_path) > 0:
                    if self.isURLAddress(file_path=file_path): url_image = {'type': 'image_url', 'image_url': {'url': file_path}}
                    else:
                        file_dictionary = self.fileToBase64(file_path=file_path)
                        image_data, _type = file_dictionary['base64_string'], file_dictionary['type']
                else: image_data, _type = file_dictionary['base64_string'], file_dictionary['type']
                if url_image is None:
                    if _type == 'jpg': _type = 'jpeg'
                    url_image = {'type': 'image_url', 'image_url': {'url': f'data:image/{_type};base64,{image_data}'}}
                messages = [{'role': 'system', 'content': system_instruction}, {'role': 'user', 'content': [{'type': 'text', 'text': instruction}, url_image]}]
                URL = 'https://api.openai.com/v1/chat/completions'  
                body = {'model': model, 'messages': messages, 'temperature': 0.5}
                headers = {'Content-Type': 'application/json', 'Authorization': 'Bearer '+api_key}
            elif service.startswith('google'):
                image_data, _type = '', 'png'
                if len(file_path) > 0: file_dictionary = self.fileToBase64(file_path=file_path)
                image_data, _type = file_dictionary['base64_string'], file_dictionary['type']
                if _type == 'jpg': _type = 'jpeg'
                image_media_type = 'image/'+_type
                URL = f'https://generativelanguage.googleapis.com/v1beta/models/{model}:generateContent?key={api_key}'
                instruction = '(SYSTEM) You must respond as if you were the following: '+system_instruction+'\n\n'+instruction
                parts = [{'inline_data': {'mimeType': image_media_type, 'data': f'{image_data}'}}, {'text': instruction}]
                body = {'contents': [{'role': 'user', 'parts': parts}], 'generation_config': {'temperature': 0.5}}
                headers = {'Content-Type': 'application/json'}
            elif service.startswith('replicate'):
                url_image = ''
                if len(file_path) > 0:
                    if self.isURLAddress(file_path=file_path): url_image = file_path
                    else:
                        _file_dictionary = self.imagePathToWebAddress(file_path=file_path, force_png=False, service_name='cloudinary')
                        url_image = _file_dictionary['url']
                else:
                    temporary_file_directory = self.getTemporaryFilesDirectory()
                    temporary_file = temporary_file_directory+f'{self.getHashCode()}.{file_dictionary["type"]}'
                    if self.base64ToFile(file_dictionary=file_dictionary, file_path=temporary_file):
                        _file_dictionary = self.imagePathToWebAddress(file_path=temporary_file, force_png=False, service_name='cloudinary')
                        url_image = _file_dictionary['url']
                        self.deleteFile(file_path=temporary_file)
                instruction = '(SYSTEM) You must respond as if you were the following: '+system_instruction+'\n\n'+instruction
                if '/' in model:
                    URL = f'https://api.replicate.com/v1/models/{model}/predictions'
                    body = {'stream': True, 'input': {'prompt': instruction, 'temperature': 0.5}}
                    headers = {'Content-Type': 'application/json', 'Authorization': 'Bearer '+api_key}
                else:
                    URL = f'https://api.replicate.com/v1/predictions'
                    body = {'version': model, 'input': {'image': url_image, 'prompt': instruction, 'temperature': 0.5}}
                    headers = {'Content-Type': 'application/json', 'Authorization': f'Token '+api_key}
            elif service.startswith('ollama'):
                if len(file_path) > 0: file_dictionary = self.fileToBase64(file_path=file_path)
                URL = 'http://localhost:11434/api/generate'
                instruction = '(SYSTEM) You must respond as if you were the following: '+system_instruction+'\n\n'+instruction
                body = {'model': model, 'prompt': instruction, 'stream': False, 'images': [file_dictionary['base64_string']]}
                headers = {'Content-Type': 'application/json'}
            else:
                image_input_model_path = str(self.__image_input_model_path).strip()
                if not user_code in self.__total_number_of_tokens: self.__total_number_of_tokens[user_code] = 0
                if len(image_input_model_path) > 0:
                    def getResponseFromSapiensInterpretation(language='en', file_path=''):
                        try: from sapiens_transformers.inference import SapiensModel
                        except:
                            self.installModule(module_name='sapiens-transformers', version='1.8.9')
                            from sapiens_transformers.inference import SapiensModel
                        sapiens_model = SapiensModel(model_path=image_input_model_path, show_errors=self.__show_errors)
                        CONTEXT_SIZE = self.getInputContextLimit(model_path=image_input_model_path, default_value=sapiens_model.CONTEXT_SIZE)
                        temporary_file = ''
                        if file_dictionary and len(file_path) < 1:
                            temporary_file_directory = self.getTemporaryFilesDirectory()
                            temporary_file = temporary_file_directory+f'{self.getHashCode()}.{file_dictionary["type"]}'
                            if self.base64ToFile(file_dictionary=file_dictionary, file_path=temporary_file): file_path = temporary_file
                        sapiens_model.add_image(path=file_path, maximum_pixels=500, merge_images=False)
                        self.__total_number_of_tokens[user_code] += 500*500
                        if temporary_file: self.deleteFile(file_path=temporary_file)
                        formatted_language = str(language).lower().strip()
                        prompt = instruction
                        try: from sapiens_infinite_context_window import SapiensInfiniteContextWindow
                        except:
                            self.installModule(module_name='sapiens-infinite-context-window', version='1.0.5')
                            from sapiens_infinite_context_window import SapiensInfiniteContextWindow
                        sapiens_infinite_context_window = SapiensInfiniteContextWindow()
                        prompt = sapiens_infinite_context_window.synthesize_tokens(text=prompt, maximum_tokens=CONTEXT_SIZE)
                        image_interpreter_answer = ''
                        self.__total_number_of_tokens[user_code] += sapiens_infinite_context_window.count_tokens(text=prompt)
                        tokens_generator = sapiens_model.generate_text(system=system_instruction, prompt=prompt, stream=True)
                        for token in tokens_generator:
                            self.__total_number_of_tokens[user_code] += 1
                            image_interpreter_answer += token
                        return image_interpreter_answer
                    interpretation, total_cost = getResponseFromSapiensInterpretation(language=language, file_path=file_path), 0.0
                else:
                    object_detection = self.objectDetection(file_path=file_path, file_dictionary=file_dictionary, service=service, max_tokens=max_tokens)
                    interpretation, total_cost = object_detection['answer'], float(object_detection['str_cost'])
                    interpretation = self.translate(string=interpretation, source_language='auto', target_language=language)
            if not service.startswith('local'): interpretation, total_cost, _ = self.executeModel(URL=URL, body=body, headers=headers)
            if not _file_dictionary is None: self.deleteImageAddress(file_dictionary=_file_dictionary)
            if self.countTokens(string=interpretation, pattern='gpt') > max_tokens: interpretation = self.getTokensSummary(string=interpretation, max_tokens=max_tokens)
            result_object['answer'], result_object['str_cost'] = interpretation, f'{total_cost:.10f}'.strip()
            return result_object
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.imageInterpreter: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'answer': '', 'str_cost': '0.0000000000'}
    def textToSpeech(self, text='', voice='', speed=1.0, pitch=0.0, service='', model='', api_key='', language=None, extension='mp3', output_path=None):
        try:
            file_dictionary = {'base64_string': '', 'type': 'mp3', 'output_path': '', 'str_cost': '0.0000000000'}
            text, total_cost = str(text).strip(), 0
            voice = voice.strip() if type(voice) == str and len(voice.strip()) > 0 else ''
            speed = min((max((0.25, float(speed))), 4.0)) if type(speed) in (bool, int, float) else 1.0
            pitch = min((max((-20.0, float(pitch))), 20.0)) if type(pitch) in (bool, int, float) else 0.0
            service = service.lower().strip() if type(service) == str and len(service.strip()) > 0 else ''
            model = model.lower().strip() if type(model) == str and len(model.strip()) > 0 else ''
            api_key = str(api_key).strip()
            if not language is None: language = str(language).strip()
            extension, dot = extension.lower().strip() if type(extension) == str else 'mp3', chr(46)
            if extension.startswith(dot): extension = extension[1:].strip()            
            output_path = output_path.strip() if type(output_path) == str else None
            is_openai, is_google = service.startswith('openai'), service.startswith('google')
            is_local = not is_openai or not is_google or len(api_key) < 1
            if is_openai:
                if voice not in ('alloy', 'echo', 'fable', 'onyx', 'nova', 'shimmer'): voice = 'onyx'
                if extension not in ('mp3', 'opus', 'aac', 'flac', 'wav', 'pcm'): extension = 'mp3'
            elif is_google:
                if language is None: language = 'en-us'
                language = language.lower()
                if language == 'en': language = 'en-us'
                elif language == 'es': language = 'es-es'
                elif language == 'pt': language = 'pt-br'
                language_ = language+'-'
                if voice.replace(language_, '') not in (
                    'neural2-a', 'neural2-c', 'neural2-d', 'neural2-e', 'neural2-f', 'neural2-g', 'neural2-h', 'neural2-i', 'neural2-j',
                    'studio-o', 'studio-q',
                    'journey-d', 'journey-f', 'journey-o',
                    'polyglot-1',
                    'wavenet-a', 'wavenet-b', 'wavenet-c', 'wavenet-d', 'wavenet-e', 'wavenet-f', 'wavenet-g', 'wavenet-h', 'wavenet-i', 'wavenet-j',
                    'news-k', 'news-l', 'news-n',
                    'standard-a', 'standard-b', 'standard-c', 'standard-d', 'standard-e', 'standard-f', 'standard-g', 'standard-h', 'standard-i', 'standard-j'
                ): voice = 'wavenet-e'
                if not voice.startswith(language_): voice = language_+voice
                if model not in (
                    'wearable-class-device', 'handset-class-device', 'headphone-class-device', 'small-bluetooth-speaker-class-device',
                    'medium-bluetooth-speaker-class-device', 'large-home-entertainment-class-device', 'large-automotive-class-device', 'telephony-class-application'
                ): model = 'wearable-class-device'
            temporary_file_directory = self.getTemporaryFilesDirectory()
            if not output_path is None:
                temporary_file = output_path
                if not dot+extension in temporary_file: temporary_file += dot+extension
            else: temporary_file = temporary_file_directory+f'{self.getHashCode()}{dot}{extension}'
            try: from requests import post
            except:
                self.installModule(module_name='requests', version='2.31.0')
                from requests import post
            def getBrokenText(text='', text_limit=4000):
                broken_text = []
                from re import sub
                def clearLineBreak(text=''): return sub(r'\s+', ' ', text.replace('\n', ' ')).strip()
                while len(text) > text_limit:
                    split_index = max(text.rfind(char, 0, text_limit) for char in ('\n', '.', ';', '!', '?'))
                    split_index = split_index if split_index!=-1 else text_limit
                    broken_text.append(text[:split_index+1])
                    text = clearLineBreak(text[split_index+1:])
                if len(text.strip()) > 0: broken_text.append(clearLineBreak(text))
                return broken_text
            def mergeAudioFiles(_paths=[], _output_path='', speed=1.0):
                try:
                    from pydub import AudioSegment
                    from pydub.effects import speedup
                except:
                    self.installModule(module_name='pydub', version='0.25.1')
                    from pydub import AudioSegment
                    from pydub.effects import speedup
                final_audio = AudioSegment.empty()
                for _path in _paths: final_audio += AudioSegment.from_file(_path)
                if speed != 1.0: final_audio = speedup(final_audio, playback_speed=speed)
                final_audio.export(_output_path, format=extension)
                return _output_path
            broken_text = getBrokenText(text=text, text_limit=4000)
            from os import path
            list_of_temporary_files = []
            if is_openai:
                text_length = len(text)
                if 'hd' in model: total_cost += (30/1000000)*text_length
                else: total_cost += (15/1000000)*text_length
                ROUTE = 'https://api.openai.com/v1/audio/speech'
                headers = {'Content-Type': 'application/json', 'Authorization': 'Bearer '+api_key}
                for text in broken_text:
                    data = {'model': model, 'input': text, 'voice': voice, 'response_format': extension, 'speed': speed}
                    response = post(ROUTE, json=data, headers=headers, timeout=self.__timeout)
                    if response.status_code in (200, 201):
                        partitioned_temporary_file = temporary_file_directory+f'{self.getHashCode()}{dot}{extension}'
                        with open(partitioned_temporary_file, 'wb') as file: file.write(response.content)
                        if path.exists(partitioned_temporary_file): list_of_temporary_files.append(partitioned_temporary_file)
            elif is_google:
                text_length = len(text.encode('utf-8'))
                if 'neural2' in voice or 'wavenet' in voice: cost_per_byte = 0.000016
                elif 'studio' in voice or 'polyglot' in voice: cost_per_byte = 0.00016
                elif 'standard' in voice: cost_per_byte = 0.000004
                else: cost_per_byte = 0.00017
                total_cost += cost_per_byte*text_length
                ROUTE = 'https://texttospeech.googleapis.com/v1/text:synthesize?key='+api_key
                headers = {'Content-Type': 'application/json'}
                for text in broken_text:
                    data = {
                        'audioConfig': {'audioEncoding': 'MP3', 'effectsProfileId': model, 'pitch': pitch, 'speakingRate': speed},
                        'input': {'text': text}, 'voice': {'languageCode': language, 'name': voice}
                    }
                    response = post(ROUTE, json=data, headers=headers, timeout=self.__timeout)
                    if response.status_code in (200, 201):
                        json_object = self.stringToDictionaryOrJSON(string=response.content)
                        audioContent = json_object['audioContent'] if 'audioContent' in json_object else ''
                        _file_dictionary = {'base64_string': audioContent, 'type': extension}
                        partitioned_temporary_file = temporary_file_directory+f'{self.getHashCode()}{dot}{extension}'
                        base64_to_file = self.base64ToFile(file_dictionary=_file_dictionary, file_path=partitioned_temporary_file)
                        if base64_to_file and path.exists(partitioned_temporary_file): list_of_temporary_files.append(partitioned_temporary_file)
            elif is_local:
                try: from gtts import gTTS
                except:
                    self.installModule(module_name='gTTS', version='2.5.3')
                    from gtts import gTTS
                for text in broken_text:
                    if not language is None: text_to_speech = gTTS(text=text, lang=language)
                    else: text_to_speech = gTTS(text=text)
                    partitioned_temporary_file = temporary_file_directory+f'{self.getHashCode()}{dot}{extension}'
                    text_to_speech.save(partitioned_temporary_file)
                    if path.exists(partitioned_temporary_file): list_of_temporary_files.append(partitioned_temporary_file)                
            _output_path = mergeAudioFiles(_paths=list_of_temporary_files, _output_path=temporary_file, speed=speed if is_local else 1.0)
            for _temporary_file in list_of_temporary_files: self.deleteFile(file_path=_temporary_file)
            if not output_path is None: file_dictionary['output_path'] = _output_path.strip()
            else:
                file_dictionary = self.fileToBase64(file_path=_output_path)
                self.deleteFile(file_path=_output_path)
            file_dictionary['str_cost'] = f'{total_cost:.10f}'.strip()
            return file_dictionary
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.textToSpeech: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'base64_string': '', 'type': 'mp3', 'output_path': '', 'str_cost': '0.0000000000'}
    def audioInterpreter(self, file_path='', file_dictionary={}, service='local', model='tiny', api_key='', max_tokens=1000, language=None):
        try:
            result_object = {'answer': '', 'str_cost': '0.0000000000'}
            file_path = str(file_path).strip()
            if type(file_dictionary) != dict or file_dictionary == {}: file_dictionary = {'base64_string': '', 'type': 'mp3'}
            service = service.lower().strip() if type(service) == str and len(service.strip()) > 0 else 'local'
            model = model.lower().strip() if type(model) == str and len(model.strip()) > 0 else 'tiny'
            api_key = str(api_key).strip()
            max_tokens = int(max_tokens) if type(max_tokens) in (bool, int, float) else 1000
            if not language is None: language = str(language).strip()
            is_url_address = self.isURLAddress(file_path=file_path)
            changed_the_path, transcription, output_paths = False, '', []
            try: from requests import get, post
            except:
                self.installModule(module_name='requests', version='2.31.0')
                from requests import get, post
            is_openai, is_deepinfra, is_replicate, has_api_key = service.startswith('openai'), service.startswith('deepinfra'), service.startswith('replicate'), len(api_key) > 0
            model_data, is_local = None, not (is_openai or is_deepinfra or is_replicate or has_api_key)
            if len(file_path) < 1:
                temporary_file_directory = self.getTemporaryFilesDirectory()
                file_name, file_type = self.getHashCode(), file_dictionary['type']
                file_path = temporary_file_directory+file_name+'.'+file_type
                self.base64ToFile(file_dictionary=file_dictionary, file_path=file_path)
                changed_the_path = True
            elif is_url_address and not is_replicate:
                response = get(file_path, timeout=self.__timeout)
                temporary_file_directory = self.getTemporaryFilesDirectory()
                def getAudioType(file_path=''):
                    substrings = ('mp3', 'wav', 'mp4', 'webm', 'm4a', 'mpeg', 'flac', 'mpga', 'oga', 'ogg')
                    for substring in substrings:
                        if substring in file_path: return substring
                    return 'mp3'
                file_name, file_type = self.getHashCode(), getAudioType(file_path=file_path)
                file_path = temporary_file_directory+file_name+'.'+file_type
                with open(file_path, 'wb') as file: file.write(response.content)
                changed_the_path = True
            def getAudioDurationInMinutes(file_path=''):
                try:
                    try: from pydub import AudioSegment
                    except:
                        self.installModule(module_name='pydub', version='0.25.1')
                        from pydub import AudioSegment
                    audio = AudioSegment.from_file(str(file_path).strip())
                    duration_in_seconds = round(len(audio)/1000)
                    duration_in_minutes = duration_in_seconds/60
                    return duration_in_minutes
                except: return 60
            if is_openai:
                URL = 'https://api.openai.com/v1/audio/transcriptions'  
                body = {'model': model}
                if not language is None: body['language'] = language
                total_cost = getAudioDurationInMinutes(file_path=file_path)*0.00600
                result_object['str_cost'] = f'{total_cost:.10f}'.strip()
            elif is_deepinfra:
                if '/' in model: URL = 'https://api.deepinfra.com/v1/inference/'+model
                else: URL = 'https://api.deepinfra.com/v1/inference/openai/'+model
                body = {'language': language} if not language is None else {}
                total_cost = getAudioDurationInMinutes(file_path=file_path)*0.00045
                result_object['str_cost'] = f'{total_cost:.10f}'.strip()
            elif is_replicate:
                if is_url_address:
                    if '/' in model:
                        URL = f'https://api.replicate.com/v1/models/{model}/predictions'
                        body = {'input': {'audio': file_path}}
                    else:
                        URL = 'https://api.replicate.com/v1/predictions'
                        body = {'version': model, 'input': {'audio': file_path}}
                    if not language is None:
                        body['input']['translate'] = True
                        body['input']['language'] = language
                    total_cost = (getAudioDurationInMinutes(file_path=file_path)*0.0135)+0.00049
                    result_object['str_cost'] = f'{total_cost:.10f}'.strip()
                else: is_local, service, model = True, 'local', 'tiny' 
            if (not is_openai and not is_deepinfra and not is_replicate) or is_local:
                try: from whisper import load_model
                except:
                    self.installModule(module_name='openai-whisper', version='20231117')
                    self.installModule(module_name='setuptools-rust', version='1.10.1')
                    from whisper import load_model
                try:
                    from os import environ
                    try: from certifi import where
                    except:
                        self.installModule(module_name='certifi', version='2024.2.2')
                        from certifi import where
                    environ['SSL_CERT_FILE'] = where()
                    from logging import getLogger, ERROR
                    getLogger('requests').setLevel(ERROR)
                except: pass
                from os import environ
                from warnings import filterwarnings
                from contextlib import redirect_stdout
                from io import StringIO
                environ['TRANSFORMERS_NO_PROGRESS_BAR'] = '1'
                filterwarnings("ignore", category=FutureWarning)
                filterwarnings('ignore', message='FP16 is not supported on CPU; using FP32 instead')
                def loadModelSilently(model_name=''):
                    model_data, temp_stdout = None, StringIO()
                    with redirect_stdout(temp_stdout): model_data = load_model(model_name.lower())
                    return model_data
                model_data = loadModelSilently(model_name=model)
            output_paths = self.splitAudioFile(file_path=file_path, max_size_bytes=26214200, output_directory=self.getTemporaryFilesDirectory())
            len_output_paths_greater_than_1 = len(output_paths) > 1
            headers = {'Authorization': 'Bearer '+api_key}
            if is_replicate: headers['Content-Type'] = 'application/json'
            from os import path
            for output_path in output_paths:
                file_name = path.basename(output_path)
                if not is_local:
                    if is_replicate and is_url_address:
                        _, _, json_object = self.executeModel(URL=URL, body=body, headers=headers)
                        if 'output' in json_object:
                            output = json_object['output']
                            if 'transcription' in output: transcription += output['transcription']
                    else:
                        if is_openai: files = {'file': (file_name, open(output_path, 'rb'))}
                        elif is_deepinfra: files = {'audio': open(output_path, 'rb')}
                        response = post(URL, data=body, files=files, headers=headers, timeout=self.__timeout)
                        if response.status_code in (200, 201):
                            json_object = response.json()
                            transcription += str(json_object['text']).strip()+'\n' if 'text' in json_object else ''
                else:
                    if not language is None: json_object = model_data.transcribe(output_path, language=language)
                    else: json_object = model_data.transcribe(output_path)
                    transcription += str(json_object['text']).strip()+'\n' if 'text' in json_object else ''
                if len_output_paths_greater_than_1: self.deleteFile(file_path=output_path)
            if changed_the_path: self.deleteFile(file_path=file_path)
            if self.countTokens(string=transcription, pattern='gpt') > max_tokens: transcription = self.getTokensSummary(string=transcription, max_tokens=max_tokens)
            result_object['answer'] = transcription.strip()
            return result_object
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.audioInterpreter: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'answer': '', 'str_cost': '0.0000000000'}
    def downloadYouTubeVideo(self, url_path='', audio_only=False, extension=None, output_path=None):
        try:
            file_dictionary, temporary_file = {'base64_string': '', 'type': 'mp4'}, ''
            url_path = str(url_path).strip()
            audio_only = bool(audio_only) if type(audio_only) in (bool, int, float) else False
            extension = extension.lower().strip() if type(extension) == str else None
            output_path, dot = output_path.strip() if type(output_path) == str else None, chr(46)
            from os import devnull
            if extension is None: extension = 'mp3' if audio_only else 'mp4'
            if extension.startswith(dot): extension = extension[1:].strip()
            if not output_path is None:
                temporary_file = output_path
                if not dot+extension in temporary_file: temporary_file += dot+extension
            else:
                temporary_file_directory = self.getTemporaryFilesDirectory()
                temporary_file = temporary_file_directory+f'{self.getHashCode()}{dot}{extension}'
            if audio_only: options = {'format': 'bestaudio/best', 'outtmpl': temporary_file.split(dot+extension)[0], 'quiet': True, 'no_warnings': True, 'postprocessors': [{'key': 'FFmpegExtractAudio', 'preferredcodec': extension, 'preferredquality': '192'}]}
            else: options = {'format': 'bestvideo+bestaudio/best', 'outtmpl': temporary_file, 'merge_output_format': extension, 'quiet': True, 'no_warnings': True}
            from contextlib import redirect_stdout, redirect_stderr
            try: from yt_dlp import YoutubeDL
            except:
                self.installModule(module_name='yt-dlp', version='2025.10.22')
                from yt_dlp import YoutubeDL
            try:
                with redirect_stdout(open(devnull, 'w')), redirect_stderr(open(devnull, 'w')):
                    with YoutubeDL(options) as YouTube: YouTube.download([url_path])
            except:
                self.updateModule(module_name='yt-dlp')
                from yt_dlp import YoutubeDL
                with redirect_stdout(open(devnull, 'w')), redirect_stderr(open(devnull, 'w')):
                    with YoutubeDL(options) as YouTube: YouTube.download([url_path])
            if not output_path is None: return temporary_file.strip()
            else:
                file_dictionary = self.fileToBase64(file_path=temporary_file)
                self.deleteFile(file_path=temporary_file)
                return file_dictionary
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.downloadYouTubeVideo: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            try: self.updateModule(module_name='yt-dlp')
            except: pass
            return '' if not output_path is None else {'base64_string': '', 'type': 'mp4'}
    def youTubeInterpreter(self, url_path='', service='local', model='', api_key='', max_tokens=1000, language=None, hour=0, minute=0, second=0, user_code=None):
        try:
            result_object = {'answer': '', 'str_cost': '0.0000000000'}
            url_path = original_url_path = str(url_path).strip()
            service = str(service).lower().strip()
            model = str(model).strip()
            api_key = str(api_key).strip()
            original_max_tokens = max_tokens = int(max_tokens) if type(max_tokens) in (bool, int, float) else 1000
            if not language is None: language = str(language).strip()
            hour = max((0, int(hour))) if type(hour) in (bool, int, float) else 0
            minute = max((0, int(minute))) if type(minute) in (bool, int, float) else 0
            second = max((0, int(second))) if type(second) in (bool, int, float) else 0
            user_code = str(user_code).strip() if user_code else '-'
            have_time, thumbnail_url, details, total_cost, visualization, answer, transcription = sum((hour, minute, second)) > 0, '', '', 0, '', '', ''
            from re import search, split
            def getYoutubeID(url=''):
                result = search(r'(?:v=|\/)([0-9A-Za-z_-]{11}).*', str(url).strip())
                return result.group(1) if result else ''
            try:
                try: from youtubesearchpython import Video, ResultMode
                except:
                    self.installModule(module_name='youtube-search-python', version='1.6.6')
                    from youtubesearchpython import Video, ResultMode
                try: video = Video.getInfo(getYoutubeID(url_path), mode=ResultMode.json)
                except:
                    url_path = split(r'\?', url_path)[0].strip() if search(r'\?', url_path) else url_path.strip()
                    video = Video.getInfo(url_path, mode=ResultMode.json)
                channel_name = str(video['channel']['name']).strip() if ('channel' in video and 'name' in video['channel']) else ''
                title = str(video['title']).strip() if 'title' in video else ''
                publish_date = str(video['publishDate']).strip() if 'publishDate' in video else ''
                description = str(video['description']).strip() if 'description' in video else ''
                thumbnails = video['thumbnails'] if 'thumbnails' in video and type(video['thumbnails']) in (tuple, list) else []
                thumbnail = thumbnails[-1] if len(thumbnails) > 0 else {}
                thumbnail_url = str(thumbnail['url']).strip() if 'url' in thumbnail else ''
                if len(channel_name) > 0: details += f'YouTube Channel Name: {channel_name}\n'
                if len(title) > 0: details += f'Video Title: {title}\n'
                if len(publish_date) > 0: details += f'Video Date: {publish_date}\n'
                if len(description) > 0: details += f'Video Description:\n{description}\n'
                if len(thumbnail_url) > 0:
                    if not user_code in self.__total_number_of_tokens: self.__total_number_of_tokens[user_code] = 0
                    max_tokens = max_tokens*(.25/2) if have_time else max_tokens*.25
                    image_input_model_path = str(self.__image_input_model_path).strip()
                    if service == 'local' and len(image_input_model_path) > 0:
                        def getResponseFromSapiensInterpretation(language='en'):
                            try: from sapiens_transformers.inference import SapiensModel
                            except:
                                self.installModule(module_name='sapiens-transformers', version='1.8.9')
                                from sapiens_transformers.inference import SapiensModel
                            sapiens_model = SapiensModel(model_path=image_input_model_path, show_errors=self.__show_errors)
                            CONTEXT_SIZE = self.getInputContextLimit(model_path=image_input_model_path, default_value=sapiens_model.CONTEXT_SIZE)
                            sapiens_model.add_image(path=thumbnail_url, maximum_pixels=500, merge_images=False)
                            formatted_language = str(language).lower().strip()
                            prompt = 'Describe this image.'
                            if formatted_language.startswith('pt'): prompt = 'Descreva essa imagem.'
                            elif formatted_language.startswith('es'): prompt = 'Describe esta imagen.'
                            try: from sapiens_infinite_context_window import SapiensInfiniteContextWindow
                            except:
                                self.installModule(module_name='sapiens-infinite-context-window', version='1.0.5')
                                from sapiens_infinite_context_window import SapiensInfiniteContextWindow
                            sapiens_infinite_context_window = SapiensInfiniteContextWindow()
                            prompt = sapiens_infinite_context_window.synthesize_tokens(text=prompt, maximum_tokens=CONTEXT_SIZE)
                            image_interpreter_answer = ''
                            self.__total_number_of_tokens[user_code] += sapiens_infinite_context_window.count_tokens(text=prompt)
                            tokens_generator = sapiens_model.generate_text(prompt=prompt, stream=True)
                            for token in tokens_generator:
                                self.__total_number_of_tokens[user_code] += 1
                                image_interpreter_answer += token
                            return image_interpreter_answer
                        image_interpreter_answer = getResponseFromSapiensInterpretation(language=language)
                    else:
                        image_interpreter = self.imageInterpreter(file_path=thumbnail_url, file_dictionary={}, service=service, model=model, api_key=api_key, max_tokens=max_tokens, language=language, user_code=user_code)
                        total_cost += float(image_interpreter['str_cost'])
                        image_interpreter_answer = str(image_interpreter['answer']).strip()
                        if len(image_interpreter_answer) > 0: image_interpreter_answer = image_interpreter_answer.replace('\n\n', '\n').replace('\n', ' - ')
                    if len(image_interpreter_answer) > 0: details += f'Video cover/thumbnail: ({image_interpreter_answer})\n'
                details = str(details).strip()
                if len(details) > 0: answer += details+'\n\n'
            except Exception as error:
                self.updateModule(module_name='youtube-search-python')
                if self.__show_errors:
                    error_message = 'ERROR in UtilitiesNLP.youTubeInterpreter (description): '+str(error)
                    print(error_message)
                    try: self.__print_exc() if self.__display_error_point else None
                    except: pass
            try:
                try: from youtube_transcript_api import YouTubeTranscriptApi
                except:
                    self.installModule(module_name='youtube-transcript-api', version='1.2.2')
                    from youtube_transcript_api import YouTubeTranscriptApi
                main_languages = ('en', 'es', 'pt')
                languages = (
                    'en', 'es', 'pt', 'zh-Hans', 'zh-Hant', 'hi', 'ar', 'bn', 'fr', 'ru', 'id', 
                    'ur', 'de', 'ja', 'sw', 'jv', 'pa', 'mr', 'te', 'vi', 'ko', 'ta', 'tr', 'fa', 
                    'th', 'it', 'pl', 'uk', 'nl', 'ms', 'fil', 'ro', 'km', 'ha', 'my', 'az', 'hu', 
                    'ceb', 'mg', 'ne', 'si', 'ml', 'cs', 'yo', 'el', 'bg', 'sv', 'sk', 'da', 'fi', 
                    'no', 'lv', 'lt', 'et', 'is', 'sl', 'hr', 'sr', 'eu', 'gl', 'ca', 'ga', 'cy', 
                    'mi', 'mt', 'bs', 'mk', 'lb', 'sm', 'to', 'mg', 'af', 'hy', 'am', 'sq', 'ne', 
                    'lo', 'ky', 'tg', 'tk', 'kk', 'mn', 'uz', 'rw', 'lg', 'so', 'st', 'tn', 'sn', 
                    'sd', 'tn', 'ss', 'ts', 've', 'xh', 'zu', 'ny', 'rw', 'om', 'gn', 'ay', 'qu', 
                    'pam', 'ceb', 'tum', 'ht', 'kri', 'haw', 'crs', 'war', 'gaa', 'luo', 'ak', 
                    'ee', 'ln', 'ig', 'lg', 'yo', 'haw', 'kha', 'ab', 'aa', 'as', 'ay', 'ba', 'bho', 
                    'br', 'dv', 'dz', 'eo', 'et', 'fo', 'fj', 'fil', 'gaa', 'gl', 'gn', 'gu', 'ht', 
                    'haw', 'hmn', 'is', 'ig', 'iw', 'kl', 'kn', 'kha', 'kri', 'ky', 'ku', 'lo', 
                    'la', 'ln', 'lt', 'luo', 'lb', 'mg', 'mi', 'mt', 'gv', 'mfe', 'oc', 'os', 
                    'om', 'ps', 'pl', 'pt-PT', 'rn', 'sg', 'sm', 'sr', 'su', 'sv', 'tg', 'ta', 
                    'tt', 'te', 'th', 'bo', 'ti', 'to', 'tk', 'tum', 'ur', 'ug', 'uz', 've', 'war', 
                    'yi', 'yo', 'zu'
                ) if language is None else (language if language not in main_languages else 'pt-PT', 'en', 'es', 'pt')
                transcript = YouTubeTranscriptApi().fetch(getYoutubeID(original_url_path), languages=languages)
                for entry in transcript: transcription += f'start: {entry.start}s - duration: {entry.duration}s - transcription: {entry.text}\n'
                if len(transcription) > 0: transcription = 'In the video you can hear the following:\n'+transcription
                max_tokens = max_tokens*.75
                if self.countTokens(string=transcription, pattern='gpt') > max_tokens: transcription = self.getTokensSummary(string=transcription, max_tokens=max_tokens)
                transcription = transcription.strip()
                if len(transcription) > 0: answer += transcription+'\n\n'
            except Exception as error:
                self.updateModule(module_name='youtube-transcript-api')
                try:
                    transcription = self.__transcribeYouTubeVideo(url_path=original_url_path, max_tokens=original_max_tokens, language=language)
                    if len(transcription) > 0:
                        transcription = 'In the video you can hear the following:\n'+transcription
                        answer += transcription+'\n\n'
                except: pass
                if self.__show_errors:
                    error_message = 'ERROR in UtilitiesNLP.youTubeInterpreter (transcription): '+str(error)
                    print(error_message)
                    try: self.__print_exc() if self.__display_error_point else None
                    except: pass
            try:
                if have_time:
                    temporary_file_directory = self.getTemporaryFilesDirectory()
                    temporary_file = temporary_file_directory+f'{self.getHashCode()}.mp4'
                    temporary_file = self.downloadYouTubeVideo(url_path=original_url_path, audio_only=False, output_path=temporary_file)
                    if len(temporary_file) > 0:
                        try: from moviepy.editor import VideoFileClip
                        except:
                            self.installModule(module_name='moviepy', version='1.0.3')
                            from moviepy.editor import VideoFileClip
                        try: from PIL import Image
                        except:
                            self.installModule(module_name='pillow', version='10.3.0')
                            from PIL import Image
                        try:
                            video = VideoFileClip(temporary_file)
                            main_time = (hour*3600)+(minute*60)+second
                            image_temporary_file_directory = self.getTemporaryFilesDirectory()
                            image_temporary_file = image_temporary_file_directory+f'{self.getHashCode()}.png'
                            frame = video.get_frame(main_time)
                            image = Image.fromarray(frame)
                            image.save(image_temporary_file)
                            max_tokens = max_tokens*(.25/2)
                            image_interpreter = self.imageInterpreter(file_path=image_temporary_file, file_dictionary={}, service=service, model=model, api_key=api_key, max_tokens=max_tokens, language=language, user_code=user_code)
                            self.deleteFile(file_path=image_temporary_file)
                            image_interpreter_answer = str(image_interpreter['answer']).strip()
                            if len(image_interpreter_answer) > 0: visualization += image_interpreter_answer.replace('\n\n', '\n').replace('\n', ' - ')
                            total_cost += float(image_interpreter['str_cost'])
                            visualization = str(visualization).strip()
                            if len(visualization) > 0: answer += f'In the video you can see the following ({hour}:{minute}:{second}): ({visualization})'
                        except Exception as error:
                            if self.__show_errors:
                                error_message = 'ERROR in UtilitiesNLP.youTubeInterpreter (time 2): '+str(error)
                                print(error_message)
                                try: self.__print_exc() if self.__display_error_point else None
                                except: pass
            except Exception as error:
                if self.__show_errors:
                    error_message = 'ERROR in UtilitiesNLP.youTubeInterpreter (time 1): '+str(error)
                    print(error_message)
                    try: self.__print_exc() if self.__display_error_point else None
                    except: pass
            if self.countTokens(string=answer, pattern='gpt') > original_max_tokens: answer = self.getTokensSummary(string=answer, max_tokens=original_max_tokens)
            result_object['answer'] = answer.strip()
            result_object['str_cost'] = f'{total_cost:.10f}'.strip()
            return result_object
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.youTubeInterpreter: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'answer': '', 'str_cost': '0.0000000000'}
    def videoInterpreter(self, file_path='', file_dictionary={}, service='local', model='', api_key='', max_tokens=1000, language=None, hour=0, minute=0, second=0, user_code=None):
        try:
            result_object = {'answer': '', 'str_cost': '0.0000000000'}
            file_path = str(file_path).strip()
            if self.isYouTubeURL(url_path=file_path): return self.youTubeInterpreter(url_path=file_path, service=service, model=model, api_key=api_key, max_tokens=max_tokens, language=language, hour=hour, minute=minute, second=second, user_code=user_code)
            if type(file_dictionary) != dict: file_dictionary = {}
            service = str(service).lower().strip()
            model = str(model).strip()
            api_key = str(api_key).strip()
            max_tokens = int(max_tokens) if type(max_tokens) in (bool, int, float) else 1000
            if not language is None: language = str(language).strip()
            hour = max((0, int(hour))) if type(hour) in (bool, int, float) else 0
            minute = max((0, int(minute))) if type(minute) in (bool, int, float) else 0
            second = max((0, int(second))) if type(second) in (bool, int, float) else 0
            user_code = str(user_code).strip() if user_code else '-'
            have_time = sum((hour, minute, second)) > 0
            created_file = False
            if len(file_path) < 1 and file_dictionary:
                temporary_path = self.getTemporaryFilesDirectory()
                hash_code = self.getHashCode()
                _type = file_dictionary['type']
                file_path = f'{temporary_path}{hash_code}.{_type}'
                created_file = self.base64ToFile(file_dictionary=file_dictionary, file_path=file_path)
            try: from moviepy.editor import VideoFileClip
            except:
                self.installModule(module_name='moviepy', version='1.0.3')
                from moviepy.editor import VideoFileClip
            try: from PIL import Image
            except:
                self.installModule(module_name='pillow', version='10.3.0')
                from PIL import Image
            descriptions, total_cost, transcriptions, answer = '', 0, '', ''
            try:
                video = VideoFileClip(file_path)
                if have_time: main_times = [(hour*3600)+(minute*60)+second]
                else:
                    duration_in_seconds = video.duration
                    middle_time = duration_in_seconds/2
                    initial_time = middle_time/2
                    end_time = middle_time+initial_time
                    main_times = (initial_time, middle_time, end_time)
                temporary_file_directory = self.getTemporaryFilesDirectory()
                temporary_file = temporary_file_directory+f'{self.getHashCode()}.png'
                for main_time in main_times:
                    try:
                        frame = video.get_frame(main_time)
                        image = Image.fromarray(frame)
                        image.save(temporary_file)
                        image_interpreter = self.imageInterpreter(file_path=temporary_file, file_dictionary={}, service=service, model=model, api_key=api_key, max_tokens=max_tokens, language=language, user_code=user_code)
                        self.deleteFile(file_path=temporary_file)
                        descriptions += image_interpreter['answer'].replace('\n\n', '\n').replace('\n', ' - ').strip()+'\n'
                        total_cost += float(image_interpreter['str_cost'])
                    except: pass
                descriptions = str(descriptions).strip()
            except: pass
            if len(descriptions) > 0: descriptions = 'In the video you can see the following:\n'+descriptions
            try:
                if service.startswith('openai'): model = 'whisper-1'
                elif service.startswith('deepinfra'): model = 'whisper-large-v3'
                elif service.startswith('replicate'): model = 'cdd97b257f93cb89dede1c7584e3f3dfc969571b357dbcee08e793740bedd854'
                else: service, model = 'local', 'tiny'
                is_video_compatible = self.getFileExtension(file_path=file_path).lower() not in ('.mp3', '.mp4', '.mpeg', '.mpga', '.m4a', '.wav', '.webm')
                if not is_video_compatible:
                    audio = video.audio
                    temporary_file = temporary_file_directory+f'{self.getHashCode()}.mp3'
                    audio.write_audiofile(temporary_file, logger=None)
                audio_interpreter = self.audioInterpreter(file_path=file_path if is_video_compatible else temporary_file, file_dictionary={}, service=service, model=model, api_key=api_key, max_tokens=max_tokens, language=language)
                if not is_video_compatible: self.deleteFile(file_path=temporary_file)
                transcriptions = audio_interpreter['answer'].replace('\n\n', '\n').replace('\n', ' ').strip()
                total_cost += float(audio_interpreter['str_cost'])
                if len(transcriptions) > 0: answer += descriptions+'\n\nIn the vídeo you can hear the following: '+transcriptions
                else: answer += descriptions
            except: answer += descriptions
            if self.countTokens(string=answer, pattern='gpt') > max_tokens: answer = self.getTokensSummary(string=answer, max_tokens=max_tokens)
            if created_file: self.deleteFile(file_path=file_path)
            result_object['answer'] = answer.strip()
            result_object['str_cost'] = f'{total_cost:.10f}'.strip()
            return result_object
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.videoInterpreter: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'answer': '', 'str_cost': '0.0000000000'}
    def extractAudioFromVideo(self, video_path='', output_directory='./extracted_audios', file_name='', format='mp3'):
        try:
            audio_path = ''
            video_path = str(video_path).strip()
            output_directory = str(output_directory).strip()
            file_name = str(file_name).strip()
            format = str(format).strip()
            from os import makedirs, unlink, path
            try: from moviepy.editor import VideoFileClip
            except:
                self.installModule(module_name='moviepy', version='1.0.3')
                from moviepy.editor import VideoFileClip
            from urllib.parse import urlparse
            try: from requests import get
            except:
                self.installModule(module_name='requests', version='2.31.0')
                from requests import get
            from tempfile import NamedTemporaryFile
            makedirs(output_directory, exist_ok=True)
            is_url = urlparse(video_path).scheme in ('http', 'https')
            if is_url:
                temp_video = NamedTemporaryFile(delete=False, suffix='.mp4')
                try:
                    response = get(video_path, stream=True)
                    response.raise_for_status()
                    for chunk in response.iter_content(chunk_size=8192): temp_video.write(chunk)
                    temp_video.close()
                    video_file_path = temp_video.name
                    if not file_name: file_name = video_path.split('/')[-1].split('.')[0]
                except:
                    temp_video.close()
                    unlink(temp_video.name)
            else:
                if not path.exists(video_path) and self.__show_errors: print(f'The path {video_path} was not found.')
                video_file_path = video_path
                if not file_name: file_name = path.splitext(path.basename(video_path))[0]
            audio_path = path.join(output_directory, f"{file_name}.{format}")
            try:
                video = VideoFileClip(video_file_path)
                audio = video.audio
                audio.write_audiofile(audio_path, codec='libmp3lame' if format == 'mp3' else None, logger=None)
                audio.close()
                video.close()
            except Exception as error:
                if self.__show_errors:
                    error_message = 'ERROR 1 in UtilitiesNLP.extractAudioFromVideo: '+str(error)
                    print(error_message)
                    try: self.__print_exc() if self.__display_error_point else None
                    except: pass
            finally:
                if is_url:
                    try: unlink(temp_video.name)
                    except: pass
            return audio_path
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR 2 in UtilitiesNLP.extractAudioFromVideo: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return ''
    def getInputContextLimit(self, model_path='', default_value=512):
        try:
            model_path = str(model_path).strip()
            default_value = int(default_value) if type(default_value) in (bool, int, float) else 512
            from pathlib import Path
            from json import load
            def search_keys(data={}, keys=[]):
                if isinstance(data, dict):
                    for key, value in data.items():
                        if key in keys: return value
                        result = search_keys(value, keys)
                        if result is not None: return result
                elif isinstance(data, list):
                    for item in data:
                        result = search_keys(item, keys)
                        if result is not None: return result
                return None
            primary_keys = ['max_position_embeddings','max_source_positions','gpt_max_text_tokens']
            fallback_keys = ['n_ctx','n_positions','max_input_length','context_window_size','gpt_max_prompt_tokens','input_context_length','max_audio_context','max_image_tokens','max_modal_context_length']
            other_files = ['tokenizer_config.json','generation_config.json','preprocessor_config.json','model_args.json','training_args.json','special_tokens_map.json','args.json']
            path = Path(model_path)
            if not path.exists(): return default_value
            config_files = list(path.rglob('config.json'))
            for file in config_files:
                try:
                    with file.open('r', encoding='utf-8') as file_open: data = load(file_open)
                    value = search_keys(data, primary_keys)
                    if value is not None: return int(value)
                    value = search_keys(data, fallback_keys)
                    if value is not None: return int(value)
                except: continue
            for file_name in other_files:
                file_path = path / file_name
                if file_path.exists():
                    try:
                        try:
                            with file_path.open('r', encoding='utf-8') as file_open: data = load(file_open)
                        except:
                            content = ''
                            with open(str(file_path), 'r', encoding='utf-8') as file_open: content = str(file_open.read()).strip()
                            data = self.stringToDictionaryOrJSON(string=content)
                        value = search_keys(data, primary_keys + fallback_keys)
                        if value is not None: return int(value)
                    except: continue
            return default_value
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.getInputContextLimit: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return default_value
    def __executeSapiensTask(self, system_instruction='', prompt='', messages=[], task_names=[], config=None, user_code=None):
        try:
            from time import time
            start_time = time()
            result_dictionary = {'answer': '', 'answer_for_files': '', 'files': [], 'sources': [], 'javascript_charts': [], 'artifacts': [], 'next_token': '', 'str_cost': '0.0000000000'}
            system_instruction = str(system_instruction).strip()
            prompt = original_prompt = str(prompt).strip()
            messages = list(messages) if type(messages) in (tuple, list) else []
            task_names = list(task_names) if type(task_names) in (tuple, list) else [str(task_names).upper().strip()]
            has_artifact = len(task_names) > 0 and any(str(task).upper().strip().startswith('ARTIFACT_') for task in task_names)
            user_code = str(user_code).strip() if user_code else '-'
            default_config = {
                'text_model_path': '',
                'code_model_path': '',
                'deep_reasoning_model_path': '',
                'image_input_model_path': '',
                'audio_input_model_path': '',
                'video_input_model_path': '',
                'image_output_model_path': '',
                'audio_output_model_path': '',
                'music_output_model_path': '',
                'video_output_model_path': '',
                'language': '',
                'hur_context_size': None,
                'show_errors': False,
                'image_paths': [],
                'audio_paths': [],
                'video_paths': [],
                'maximum_pixels': 500,
                'merge_images': True,
                'temperature': 0.5,
                'max_new_tokens': None,
                'width': 512,
                'height': 512,
                'fidelity_to_the_prompt': None,
                'precision': None,
                'seed': None,
                'voice_file_path': '',
                'duration_seconds': 5,
                'fps': None,
                'number_of_frames': None,
                'progress': False,
                'url_path': '',
                'media_metadata': [],
                'javascript_chart': False,
                'mathematical_solution': False,
                'creativity': 0.5,
                'humor': 0.5,
                'formality': 0.5,
                'political_spectrum': 0.5,
                'cost_per_execution': 0.0,
                'cost_per_token': 0.0,
                'cost_per_million_tokens': 0.0,
                'cost_per_second': 0.0,
                'disabled_tasks': []
            }
            config = config if type(config) == dict else default_config
            text_model_path = str(config.get('text_model_path', '')).strip()
            code_model_path = str(config.get('code_model_path', '')).strip()
            deep_reasoning_model_path = str(config.get('deep_reasoning_model_path', '')).strip()
            image_input_model_path = str(config.get('image_input_model_path', '')).strip()
            audio_input_model_path = str(config.get('audio_input_model_path', '')).strip()
            video_input_model_path = str(config.get('video_input_model_path', '')).strip()
            image_output_model_path = str(config.get('image_output_model_path', '')).strip()
            audio_output_model_path = str(config.get('audio_output_model_path', '')).strip()
            music_output_model_path = str(config.get('music_output_model_path', '')).strip()
            video_output_model_path = str(config.get('video_output_model_path', '')).strip()
            if ((not config or config == default_config) or (not text_model_path and not code_model_path and not deep_reasoning_model_path and not image_input_model_path and not audio_input_model_path
            and not video_input_model_path and not image_output_model_path and not audio_output_model_path and not music_output_model_path and not video_output_model_path)):
                default_answer = self.getDefaultAnswer(prompt=prompt, messages=messages, language=None)
                tokens = self.stringToTokens(string=default_answer)
                for token in tokens:
                    result_dictionary['answer'] += token
                    result_dictionary['answer_for_files'] += token
                    result_dictionary['next_token'] = token
                    yield result_dictionary
                return
            language = str(config.get('language', '')).strip()
            hur_context_size = config.get('hur_context_size', None)
            if hur_context_size: hur_context_size = int(hur_context_size)
            show_errors = bool(config.get('show_errors', False))
            image_paths = list(config.get('image_paths', []))
            audio_paths = list(config.get('audio_paths', []))
            video_paths = list(config.get('video_paths', []))
            maximum_pixels = int(config.get('maximum_pixels', 500))
            merge_images = bool(config.get('merge_images', True))
            temperature = float(config.get('temperature', 0.5))
            max_new_tokens, default_limit = config.get('max_new_tokens', None), 1010000
            if max_new_tokens: max_new_tokens = int(max_new_tokens)
            width = int(config.get('width', 512))
            height = int(config.get('height', 512))
            fidelity_to_the_prompt = config.get('fidelity_to_the_prompt', None)
            if fidelity_to_the_prompt: fidelity_to_the_prompt = float(fidelity_to_the_prompt)
            precision = config.get('precision', None)
            if precision: precision = float(precision)
            seed = config.get('seed', None)
            if seed: seed = int(seed)
            voice_file_path = str(config.get('voice_file_path', '')).strip()
            duration_seconds = int(config.get('duration_seconds', 5))
            fps = config.get('fps', None)
            if fps: fps = int(fps)
            number_of_frames = config.get('number_of_frames', None)
            if number_of_frames: number_of_frames = int(number_of_frames)
            progress = bool(config.get('progress', False))
            url_path = str(config.get('url_path', '')).strip()
            media_metadata = list(config.get('media_metadata', []))
            javascript_chart = bool(config.get('javascript_chart', False))
            mathematical_solution = bool(config.get('mathematical_solution', False))
            creativity = float(config.get('creativity', 0.5))
            humor = float(config.get('humor', 0.5))
            formality = float(config.get('formality', 0.5))
            political_spectrum = float(config.get('political_spectrum', 0.5))
            cost_per_execution = float(config.get('cost_per_execution', 0.0))
            cost_per_token = float(config.get('cost_per_token', 0.0))
            cost_per_million_tokens = float(config.get('cost_per_million_tokens', 0.0))
            cost_per_second = float(config.get('cost_per_second', 0.0))
            disabled_tasks = list(config.get('disabled_tasks', []))
            image_paths = list(set(image_paths))
            audio_paths = list(set(audio_paths))
            video_paths = list(set(video_paths))
            self.__image_input_model_path = image_input_model_path
            from os import environ
            environ['TOKENIZERS_PARALLELISM'] = 'true'
            environ['KMP_WARNINGS'] = 'false'
            environ['KMP_SETTINGS'] = 'false'
            environ['OMP_DISPLAY_ENV'] = 'false'
            if cost_per_execution > 0: result_dictionary['str_cost'] = f'{cost_per_execution:.10f}'
            def get_prompt(prompt='', messages=[]):
                prompt_content = str(prompt).strip()
                if not prompt_content:
                    prompt_messages = messages[-1] if len(messages) > 0 else {'role': 'user', 'content': ''}
                    prompt_role, prompt_content = str(prompt_messages.get('role', '')).lower().strip(), ''
                    if prompt_role == 'user': prompt_content = str(prompt_messages.get('content', '')).strip()
                return prompt_content
            prompt_obtained = get_prompt(prompt=prompt, messages=messages)
            task_names = self.__removeTasks(prompt=prompt_obtained, task_names=task_names, disabled_tasks=disabled_tasks)
            if len(task_names) > 0 and any(task.startswith('TRANSLATION_') for task in task_names):
                number_of_tokens_in_the_prompt = self.countTokens(string=prompt_obtained, pattern='gpt-4')
                tokens_limit = max_new_tokens if type(max_new_tokens) == int else 4096
                if number_of_tokens_in_the_prompt > tokens_limit:
                    language_abbreviation = [task for task in task_names if task.startswith('TRANSLATION_')][0].split('_')[-1].lower().strip()
                    translated_answer = self.translate(string=self.formatPrompt(prompt=prompt_obtained, task_names=task_names, language=language), source_language='auto', target_language=language_abbreviation)
                    if translated_answer and translated_answer.lower().strip() not in prompt_obtained and prompt_obtained.lower().strip() not in translated_answer:
                        result_dictionary['answer'] = translated_answer
                        answer = result_dictionary['answer']
                        result_dictionary['answer'] = ''
                        tokens = self.stringToTokens(string=answer)
                        for token in tokens:
                            result_dictionary['answer'] += token
                            result_dictionary['next_token'] = token
                            yield result_dictionary
                        return
            if mathematical_solution:
                if messages: messages[-1]['content'] = self.mathematicalSolution(prompt=prompt_obtained)
                elif prompt: prompt = self.mathematicalSolution(prompt=prompt)
            base64_string, _type, full_text = '', '', ''
            hour, minute, second = 0, 0, 0
            for task_name in task_names:
                if task_name.startswith('WIDTH_HEIGHT'):
                    WIDTH_HEIGHT = task_name.split('WIDTH_HEIGHT_')[-1]
                    WIDTH_HEIGHT = WIDTH_HEIGHT.split('_')
                    width, height = int(float(WIDTH_HEIGHT[0])), int(float(WIDTH_HEIGHT[-1]))
                elif task_name.startswith('TIME_'):
                    TIME = task_name.split('TIME_')[-1]
                    TIME = TIME.split('_')
                    hour, minute, second = int(float(TIME[0])), int(float(TIME[1])), int(float(TIME[-1]))
            system_customization, file_directory, created_local_file, extensions, result = {}, '', False, [], {}
            try: from sapiens_infinite_context_window import SapiensInfiniteContextWindow
            except:
                self.installModule(module_name='sapiens-infinite-context-window', version='1.0.5')
                from sapiens_infinite_context_window import SapiensInfiniteContextWindow
            sapiens_infinite_context_window = SapiensInfiniteContextWindow()
            system_tokens_number = sapiens_infinite_context_window.count_tokens(text=system_instruction)
            prompt_tokens_number = sapiens_infinite_context_window.count_tokens(text=prompt)
            messages_tokens_number = sapiens_infinite_context_window.count_tokens(text=str(messages))
            if len(text_model_path) > 0: default_limit = self.getInputContextLimit(model_path=text_model_path, default_value=default_limit)
            if system_tokens_number > default_limit: system_instruction = sapiens_infinite_context_window.synthesize_tokens(text=system_instruction, maximum_tokens=default_limit)
            if prompt_tokens_number > default_limit: prompt = sapiens_infinite_context_window.synthesize_tokens(text=prompt, maximum_tokens=default_limit)
            if messages_tokens_number > default_limit: messages = sapiens_infinite_context_window.synthesize_messages(prompt=prompt, messages=messages, maximum_tokens=default_limit)['synthesis']
            total_tokens = system_tokens_number+prompt_tokens_number+messages_tokens_number
            if cost_per_token > 0: result_dictionary['str_cost'] = f'{float(cost_per_token*total_tokens):.10f}'
            elif cost_per_million_tokens > 0: result_dictionary['str_cost'] = f'{float((cost_per_million_tokens/1000000)*total_tokens):.10f}'                
            def get_total_tokens(system_instruction='', prompt='', messages=[]):
                system_tokens_number = sapiens_infinite_context_window.count_tokens(text=system_instruction)
                prompt_tokens_number = sapiens_infinite_context_window.count_tokens(text=prompt)
                messages_tokens_number = sapiens_infinite_context_window.count_tokens(text=str(messages))
                return system_tokens_number+prompt_tokens_number+messages_tokens_number
            def get_prompts(system_instruction='', original_prompt='', prompt='', messages=[]):
                def differents(string1='', string2=''):
                    normalized_string1 = str(string1).lower().strip()
                    normalized_string2 = str(string2).lower().strip()
                    if not normalized_string1 and normalized_string2: return True
                    elif not normalized_string2 and normalized_string1: return True
                    return normalized_string1 not in normalized_string2 and normalized_string2 not in normalized_string1
                system_messages, prompt_messages = messages[0] if len(messages) > 1 else {'role': 'system', 'content': ''}, messages[-1] if len(messages) > 0 else {'role': 'user', 'content': ''}
                system_role, system_content, original_system_content = str(system_messages.get('role', '')).lower().strip(), '', ''
                if system_role == 'system': system_content = original_system_content = str(system_messages.get('content', '')).strip()
                if system_instruction and differents(system_content, system_instruction): system_content = str(system_instruction+'\n\n'+system_content).strip()
                if system_role == 'system' and system_content and original_system_content: messages[0]['content'] = system_content
                elif system_content: messages = [{'role': 'system', 'content': system_content}]+messages
                prompt_role, prompt_content, original_prompt_content = str(prompt_messages.get('role', '')).lower().strip(), '', ''
                if prompt_role == 'user': prompt_content = original_prompt_content = str(prompt_messages.get('content', '')).strip()
                if prompt and differents(prompt_content, prompt): prompt_content = str(prompt+'\n\n'+prompt_content).strip()
                if original_prompt and differents(prompt_content, original_prompt): prompt_content = str(prompt_content+'\n\n'+original_prompt).strip()
                if prompt_role == 'user' and original_prompt_content: messages[-1]['content'] = prompt_content
                else: messages.append({'role': 'user', 'content': prompt_content})
                if not original_prompt and prompt_content: original_prompt = prompt_content
                return (system_instruction, original_prompt, prompt, messages)
            def get_infinite_context(system_instruction='', prompt='', messages=[], max_tokens=1000000, task_names=[], disabled_tasks=[]):
                all_the_text_of_the_tokens = ''
                if system_instruction: all_the_text_of_the_tokens += system_instruction+'\n'
                if prompt: all_the_text_of_the_tokens += prompt+'\n'
                if messages: all_the_text_of_the_tokens += str(messages)+'\n'
                all_the_text_of_the_tokens = all_the_text_of_the_tokens.strip()
                original_text_tokens_number = sapiens_infinite_context_window.count_tokens(text=all_the_text_of_the_tokens)
                if type(max_new_tokens) == int: max_tokens -= max_new_tokens
                max_tokens = max(64, max_tokens)
                if prompt: prompt = self.getPromptWithoutIgnoringTheSystem(prompt=prompt, task_names=task_names)
                if messages:
                    prompt_content = get_prompt(prompt=prompt, messages=messages)
                    prompt_content = self.getPromptWithoutIgnoringTheSystem(prompt=prompt_content, task_names=task_names)
                    number_of_prompt_tokens = sapiens_infinite_context_window.count_tokens(text=prompt_content)
                    if number_of_prompt_tokens >= max_tokens:
                        task_names = self.getTasks(prompt=prompt_content)
                        task_names = self.__removeTasks(prompt=prompt_content, task_names=task_names, disabled_tasks=disabled_tasks)
                        code_tasks = 'CODE_CREATION' in task_names or 'CODE_EDITING' in task_names
                        programming_language = self.isProgrammingLanguage(string=prompt_content)
                        if code_tasks and programming_language:
                            number_of_messages_tokens = sapiens_infinite_context_window.count_tokens(text=str(messages[:-1]))
                            max_tokens_messages = max(64, max_tokens-number_of_messages_tokens)
                            prompt_content = sapiens_infinite_context_window.synthesize_code(text=prompt_content, maximum_tokens=max_tokens_messages)
                    messages[-1]['content'] = prompt_content
                if original_text_tokens_number >= max_tokens:
                    number_of_system_tokens = sapiens_infinite_context_window.count_tokens(text=system_instruction)
                    number_of_prompt_tokens = sapiens_infinite_context_window.count_tokens(text=prompt)
                    max_tokens_system = max_tokens
                    max_tokens_prompt = max(64, max_tokens-max_tokens_system)
                    max_tokens_messages = max(64, max_tokens-max_tokens_system-max_tokens_prompt)
                    system_instruction = sapiens_infinite_context_window.synthesize_tokens(text=system_instruction, maximum_tokens=max_tokens_system)
                    prompt = sapiens_infinite_context_window.synthesize_tokens(text=prompt, maximum_tokens=max_tokens_prompt)
                    messages = sapiens_infinite_context_window.synthesize_messages(prompt=prompt, messages=messages, maximum_tokens=max_tokens_messages)['synthesis']
                return (system_instruction, prompt, messages)
            system_instruction = self.addSystemIntonation(system_instruction=system_instruction, creativity=creativity, humor=humor, formality=formality, political_spectrum=political_spectrum, language=language)
            system_instruction, original_prompt, prompt, messages = get_prompts(system_instruction=system_instruction, original_prompt=original_prompt, prompt=prompt, messages=messages)
            EDITION, CREATION = len(task_names) > 0 and any(str(task).upper().strip().endswith('_EDITING') for task in task_names), False
            if EDITION:
                if not user_code in self.__total_number_of_tokens: self.__total_number_of_tokens[user_code] = 0
                def get_last_prompt():
                    prompt_content, new_messages = original_prompt, messages
                    if len(messages) > 0:
                        system_messages = messages[0]
                        system_role = str(system_messages.get('role', '')).lower().strip()
                        if system_role == 'system': new_messages = messages[1:]
                    if len(new_messages) >= 3:
                        prompt_messages = new_messages[-3]
                        prompt_role = str(prompt_messages.get('role', '')).lower().strip()
                        if prompt_role == 'user': prompt_content = str(prompt_messages.get('content', '')).strip()
                    return prompt_content
                def executeNewPrompt(last_prompt='', prompt='', task_names=[], disabled_tasks=[]):
                    return_prompt = prompt
                    insert = 'Replace the terms in the prompt above with the changes suggested in the prompt below, deleting the old terms and generating a new prompt with the changes from the prompt below applied to the prompt above.'
                    new_prompt = f'```python\nprompt = "{last_prompt}"\n```\n\n{insert}\n\n```python\nprompt = "{prompt}"\n```'
                    try: from sapiens_transformers.inference import SapiensModel
                    except:
                        self.installModule(module_name='sapiens-transformers', version='1.8.9')
                        from sapiens_transformers.inference import SapiensModel
                    sapiens_model = SapiensModel(model_path=text_model_path, hur_context_size=hur_context_size, show_errors=show_errors)
                    CONTEXT_SIZE = self.getInputContextLimit(model_path=text_model_path, default_value=sapiens_model.CONTEXT_SIZE)
                    _, new_prompt, _ = get_infinite_context(prompt=new_prompt, max_tokens=CONTEXT_SIZE, task_names=task_names, disabled_tasks=disabled_tasks)
                    generated_text = ''
                    self.__total_number_of_tokens[user_code] += sapiens_infinite_context_window.count_tokens(text=new_prompt)
                    tokens_generator = sapiens_model.generate_text(prompt=new_prompt, temperature=temperature, max_new_tokens=max_new_tokens, stream=True)
                    for token in tokens_generator: generated_text += token
                    self.__total_number_of_tokens[user_code] += sapiens_infinite_context_window.count_tokens(text=generated_text)
                    code_list = self.getCodeList(string=generated_text, html=False)
                    last_code = code_list[-1] if code_list else ''
                    from re import findall
                    result_findall = findall(r'"(.*?)"', last_code)
                    if not result_findall: result_findall = findall(r"'(.*?)'", last_code)
                    if not result_findall: result_findall = last_code.split('=')[-1].strip()
                    _last_prompt = str(last_prompt).lower().strip()
                    for element_findall in result_findall:
                        _element_findall = str(element_findall).lower().strip()
                        if _last_prompt not in _element_findall and _element_findall not in _last_prompt:
                            return_prompt = _element_findall
                            break
                    return_prompt = result_findall[-1] if result_findall else prompt
                    return return_prompt
                last_prompt = get_last_prompt()
                last_tasks = self.getTasks(prompt=last_prompt)
                last_tasks = self.__removeTasks(prompt=last_prompt, task_names=last_tasks, disabled_tasks=disabled_tasks)
                CREATION = len(last_tasks) > 0 and any(str(task).upper().strip().endswith('_CREATION') for task in last_tasks)
                if CREATION:
                    _get_prompt = get_prompt(prompt=prompt, messages=messages)
                    new_prompt = executeNewPrompt(last_prompt=last_prompt, prompt=_get_prompt, task_names=task_names, disabled_tasks=disabled_tasks)
                    if new_prompt == _get_prompt: new_prompt = f'{last_prompt.strip()}\n{_get_prompt.strip()}'
                    if new_prompt and new_prompt != _get_prompt: messages[-1]['content'], task_names = new_prompt, last_tasks+['NO_BACKGROUND'] if 'NO_BACKGROUND' in task_names else last_tasks
                elif sum((len(image_paths), len(audio_paths), len(video_paths), len(url_path))) < 1:
                    for index, task_name in enumerate(task_names):
                        formatted_task = str(task_name).upper().strip()
                        if formatted_task.endswith('_EDITING'): task_names[index] = formatted_task.replace('_EDITING', '_CREATION')
            code_creation_task = ('CODE_CREATION' in task_names or 'CODE_EDITING' in task_names) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='CODE_CREATION', disabled_tasks=disabled_tasks) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='CODE_EDITING', disabled_tasks=disabled_tasks)
            image_interpretation_task = ('IMAGE_INTERPRETATION' in task_names) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='IMAGE_INTERPRETATION', disabled_tasks=disabled_tasks)
            audio_interpretation_task = ('AUDIO_INTERPRETATION' in task_names) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='AUDIO_INTERPRETATION', disabled_tasks=disabled_tasks)
            video_interpretation_task = ('VIDEO_INTERPRETATION' in task_names) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='VIDEO_INTERPRETATION', disabled_tasks=disabled_tasks)
            youtube_download_task = ('YOUTUBE_VIDEO_DOWNLOAD' in task_names or 'YOUTUBE_AUDIO_DOWNLOAD' in task_names) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='YOUTUBE_VIDEO_DOWNLOAD', disabled_tasks=disabled_tasks) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='YOUTUBE_AUDIO_DOWNLOAD', disabled_tasks=disabled_tasks)
            image_creation_task = ('IMAGE_CREATION' in task_names) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='IMAGE_CREATION', disabled_tasks=disabled_tasks)
            logo_creation_task = ('LOGO_CREATION' in task_names) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='LOGO_CREATION', disabled_tasks=disabled_tasks)
            no_background_task = ('NO_BACKGROUND' in task_names) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='NO_BACKGROUND', disabled_tasks=disabled_tasks)
            upscale_image = ('UPSCALE_IMAGE' in task_names) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='UPSCALE_IMAGE', disabled_tasks=disabled_tasks)
            audio_creation_task = ('AUDIO_CREATION' in task_names) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='AUDIO_CREATION', disabled_tasks=disabled_tasks)
            music_creation_task = ('MUSIC_CREATION' in task_names) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='MUSIC_CREATION', disabled_tasks=disabled_tasks)
            video_creation_task = ('VIDEO_CREATION' in task_names) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='VIDEO_CREATION', disabled_tasks=disabled_tasks)
            deep_reasoning_task = ('DEEP_REASONING' in task_names) and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='DEEP_REASONING', disabled_tasks=disabled_tasks)
            if image_interpretation_task and len(url_path) > 0 and len(image_paths) < 1: image_paths = [url_path]
            elif no_background_task and len(url_path) > 0 and len(image_paths) < 1: image_paths = [url_path]
            elif upscale_image and len(url_path) > 0 and len(image_paths) < 1: image_paths = [url_path]
            elif audio_interpretation_task and len(url_path) > 0 and len(audio_paths) < 1: audio_paths = [url_path]
            elif video_interpretation_task and len(url_path) > 0 and len(video_paths) < 1: video_paths = [url_path]
            is_youtube_url = False
            if youtube_download_task and len(video_paths) > 0:
                for video_path in video_paths:
                    if self.isYouTubeURL(url_path=video_path):
                        url_path = video_path
                        is_youtube_url = True
                        break
            elif youtube_download_task and len(url_path) > 0 and self.isYouTubeURL(url_path=url_path): is_youtube_url = True
            if youtube_download_task and not is_youtube_url:
                prompt_content = get_prompt(prompt=prompt, messages=messages)
                link_paths = self.getLinks(string=prompt_content, check_existence=False)
                for link_path in link_paths:
                    if self.isYouTubeURL(url_path=link_path):
                        url_path = link_path
                        is_youtube_url = True
                        break
            ARTIFACT = has_artifact and len(code_model_path) > 0
            IMAGE_INTERPRETATION = image_interpretation_task and len(image_paths) > 0
            AUDIO_INTERPRETATION = audio_interpretation_task and len(audio_paths) > 0
            VIDEO_INTERPRETATION = video_interpretation_task and len(video_paths) > 0
            IMAGE_CREATION = image_creation_task and len(image_output_model_path) > 0
            LOGO_CREATION = logo_creation_task and len(image_output_model_path) > 0
            NO_BACKGROUND = no_background_task and (len(image_output_model_path) > 0 or len(image_paths) > 0)
            UPSCALE_IMAGE = upscale_image and (len(image_output_model_path) > 0 or len(image_paths) > 0)
            AUDIO_CREATION = audio_creation_task and len(audio_output_model_path) > 0
            MUSIC_CREATION = music_creation_task and len(music_output_model_path) > 0
            VIDEO_CREATION = video_creation_task and len(video_output_model_path) > 0
            creation = IMAGE_CREATION or LOGO_CREATION or AUDIO_CREATION or MUSIC_CREATION or VIDEO_CREATION
            YOUTUBE_DOWNLOAD = is_youtube_url and not creation
            CODE_CREATION = code_creation_task and len(code_model_path) > 0 and not creation and len(image_paths) < 1 and len(audio_paths) < 1 and len(video_paths) < 1
            DEEP_REASONING = deep_reasoning_task and len(deep_reasoning_model_path) > 0 and not creation and len(image_paths) < 1 and len(audio_paths) < 1 and len(video_paths) < 1
            ABOUT_SYSTEM = not creation and 'ABOUT_SYSTEM' in task_names
            not_all_tasks = not (code_creation_task or image_interpretation_task or audio_interpretation_task or video_interpretation_task or image_creation_task or
            logo_creation_task or no_background_task or upscale_image or audio_creation_task or music_creation_task or video_creation_task or deep_reasoning_task)
            IMAGE_EDITING = 'IMAGE_EDITING' in task_names and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='IMAGE_EDITING', disabled_tasks=disabled_tasks)
            AUDIO_EDITING = 'AUDIO_EDITING' in task_names and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='AUDIO_EDITING', disabled_tasks=disabled_tasks)
            VIDEO_EDITING = 'VIDEO_EDITING' in task_names and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='VIDEO_EDITING', disabled_tasks=disabled_tasks)
            WEBPAGE_ACCESS = 'WEBPAGE_ACCESS' in task_names and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='WEBPAGE_ACCESS', disabled_tasks=disabled_tasks)
            TEXT_SUMMARY = 'TEXT_SUMMARY' in task_names and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='TEXT_SUMMARY', disabled_tasks=disabled_tasks)
            TEXT_SUMMARY_WITH_BULLET_POINTS = 'TEXT_SUMMARY_WITH_BULLET_POINTS' in task_names and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='TEXT_SUMMARY_WITH_BULLET_POINTS', disabled_tasks=disabled_tasks)
            if len(image_paths) > 0 and not_all_tasks and not IMAGE_EDITING: IMAGE_INTERPRETATION = True
            if len(audio_paths) > 0 and not_all_tasks and not AUDIO_EDITING: AUDIO_INTERPRETATION = True
            if len(video_paths) > 0 and not_all_tasks and not VIDEO_EDITING: VIDEO_INTERPRETATION = True
            if len(image_paths) > 0 and video_interpretation_task: IMAGE_INTERPRETATION, VIDEO_INTERPRETATION = True, False
            if len(video_paths) > 0 and image_interpretation_task: VIDEO_INTERPRETATION, IMAGE_INTERPRETATION = True, False
            if YOUTUBE_DOWNLOAD: IMAGE_INTERPRETATION = AUDIO_INTERPRETATION = VIDEO_INTERPRETATION = False
            if prompt_tokens_number <= 50 and ('RECENT_DATA' in task_names and not 'WEB_SEARCH' in task_names and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='RECENT_DATA', disabled_tasks=disabled_tasks)): task_names.append('WEB_SEARCH')
            elif 'IMAGE_FILTER_APPLICATION' in task_names and not 'IMAGE_EDITING' in task_names and not self.__inDisabledTasks(prompt=prompt_obtained, task_name='IMAGE_FILTER_APPLICATION', disabled_tasks=disabled_tasks):
                task_names.append('IMAGE_EDITING')
                IMAGE_INTERPRETATION, CODE_CREATION = False, len(code_model_path) > 0
            if ABOUT_SYSTEM:
                prompt_content = get_prompt(prompt=prompt, messages=messages)
                prompt_system = self.getSystemInstructions(language=language)
                messages[-1]['content'] = f'{prompt_system}\n\n{prompt_content}'
            background_removal_only, upscale_image_only = False, False
            if IMAGE_INTERPRETATION or AUDIO_INTERPRETATION or VIDEO_INTERPRETATION:
                def transcribeAudio(audio_paths=[], video=False):
                    transcription = ''
                    if len(audio_input_model_path) < 1:
                        insert = '# VIDEO AUDIO CONTENT:\n' if video else '# AUDIO CONTENT:\n'
                        for path in audio_paths: transcription += insert+self.audioInterpreter(file_path=path, service='local', max_tokens=1010000, language=language if language else None)['answer']+'\n\n'
                    else:
                        try: from sapiens_transformers.inference import SapiensModel
                        except:
                            self.installModule(module_name='sapiens-transformers', version='1.8.9')
                            from sapiens_transformers.inference import SapiensModel
                        sapiens_model = SapiensModel(model_path=audio_input_model_path, hur_context_size=hur_context_size, show_errors=show_errors)
                        CONTEXT_SIZE = self.getInputContextLimit(model_path=audio_input_model_path, default_value=sapiens_model.CONTEXT_SIZE)
                        sapiens_model.CONTEXT_SIZE = CONTEXT_SIZE
                        temporary_files_directory = self.getTemporaryFilesDirectory()
                        for audio_path in audio_paths:
                            output_paths = self.splitAudioFile(file_path=audio_path, max_size_bytes=26214200, output_directory=temporary_files_directory)
                            local_transcription = ''
                            for index, output_path in enumerate(output_paths):
                                sapiens_model.add_audio(path=output_path)
                                _local_transcription = ''
                                tokens_generator = sapiens_model.generate_text(stream=True)
                                for token in tokens_generator: _local_transcription += token
                                local_transcription += _local_transcription
                                if video: local_transcription = f'audio(s) from the video(s): {local_transcription} ' if index < 1 else local_transcription+' '
                                else: local_transcription = f'audio(s): {local_transcription} ' if index < 1 else local_transcription+' '
                                if output_path not in audio_paths: self.deleteFile(file_path=output_path)
                            transcription += local_transcription.strip()+'\n\n'
                    transcription = transcription.strip()
                    self.__total_number_of_tokens[user_code] += sapiens_infinite_context_window.count_tokens(text=transcription)
                    return transcription
                def getResponseFromSapiensInterpretation(messages=[], _IMAGE_INTERPRETATION=False, _VIDEO_INTERPRETATION=False, task_names=[], disabled_tasks=[]):
                    try: from sapiens_transformers.inference import SapiensModel
                    except:
                        self.installModule(module_name='sapiens-transformers', version='1.8.9')
                        from sapiens_transformers.inference import SapiensModel
                    model_path = text_model_path
                    if _IMAGE_INTERPRETATION and len(image_input_model_path) > 0: model_path = image_input_model_path
                    if _VIDEO_INTERPRETATION:
                        is_youtube = False
                        for path in video_paths:
                            if self.isURLAddress(file_path=path) and self.isYouTubeURL(url_path=path):
                                is_youtube = True
                                break
                        if not is_youtube and len(video_input_model_path) > 0: model_path = video_input_model_path
                    sapiens_model = SapiensModel(model_path=model_path, hur_context_size=hur_context_size, show_errors=show_errors)
                    CONTEXT_SIZE = self.getInputContextLimit(model_path=model_path, default_value=sapiens_model.CONTEXT_SIZE)
                    all_detections = ''
                    if _IMAGE_INTERPRETATION:
                        for path in image_paths:
                            if len(image_input_model_path) < 1: all_detections += '# IMAGE CONTENT:\n'+self.imageInterpreter(file_path=path, service='local', max_tokens=CONTEXT_SIZE, language=language if language else None, user_code=user_code)['answer']+'\n\n'
                            else: sapiens_model.add_image(path=path, maximum_pixels=maximum_pixels, merge_images=merge_images)
                            self.__total_number_of_tokens[user_code] += maximum_pixels*maximum_pixels
                    if _VIDEO_INTERPRETATION:
                        if type(max_new_tokens) == int: CONTEXT_SIZE = abs(CONTEXT_SIZE-max_new_tokens)
                        youtube_interpreter, youtube_prompt = '', ''
                        for path in video_paths:
                            _maximum_pixels = 100 if maximum_pixels == 500 else max(maximum_pixels, 500)
                            if self.isURLAddress(file_path=path) and self.isYouTubeURL(url_path=path): youtube_interpreter += self.youTubeInterpreter(url_path=path, service='local', max_tokens=CONTEXT_SIZE, language=language if language else None, hour=hour, minute=minute, second=second, user_code=user_code)['answer']+'\n\n'
                            elif len(video_input_model_path) < 1: all_detections += '# VIDEO CONTENT:\n'+self.videoInterpreter(file_path=path, service='local', max_tokens=CONTEXT_SIZE, language=language if language else None, hour=hour, minute=minute, second=second, user_code=user_code)['answer']+'\n\n'
                            else: sapiens_model.add_video(path=path, maximum_pixels=_maximum_pixels)
                            self.__total_number_of_tokens[user_code] += (_maximum_pixels*_maximum_pixels)*3
                        youtube_interpreter = youtube_interpreter.strip()
                        if len(youtube_interpreter) > 0:
                            youtube_prompt = get_prompt(prompt=prompt, messages=messages)
                            messages[-1]['content'] = f'{youtube_interpreter}\n\n{youtube_prompt}'
                    all_detections = all_detections.strip()
                    if all_detections:
                        prompt_detections = get_prompt(prompt=prompt, messages=messages)
                        messages[-1]['content'] = f'{all_detections}\n\n{prompt_detections}'
                    _system_instruction, _, _messages = get_infinite_context(system_instruction=system_instruction, messages=messages, max_tokens=CONTEXT_SIZE, task_names=task_names, disabled_tasks=disabled_tasks)
                    self.__total_number_of_tokens[user_code] += get_total_tokens(system_instruction=_system_instruction, messages=_messages)
                    generated_text = sapiens_model.generate_text(system=_system_instruction, messages=_messages, temperature=temperature, max_new_tokens=max_new_tokens, stream=True)
                    for token in generated_text:
                        self.__total_number_of_tokens[user_code] += 1
                        yield token
                result_dictionary['answer'] = ''
                if IMAGE_INTERPRETATION:
                    generated_text = getResponseFromSapiensInterpretation(messages=messages, _IMAGE_INTERPRETATION=IMAGE_INTERPRETATION, task_names=task_names, disabled_tasks=disabled_tasks)
                    for token in generated_text:
                        result_dictionary['answer'] += token
                        result_dictionary['next_token'] = token
                        yield result_dictionary
                    result_dictionary['answer'] += '\n\n'
                    result_dictionary['next_token'] = '\n\n'
                    yield result_dictionary
                if AUDIO_INTERPRETATION:
                    transcription = transcribeAudio(audio_paths=audio_paths)
                    new_prompt = get_prompt(prompt=prompt, messages=messages)
                    new_prompt = f'{transcription}\n\n{new_prompt}'
                    messages[-1]['content'] = new_prompt
                    generated_text = getResponseFromSapiensInterpretation(messages=messages, task_names=task_names, disabled_tasks=disabled_tasks)
                    for token in generated_text:
                        result_dictionary['answer'] += token
                        result_dictionary['next_token'] = token
                        yield result_dictionary
                    result_dictionary['answer'] += '\n\n'
                    result_dictionary['next_token'] = '\n\n'
                    yield result_dictionary
                if VIDEO_INTERPRETATION:
                    temporary_audios, transcription = [], ''
                    for video_path in video_paths:
                        if not self.isYouTubeURL(url_path=video_path):
                            video_path = self.extractAudioFromVideo(video_path=video_path, output_directory=self.getTemporaryFilesDirectory())
                            if video_path: temporary_audios.append(video_path)
                    if len(temporary_audios) > 0 and len(video_input_model_path) > 0:
                        transcription = transcribeAudio(audio_paths=temporary_audios, video=True)
                        from os import path, remove
                        for temporary_audio in temporary_audios:
                            if path.exists(temporary_audio):
                                try: remove(temporary_audio)
                                except: pass
                    transcription = transcription.strip()
                    if len(transcription) > 1:
                        new_prompt = get_prompt(prompt=prompt, messages=messages)
                        new_prompt = f'{transcription}\n\n{new_prompt}'
                        messages[-1]['content'] = new_prompt
                    generated_text = getResponseFromSapiensInterpretation(messages=messages, _VIDEO_INTERPRETATION=VIDEO_INTERPRETATION, task_names=task_names, disabled_tasks=disabled_tasks)
                    for token in generated_text:
                        result_dictionary['answer'] += token
                        result_dictionary['next_token'] = token
                        yield result_dictionary
                    result_dictionary['answer'] += '\n\n'
                    result_dictionary['next_token'] = '\n\n'
                    yield result_dictionary
                return
            elif IMAGE_CREATION or LOGO_CREATION or AUDIO_CREATION or MUSIC_CREATION or VIDEO_CREATION:
                def generateMedia(task_names=[], disabled_tasks=[]):
                    try: from sapiens_transformers.inference import SapiensModel
                    except:
                        self.installModule(module_name='sapiens-transformers', version='1.8.9')
                        from sapiens_transformers.inference import SapiensModel
                    if LOGO_CREATION: model_path, _type = image_output_model_path, 'png'
                    elif AUDIO_CREATION: model_path, _type = audio_output_model_path, 'mp3'
                    elif MUSIC_CREATION: model_path, _type = music_output_model_path, 'mp3'
                    elif VIDEO_CREATION: model_path, _type = video_output_model_path, 'mp4'
                    else: model_path, _type = image_output_model_path, 'png'
                    sapiens_model = SapiensModel(model_path=model_path, hur_context_size=hur_context_size, show_errors=show_errors)
                    CONTEXT_SIZE = self.getInputContextLimit(model_path=model_path, default_value=sapiens_model.CONTEXT_SIZE)
                    format_prompt = self.formatPrompt(prompt=get_prompt(prompt=prompt, messages=messages), task_names=task_names, language=language)
                    if not AUDIO_CREATION: format_prompt = self.translate(string=format_prompt, source_language='auto', target_language='en')
                    if LOGO_CREATION: format_prompt = 'A logo of: '+format_prompt
                    _, format_prompt, _ = get_infinite_context(prompt=format_prompt, max_tokens=CONTEXT_SIZE, task_names=task_names, disabled_tasks=disabled_tasks)
                    self.__total_number_of_tokens[user_code] += sapiens_infinite_context_window.count_tokens(text=format_prompt)
                    if LOGO_CREATION: base64_string = sapiens_model.generate_base64_image(prompt=format_prompt, width=width, height=height, fidelity_to_the_prompt=fidelity_to_the_prompt, precision=precision, seed=seed)
                    elif AUDIO_CREATION: base64_string = sapiens_model.generate_base64_audio(prompt=format_prompt, voice_file_path=voice_file_path, language=language if language else 'en')
                    elif MUSIC_CREATION: base64_string = sapiens_model.generate_base64_music(prompt=format_prompt, duration_seconds=duration_seconds, fidelity_to_the_prompt=fidelity_to_the_prompt)
                    elif VIDEO_CREATION: base64_string = sapiens_model.generate_base64_video(prompt=format_prompt, width=width, height=height, fidelity_to_the_prompt=fidelity_to_the_prompt, precision=precision, fps=fps, number_of_frames=number_of_frames, seed=seed, progress=progress)
                    else: base64_string = sapiens_model.generate_base64_image(prompt=format_prompt, width=width, height=height, fidelity_to_the_prompt=fidelity_to_the_prompt, precision=precision, seed=seed)
                    self.__total_number_of_tokens[user_code] += (width*height if not AUDIO_CREATION else 1)*(duration_seconds if MUSIC_CREATION else 1)
                    if NO_BACKGROUND and (LOGO_CREATION or IMAGE_CREATION):
                        file_dictionary = {'base64_string': base64_string, 'type': _type}
                        file_dictionary = self.removeImageBackground(file_dictionary=file_dictionary)
                        base64_string, _type = file_dictionary['base64_string'], file_dictionary['type']
                    if UPSCALE_IMAGE and (LOGO_CREATION or IMAGE_CREATION):
                        file_dictionary = {'base64_string': base64_string, 'type': _type}
                        file_dictionary = self.upscaleImage(file_dictionary=file_dictionary)
                        base64_string, _type = file_dictionary['base64_string'], file_dictionary['type']                        
                    return base64_string, _type
                base64_string, _type = generateMedia(task_names=task_names, disabled_tasks=disabled_tasks)
            elif NO_BACKGROUND and len(image_paths) > 0:
                for image_path in image_paths:
                    file_dictionary = self.fileToBase64(file_path=image_path)
                    file_dictionary = self.removeImageBackground(file_dictionary=file_dictionary)
                    file_dictionary = self.updateMediaMetadata(file_dictionary=file_dictionary, metadata=media_metadata)
                    result_dictionary['files'].append(file_dictionary)
                background_removal_only = True
            elif UPSCALE_IMAGE and len(image_paths) > 0:
                for image_path in image_paths:
                    file_dictionary = self.fileToBase64(file_path=image_path)
                    file_dictionary = self.upscaleImage(file_dictionary=file_dictionary)
                    file_dictionary = self.updateMediaMetadata(file_dictionary=file_dictionary, metadata=media_metadata)
                    result_dictionary['files'].append(file_dictionary)
                upscale_image_only = True
            use_code_model_path = False
            if not creation and not background_removal_only and not upscale_image_only:
                if not url_path and image_paths: url_path = str(image_paths[-1]).strip()
                elif not url_path and audio_paths: url_path = str(audio_paths[-1]).strip()
                elif not url_path and video_paths: url_path = str(video_paths[-1]).strip()
                def get_javascript_chart(javascript_charts=[]):
                    if javascript_charts:
                        for index, javascript_chart in enumerate(javascript_charts):
                            if 'https://cdn.plot.ly/plotly-2.35.2.min.js' not in javascript_chart.lower():
                                javascript_chart = f'<script src="https://cdn.plot.ly/plotly-2.35.2.min.js" charset="utf-8"></script>\n{javascript_chart}'
                                javascript_charts[index] = javascript_chart
                            if '<meta charset=' not in javascript_chart.lower():
                                javascript_chart = f'<meta charset="UTF-8">\n{javascript_chart}'
                                javascript_charts[index] = javascript_chart                                
                    return javascript_charts
                system_customization = self.__getSystemCustomization(system_instruction=system_instruction, prompt=get_prompt(prompt=prompt, messages=messages), language=language, max_tokens=max_new_tokens, url_path=url_path, javascript_chart=javascript_chart, task_names=task_names)
                system_instruction = str(system_customization.get('system_instruction', system_instruction)).strip()
                file_directory = str(system_customization.get('file_directory', '')).strip()
                created_local_file = bool(system_customization.get('created_local_file', False))
                extensions = list(system_customization.get('extensions', []))
                prompt = str(system_customization.get('prompt', prompt)).strip()
                result = dict(system_customization.get('result', {}))
                if result:
                    _result_answer = str(result.get('answer', '')).strip()
                    _result_answer_for_files = str(result.get('answer_for_files', '')).strip()
                    _result_files = list(result.get('files', []))
                    _result_sources = list(result.get('sources', []))
                    _result_javascript_charts = list(result.get('javascript_charts', []))
                    if _result_answer: result_dictionary['answer'] = _result_answer
                    if _result_answer_for_files: result_dictionary['answer_for_files'] = _result_answer_for_files
                    if _result_files: result_dictionary['files'] = _result_files
                    if _result_sources: result_dictionary['sources'] = _result_sources
                    if _result_javascript_charts: result_dictionary['javascript_charts'] = get_javascript_chart(javascript_charts=_result_javascript_charts)
                use_code_model_path = len(code_model_path) > 0 and created_local_file
                if EDITION and CREATION: prompt = original_prompt = ''
                system_instruction, original_prompt, prompt, messages = get_prompts(system_instruction=system_instruction, original_prompt=original_prompt, prompt=prompt, messages=messages)
                def getResponseFromSapiens(task_names=[], disabled_tasks=[]):
                    try: from sapiens_transformers.inference import SapiensModel
                    except:
                        self.installModule(module_name='sapiens-transformers', version='1.8.9')
                        from sapiens_transformers.inference import SapiensModel
                    if DEEP_REASONING: model_path = deep_reasoning_model_path
                    else: model_path = code_model_path if CODE_CREATION or ARTIFACT or use_code_model_path else text_model_path
                    def get_default_answer():
                        generated_text = self.getDefaultAnswer(prompt=prompt, messages=messages, language=None)
                        for token in generated_text: yield token
                    from os import path
                    existing_model, generated_text = path.exists(model_path), ''
                    if existing_model:
                        sapiens_model = SapiensModel(model_path=model_path, hur_context_size=hur_context_size, show_errors=show_errors)
                        CONTEXT_SIZE = self.getInputContextLimit(model_path=model_path, default_value=sapiens_model.CONTEXT_SIZE)
                        content_from_web_links = ''
                        prompt_content = get_prompt(prompt=prompt, messages=messages)
                        if WEBPAGE_ACCESS and not url_path:
                            link_paths = self.getLinks(string=prompt_content, check_existence=True)
                            inserted_content = False
                            for link_path in link_paths:
                                file_category, content_from_web_links = self.getFileCategory(file_path=link_path), ''
                                if file_category == 'WEBPAGE_FILE': content_from_web_links = self.getContentFromWEBLinks(list_of_links=[link_path], max_tokens=CONTEXT_SIZE)['text'].strip()
                                elif file_category == 'VIDEO_FILE' and self.isYouTubeURL(url_path=link_path): content_from_web_links = self.youTubeInterpreter(url_path=link_path, service='local', max_tokens=CONTEXT_SIZE, language=language if language else None, hour=hour, minute=minute, second=second, user_code=user_code)['answer'].strip()
                                if content_from_web_links:
                                    prompt_content = prompt_content.replace(link_path, f'\n\n{content_from_web_links}\n\n')
                                    inserted_content = True
                            if inserted_content: messages[-1]['content'] = prompt_content.strip()
                        elif (url_path and self.isURLAddress(file_path=url_path)) and not 'WEBPAGE_ACCESS' in disabled_tasks:
                            if self.isYouTubeURL(url_path=url_path): content_from_web_links = self.youTubeInterpreter(url_path=url_path, service='local', max_tokens=CONTEXT_SIZE, language=language if language else None, hour=hour, minute=minute, second=second, user_code=user_code)['answer'].strip()
                            else: content_from_web_links = self.getContentFromWEBLinks(list_of_links=[url_path], max_tokens=CONTEXT_SIZE)['text'].strip()
                            prompt_content = content_from_web_links+'\n\n'+prompt_content
                            messages[-1]['content'] = prompt_content.strip()
                        if prompt_content.startswith('/'):
                            message_content = messages[-1]['content']
                            message_content = self.translate(string=self.formatPrompt(prompt=message_content, task_names=task_names, language=language))
                            messages[-1]['content'] = message_content
                        _system_instruction, _, _messages = get_infinite_context(system_instruction=system_instruction, messages=messages, max_tokens=CONTEXT_SIZE, task_names=task_names, disabled_tasks=disabled_tasks)
                        self.__total_number_of_tokens[user_code] += get_total_tokens(system_instruction=_system_instruction, messages=_messages)
                        generated_text = sapiens_model.generate_text(system=_system_instruction, messages=_messages, temperature=temperature, max_new_tokens=max_new_tokens, stream=True)
                    if not generated_text: generated_text = get_default_answer()
                    for token in generated_text:
                        self.__total_number_of_tokens[user_code] += 1
                        yield token
                def executeCorrection(prompt='', disabled_tasks=[]):
                    prompt = f'Fix the code below and return the fix in a single code:\n\n{prompt}\n\nReturn only the corrected code, without any explanatory text.'
                    try: from sapiens_transformers.inference import SapiensModel
                    except:
                        self.installModule(module_name='sapiens-transformers', version='1.8.9')
                        from sapiens_transformers.inference import SapiensModel
                    model_path = code_model_path if CODE_CREATION or ARTIFACT or use_code_model_path else text_model_path
                    sapiens_model = SapiensModel(model_path=model_path, hur_context_size=hur_context_size, show_errors=show_errors)
                    CONTEXT_SIZE = self.getInputContextLimit(model_path=model_path, default_value=sapiens_model.CONTEXT_SIZE)
                    _, prompt, _ = get_infinite_context(prompt=prompt, max_tokens=CONTEXT_SIZE, task_names=task_names, disabled_tasks=disabled_tasks)
                    generated_text = ''
                    self.__total_number_of_tokens[user_code] += sapiens_infinite_context_window.count_tokens(text=prompt)
                    tokens_generator = sapiens_model.generate_text(prompt=prompt, temperature=temperature, max_new_tokens=max_new_tokens, stream=True)
                    for token in tokens_generator:
                        self.__total_number_of_tokens[user_code] += 1
                        generated_text += token
                    return generated_text
                def format_matplotlib(string_code=''):
                    string_code = str(string_code).strip()
                    if 'matplotlib' in string_code:
                        code_lines, updated = string_code.split('\n'), False
                        for index, code_line in enumerate(code_lines):
                            if len(code_line.strip()) > 0:
                                if 'show()' in code_line: code_lines[index], updated = code_line.replace('show()', f"savefig('{file_directory}temp.png')"), True
                                if 'imshow(' in code_line: code_lines[index], updated = f'# {code_line}', True
                        if updated: string_code = str('\n'.join(code_lines)).strip()
                    return string_code
                new_prompt = get_prompt(prompt=prompt, messages=messages)
                if not 'WEBPAGE_ACCESS' in disabled_tasks:
                    pieces, link_paths = [], []
                    if '\n' in new_prompt: pieces += [line.strip() for line in new_prompt.split('\n')]
                    if ':' in new_prompt: pieces += [line.strip() for line in new_prompt.split(':')]
                    for piece in pieces:
                        if piece.startswith(('https://', 'http://', 'www.')):
                            link_paths = self.getLinks(string=piece, check_existence=False)
                            break
                    if len(link_paths) > 0: WEBPAGE_ACCESS = True
                messages[-1]['content'] = new_prompt
                if YOUTUBE_DOWNLOAD:
                    generated_text, full_text = '', '_'
                    if len(result_dictionary['files']) < 1:
                        default_answer = self.getDefaultAnswer(prompt=prompt, messages=messages, language=None)
                        tokens = self.stringToTokens(string=default_answer)
                        for token in tokens:
                            result_dictionary['answer'] += token
                            result_dictionary['answer_for_files'] += token
                            result_dictionary['next_token'] = token
                            yield result_dictionary
                        return
                else: generated_text = getResponseFromSapiens(task_names=task_names, disabled_tasks=disabled_tasks)
                if cost_per_second > 0: result_dictionary['str_cost'] = f'{float(abs(time()-start_time)*cost_per_second):.10f}'
                has_file = type(result_dictionary['files']) == list and len(result_dictionary['files']) > 0
                if has_artifact or javascript_chart or created_local_file or has_file:
                    for token in generated_text: full_text += token
                    result_dictionary['answer'] = full_text.strip()
                else:
                    result_dictionary['answer'] = ''
                    for token in generated_text:
                        result_dictionary['answer'] += token
                        result_dictionary['next_token'] = token
                        yield result_dictionary
                    return
                if (has_artifact or javascript_chart) and full_text:
                    html_code = self.getCodeList(string=full_text, html=True)
                    if 'plotly' in str('\n'.join(html_code)).lower().strip(): result_dictionary['javascript_charts'] = get_javascript_chart(javascript_charts=html_code)
                    else: result_dictionary['artifacts'] = html_code
                if created_local_file and full_text:
                    def in_code(disregarded_files=[], code=''):
                        for disregarded_file in disregarded_files:
                            if disregarded_file and self.getFileNameWithExtension(file_path=disregarded_file) in code: return True
                        return False
                    code_list, delete_files = self.getCodeList(string=full_text, html=False), []
                    disregarded_files = self.getFilePathsFromDirectory(directory_path=self.__getRootDirectory(), extensions=[], recursive=False)
                    for code in code_list:
                        _in_code = in_code(disregarded_files=disregarded_files, code=code)
                        execution_result = self.executePythonCode(string_code=format_matplotlib(string_code=code)) if not _in_code else ''
                        if len(execution_result.strip()) < 1:
                            file_list = self.getFilePathsFromDirectory(directory_path=file_directory, extensions=extensions)
                            if not file_list: file_list = self.getFilePathsFromDirectory(directory_path=self.__getRootDirectory(), extensions=extensions)
                            for temporary_path in file_list:
                                if self.getFileNameWithExtension(file_path=temporary_path) in code:
                                    file_dictionary = self.fileToBase64(file_path=temporary_path)
                                    result_dictionary['files'].append(file_dictionary)
                                    delete_files.append(temporary_path)
                        else:
                            correction_content = f'```python\n{code}\n```\n\n{execution_result}'
                            main_result = executeCorrection(prompt=correction_content, disabled_tasks=disabled_tasks) if not _in_code else ''
                            _code_list = self.getCodeList(string=main_result, html=False)
                            for _code in _code_list:
                                execution_result = self.executePythonCode(string_code=format_matplotlib(string_code=_code)) if not _in_code else ''
                                if len(execution_result.strip()) < 1:
                                    file_list = self.getFilePathsFromDirectory(directory_path=file_directory, extensions=extensions)
                                    if not file_list: file_list = self.getFilePathsFromDirectory(directory_path=self.__getRootDirectory(), extensions=extensions)
                                    for temporary_path in file_list:
                                        if self.getFileNameWithExtension(file_path=temporary_path) in code:
                                            file_dictionary = self.fileToBase64(file_path=temporary_path)
                                            result_dictionary['files'].append(file_dictionary)
                                            delete_files.append(temporary_path)
                    if file_directory: self.deleteDirectory(directory_path=file_directory)
                    for delete_file in delete_files:
                        if delete_file not in disregarded_files: self.deleteFile(file_path=delete_file)
                    new_disregarded_files = self.getFilePathsFromDirectory(directory_path=self.__getRootDirectory(), extensions=[], recursive=False)
                    if len(disregarded_files) != len(new_disregarded_files):
                        for new_disregarded_file in new_disregarded_files:
                            if new_disregarded_file not in disregarded_files: self.deleteFile(file_path=new_disregarded_file)
                    if file_directory in result_dictionary['answer']: result_dictionary['answer'] = result_dictionary['answer'].replace(file_directory, '').strip()
                    if url_path in result_dictionary['answer']: result_dictionary['answer'] = result_dictionary['answer'].replace(url_path, '').strip()
            if base64_string and _type:
                file_dictionary = {'base64_string': base64_string, 'type': _type}
                if len(media_metadata) > 0: file_dictionary = self.updateMediaMetadata(file_dictionary=file_dictionary, metadata=media_metadata)
                result_dictionary['files'].append(file_dictionary)
            if len(result_dictionary['files']) > 0 or len(result_dictionary['artifacts']) > 0 or len(result_dictionary['javascript_charts']) > 0: result_dictionary['answer_for_files'] = self.getAnswerForFiles(prompt=original_prompt, language=language if len(language) > 0 else None)
            elif len(result_dictionary['answer']) < 1 and len(result_dictionary['answer_for_files']) < 1 and TEXT_SUMMARY_WITH_BULLET_POINTS: result_dictionary['answer'] = self.summary(string=self.formatPrompt(prompt=original_prompt, task_names=['TEXT_SUMMARY_WITH_BULLET_POINTS'], language=language), topics=True)
            elif len(result_dictionary['answer']) < 1 and len(result_dictionary['answer_for_files']) < 1 and TEXT_SUMMARY: result_dictionary['answer'] = self.summary(string=self.formatPrompt(prompt=original_prompt, task_names=['TEXT_SUMMARY'], language=language), topics=False)
            if (full_text and len(result_dictionary['answer_for_files']) > 1) or background_removal_only or upscale_image_only or IMAGE_CREATION or LOGO_CREATION or AUDIO_CREATION or MUSIC_CREATION or VIDEO_CREATION:
                answer_for_files = result_dictionary['answer_for_files']
                if file_directory in answer_for_files: answer_for_files = answer_for_files.replace(file_directory, '').strip()
                if url_path in answer_for_files: answer_for_files = answer_for_files.replace(url_path, '').strip()
                result_dictionary['answer_for_files'] = ''
                tokens = self.stringToTokens(string=answer_for_files)
                for token in tokens:
                    result_dictionary['answer_for_files'] += token
                    result_dictionary['next_token'] = token
                    yield result_dictionary
                return
            elif full_text and len(result_dictionary['answer']) > 1:
                answer = result_dictionary['answer']
                if file_directory in answer: answer = answer.replace(file_directory, '').strip()
                if url_path in answer: answer = answer.replace(url_path, '').strip()
                result_dictionary['answer'] = ''
                tokens = self.stringToTokens(string=answer)
                for token in tokens:
                    result_dictionary['answer'] += token
                    result_dictionary['next_token'] = token
                    yield result_dictionary
                return
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.__executeSapiensTask: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'answer': '', 'answer_for_files': '', 'files': [], 'sources': [], 'javascript_charts': [], 'artifacts': [], 'next_token': '', 'str_cost': '0.0000000000'}          
    def __executeTask(self, system_instruction='', prompt='', messages=[], task_names=[], config=None):
        try:
            result = {'answer': '', 'answer_for_files': '', 'files': [], 'sources': [], 'javascript_charts': [], 'artifacts': [], 'next_token': '', 'str_cost': '0.0000000000'}
            system_instruction = str(system_instruction).strip()
            prompt = original_prompt = str(prompt).strip()
            messages = list(messages) if type(messages) in (tuple, list) else []
            task_names = list(task_names) if type(task_names) in (tuple, list) else [str(task_names).upper().strip()]
            if len(task_names) > 0 and any(task.startswith('TRANSLATION_') for task in task_names):
                number_of_tokens_in_the_prompt = self.countTokens(string=prompt, pattern='gpt-4')
                if number_of_tokens_in_the_prompt > 4096:
                    language_abbreviation = [task for task in task_names if task.startswith('TRANSLATION_')][0].split('_')[-1].lower().strip()
                    result['answer'] = self.translate(string=self.formatPrompt(prompt=prompt, task_names=task_names, language=language), source_language='auto', target_language=language_abbreviation)
                    return result
            has_artifact = len(task_names) > 0 and any(str(task).upper().strip().startswith('ARTIFACT_') for task in task_names)
            default_config = {
                'model_name': '',
                'api_key': '',
                'version': '',
                'model_route': '',
                'max_tokens': None,
                'language': '',
                'aspect_ratio': '3:2',
                'width': 768,
                'height': 768,
                'duration_in_seconds': 30,
                'audio_format': 'mp3',
                'seed': 0,
                'url_path': '',
                'image_paths': [],
                'media_metadata': [],
                'javascript_chart': False,
                'creativity': 0.5,
                'humor': 0.5,
                'formality': 0.5,
                'political_spectrum': 0.5,
                'cost_per_execution': 0,
                'cost_per_token': 0,
                'cost_per_million_tokens': 0,
                'cost_per_second': 0
            }
            config = config if type(config) == dict else default_config
            model_name = str(config['model_name']).strip() if 'model_name' in config else ''
            api_key = str(config['api_key']).strip() if 'api_key' in config else ''
            version = str(config['version']).strip() if 'version' in config else ''
            model_route = str(config['model_route']).strip() if 'model_route' in config else ''
            max_tokens = int(config['max_tokens']) if 'max_tokens' in config and type(config['max_tokens']) in (bool, int, float) else -1
            if max_tokens < 1:
                if model_name.find('claude-instant') >= 0: max_tokens = 100000
                elif model_name.find('claude-2.1') >= 0: max_tokens = 200000
                elif model_name.find('claude-2') >= 0: max_tokens = 100000
                elif model_name.find('claude-3') >= 0: max_tokens = 200000
                elif model_name.find('claude') >= 0: max_tokens = 200000
                elif model_name.find('gpt-3.5-turbo-instruct') >= 0: max_tokens = 4096
                elif model_name.find('gpt-3.5-turbo') >= 0: max_tokens = 16000
                elif model_name.find('gpt-4-turbo') >= 0: max_tokens = 128000
                elif model_name.find('gpt-4-0125') >= 0: max_tokens = 128000
                elif model_name.find('gpt-4-1106') >= 0: max_tokens = 128000
                elif model_name.find('gpt-4-32k') >= 0: max_tokens = 32000
                elif model_name.find('gpt-4.1') >= 0: max_tokens = 1047576
                elif model_name.find('gpt-4') >= 0: max_tokens = 8192
                elif model_name.find('gpt-5-nano') >= 0: max_tokens = 400000
                elif model_name.find('gpt-5-mini') >= 0: max_tokens = 400000
                elif model_name.find('gpt-5-pro') >= 0: max_tokens = 400000
                elif model_name.find('gpt-5') >= 0: max_tokens = 400000
                elif model_name.find('gemini-pro-vision') >= 0: max_tokens = 12288
                elif model_name.find('gemini-pro') >= 0: max_tokens = 30720
                elif model_name.find('gemini-1.0-pro') >= 0: max_tokens = 30720
                elif model_name.find('gemini-1.5-pro') >= 0: max_tokens = 2097152
                elif model_name.find('gemini-1.5-flash') >= 0: max_tokens = 1048576
                elif model_name.find('gemini-2.5-flash') >= 0: max_tokens = 1048576
                elif model_name.find('gemini-2.5-pro') >= 0: max_tokens = 1048576
                elif model_name.find('llama3.1') >= 0 or model_name.find('llama-3.1') >= 0: max_tokens = 131072
                elif model_name.find('llama3') >= 0 or model_name.find('llama-3') >= 0: max_tokens = 8192
                elif model_name.find('mistral-nemo') >= 0: max_tokens = 1024000
                elif model_name.find('mistral') >= 0: max_tokens = 32768
                elif model_name.find('mixtral') >= 0: max_tokens = 32768
                elif model_name.find('gemma') >= 0: max_tokens = 8192
                elif model_name.find('phi3') >= 0 or model_name.find('phi-3') >= 0: max_tokens = 131072
                elif model_name.find('yi') >= 0: max_tokens = 4096
                elif model_name.find('falcon') >= 0: max_tokens = 2048
                elif model_name.find('stablelm2') >= 0 or model_name.find('stablelm-2') >= 0: max_tokens = 4096
                else: max_tokens = 750
            language = str(config['language']).strip() if 'language' in config else ''
            aspect_ratio = str(config['aspect_ratio']).strip() if 'aspect_ratio' in config else '3:2'
            if aspect_ratio not in ('1:1', '16:9', '21:9', '3:2', '2:3', '4:5', '5:4', '9:16', '9:21'): aspect_ratio = '3:2'
            width = int(config['width']) if 'width' in config and type(config['width']) in (bool, int, float) else 768
            width = min((max((8, width)), 1024))
            height = int(config['height']) if 'height' in config and type(config['height']) in (bool, int, float) else 768
            height = min((max((8, height)), 1024))
            width, height = (width + 7) // 8 * 8, (height + 7) // 8 * 8
            duration_in_seconds = int(config['duration_in_seconds']) if 'duration_in_seconds' in config and type(config['duration_in_seconds']) in (bool, int, float) else 30
            duration_in_seconds = min((max((5, duration_in_seconds)), 600 if '-musicgen' in model_name else 120))
            audio_format = str(config['audio_format']).lower().strip() if 'audio_format' in config else 'mp3'
            if audio_format not in ('wav', 'mp3'): audio_format = 'mp3'
            seed = int(config['seed']) if 'seed' in config and type(config['seed']) in (bool, int, float) else 0
            from datetime import datetime
            if seed == 0: seed = (datetime.now().hour % 12) + 1
            url_path, model_key = str(config['url_path']).strip() if 'url_path' in config else '', model_name.lower()
            image_paths = config['image_paths'] if 'image_paths' in config else []
            image_paths = list(image_paths) if type(image_paths) in (tuple, list) else []
            if model_key.startswith('replicate') and len(image_paths) > 0 and len(str(url_path).strip()) < 1: url_path = str(image_paths[0]).strip()
            url_path_lenght = len(url_path)
            temporary_dictionary_file = self.imagePathToWebAddress(file_path=url_path, force_png=False, service_name='cloudinary') if url_path_lenght > 0 and self.isImageAddress(file_path=url_path) else {'client_id': '', 'url': '', 'deletehash': ''}
            cost_per_execution = float(config['cost_per_execution']) if 'cost_per_execution' in config and type(config['cost_per_execution']) in (bool, int, float) else 0
            cost_per_token = float(config['cost_per_token']) if 'cost_per_token' in config and type(config['cost_per_token']) in (bool, int, float) else 0
            cost_per_million_tokens = float(config['cost_per_million_tokens']) if 'cost_per_million_tokens' in config and type(config['cost_per_million_tokens']) in (bool, int, float) else 0
            cost_per_second = float(config['cost_per_second']) if 'cost_per_second' in config and type(config['cost_per_second']) in (bool, int, float) else 0
            number_of_system_tokens = self.countTokens(string=system_instruction, pattern='gpt')
            number_of_prompt_tokens = self.countTokens(string=prompt, pattern='gpt')
            number_of_messages_tokens = self.countTokens(string=str(messages), pattern='gpt')
            if (number_of_system_tokens+number_of_prompt_tokens+number_of_messages_tokens) > max_tokens:
                if number_of_messages_tokens > number_of_prompt_tokens: max_tokens_for_system, max_tokens_for_prompt, max_tokens_for_messages = int(max_tokens*.10), int(max_tokens*.20), int(max_tokens*.70)
                elif number_of_prompt_tokens > number_of_messages_tokens: max_tokens_for_system, max_tokens_for_messages, max_tokens_for_prompt = int(max_tokens*.10), int(max_tokens*.20), int(max_tokens*.70)
                system_instruction = self.getTokensSummary(string=system_instruction, max_tokens=max_tokens_for_system)
                prompt = self.getTokensSummary(string=prompt, max_tokens=max_tokens_for_prompt)
                messages = self.getMessagesSummary(messages=messages, max_tokens=max_tokens_for_messages)
            elif number_of_system_tokens > max_tokens or number_of_prompt_tokens > max_tokens or number_of_messages_tokens > max_tokens:
                if number_of_system_tokens > max_tokens: system_instruction = self.getTokensSummary(string=system_instruction, max_tokens=max_tokens)
                elif number_of_prompt_tokens > max_tokens: prompt = self.getTokensSummary(string=prompt, max_tokens=max_tokens)
                elif number_of_messages_tokens > max_tokens: messages = self.getMessagesSummary(messages=messages, max_tokens=max_tokens)
            def checkModelInString(model_name=''):
                model_strings = (
                    'llava-13b',
                    'llava-v1.6-vicuna-13b',
                    'llava-v1.6-mistral-7b',
                    'llava-v1.6-34b',
                    'moe-llava',
                    'video-llava',
                    'llava-v1.6-vicuna-7b',
                    'llava-phi-3-mini',
                    'llava-lies',
                    'bakllava',
                    'llava-next-video',
                    'pllava',
                    'llava-birds',
                    'dolphin-2.9-llama3-70b-gguf',
                    'dolphin-2.9-llama3-8b-gguf',
                    'dolphin-2.9.1-llama3-8b-gguf',
                    'mistral-7b-instruct-v0.1',
                    'gemma-7b-it',
                    'gemma-2b-it',
                    'gemma-7b',
                    'gemma-2b',
                    'gemma2-27b-it',
                    'gemma2-9b-it',
                    'phi-3-mini-4k-instruct',
                    'phi-3-mini-128k-instruct',
                    'yi-34b-chat',
                    'yi-6b-chat',
                    'yi-34b-200k',
                    'yi-34b',
                    'yi-6b',
                    'falcon-40b-instruct',
                    'oasst-falcon-7b-sft-top1-696',
                    'stable-diffusion-img2img',
                    'stable-diffusion-inpainting',
                    'stable-diffusion',
                    'sdxl',
                    'logoai',
                    'remove-bg',
                    'ic-light',
                    'real-esrgan',
                    'musicgen',
                    'animatediff-illusions',
                )
                for model in model_strings:
                    if model in model_name: return True
                return False
            use_time = (cost_per_execution > 0 and cost_per_second > 0) or (cost_per_second > 0 and cost_per_token <= 0) or checkModelInString(model_name=model_name)
            media_metadata = config['media_metadata'] if 'media_metadata' in config else []
            media_metadata = list(media_metadata) if type(media_metadata) in (tuple, list) else []
            javascript_chart = config['javascript_chart'] if 'javascript_chart' in config else False
            javascript_chart = bool(javascript_chart) if type(javascript_chart) in (bool, int, float) else False
            creativity = min((1, max((0, float(config['creativity']))))) if 'creativity' in config and type(config['creativity']) in (bool, int, float) else 0.5
            humor = min((1, max((0, float(config['humor']))))) if 'humor' in config and type(config['humor']) in (bool, int, float) else 0.5
            formality = min((1, max((0, float(config['formality']))))) if 'formality' in config and type(config['formality']) in (bool, int, float) else 0.5
            political_spectrum = min((1, max((0, float(config['political_spectrum']))))) if 'political_spectrum' in config and type(config['political_spectrum']) in (bool, int, float) else 0.5
            if len(task_names) > 0 and task_names != ['TEXT_INTERPRETATION']: political_spectrum = 0.5
            system_instruction = self.addSystemIntonation(system_instruction=system_instruction, creativity=creativity, humor=humor, formality=formality, political_spectrum=political_spectrum, language=language)
            political_character = True if political_spectrum != 0.5 else False
            if url_path_lenght < 1 and 'llava' in model_key and len(image_paths) > 0: temporary_dictionary_file = self.imagePathToWebAddress(file_path=image_paths[0], force_png=False, service_name='cloudinary')
            web_address = temporary_dictionary_file['url']
            if len(web_address) < 1:
                temporary_dictionary_file = self.imagePathToWebAddress(file_path=url_path, force_png=False, service_name='imgbb') if url_path_lenght > 0 and self.isImageAddress(file_path=url_path) else {'client_id': '', 'url': '', 'deletehash': ''}
                web_address = temporary_dictionary_file['url']
            generated_text, file_data, total_response_cost, total_cost, file_directory, created_local_file, extensions = '', '', 0, 0, '', False, []
            if cost_per_million_tokens > 0: cost_per_token = cost_per_million_tokens/1000000
            without_images_interpretation = ('IMAGE_EDITING', 'IMAGE_FILTER_APPLICATION', 'NO_BACKGROUND', 'UPSCALE_IMAGE')
            not_without_images_interpretation = not self.hasCoincidentElements(vector1=task_names, vector2=without_images_interpretation)
            if political_character:
                if language.lower().startswith('es'): political_instructions = 'Primero, responde con una respuesta más corta como tú mismo después del marcador [ASSISTANT], y finalmente responde con una respuesta más larga como un personaje que sigue con exactitud el sesgo político y la forma de hablar definida en las instrucciones del sistema después del marcador [CHARACTER].'
                elif language.lower().startswith('pt'): political_instructions = 'Primeiro, responda em uma resposta mais curta como você mesmo depois do marcador [ASSISTANT], e por último responda em uma resposta mais longa como um personagem que segue com exatidão o viés político e a forma de falar definida nas instruções de sistema depois do marcador [CHARACTER].'
                else: political_instructions = 'First, respond with a shorter answer as yourself after the [ASSISTANT] marker, and finally respond with a longer answer as a character that strictly follows the political bias and speaking style defined in the system instructions after the [CHARACTER] marker.'
                prompt += '\n\n'+political_instructions
            def get_prompts(system_instruction='', original_prompt='', prompt='', messages=[]):
                def differents(string1='', string2=''):
                    normalized_string1 = str(string1).lower().strip()
                    normalized_string2 = str(string2).lower().strip()
                    if not normalized_string1 and normalized_string2: return True
                    elif not normalized_string2 and normalized_string1: return True
                    return normalized_string1 not in normalized_string2 and normalized_string2 not in normalized_string1
                system_messages, prompt_messages = messages[0] if len(messages) > 1 else {'role': 'system', 'content': ''}, messages[-1] if len(messages) > 0 else {'role': 'user', 'content': ''}
                system_role, system_content, original_system_content = str(system_messages.get('role', '')).lower().strip(), '', ''
                if system_role == 'system': system_content = original_system_content = str(system_messages.get('content', '')).strip()
                if system_instruction and differents(system_content, system_instruction): system_content = str(system_instruction+'\n\n'+system_content).strip()
                if system_role == 'system' and system_content and original_system_content: messages[0]['content'] = system_content
                elif system_content: messages = [{'role': 'system', 'content': system_content}]+messages
                prompt_role, prompt_content, original_prompt_content = str(prompt_messages.get('role', '')).lower().strip(), '', ''
                if prompt_role == 'user': prompt_content = original_prompt_content = str(prompt_messages.get('content', '')).strip()
                if prompt and differents(prompt_content, prompt): prompt_content = str(prompt+'\n\n'+prompt_content).strip()
                if original_prompt and differents(prompt_content, original_prompt): prompt_content = str(prompt_content+'\n\n'+original_prompt).strip()
                if prompt_role == 'user' and original_prompt_content: messages[-1]['content'] = prompt_content
                else: messages.append({'role': 'user', 'content': prompt_content})
                if not original_prompt and prompt_content: original_prompt = prompt_content
                return (system_content, original_prompt, prompt_content, messages)
            def get_prompt(prompt='', messages=[]):
                prompt_content = str(prompt).strip()
                if not prompt_content:
                    prompt_messages = messages[-1] if len(messages) > 0 else {'role': 'user', 'content': ''}
                    prompt_role, prompt_content = str(prompt_messages.get('role', '')).lower().strip(), ''
                    if prompt_role == 'user': prompt_content = str(prompt_messages.get('content', '')).strip()
                return prompt_content
            system_instruction, original_prompt, prompt, messages = get_prompts(system_instruction=system_instruction, original_prompt=original_prompt, prompt=prompt, messages=messages)
            def getTime():
                from time import time
                return time()
            def getCostPerExecution(start_time=0, end_time=0, cost_per_execution=0, cost_per_second=0):
                if cost_per_execution <= 0 or cost_per_second <= 0:
                    if model_name.find('llava-13b') >= 0: cost_per_execution, cost_per_second = 0.0012, 0.000725
                    elif model_name.find('llava-v1.6-vicuna-13b') >= 0: cost_per_execution, cost_per_second = 0.0035, 0.000725
                    elif model_name.find('llava-v1.6-mistral-7b') >= 0: cost_per_execution, cost_per_second = 0.034, 0.000725
                    elif model_name.find('llava-v1.6-34b') >= 0: cost_per_execution, cost_per_second = 0.0023, 0.001400
                    elif model_name.find('moe-llava') >= 0: cost_per_execution, cost_per_second = 0.0029, 0.000575
                    elif model_name.find('video-llava') >= 0: cost_per_execution, cost_per_second = 0.0061, 0.000725
                    elif model_name.find('llava-v1.6-vicuna-7b') >= 0: cost_per_execution, cost_per_second = 0.0062, 0.000725
                    elif model_name.find('llava-phi-3-mini') >= 0: cost_per_execution, cost_per_second = 0.0034, 0.000575
                    elif model_name.find('llava-lies') >= 0: cost_per_execution, cost_per_second = 0.0044, 0.000725
                    elif model_name.find('bakllava') >= 0: cost_per_execution, cost_per_second = 0.0061, 0.000725
                    elif model_name.find('llava-next-video') >= 0: cost_per_execution, cost_per_second = 0.0071, 0.001400
                    elif model_name.find('pllava') >= 0: cost_per_execution, cost_per_second = 0.0023, 0.000725
                    elif model_name.find('llava-birds') >= 0: cost_per_execution, cost_per_second = 0.00078, 0.000725
                    elif model_name.find('dolphin-2.9-llama3-70b-gguf') >= 0: cost_per_execution, cost_per_second = 0.012, 0.001400
                    elif model_name.find('dolphin-2.9-llama3-8b-gguf') >= 0: cost_per_execution, cost_per_second = 0.00084, 0.000225
                    elif model_name.find('dolphin-2.9.1-llama3-8b-gguf') >= 0: cost_per_execution, cost_per_second = 0.00084, 0.000225
                    elif model_name.find('mistral-7b-instruct-v0.1') >= 0: cost_per_execution, cost_per_second = 0.0018, 0.000725
                    elif model_name.find('gemma-7b-it') >= 0: cost_per_execution, cost_per_second = 0.0016, 0.000725
                    elif model_name.find('gemma-2b-it') >= 0: cost_per_execution, cost_per_second = 0.00088, 0.000725
                    elif model_name.find('gemma-7b') >= 0: cost_per_execution, cost_per_second = 0.0079, 0.000725
                    elif model_name.find('gemma-2b') >= 0: cost_per_execution, cost_per_second = 0.0035, 0.000725
                    elif model_name.find('gemma2-27b-it') >= 0: cost_per_execution, cost_per_second = 0.036, 0.001400
                    elif model_name.find('gemma2-9b-it') >= 0: cost_per_execution, cost_per_second = 0.0069, 0.000725
                    elif model_name.find('phi-3-mini-4k-instruct') >= 0: cost_per_execution, cost_per_second = 0.038, 0.000725
                    elif model_name.find('phi-3-mini-128k-instruct') >= 0: cost_per_execution, cost_per_second = 0.038, 0.000725
                    elif model_name.find('yi-34b-chat') >= 0: cost_per_execution, cost_per_second = 0.011, 0.001400
                    elif model_name.find('yi-6b-chat') >= 0: cost_per_execution, cost_per_second = 0.0027, 0.000725
                    elif model_name.find('yi-34b-200k') >= 0: cost_per_execution, cost_per_second = 0.00072, 0.000725
                    elif model_name.find('yi-34b') >= 0: cost_per_execution, cost_per_second = 0.00072, 0.000725
                    elif model_name.find('yi-6b') >= 0: cost_per_execution, cost_per_second = 0.00072, 0.000725
                    elif model_name.find('falcon-40b-instruct') >= 0: cost_per_execution, cost_per_second = 0.15, 0.005600
                    elif model_name.find('oasst-falcon-7b-sft-top1-696') >= 0: cost_per_execution, cost_per_second = 0.32, 0.001400                    
                    elif model_name.find('stable-diffusion-img2img') >= 0: cost_per_execution, cost_per_second = 0.014, 0.001400
                    elif model_name.find('stable-diffusion-inpainting') >= 0: cost_per_execution, cost_per_second = 0.0062, 0.001400
                    elif model_name.find('stable-diffusion') >= 0: cost_per_execution, cost_per_second = 0.0016, 0.001400
                    elif model_name.find('sdxl') >= 0: cost_per_execution, cost_per_second = 0.011, 0.000725
                    elif model_name.find('logoai') >= 0: cost_per_execution, cost_per_second = 0.0075, 0.000725
                    elif model_name.find('remove-bg') >= 0: cost_per_execution, cost_per_second = 0.00044, 0.000225
                    elif model_name.find('ic-light') >= 0: cost_per_execution, cost_per_second = 0.012, 0.000725
                    elif model_name.find('real-esrgan') >= 0: cost_per_execution, cost_per_second = 0.065, 0.000225
                    elif model_name.find('musicgen') >= 0: cost_per_execution, cost_per_second = 0.030, 0.001400
                    elif model_name.find('animatediff-illusions') >= 0: cost_per_execution, cost_per_second = 0.090, 0.000725
                    else: cost_per_execution, cost_per_second = 0.32, 0.005600
                return cost_per_execution+(abs(end_time-start_time)*cost_per_second)
            def getCostPerToken(cost_per_token=0):
                if cost_per_token <= 0:
                    if model_name.find('meta-llama-3-8b-instruct') >= 0: cost_per_token = (0.05+0.25)/1000000
                    elif model_name.find('meta-llama-3-70b-instruct') >= 0: cost_per_token = (0.65+2.75)/1000000
                    elif model_name.find('meta-llama-3-8b') >= 0: cost_per_token = (0.05+0.25)/1000000
                    elif model_name.find('meta-llama-3-70b') >= 0: cost_per_token = (0.65+2.75)/1000000
                    elif model_name.find('mixtral-8x7b-instruct-v0.1') >= 0: cost_per_token = (0.30+1.00)/1000000
                    elif model_name.find('mistral-7b-instruct-v0.2') >= 0: cost_per_token = (0.05+0.25)/1000000
                    elif model_name.find('mistral-7b-v0.1') >= 0: cost_per_token = (0.05+0.25)/1000000
                    elif model_name.find('meta-llama-3.1-405b-instruct') >= 0: cost_per_token = (9.50+9.50)/1000000
                    else: cost_per_token = (9.50+9.50)/1000000
                return cost_per_token
            def getResponseFromClaude4API(system_instruction='', messages=[], image_paths=[], cost_per_token=0):
                generated_text, total_cost = '', 0
                prompt_length, vision_messages, vision_messages_length = len(prompt), [], 0
                if url_path_lenght > 0 and self.isImageAddress(file_path=url_path) and not_without_images_interpretation: image_paths = [url_path]+image_paths
                if len(image_paths) > 0:
                    content, number_of_images = [], len(image_paths)
                    for index, image_path in enumerate(image_paths):
                        base64Properties = self.fileToBase64(file_path=image_path)
                        image_data = base64Properties['base64_string']
                        _type = base64Properties['type']
                        if _type == 'jpg': _type = 'jpeg'
                        image_media_type = 'image/'+_type
                        content.append({'type': 'image', 'source': {'type': 'base64', 'media_type': image_media_type, 'data': image_data}})
                        image_number = index+1
                        if image_number >= number_of_images:
                            if prompt_length > 0: content.append({'type': 'text', 'text': prompt})
                            else: content.append({'type': 'text', 'text': 'Describe what you see.'})
                        else: content.append({'type': 'text', 'text': str(image_number)+'st view:'})
                    if len(content) > 0: vision_messages.append({'role': 'user', 'content': content})
                    vision_messages_length = len(vision_messages)
                if vision_messages_length > 0: messages = messages+vision_messages
                first_message = messages[0] if len(messages) > 0 else None
                if first_message and 'system' in list(first_message.values()) and 'content'in first_message: system_instruction, messages = str(first_message['content']).strip(), messages[1:]
                if prompt_length > 0 and vision_messages_length < 1: messages.append({'role': 'user', 'content': prompt})
                ROUTE = 'https://api.anthropic.com/v1/messages'
                haiku, sonnet, opus, max_tokens = True, True, True, 4096
                if model_name.find('haiku') >= 0: haiku, max_tokens = True, 64000
                elif model_name.find('sonnet') >= 0: sonnet, max_tokens = True, 64000
                elif model_name.find('opus') >= 0: opus, max_tokens = True, 32000
                data = {'model': model_name, 'temperature': creativity, 'max_tokens': max_tokens, 'system': system_instruction, 'messages': messages}
                headers = {'x-api-key': api_key, 'anthropic-version': '2023-06-01', 'content-type': 'application/json'}
                try: from requests import post
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import post
                from json import dumps
                response = post(ROUTE, data=dumps(data), headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_data = response.json()
                    if 'content' in response_data:
                        content = response_data['content']
                        if type(content) in (tuple, list) and len(content) > 0:
                            content = content[0]
                            if 'text' in content: generated_text, total_tokens = str(content['text']).strip(), 0
                    if 'usage' in response_data:
                        usage = response_data['usage']
                        total_tokens += int(usage['input_tokens']) if 'input_tokens' in usage and type(usage['input_tokens']) in (int, float) else 0
                        total_tokens += int(usage['output_tokens']) if 'output_tokens' in usage and type(usage['output_tokens']) in (int, float) else 0
                        if cost_per_token <= 0:
                            if haiku: cost_per_token = (1+5)/1000000
                            elif sonnet: cost_per_token = (3+15)/1000000
                            elif opus: cost_per_token = (15+75)/1000000
                            else: cost_per_token = (15+80)/1000000
                        total_cost += total_tokens*cost_per_token
                        if vision_messages_length > 0: total_cost += 0.0048
                elif response is not None and self.__show_errors: print(f'ERROR {response.status_code} in getResponseFromClaude4API: {response.text}')
                return generated_text, total_cost
            def getResponseFromClaude3API(system_instruction='', messages=[], image_paths=[], cost_per_token=0):
                generated_text, total_cost = '', 0
                prompt_length, vision_messages, vision_messages_length = len(prompt), [], 0
                if url_path_lenght > 0 and self.isImageAddress(file_path=url_path) and not_without_images_interpretation: image_paths = [url_path]+image_paths
                if len(image_paths) > 0:
                    content, number_of_images = [], len(image_paths)
                    for index, image_path in enumerate(image_paths):
                        base64Properties = self.fileToBase64(file_path=image_path)
                        image_data = base64Properties['base64_string']
                        _type = base64Properties['type']
                        if _type == 'jpg': _type = 'jpeg'
                        image_media_type = 'image/'+_type
                        content.append({'type': 'image', 'source': {'type': 'base64', 'media_type': image_media_type, 'data': image_data}})
                        image_number = index+1
                        if image_number >= number_of_images:
                            if prompt_length > 0: content.append({'type': 'text', 'text': prompt})
                            else: content.append({'type': 'text', 'text': 'Describe what you see.'})
                        else: content.append({'type': 'text', 'text': str(image_number)+'st view:'})
                    if len(content) > 0: vision_messages.append({'role': 'user', 'content': content})
                    vision_messages_length = len(vision_messages)
                if vision_messages_length > 0: messages = messages+vision_messages
                first_message = messages[0] if len(messages) > 0 else None
                if first_message and 'system' in list(first_message.values()) and 'content'in first_message: system_instruction, messages = str(first_message['content']).strip(), messages[1:]
                if prompt_length > 0 and vision_messages_length < 1: messages.append({'role': 'user', 'content': prompt})
                ROUTE = 'https://api.anthropic.com/v1/messages'
                data = {'model': model_name, 'temperature': creativity, 'max_tokens': 4096, 'system': system_instruction, 'messages': messages}
                headers = {'x-api-key': api_key, 'anthropic-version': '2023-06-01', 'content-type': 'application/json'}
                try: from requests import post
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import post
                from json import dumps
                response = post(ROUTE, data=dumps(data), headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_data = response.json()
                    if 'content' in response_data:
                        content = response_data['content']
                        if type(content) in (tuple, list) and len(content) > 0:
                            content = content[0]
                            if 'text' in content: generated_text, total_tokens = str(content['text']).strip(), 0
                    if 'usage' in response_data:
                        usage = response_data['usage']
                        total_tokens += int(usage['input_tokens']) if 'input_tokens' in usage and type(usage['input_tokens']) in (int, float) else 0
                        total_tokens += int(usage['output_tokens']) if 'output_tokens' in usage and type(usage['output_tokens']) in (int, float) else 0
                        if cost_per_token <= 0:
                            if model_name.find('haiku') >= 0: cost_per_token = (0.25+1.25)/1000000
                            elif model_name.find('sonnet') >= 0: cost_per_token = (3+15)/1000000
                            elif model_name.find('opus') >= 0: cost_per_token = (15+75)/1000000
                            else: cost_per_token = (15+80)/1000000
                        total_cost += total_tokens*cost_per_token
                        if vision_messages_length > 0: total_cost += 0.0048
                elif response is not None and self.__show_errors: print(f'ERROR {response.status_code} in getResponseFromClaude3API: {response.text}')
                return generated_text, total_cost
            def getResponseFromClaude2API(prompt='', messages=[], cost_per_token=0):
                generated_text, total_cost = '', 0
                if len(messages) > 0:
                    if len(system_instruction) > 0: messages = [{'role': 'system', 'content': system_instruction}]+messages
                    if len(prompt) > 0: messages.append({'role': 'user', 'content': prompt})
                    full_list, completions = [], ''
                    for message in messages:
                        role = str(message['role']).lower().strip() if 'role' in message else 'user'
                        content = str(message['content']).strip() if 'content' in message else ''
                        if role == 'user' and len(content) > 0: full_list.append('Human: '+content)
                        elif role == 'assistant' and len(content) > 0: full_list.append('Assistant: '+content)
                        elif role == 'system' and len(content) > 0: full_list = ['System: '+content]+full_list
                    if len(full_list) > 0: completions = '\n\n'.join(full_list)
                    if len(completions) > 0:
                        completions = '\n\n'+completions+'\n\nAssistant:'
                        prompt = completions
                if '\n\nHuman: ' not in prompt: prompt = '\n\nHuman: '+prompt
                if '\n\nAssistant:' not in prompt: prompt = prompt+'\n\nAssistant:'
                if len(system_instruction) > 0 and '\n\nSystem: ' not in prompt: prompt = '\n\nSystem: '+system_instruction+prompt
                ROUTE = 'https://api.anthropic.com/v1/complete'
                data = {'model': model_name, 'temperature': creativity, 'max_tokens_to_sample': 1000000, 'prompt': prompt}
                headers = {'x-api-key': api_key, 'anthropic-version': '2023-06-01', 'content-type': 'application/json'}
                try: from requests import post
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import post
                from json import dumps
                response = post(ROUTE, data=dumps(data), headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_data = response.json()
                    generated_text = str(response_data['completion']).strip() if 'completion' in response_data else ''
                    input_output = prompt.strip()+' '+generated_text
                    total_tokens = self.countTokens(string=input_output, pattern='gpt-3')
                    if cost_per_token <= 0:
                        if model_name.find('claude-instant') >= 0: cost_per_token = (0.8+2.4)/1000000
                        elif model_name.find('claude-2') >= 0: cost_per_token = (8+24)/1000000
                        else: cost_per_token = (8+25)/1000000
                    total_cost += total_tokens*cost_per_token
                elif response is not None and self.__show_errors: print(f'ERROR {response.status_code} in getResponseFromClaude2API: {response.text}')
                return generated_text, total_cost
            def getResponseFromChatGPTAPI(messages=[], image_paths=[], cost_per_token=0):
                generated_text, total_cost = '', 0
                prompt_length, vision_messages, vision_messages_length = len(prompt), [], 0
                if url_path_lenght > 0 and self.isImageAddress(file_path=url_path) and not_without_images_interpretation: image_paths = [url_path]+image_paths
                if len(image_paths) > 0:
                    text = prompt if prompt_length > 0 else 'Describe what you see.'
                    content = [{'type': 'text', 'text': text}]
                    for image_path in image_paths:
                        if self.isURLAddress(file_path=image_path): content.append({'type': 'image_url', 'image_url': {'url': image_path}})
                        else:
                            base64Properties = self.fileToBase64(file_path=image_path)
                            image_data = base64Properties['base64_string']
                            _type = base64Properties['type']
                            if _type == 'jpg': _type = 'jpeg'
                            image_media_type = 'image/'+_type
                            content.append({'type': 'image_url', 'image_url': {'url': f'data:{image_media_type};base64,{image_data}'}})
                    if len(content) > 0: vision_messages.append({'role': 'user', 'content': content})
                    vision_messages_length = len(vision_messages)
                if vision_messages_length > 0: messages = messages+vision_messages
                if len(system_instruction) > 0: messages = [{'role': 'system', 'content': system_instruction}]+messages
                if prompt_length > 0 and vision_messages_length < 1: messages.append({'role': 'user', 'content': prompt})
                ROUTE = 'https://api.openai.com/v1/chat/completions'
                data = {'model': model_name, 'messages': messages, 'temperature': creativity}
                headers = {'Content-Type': 'application/json', 'Authorization': 'Bearer '+api_key}
                try: from requests import post
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import post
                response = post(ROUTE, json=data, headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_data = response.json()
                    if 'choices' in response_data:
                        choices = response_data['choices']
                        if type(choices) in (tuple, list) and len(choices) > 0:
                            choices = choices[0]
                            if 'message' in choices:
                                message = choices['message']
                                generated_text = str(message['content']).strip() if 'content' in message else ''
                    if 'usage' in response_data:
                        usage = response_data['usage']
                        total_tokens = int(usage['total_tokens']) if 'total_tokens' in usage and type(usage['total_tokens']) in (int, float) else 0
                        if cost_per_token <= 0:
                            if model_name.find('gpt-4o-mini') >= 0: cost_per_token = (0.60+0.30)/1000000
                            elif model_name.find('gpt-4o') >= 0: cost_per_token = (15+7.50)/1000000
                            elif model_name.find('gpt-3.5') >= 0: cost_per_token = (3+4)/1000000
                            elif model_name.find('gpt-4-turbo') >= 0: cost_per_token = (10+30)/1000000
                            elif model_name.find('gpt-4-32k') >= 0: cost_per_token = (60+120)/1000000
                            elif model_name.find('gpt-4.1') >= 0: cost_per_token = (2.50+8.00)/1000000
                            elif model_name.find('gpt-4') >= 0: cost_per_token = (30+60)/1000000
                            elif model_name.find('gpt-5-nano') >= 0: cost_per_token = (0.055+0.40)/1000000
                            elif model_name.find('gpt-5-mini') >= 0: cost_per_token = (0.275+2.00)/1000000
                            elif model_name.find('gpt-5-pro') >= 0: cost_per_token = (15+120)/1000000
                            elif model_name.find('gpt-5') >= 0: cost_per_token = (1.375+10)/1000000
                            else: cost_per_token = (60+125)/1000000
                        total_cost += total_tokens*cost_per_token
                        if vision_messages_length > 0: total_cost += 0.007225
                elif response is not None and self.__show_errors: print(f'ERROR {response.status_code} in getResponseFromChatGPTAPI: {response.text}')
                return generated_text, total_cost
            def getResponseFromDALLEAPI(prompt='', cost_per_execution=0):
                image_address, total_cost, temporary_cost_per_run = '', 0, 0
                ROUTE = 'https://api.openai.com/v1/images/generations'
                size = f'{width}x{height}'
                sizes_dalle2, sizes_dalle3 = ('256x256', '512x512', '1024x1024'), ('1024x1024', '1792x1024', '1024x1792')
                if model_name.find('dall-e-2') >= 0 and size not in sizes_dalle2:
                    if width <= 256: size, temporary_cost_per_run = '256x256', 0.016
                    elif width <= 512: size, temporary_cost_per_run = '512x512', 0.018
                    else: size, temporary_cost_per_run = '1024x1024', 0.020
                elif model_name.find('dall-e-3') >= 0 and size not in sizes_dalle3:
                    if width <= 1024: size, temporary_cost_per_run = '1024x1024', 0.080
                    elif width <= 1792: size, temporary_cost_per_run = '1792x1024', 0.120
                    else: size, temporary_cost_per_run = '1024x1792', 0.120
                else: temporary_cost_per_run = 0.120
                data = {'model': model_name, 'prompt': prompt, 'n': 1, 'size': size, 'quality': 'hd'}
                headers = {'Content-Type': 'application/json', 'Authorization': 'Bearer '+api_key}
                try: from requests import post
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import post
                response = post(ROUTE, json=data, headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_data = response.json()
                    if 'data' in response_data:
                        data = response_data['data']
                        if type(data) in (tuple, list) and len(data) > 0:
                            data = data[0]
                            if 'url' in data: image_address = str(data['url']).strip()
                elif response is not None and self.__show_errors: print(f'ERROR {response.status_code} in getResponseFromDALLEAPI: {response.text}')
                if cost_per_execution <= 0: cost_per_execution = temporary_cost_per_run
                total_cost += cost_per_execution
                return image_address.strip(), total_cost
            def getResponseFromGeminiAPI(system_instruction='', messages=[], image_paths=[], cost_per_token=0):
                generated_text, total_cost, contents = '', 0, []
                prompt_length, vision_messages, vision_messages_length = len(prompt), [], 0
                if url_path_lenght > 0 and self.isImageAddress(file_path=url_path) and not_without_images_interpretation: image_paths = [url_path]+image_paths
                response_instructions = '(Instructions on how you should respond) '
                if len(image_paths) > 0:
                    first_message = messages[0] if len(messages) > 0 else None
                    if first_message and 'system' in list(first_message.values()) and 'content'in first_message: system_instruction += '\n'+str(first_message['content']).strip()
                    text, messages = prompt if prompt_length > 0 else 'Describe what you see.', []
                    parts = [{'text': response_instructions+system_instruction+'\n'+text}] if len(system_instruction) > 0 else [{'text': text}]
                    for image_path in image_paths:
                        base64Properties = self.fileToBase64(file_path=image_path)
                        image_data = base64Properties['base64_string']
                        _type = base64Properties['type']
                        if _type == 'jpg': _type = 'jpeg'
                        image_media_type = 'image/'+_type
                        parts.append({'inline_data': {'mimeType': image_media_type, 'data': image_data}})
                    if len(parts) > 0: vision_messages.append({'role': 'user', 'parts': parts})
                    vision_messages_length = len(vision_messages)
                if vision_messages_length > 0: messages = messages+vision_messages
                if len(messages) > 0 and 'parts' not in messages[0]:
                    if len(system_instruction) > 0: contents = [{'role': 'user', 'parts': [{'text': response_instructions+system_instruction}]}]
                    for message in messages:
                        role = str(message['role']).lower().strip() if 'role' in message else 'user'
                        content = str(message['content']).strip() if 'content' in message else ''
                        if role == 'user' and len(content) > 0: contents.append({'role': 'user', 'parts': [{'text': content}]})
                        elif role == 'assistant' and len(content) > 0: contents.append({'role': 'model', 'parts': [{'text': content}]})
                        elif role == 'system' and len(content) > 0: contents.append({'role': 'user', 'parts': [{'text': response_instructions+content}]})
                    if len(contents) > 0: messages = contents.copy()
                elif len(system_instruction) > 0: messages = [{'role': 'user', 'parts': [{'text': response_instructions+system_instruction}]}]+messages
                if prompt_length > 0 and vision_messages_length < 1: messages.append({'role': 'user', 'parts': [{'text': prompt}]})
                ROUTE = f'https://generativelanguage.googleapis.com/v1beta/models/{model_name}:generateContent?key={api_key}'
                if vision_messages_length > 0: messages = messages+vision_messages
                data = {'contents': messages, 'generation_config': {'temperature': creativity}}
                headers = {'Content-Type': 'application/json'}
                try: from requests import post
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import post
                response = post(ROUTE, json=data, headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_data = response.json()
                    if 'candidates' in response_data:
                        candidates = response_data['candidates']
                        if type(candidates) in (tuple, list) and len(candidates) > 0:
                            candidates = candidates[0]
                            if 'content' in candidates:
                                content = candidates['content']
                                if 'parts' in content:
                                    parts = content['parts']
                                    if type(parts) in (tuple, list) and len(parts) > 0:
                                        parts = parts[0]
                                        generated_text = str(parts['text']).strip() if 'text' in parts else ''
                    if 'usageMetadata' in response_data:
                        usageMetadata = response_data['usageMetadata']
                        total_tokens = int(usageMetadata['totalTokenCount']) if 'totalTokenCount' in usageMetadata and type(usageMetadata['totalTokenCount']) in (int, float) else 0
                        if cost_per_token <= 0:
                            if model_name.find('gemini-1.5-flash') >= 0: cost_per_token = (0.70+2.10)/1000000
                            elif model_name == 'gemini-pro': cost_per_token = (0.50+1.50)/1000000
                            elif model_name.find('gemini-1.5-pro') >= 0: cost_per_token = (7+21)/1000000
                            elif model_name.find('gemini-2.5-flash') >= 0: cost_per_token = (0.30+2.50)/1000000
                            elif model_name.find('gemini-2.5-pro') >= 0: cost_per_token = (2.50+15.00)/1000000
                            else: cost_per_token = (7+25)/1000000
                        total_cost += total_tokens*cost_per_token
                        if vision_messages_length > 0: total_cost += 0.00263
                elif response is not None and self.__show_errors: print(f'ERROR {response.status_code} in getResponseFromGeminiAPI: {response.text}')
                return generated_text, total_cost
            def getResponseFromOllamaLLM(model_name='', template='', image_paths=[]):
                generated_text, total_cost = '', 0
                ROUTE = 'http://localhost:11434/api/generate'
                data, headers = {'model': model_name, 'prompt': template, 'stream': False}, {'Content-Type': 'application/json'}
                if 'llava' in model_key or 'ollama' in model_key:
                    if url_path_lenght > 0 and self.isImageAddress(file_path=url_path) and not_without_images_interpretation: image_paths = [url_path]+image_paths
                    if len(image_paths) > 0:
                        images = []
                        for image_path in image_paths:
                            base64Properties = self.fileToBase64(file_path=image_path)
                            images.append(base64Properties['base64_string'])
                        if len(images) > 0: data['images'] = images
                try: from requests import post
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import post
                response = post(ROUTE, json=data, headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_data = response.json()
                    generated_text = response_data['response']
                    input_output = template+generated_text
                    total_tokens = self.countTokens(string=input_output, pattern='gpt-3')
                    total_cost += total_tokens*cost_per_token
                elif response is not None and self.__show_errors: print(f'ERROR {response.status_code} in getResponseFromOllamaLLM: {response.text}')
                return generated_text, total_cost
            def getResponseFromReplicateLLMWithModelRoute(template='', cost_per_token=0, cost_per_execution=0, cost_per_second=0):
                generated_text, total_cost = '', 0
                ROUTE = f'https://api.replicate.com/v1/models/{model_route}/predictions'
                data = {'stream': True, 'input': {'prompt': template, 'temperature': creativity}}
                headers = {'Authorization': 'Bearer '+api_key, 'Content-Type': 'application/json'}
                try: from requests import post, get
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import post, get
                from json import dumps
                if use_time: start_time = getTime()
                response = post(ROUTE, data=dumps(data), headers=headers, timeout=self.__timeout)
                response_json = response.json()
                if 'urls' in response_json:
                    urls = response_json['urls']
                    if 'stream' in urls:
                        stream_url = urls['stream']
                        stream_headers = {'Accept': 'text/event-stream', 'Cache-Control': 'no-store'}
                        stream_response, tokens = get(stream_url, headers=stream_headers, stream=True, timeout=self.__timeout), []
                        for line in stream_response.iter_lines():
                            if line:
                                decoded_line = line.decode('utf-8')
                                if decoded_line.startswith('data: '):
                                    token = decoded_line[6:]
                                    if token.endswith('{}'): token = token[:-2]
                                    tokens.append(token)
                        generated_text = ''.join(tokens)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else:
                    input_output = template+generated_text
                    total_tokens = self.countTokens(string=input_output, pattern='gpt-3')
                    if cost_per_token <= 0: cost_per_token = getCostPerToken(cost_per_token=cost_per_token)
                    total_cost += total_tokens*cost_per_token
                if len(str(generated_text).strip()) < 1 and response is not None and self.__show_errors: print(f'ERROR {response.status_code} in getResponseFromReplicateLLMWithModelRoute: {response.text}')
                return generated_text, total_cost
            def getResponseFromReplicateLLM(template='', cost_per_token=0, cost_per_execution=0, cost_per_second=0):
                if len(model_route) > 0: return getResponseFromReplicateLLMWithModelRoute(template=template, cost_per_token=cost_per_token, cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                generated_text, total_cost = '', 0
                ROUTE = 'https://api.replicate.com/v1/predictions'
                data = {'version': version, 'input': {'prompt': template, 'temperature': creativity, 'max_new_tokens': 200000}}
                headers = {'Authorization': 'Token '+api_key, 'Content-Type': 'application/json'}
                try: from requests import post, get
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import post, get
                from json import dumps
                if use_time: start_time = getTime()
                response = post(ROUTE, data=dumps(data), headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_data = response.json()
                    status = str(response_data['status']).lower().strip() if 'status' in response_data else ''
                    if status == 'starting':
                        if 'urls' in response_data:
                            urls = response_data['urls']
                            if 'get' in urls:
                                url = str(urls['get']).strip()
                                response = get(url, headers=headers, timeout=self.__timeout)
                                if response.status_code in (200, 201):
                                    response_data = response.json()
                                    status, output = str(response_data['status']).lower().strip() if 'status' in response_data else '', []
                                    while status in ('starting', 'processing'):
                                        response = get(url, headers=headers, timeout=self.__timeout)
                                        if response.status_code in (200, 201):
                                            response_data = response.json()
                                            status = str(response_data['status']).lower().strip() if 'status' in response_data else ''
                                            if 'output' in response_data: output = list(response_data['output']) if not response_data['output'] is None else []
                                        else: break
                                    if len(output) > 0: generated_text = ''.join(output)
                    if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                    else:
                        input_output = template+generated_text
                        total_tokens = self.countTokens(string=input_output, pattern='gpt-3')
                        if cost_per_token <= 0: cost_per_token = getCostPerToken(cost_per_token=cost_per_token)
                        total_cost += total_tokens*cost_per_token
                elif response is not None and self.__show_errors: print(f'ERROR {response.status_code} in getResponseFromReplicateLLM: {response.text}')
                return generated_text, total_cost
            def getResponseFromReplicateWithModelRoute(ROUTE='', data={}):
                url_address_or_text = ''
                try: from requests import post, get
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import post, get
                headers = {'Authorization': 'Bearer '+api_key, 'Content-Type': 'application/json'}
                response = post(ROUTE, json=data, headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_json = response.json()
                    if 'urls' in response_json:
                        urls = response_json['urls']
                        if 'get' in urls:
                            get_url = urls['get']
                            status, response, output = 'starting', get(get_url, headers=headers, timeout=self.__timeout), ''
                            is_list_of_tokens, output_is_tuple_or_list, output_length = False, False, 0
                            while status in ('starting', 'processing'):
                                response = get(get_url, headers=headers, timeout=self.__timeout)
                                if response.status_code in (200, 201):
                                    response_json = response.json()
                                    status = str(response_json['status']).lower().strip() if 'status' in response_json else ''
                                    if 'output' in response_json:
                                        if not response_json['output'] is None: output = response_json['output']
                                        output_is_tuple_or_list = type(output) in (tuple, list)
                                        output_length = len(output) if output_is_tuple_or_list else len(str(output))
                                        if output_is_tuple_or_list and output_length > 0:
                                            output_string = str(output[0]).lower().strip()
                                            if not self.isURLAddress(file_path=output_string): is_list_of_tokens = True
                                    if output_length > 0 and not is_list_of_tokens: break
                                else: break
                            if output_length > 1 and is_list_of_tokens: url_address_or_text = ''.join(output)
                            elif output_length > 0: url_address_or_text = list(output)[0] if output_is_tuple_or_list else str(output).strip()
                elif response is not None and self.__show_errors: print(f'ERROR {response.status_code} in getResponseFromReplicateWithModelRoute: {response.text}')
                return url_address_or_text
            def getResponseFromReplicate(data={}):
                url_address_or_text = ''
                try: from requests import post, get
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import post, get
                from json import dumps
                ROUTE = 'https://api.replicate.com/v1/predictions'
                headers = {'Authorization': 'Token '+api_key, 'Content-Type': 'application/json'}
                response = post(ROUTE, data=dumps(data), headers=headers, timeout=self.__timeout)
                if response.status_code in (200, 201):
                    response_data = response.json()
                    status = str(response_data['status']).lower().strip() if 'status' in response_data else ''
                    if status == 'starting':
                        if 'urls' in response_data:
                            urls = response_data['urls']
                            if 'get' in urls:
                                url = str(urls['get']).strip()
                                response = get(url, headers=headers, timeout=self.__timeout)
                                if response.status_code in (200, 201):
                                    response_data = response.json()
                                    status, output = str(response_data['status']).lower().strip() if 'status' in response_data else '', []
                                    is_list_of_tokens, output_is_tuple_or_list, output_length = False, False, 0
                                    while status in ('starting', 'processing'):
                                        response = get(url, headers=headers, timeout=self.__timeout)
                                        if response.status_code in (200, 201):
                                            response_data = response.json()
                                            status = str(response_data['status']).lower().strip() if 'status' in response_data else ''
                                            if 'output' in response_data:
                                                if not response_data['output'] is None: output = response_data['output']
                                                output_is_tuple_or_list = type(output) in (tuple, list)
                                                output_length = len(output) if output_is_tuple_or_list else len(str(output))
                                                if output_is_tuple_or_list and output_length > 0:
                                                    output_string = str(output[0]).lower().strip()
                                                    if not self.isURLAddress(file_path=output_string): is_list_of_tokens = True
                                            if output_length > 0 and not is_list_of_tokens: break
                                        else: break
                                    if output_length > 1 and is_list_of_tokens: url_address_or_text = ''.join(output)
                                    elif output_length > 0: url_address_or_text = list(output)[0] if output_is_tuple_or_list else str(output).strip()
                elif response is not None and self.__show_errors: print(f'ERROR {response.status_code} in getResponseFromReplicate: {response.text}')
                return url_address_or_text
            def getResponseFromDeepinfraLLMAPI(messages=[]):
                generated_text, total_cost = '', 0
                def itsTemplate():
                    pairs = (
                        ('\n\nHuman:', '\n\nAssistant:', '\n\n', '\n\nSystem: [SYSTEM]'),
                        ('<|start_header_id|>', '<|end_header_id|>', '<|eot_id|>', '<|begin_of_text|><|start_header_id|>system<|end_header_id|>\n\n[SYSTEM]<|eot_id|>'),
                        ('[INST]', '[/INST]', '</s>', '[INST][SYSTEM][/INST]'),
                        ('<start_of_turn>', '<end_of_turn>', '<end_of_turn>', '<start_of_turn>user\n(Instructions on how you should respond) [SYSTEM]<end_of_turn>\n'),
                        ('<|user|>', '<|assistant|>', '<|end|>', '<|system|>\n[SYSTEM]<|end|>\n'),
                        ('<|im_start|>', '<|im_end|>', '<|im_end|>', '<|im_start|>system\n[SYSTEM]<|im_end|>\n'),
                        ('User: ', 'Assistant: ', '\n', '[SYSTEM]\n'),
                        ('User: ', 'Falcon: ', '\n', 'System: [SYSTEM]\n')
                    )
                    for pair in pairs:
                        if pair[0] in prompt and pair[1] in prompt: return True, pair[2], pair[3]
                    return False, '\n', 'System: [SYSTEM]\n'
                its_template, stop, system_template = itsTemplate()
                try: from requests import post
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import post
                response = None
                if its_template:
                    ROTE = 'https://api.deepinfra.com/v1/inference/'+model_route
                    template = system_template.replace('[SYSTEM]', system_instruction)+prompt if len(system_instruction) > 0 else prompt
                    data = {'input': template, 'stop': [stop]}
                    headers = {'Content-Type': 'application/json', 'Authorization': 'Bearer '+api_key}
                    response = post(ROTE, json=data, headers=headers, timeout=self.__timeout)
                    response_data = response.json()
                    if 'results' in response_data:
                        results = response_data['results']
                        if type(results) in (tuple, list) and len(results) > 0:
                            results = results[0]
                            generated_text = str(results['generated_text']).strip() if 'generated_text' in results else ''
                    if 'inference_status' in response_data:
                        inference_status = response_data['inference_status']
                        total_cost += float(inference_status['cost']) if 'cost' in inference_status and type(inference_status['cost']) in (int, float) else 0
                else:
                    if len(system_instruction) > 0: messages = [{'role': 'system', 'content': system_instruction}]+messages
                    if len(prompt) > 0: messages.append({'role': 'user', 'content': prompt})
                    ROTE = 'https://api.deepinfra.com/v1/openai/chat/completions'
                    data = {'model': model_route, 'messages': messages}
                    headers = {'Content-Type': 'application/json', 'Authorization': 'Bearer '+api_key}
                    response = post(ROTE, json=data, headers=headers, timeout=self.__timeout)
                    response_data = response.json()
                    if 'choices' in response_data:
                        choices = response_data['choices']
                        if type(choices) in (tuple, list) and len(choices) > 0:
                            choices = choices[0]
                            if 'message' in choices:
                                message = choices['message']
                                generated_text = str(message['content']).strip() if 'content' in message else ''
                    if 'usage' in response_data:
                        usage = response_data['usage']
                        total_cost += float(usage['estimated_cost']) if 'estimated_cost' in usage and type(usage['estimated_cost']) in (int, float) else 0
                if len(str(generated_text).strip()) < 1 and response is not None and self.__show_errors: print(f'ERROR {response.status_code} in getResponseFromDeepinfraLLMAPI: {response.text}')
                return generated_text, total_cost
            def getResponseFromDeepinfraImageGenerationAPI():
                image_data, total_cost, _type = '', 0, ''
                try: from requests import post
                except:
                    self.installModule(module_name='requests', version='2.31.0')
                    from requests import post
                formatted_prompt = self.formatPrompt(prompt=prompt, task_names=task_names, language=language)    
                if len(image_paths) > 0 or url_path_lenght > 0:
                    url = 'https://api.deepinfra.com/v1/openai/images/edits'
                    headers = {'Authorization': 'Bearer '+api_key}
                    image_path = image_paths[0] if len(image_paths) > 0 else url_path
                    files = {'image': open(image_path, 'rb')}
                    data = {'n': '1', 'size': '1024x1024', 'model': model_route, 'prompt': formatted_prompt}
                    response = post(url, headers=headers, files=files, data=data, timeout=self.__timeout)
                else:
                    ROTE = 'https://api.deepinfra.com/v1/inference/'+model_route
                    sdxl, custom_diffusion = 'sdxl' in model_route, 'custom-diffusion' in model_route
                    if not sdxl and not custom_diffusion:
                        data = {'prompt': formatted_prompt, 'width': width, 'height': height, 'seed': seed}
                        headers = {'Content-Type': 'application/json', 'Authorization': 'Bearer '+api_key}
                    elif not sdxl and custom_diffusion:
                        models = (
                            'AnythingV5_v5PrtRE.safetensors',
                            'pastelmix-better-vae-fp16.safetensors',
                            'meinamix_meinaV11.safetensors',
                            'cetusMix_Whalefall2.safetensors',
                            'CounterfeitV30_v30.safetensors',
                            'abyssorangemix3AOM3_aom3a1b.safetensors'
                        )
                        model = self.__shuffleTuple(original_tuple=models)[0]
                        data = {'input': {'model': model, 'prompt': formatted_prompt, 'width': width, 'height': height, 'seed': seed}}
                        headers = {'Content-Type': 'application/json', 'Authorization': 'Bearer '+api_key}
                    else:
                        data = {'input': {'prompt': formatted_prompt, 'width': width, 'height': height, 'seed': seed}}
                        headers = {'Content-Type': 'application/json', 'Authorization': 'Bearer '+api_key}
                    response = post(ROTE, json=data, headers=headers, timeout=self.__timeout)
                response_data = response.json()
                if 'output' in response_data:
                    output = response_data['output']
                    image_data = output[0] if type(output) in (tuple, list) and len(output) > 0 else ''
                elif 'images' in response_data:
                    images = response_data['images']
                    image_data = images[0] if type(images) in (tuple, list) and len(images) > 0 else ''
                elif 'image_url' in response_data:
                    images = response_data['image_url']
                    if type(images) == str: image_data = images.strip()
                    elif type(images) in (tuple, list) and len(images) > 0: image_data = images[0]
                    else: image_data = ''
                elif 'data' in response_data:
                    data = response_data['data']
                    if type(data) in (tuple, list): data = data[0]
                    if 'b64_json' in data: image_data, _type = data['b64_json'], 'png'
                if 'inference_status' in response_data:
                    inference_status = response_data['inference_status']
                    total_cost += float(inference_status['cost']) if 'cost' in inference_status and type(inference_status['cost']) in (int, float) else 0
                if total_cost <= 0: total_cost = 0.02
                if len(str(image_data).strip()) < 1 and response is not None and self.__show_errors: print(f'ERROR {response.status_code} in getResponseFromDeepinfraImageGenerationAPI: {response.text}')
                return image_data.strip(), total_cost, _type
            def getResponseFromReplicateImageGenerationWithModelRoute(prompt='', cost_per_execution=0):
                image_address, total_cost = '', 0
                ROUTE = f'https://api.replicate.com/v1/models/{model_route}/predictions'
                data = {'input': {'prompt': prompt, 'aspect_ratio': aspect_ratio, 'output_format': 'png', 'seed': seed}}
                if use_time: start_time = getTime()
                image_address = getResponseFromReplicateWithModelRoute(ROUTE=ROUTE, data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else:
                    if cost_per_execution <= 0 and model_name.find('stable-diffusion-3') >= 0: cost_per_execution += 0.035
                    elif cost_per_execution <= 0 and model_name.find('flux-schnell') >= 0: cost_per_execution += 0.003
                    elif cost_per_execution <= 0 and model_name.find('flux-dev') >= 0: cost_per_execution += 0.030
                    elif cost_per_execution <= 0 and model_name.find('flux-pro') >= 0: cost_per_execution += 0.055
                    total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.04
                return image_address.strip(), total_cost
            def getResponseFromReplicateImageGenerationAPI(prompt='', cost_per_execution=0):
                prompt_language = self.getLanguage(string=prompt).lower()
                if not prompt_language.startswith('en'): prompt = self.translate(string=prompt, source_language='auto', target_language='en')
                if len(model_route) > 0: return getResponseFromReplicateImageGenerationWithModelRoute(prompt=prompt, cost_per_execution=cost_per_execution)
                image_address, total_cost = '', 0
                data = {'version': version, 'input': {'prompt': prompt, 'width': width, 'height': height, 'seed': seed}}
                if use_time: start_time = getTime()
                image_address = getResponseFromReplicate(data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.04
                return image_address.strip(), total_cost
            def getResponseFromReplicateLogoGenerationWithModelRoute(_input={}):
                logo_address, total_cost = '', 0
                ROUTE = f'https://api.replicate.com/v1/models/{model_route}/predictions'
                data = {'input': _input}
                if use_time: start_time = getTime()
                logo_address = getResponseFromReplicateWithModelRoute(ROUTE=ROUTE, data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.04
                return logo_address.strip(), total_cost
            def getResponseFromReplicateLogoGenerationAPI(prompt=''):
                prompt_language = self.getLanguage(string=prompt).lower()
                if not prompt_language.startswith('en'): prompt = self.translate(string=prompt, source_language='auto', target_language='en')
                _input = {'prompt': prompt, 'width': width, 'height': height, 'seed': seed, 'apply_watermark': False}
                if len(model_route) > 0: return getResponseFromReplicateLogoGenerationWithModelRoute(_input=_input)
                logo_address, total_cost = '', 0
                data = {'version': version, 'input': _input}
                if use_time: start_time = getTime()
                logo_address = getResponseFromReplicate(data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.04
                return logo_address.strip(), total_cost
            def getResponseFromReplicateRemoveBackgroundWithModelRoute(_input={}):
                image_address, total_cost = '', 0
                ROUTE = f'https://api.replicate.com/v1/models/{model_route}/predictions'
                data = {'input': _input}
                if use_time: start_time = getTime()
                image_address = getResponseFromReplicateWithModelRoute(ROUTE=ROUTE, data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.04
                return image_address.strip(), total_cost
            def getResponseFromReplicateRemoveBackgroundAPI():
                _input = {'image': web_address}
                if len(model_route) > 0: return getResponseFromReplicateRemoveBackgroundWithModelRoute(_input=_input)
                image_address, total_cost = '', 0
                data = {'version': version, 'input': _input}
                if use_time: start_time = getTime()
                image_address = getResponseFromReplicate(data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.04
                return image_address.strip(), total_cost
            def getResponseFromReplicateApplyStyleWithModelRoute(_input={}):
                image_address, total_cost = '', 0
                ROUTE = f'https://api.replicate.com/v1/models/{model_route}/predictions'
                data = {'input': _input}
                if use_time: start_time = getTime()
                image_address = getResponseFromReplicateWithModelRoute(ROUTE=ROUTE, data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.04
                return image_address.strip(), total_cost
            def getResponseFromReplicateApplyStyleAPI():
                _input = {
                    'subject_image': web_address,
                    'prompt': self.formatPrompt(prompt=prompt, task_names=['IMAGE_FILTER_APPLICATION'], language=language),
                    'width': width,
                    'height': height,
                    'light_source': 'Left Light',
                    'seed': seed,
                    'output_format': 'png',
                    'output_quality': 100
                }
                if len(model_route) > 0: return getResponseFromReplicateApplyStyleWithModelRoute(_input=_input)
                image_address, total_cost = '', 0
                data = {'version': version, 'input': _input}
                if use_time: start_time = getTime()
                image_address = getResponseFromReplicate(data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.04
                return image_address.strip(), total_cost
            def getResponseFromReplicateUpScaleWithModelRoute(_input={}):
                image_address, total_cost = '', 0
                ROUTE = f'https://api.replicate.com/v1/models/{model_route}/predictions'
                data = {'input': _input}
                if use_time: start_time = getTime()
                image_address = getResponseFromReplicateWithModelRoute(ROUTE=ROUTE, data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.04
                return image_address.strip(), total_cost
            def getResponseFromReplicateUpScaleAPI():
                _input = {'image': web_address, 'scale': 10, 'face_enhance': False}
                if len(model_route) > 0: return getResponseFromReplicateUpScaleWithModelRoute(_input=_input)
                image_address, total_cost = '', 0
                data = {'version': version, 'input': _input}
                if use_time: start_time = getTime()
                image_address = getResponseFromReplicate(data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.04
                return image_address.strip(), total_cost
            def getResponseFromReplicateImageInterpretationWithModelRoute(_input={}):
                generated_text, total_cost = '', 0
                ROUTE = f'https://api.replicate.com/v1/models/{model_route}/predictions'
                data = {'input': _input}
                if use_time: start_time = getTime()
                generated_text = getResponseFromReplicateWithModelRoute(ROUTE=ROUTE, data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.04
                return generated_text.strip(), total_cost
            def getResponseFromReplicateImageInterpretationAPI():
                _input = {'image': web_address, 'prompt': prompt, 'temperature': creativity, 'max_tokens': 200000}
                is_image_address = self.isImageAddress(file_path=web_address)
                if 'moe-llava' in model_key: _input = {'input_image': web_address, 'input_text': prompt}
                elif 'video-llava' in model_key:
                    if is_image_address: _input = {'image_path': web_address, 'text_prompt': prompt}
                    else: _input = {'video_path': web_address, 'text_prompt': prompt}
                elif 'llava-next-video' in model_key and not is_image_address: _input = {'video': web_address, 'prompt': prompt, 'num_frames': 3, 'max_new_tokens': 200000}
                if len(model_route) > 0: return getResponseFromReplicateImageInterpretationWithModelRoute(_input=_input)
                generated_text, total_cost = '', 0
                if 'llava-next-video' in model_key and is_image_address: data = {'version': '80537f9eead1a5bfa72d5ac6ea6414379be41d4d4f6679fd776e9535d1eb58bb', 'input': _input}
                else: data = {'version': version, 'input': _input}
                if use_time: start_time = getTime()
                generated_text = getResponseFromReplicate(data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.04
                return generated_text.strip(), total_cost
            def getResponseFromReplicateMusicGenerationWithModelRoute(_input={}):
                music_address, total_cost = '', 0
                ROUTE = f'https://api.replicate.com/v1/models/{model_route}/predictions'
                data = {'input': _input}
                if use_time: start_time = getTime()
                music_address = getResponseFromReplicateWithModelRoute(ROUTE=ROUTE, data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.087*duration_in_seconds
                return music_address.strip(), total_cost
            def getResponseFromReplicateMusicGenerationAPI(prompt=''):
                prompt_language = self.getLanguage(string=prompt).lower()
                if not prompt_language.startswith('en'): prompt = self.translate(string=prompt, source_language='auto', target_language='en')
                _input = {'prompt': prompt, 'duration': duration_in_seconds, 'output_format': audio_format, 'seed': seed}
                if len(model_route) > 0: return getResponseFromReplicateMusicGenerationWithModelRoute(_input=_input)
                music_address, total_cost = '', 0
                data = {'version': version, 'input': _input}
                if use_time: start_time = getTime()
                music_address = getResponseFromReplicate(data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.087*duration_in_seconds
                return music_address.strip(), total_cost
            def getResponseFromReplicateVideoGenerationWithModelRoute(_input={}):
                video_address, total_cost = '', 0
                ROUTE = f'https://api.replicate.com/v1/models/{model_route}/predictions'
                data = {'input': _input}
                if use_time: start_time = getTime()
                video_address = getResponseFromReplicateWithModelRoute(ROUTE=ROUTE, data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.40*duration_in_seconds
                return video_address.strip(), total_cost
            def getResponseFromReplicateVideoGenerationAPI(prompt=''):
                prompt_language = self.getLanguage(string=prompt).lower()
                if not prompt_language.startswith('en'): prompt = self.translate(string=prompt, source_language='auto', target_language='en')
                _input = {
                    'seed': seed,
                    'width': width,
                    'frames': int(round(4.26*duration_in_seconds)),
                    'height': height,
                    'prompt_map': '',
                    'head_prompt': prompt,
                    'prompt': prompt,
                    'tail_prompt': '',
                    'negative_prompt': 'ugly, broken',
                    'playback_frames_per_second': 32,
                    'enable_qr_code_monster_v2': False,
                    'qr_code_monster_v2_preprocessor': False,
                    'qr_code_monster_v2_guess_mode': False,
                    'controlnet_conditioning_scale': 0.25
                }
                if len(model_route) > 0: return getResponseFromReplicateVideoGenerationWithModelRoute(_input=_input)
                video_address, total_cost = '', 0
                data = {'version': version, 'input': _input}
                if use_time: start_time = getTime()
                video_address = getResponseFromReplicate(data=data)
                if use_time: total_cost += getCostPerExecution(start_time=start_time, end_time=getTime(), cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                else: total_cost += cost_per_execution
                if total_cost <= 0: total_cost = 0.40*duration_in_seconds
                return video_address.strip(), total_cost
            def getResponseFromReplicateFilesAPI(model_key='', prompt=''):
                file_data, total_response_cost = '', 0
                formatted_prompt = self.formatPrompt(prompt=prompt, task_names=task_names, language=language)
                if '-stable-diffusion' in model_key or '-sd' in model_key or 'flux-' in model_key or 'IMAGE_CREATION' in task_names: file_data, total_response_cost = getResponseFromReplicateImageGenerationAPI(prompt=formatted_prompt, cost_per_execution=cost_per_execution)
                elif '-logoai' in model_key or 'LOGO_CREATION' in task_names: file_data, total_response_cost = getResponseFromReplicateLogoGenerationAPI(prompt=formatted_prompt)
                elif '-remove-bg' in model_key or 'NO_BACKGROUND' in task_names: file_data, total_response_cost = getResponseFromReplicateRemoveBackgroundAPI()
                elif '-ic-light' in model_key or 'IMAGE_FILTER_APPLICATION' in task_names: file_data, total_response_cost = getResponseFromReplicateApplyStyleAPI()
                elif '-real-esrgan' in model_key or 'UPSCALE_IMAGE' in task_names: file_data, total_response_cost = getResponseFromReplicateUpScaleAPI()
                elif '-llava' in model_key: file_data, total_response_cost = getResponseFromReplicateImageInterpretationAPI()
                elif '-musicgen' in model_key or 'MUSIC_CREATION' in task_names: file_data, total_response_cost = getResponseFromReplicateMusicGenerationAPI(prompt=formatted_prompt)
                elif '-animatediff-illusions' in model_key or 'VIDEO_CREATION' in task_names: file_data, total_response_cost = getResponseFromReplicateVideoGenerationAPI(prompt=formatted_prompt)
                return file_data, total_response_cost
            def hasKeys(prompt='', keys=[]):
                result = True
                for key in keys:
                    if key not in prompt:
                        result = False
                        break
                return result
            def getTemplate(pattern=''):
                template = ''
                pattern = str(pattern).lower().strip()
                if pattern == 'llama3':
                    template_prompt = hasKeys(prompt=prompt, keys=['<|begin_of_text|>', '<|start_header_id|>', '<|end_header_id|>', '<|eot_id|>'])
                    template = prompt if template_prompt else ''
                    if len(system_instruction) > 0: template = f'<|begin_of_text|><|start_header_id|>system<|end_header_id|>\n\n{system_instruction}<|eot_id|>{template}'
                    if len(messages) > 0:
                        for message in messages:
                            role = str(message['role']).lower().strip() if 'role' in message else 'user'
                            content = str(message['content']).strip() if 'content' in message else ''
                            if role == 'user' and len(content) > 0: template += f'<|start_header_id|>user<|end_header_id|>\n\n{content}<|eot_id|>'
                            elif role == 'assistant' and len(content) > 0: template += f'<|start_header_id|>assistant<|end_header_id|>\n\n{content}<|eot_id|>'
                            elif role == 'system' and len(content) > 0: template = f'<|begin_of_text|><|start_header_id|>system<|end_header_id|>\n\n{content}<|eot_id|>'+template
                    if len(prompt) > 0 and not template_prompt: template += f'<|start_header_id|>user<|end_header_id|>\n\n{prompt}<|eot_id|><|start_header_id|>assistant<|end_header_id|>\n\n'
                elif pattern == 'mistral':
                    template_prompt = hasKeys(prompt=prompt, keys=['[INST]', '[/INST]'])
                    template = prompt if template_prompt else ''
                    if len(system_instruction) > 0: template = f'[INST]{system_instruction}[/INST]{template}'
                    if len(messages) > 0:
                        for message in messages:
                            role = str(message['role']).lower().strip() if 'role' in message else 'user'
                            content = str(message['content']).strip() if 'content' in message else ''
                            if role == 'user' and len(content) > 0: template += '<s>[INST]'+content
                            elif role == 'assistant' and len(content) > 0: template += f'[/INST]{content}</s>'
                            elif role == 'system' and len(content) > 0: template = f'[INST]{content}[/INST]'+template
                    if len(prompt) > 0 and not template_prompt: template += f'<s>[INST]{prompt}[/INST]'
                    if not template.endswith('[/INST]'): template += '[/INST]'
                elif pattern == 'gemma':
                    template_prompt = hasKeys(prompt=prompt, keys=['<start_of_turn>', '<end_of_turn>'])
                    response_instructions = '(Instructions on how you should respond) '
                    template = prompt if template_prompt else ''
                    if len(system_instruction) > 0: template = f'<start_of_turn>user\n{response_instructions}{system_instruction}<end_of_turn>\n{template}'
                    if len(messages) > 0:
                        for message in messages:
                            role = str(message['role']).lower().strip() if 'role' in message else 'user'
                            content = str(message['content']).strip() if 'content' in message else ''
                            if role == 'user' and len(content) > 0: template += f'<start_of_turn>user\n{content}<end_of_turn>\n'
                            elif role == 'assistant' and len(content) > 0: template += f'<start_of_turn>model\n{content}<end_of_turn>\n'
                            elif role == 'system' and len(content) > 0: template = f'<start_of_turn>user\n{response_instructions}{content}<end_of_turn>\n'+template
                    if len(prompt) > 0 and not template_prompt: template += f'<start_of_turn>user\n{prompt}<end_of_turn>\n<start_of_turn>model'
                    if not template.endswith('<start_of_turn>model'): template += '<start_of_turn>model'
                elif pattern == 'phi3':
                    template_prompt = hasKeys(prompt=prompt, keys=['<|user|>', '<|assistant|>', '<|end|>'])
                    template = prompt if template_prompt else ''
                    if len(system_instruction) > 0: template = f'<|system|>\n{system_instruction}<|end|>\n{template}'
                    if len(messages) > 0:
                        for message in messages:
                            role = str(message['role']).lower().strip() if 'role' in message else 'user'
                            content = str(message['content']).strip() if 'content' in message else ''
                            if role == 'user' and len(content) > 0: template += f'<|user|>\n{content}<|end|>\n'
                            elif role == 'assistant' and len(content) > 0: template += f'<|assistant|>\n{content}<|end|>\n'
                            elif role == 'system' and len(content) > 0: template = f'<|system|>\n{content}<|end|>\n'+template
                    if len(prompt) > 0 and not template_prompt: template += f'<|user|>\n{prompt}<|end|>\n<|assistant|>'
                    if not template.endswith('<|assistant|>'): template += '<|assistant|>'
                elif pattern in 'yi-stablelm2':
                    template_prompt = hasKeys(prompt=prompt, keys=['<|im_start|>', '<|im_end|>'])
                    template = prompt if template_prompt else ''
                    if len(system_instruction) > 0: template = f'<|im_start|>system\n{system_instruction}<|im_end|>\n{template}'
                    if len(messages) > 0:
                        for message in messages:
                            role = str(message['role']).lower().strip() if 'role' in message else 'user'
                            content = str(message['content']).strip() if 'content' in message else ''
                            if role == 'user' and len(content) > 0: template += f'<|im_start|>user\n{content}<|im_end|>\n'
                            elif role == 'assistant' and len(content) > 0: template += f'<|im_start|>assistant\n{content}<|im_end|>\n'
                            elif role == 'system' and len(content) > 0: template = f'<|im_start|>system\n{content}<|im_end|>\n'+template
                    if len(prompt) > 0 and not template_prompt: template += f'<|im_start|>user\n{prompt}<|im_end|>\n<|im_start|>assistant'
                    if not template.endswith('<|im_start|>assistant'): template += '<|im_start|>assistant'
                elif pattern == 'falcon':
                    template_prompt = hasKeys(prompt=prompt, keys=['User: ', 'Assistant: '])
                    template = prompt if template_prompt else ''
                    if len(system_instruction) > 0: template = system_instruction+'\n'+template
                    if len(messages) > 0:
                        for message in messages:
                            role = str(message['role']).lower().strip() if 'role' in message else 'user'
                            content = str(message['content']).strip() if 'content' in message else ''
                            if role == 'user' and len(content) > 0: template += f'User: {content}\n'
                            elif role == 'assistant' and len(content) > 0: template += f'Assistant: {content}\n'
                            elif role == 'system' and len(content) > 0: template = content+'\n'+template
                    if len(prompt) > 0 and not template_prompt: template += f'User: {prompt}\nAssistant:'
                    if not template.endswith('Assistant:'): template += 'Assistant:'
                elif pattern == 'falcon2':
                    template_prompt = hasKeys(prompt=prompt, keys=['User: ', 'Falcon: '])
                    template = prompt if template_prompt else ''
                    if len(system_instruction) > 0: template = f'User: {system_instruction}\n{template}'
                    if len(messages) > 0:
                        for message in messages:
                            role = str(message['role']).lower().strip() if 'role' in message else 'user'
                            content = str(message['content']).strip() if 'content' in message else ''
                            if role == 'user' and len(content) > 0: template += f'User: {content}\n'
                            elif role == 'assistant' and len(content) > 0: template += f'Falcon: {content}\n'
                            elif role == 'system' and len(content) > 0: template = f'User: {content}\n'+template
                    if len(prompt) > 0 and not template_prompt: template += f'User: {prompt}\nFalcon:'
                    if not template.endswith('Falcon:'): template += 'Falcon:'
                elif pattern == 'ollama':
                    template = str(prompt).strip()
                    if messages:
                        template = ''
                        for message in messages:
                            role = str(message.get('role', '')).lower().strip()
                            content = str(message.get('content', '')).lower().strip()
                            template += f'{role}:\n{content}\n'
                        template += 'assistant:\n'
                    template = template.strip()
                return template
            def getResponseFromLlama3():
                generated_text, total_cost = '', 0
                model_server = model_name.split('-', 1)
                server_name, _model_name = model_server[0].lower().strip(), model_server[1].strip()
                template = getTemplate(pattern='llama3')
                if server_name == 'ollama': generated_text, total_cost = getResponseFromOllamaLLM(model_name=_model_name, template=template)
                elif server_name == 'replicate': generated_text, total_cost = getResponseFromReplicateLLM(template=template, cost_per_token=cost_per_token, cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                return generated_text, total_cost
            def getResponseFromMistral():
                generated_text, total_cost = '', 0
                model_server = model_name.split('-', 1)
                server_name, _model_name = model_server[0].lower().strip(), model_server[1].strip()
                template = getTemplate(pattern='mistral')
                if server_name == 'ollama': generated_text, total_cost = getResponseFromOllamaLLM(model_name=_model_name, template=template)
                elif server_name == 'replicate': generated_text, total_cost = getResponseFromReplicateLLM(template=template, cost_per_token=cost_per_token, cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                if '[INST]' in generated_text: generated_text = generated_text.replace('[INST]', ' ')
                if '[/INST]' in generated_text: generated_text = generated_text.split('[/INST]')[0]
                return generated_text, total_cost
            def getResponseFromGemma():
                generated_text, total_cost = '', 0
                model_server = model_name.split('-', 1)
                server_name, _model_name = model_server[0].lower().strip(), model_server[1].strip()
                template = getTemplate(pattern='gemma')
                if server_name == 'ollama': generated_text, total_cost = getResponseFromOllamaLLM(model_name=_model_name, template=template)
                elif server_name == 'replicate': generated_text, total_cost = getResponseFromReplicateLLM(template=template, cost_per_token=cost_per_token, cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                return generated_text, total_cost
            def getResponseFromPhi3():
                generated_text, total_cost = '', 0
                model_server = model_name.split('-', 1)
                server_name, _model_name = model_server[0].lower().strip(), model_server[1].strip()
                template = getTemplate(pattern='phi3')
                if server_name == 'ollama': generated_text, total_cost = getResponseFromOllamaLLM(model_name=_model_name, template=template)
                elif server_name == 'replicate': generated_text, total_cost = getResponseFromReplicateLLM(template=template, cost_per_token=cost_per_token, cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                return generated_text, total_cost
            def getTemplateYiAndStableLM2(): return getTemplate(pattern='yi-stablelm2')
            def getResponseFromYi():
                generated_text, total_cost = '', 0
                model_server = model_name.split('-', 1)
                server_name, _model_name = model_server[0].lower().strip(), model_server[1].strip()
                template = getTemplateYiAndStableLM2()
                if server_name == 'ollama': generated_text, total_cost = getResponseFromOllamaLLM(model_name=_model_name, template=template)
                elif server_name == 'replicate': generated_text, total_cost = getResponseFromReplicateLLM(template=template, cost_per_token=cost_per_token, cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                return generated_text, total_cost
            def getResponseFromFalcon():
                generated_text, total_cost = '', 0
                model_server = model_name.split('-', 1)
                server_name, _model_name = model_server[0].lower().strip(), model_server[1].strip()
                template = getTemplate(pattern='falcon')
                if server_name == 'ollama': generated_text, total_cost = getResponseFromOllamaLLM(model_name=_model_name, template=template)
                elif server_name == 'replicate': generated_text, total_cost = getResponseFromReplicateLLM(template=template, cost_per_token=cost_per_token, cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                if generated_text.strip().endswith('Assistant:'): generated_text = ''.join(generated_text.rsplit('Assistant:', 1))
                if 'Assistant: ' in generated_text: generated_text = generated_text.split('Assistant: ')[-1]
                return generated_text, total_cost
            def getResponseFromFalcon2():
                generated_text, total_cost = '', 0
                model_server = model_name.split('-', 1)
                server_name, _model_name = model_server[0].lower().strip(), model_server[1].strip()
                template = getTemplate(pattern='falcon2')
                if server_name == 'ollama': generated_text, total_cost = getResponseFromOllamaLLM(model_name=_model_name, template=template)
                elif server_name == 'replicate': generated_text, total_cost = getResponseFromReplicateLLM(template=template, cost_per_token=cost_per_token, cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                if generated_text.strip().endswith('Falcon:'): generated_text = ''.join(generated_text.rsplit('Falcon:', 1))
                if 'Falcon: ' in generated_text: generated_text = generated_text.split('Falcon: ')[-1]
                return generated_text, total_cost
            def getResponseFromStableLM2():
                generated_text, total_cost = '', 0
                model_server = model_name.split('-', 1)
                server_name, _model_name = model_server[0].lower().strip(), model_server[1].strip()
                template = getTemplateYiAndStableLM2()
                if server_name == 'ollama': generated_text, total_cost = getResponseFromOllamaLLM(model_name=_model_name, template=template)
                elif server_name == 'replicate': generated_text, total_cost = getResponseFromReplicateLLM(template=template, cost_per_token=cost_per_token, cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
                return generated_text, total_cost
            def getResponseFromLlava():
                generated_text, total_cost = '', 0
                model_server = model_name.split('-', 1)
                server_name, _model_name = model_server[0].lower().strip(), model_server[1].strip()
                if 'llava-llama3' in _model_name: template = getTemplate(pattern='llama3')
                elif 'bakllava' in _model_name: template = getTemplate(pattern='falcon').replace('User:', 'USER:').replace('Assistant:', 'ASSISTANT:')
                elif 'llava-phi3' in _model_name: template = getTemplate(pattern='phi3')
                else: template = getTemplate(pattern='mistral')
                if server_name == 'ollama': generated_text, total_cost = getResponseFromOllamaLLM(model_name=_model_name, template=template, image_paths=image_paths)
                elif server_name == 'replicate': generated_text, total_cost = getResponseFromReplicateFilesAPI(model_key=model_key, prompt=prompt)
                return generated_text, total_cost
            def getResponseFromOllama():
                generated_text, total_cost = '', 0
                model_server = model_name.split('-', 1)
                server_name, _model_name = model_server[0].lower().strip(), model_server[1].strip()
                template = getTemplate(pattern='ollama')
                generated_text, total_cost = getResponseFromOllamaLLM(model_name=_model_name, template=template, image_paths=image_paths)
                return generated_text, total_cost
            youtube_download = False
            if self.hasCoincidentElements(vector1=task_names, vector2=['YOUTUBE_VIDEO_DOWNLOAD', 'YOUTUBE_AUDIO_DOWNLOAD']):
                if url_path_lenght < 1:
                    links = self.getLinks(string=original_prompt, check_existence=False)
                    for link in links:
                        if self.isYouTubeURL(url_path=link):
                            url_path = link
                            youtube_download = True
                            break
                elif self.isYouTubeURL(url_path=url_path): youtube_download = True
            system_customization, _type = self.__getSystemCustomization(system_instruction=system_instruction, prompt=prompt, language=language, max_tokens=max_tokens, url_path=url_path, javascript_chart=javascript_chart, task_names=task_names), ''
            _system_instruction = str(system_customization.get('system_instruction', system_instruction)).strip()
            _file_directory = str(system_customization.get('file_directory', '')).strip()
            _created_local_file = bool(system_customization.get('created_local_file', False))
            _extensions = list(system_customization.get('extensions', []))
            _prompt = str(system_customization.get('prompt', prompt)).strip()
            _result = dict(system_customization.get('result', {}))
            if _system_instruction: system_instruction = _system_instruction
            if _file_directory: file_directory = _file_directory
            if _created_local_file: created_local_file = _created_local_file
            if _extensions: extensions = _extensions
            if _prompt: prompt = _prompt
            if _result:
                _result_answer = str(_result.get('answer', '')).strip()
                _result_answer_for_files = str(_result.get('answer_for_files', '')).strip()
                _result_files = list(_result.get('files', []))
                _result_sources = list(_result.get('sources', []))
                _result_javascript_charts = list(_result.get('javascript_charts', []))
                _result_str_cost = str(_result.get('str_cost', '0.0000000000')).strip()
                if _result_answer: result['answer'] = _result_answer
                if _result_answer_for_files: result['answer_for_files'] = _result_answer_for_files
                if _result_files: result['files'] = _result_files
                if _result_sources: result['sources'] = _result_sources
                if _result_javascript_charts: result['javascript_charts'] = _result_javascript_charts
                if _result_str_cost and float(_result_str_cost) > 0: result['str_cost'] += _result_str_cost
            if not prompt: prompt = get_prompt(prompt=prompt, messages=messages)
            if result and youtube_download: return result
            if self.hasCoincidentElements(vector1=task_names, vector2=['VIDEO_INTERPRETATION']):
                youtube_interpretation = False
                if url_path_lenght < 1:
                    links = self.getLinks(string=original_prompt, check_existence=False)
                    for link in links:
                        if self.isYouTubeURL(url_path=link):
                            url_path = link
                            youtube_interpretation = True
                            break
                elif self.isYouTubeURL(url_path=url_path): youtube_interpretation = True
                if youtube_interpretation:
                    hour, minute, second = 0, 0, 0
                    def toInt(str_numer=''):
                        try: return int(float(str_numer))
                        except: return 0
                    for task_name in task_names:
                        if task_name.startswith('TIME_'):
                            times = str(task_name.split('TIME_')[-1]).strip().split('_')
                            if len(times) > 2: hour, minute, second = toInt(times[0]), toInt(times[1]), toInt(times[-1])
                    service, model = 'local', 'tiny'
                    if model_key.startswith('claude'): service, model = 'anthropic', model_name
                    elif model_key.startswith('gpt'): service, model = 'openai', model_name
                    elif model_key.startswith('gemini'): service, model = 'google', model_name
                    youtube_interpreter = self.youTubeInterpreter(url_path=url_path, service=service, model=model, api_key=api_key, max_tokens=max_tokens, language=language if language else None, hour=hour, minute=minute, second=second)['answer']
                    if (url_path in prompt or url_path in original_prompt) or (len(messages) > 0 and url_path in messages[-1]['content']):
                        prompt = prompt.replace(url_path, f'\n{youtube_interpreter}\n\n')
                        original_prompt = original_prompt.replace(url_path, f'\n{youtube_interpreter}\n\n')
                        if messages: messages[-1]['content'] = messages[-1]['content'].replace(url_path, f'\n{youtube_interpreter}\n\n')
                    else:
                        prompt = f'{youtube_interpreter}\n\n{prompt}'
                        original_prompt = f'{youtube_interpreter}\n\n{original_prompt}'
                        if messages: messages[-1]['content'] = f'{youtube_interpreter}\n\n{messages[-1]["content"]}'
            if self.hasCoincidentElements(vector1=task_names, vector2=['WEBPAGE_ACCESS']):
                list_of_links, webpage_access = [], False
                if url_path_lenght < 1:
                    links = self.getLinks(string=original_prompt, check_existence=True)
                    for link in links:
                        if not self.isYouTubeURL(url_path=link):
                            list_of_links.append(link)
                            webpage_access = True
                elif self.isURLAddress(file_path=url_path) and not self.isYouTubeURL(url_path=url_path):
                    list_of_links.append(url_path)
                    webpage_access = True
                if webpage_access:
                    for link in list_of_links:
                        content_from_web_links = self.getContentFromWEBLinks(list_of_links=[link], max_tokens=max_tokens)['text'].strip()
                        if (link in prompt or link in original_prompt) or (len(messages) > 0 and link in messages[-1]['content']):
                            prompt = prompt.replace(link, f'\n{content_from_web_links}\n\n')
                            original_prompt = original_prompt.replace(link, f'\n{content_from_web_links}\n\n')
                            if messages: messages[-1]['content'] = messages[-1]['content'].replace(link, f'\n{content_from_web_links}\n\n')
                        else:
                            prompt = f'{content_from_web_links}\n\n{prompt}'
                            original_prompt = f'{content_from_web_links}\n\n{original_prompt}'
                            if messages: messages[-1]['content'] = f'{content_from_web_links}\n\n{messages[-1]["content"]}'
            if model_key.startswith('claude-3'): generated_text, total_response_cost = getResponseFromClaude3API(system_instruction=system_instruction, messages=messages, image_paths=image_paths, cost_per_token=cost_per_token)
            elif model_key.startswith('claude-2'): generated_text, total_response_cost = getResponseFromClaude2API(prompt=prompt, messages=messages, cost_per_token=cost_per_token)
            elif model_key.startswith('claude'): generated_text, total_response_cost = getResponseFromClaude4API(system_instruction=system_instruction, messages=messages, image_paths=image_paths, cost_per_token=cost_per_token)
            elif model_key.startswith('gpt'): generated_text, total_response_cost = getResponseFromChatGPTAPI(messages=messages, image_paths=image_paths, cost_per_token=cost_per_token)
            elif model_key.startswith('dall-e'): file_data, total_response_cost = getResponseFromDALLEAPI(prompt=prompt, cost_per_execution=cost_per_execution)
            elif model_key.startswith('gemini'): generated_text, total_response_cost = getResponseFromGeminiAPI(system_instruction=system_instruction, messages=messages, image_paths=image_paths, cost_per_token=cost_per_token)
            elif model_key.startswith('deepinfra'):
                routes_for_imaging = (
                    'stability-ai/sdxl',
                    'compvis/stable-diffusion-v1-4',
                    'lykon/dreamshaper',
                    'xpuct/deliberate',
                    'prompthero/openjourney',
                    'runwayml/stable-diffusion-v1-5',
                    'stabilityai/stable-diffusion-2-1',
                    'uwulewd/custom-diffusion'
                )
                task_names_for_images = ('IMAGE_CREATION', 'IMAGE_EDITING', 'LOGO_CREATION', 'LOGO_EDITING', 'IMAGE_FILTER_APPLICATION')
                if (model_route.lower() in routes_for_imaging) or (len(task_names) > 0 and self.hasCoincidentElements(vector1=task_names, vector2=task_names_for_images)): file_data, total_response_cost, _type = getResponseFromDeepinfraImageGenerationAPI()
                else: generated_text, total_response_cost = getResponseFromDeepinfraLLMAPI(messages=messages)
            elif '-llama3' in model_key or '-llama-3' in model_key: generated_text, total_response_cost = getResponseFromLlama3()
            elif 'llava' in model_key: generated_text, total_response_cost = getResponseFromLlava()
            elif '-mistral' in model_key or '-mixtral' in model_key: generated_text, total_response_cost = getResponseFromMistral()
            elif '-gemma' in model_key: generated_text, total_response_cost = getResponseFromGemma()
            elif '-phi3' in model_key or '-phi-3' in model_key: generated_text, total_response_cost = getResponseFromPhi3()
            elif '-yi' in model_key: generated_text, total_response_cost = getResponseFromYi()
            elif '-falcon2' in model_key or '-falcon-2' in model_key: generated_text, total_response_cost = getResponseFromFalcon2()
            elif '-falcon' in model_key: generated_text, total_response_cost = getResponseFromFalcon()
            elif '-stablelm2' in model_key or '-stablelm-2' in model_key: generated_text, total_response_cost = getResponseFromStableLM2()
            elif model_key.startswith('replicate'):
                task_names_for_media = ('IMAGE_CREATION', 'IMAGE_EDITING', 'LOGO_CREATION', 'LOGO_EDITING', 'AUDIO_CREATION', 'AUDIO_EDITING', 'MUSIC_CREATION', 'MUSIC_EDITING', 'VIDEO_CREATION', 'VIDEO_EDITING', 'NO_BACKGROUND', 'IMAGE_FILTER_APPLICATION', 'UPSCALE_IMAGE')
                if len(task_names) > 0 and self.hasCoincidentElements(vector1=task_names, vector2=task_names_for_media): generated_text, total_response_cost = getResponseFromReplicateFilesAPI(model_key=model_key, prompt=prompt)
                else: generated_text, total_response_cost = getResponseFromReplicateLLM(template=prompt, cost_per_token=cost_per_token, cost_per_execution=cost_per_execution, cost_per_second=cost_per_second)
            elif model_key.startswith('ollama'): generated_text, total_response_cost = getResponseFromOllama()
            if len(web_address) > 0: self.deleteImageAddress(file_dictionary=temporary_dictionary_file)
            is_url_address, return_in_base64 = self.isURLAddress(file_path=generated_text), False
            if is_url_address: file_data, generated_text = generated_text, ''
            elif len(file_data) > 0: return_in_base64 = not self.isURLAddress(file_path=file_data)
            if political_character and '[CHARACTER]' in generated_text: generated_text = generated_text.split('[CHARACTER]')[-1]
            result['answer'] = generated_text.strip()
            if has_artifact or javascript_chart:
                html_code = self.getCodeList(string=generated_text, html=True)
                if 'plotly' in str('\n'.join(html_code)).lower().strip(): result['javascript_charts'] = html_code
                else: result['artifacts'] = html_code
            if created_local_file:
                from os import path, walk
                def getFileList(files_path='', extensions=[]):
                    file_list = []
                    files_path = str(files_path).strip()
                    extensions = list(extensions) if type(extensions) in (tuple, list) else []
                    extensions_length = len(extensions)
                    if path.isdir(files_path):
                        for root, _, files in walk(files_path):
                            for file in files:
                                if extensions_length > 0:
                                    if any(file.lower().endswith(extension.lower()) for extension in extensions): file_list.append(path.join(root, file))
                                else: file_list.append(path.join(root, file))
                    return file_list
                code_list = self.getCodeList(string=generated_text, html=False)
                for code in code_list:
                    execution_result = self.executePythonCode(string_code=code)
                    if len(execution_result.strip()) < 1:
                        file_list = getFileList(files_path=file_directory, extensions=extensions)
                        for temporary_path in file_list:
                            file_dictionary = self.fileToBase64(file_path=temporary_path)
                            result['files'].append(file_dictionary)
                    else:
                        correction_content = f'```python\n{code}\n```\n\n{execution_result}'
                        if model_key.startswith('claude-3'):
                            URL = 'https://api.anthropic.com/v1/messages'  
                            body = {'model': model_name, 'temperature': creativity, 'max_tokens': 4096, 'system': system_instruction, 'messages': [{'role': 'user', 'content': correction_content}]}
                            headers = {'x-api-key': api_key, 'anthropic-version': '2023-06-01', 'content-type': 'application/json'}
                        elif model_key.startswith('claude-2'):
                            URL = 'https://api.anthropic.com/v1/complete'
                            body = {'model': model_name, 'temperature': creativity, 'max_tokens_to_sample': 1000000, 'prompt': correction_content}
                            headers = {'x-api-key': api_key, 'anthropic-version': '2023-06-01', 'content-type': 'application/json'}
                        elif model_key.startswith('gpt'):
                            URL = 'https://api.openai.com/v1/chat/completions'
                            body = {'model': model_name, 'messages': [{'role': 'user', 'content': correction_content}], 'temperature': creativity}
                            headers = {'Content-Type': 'application/json', 'Authorization': 'Bearer '+api_key}
                        elif model_key.startswith('gemini'):
                            URL = f'https://generativelanguage.googleapis.com/v1beta/models/{model_name}:generateContent?key={api_key}'
                            body = {'contents': [{'role': 'user', 'parts': [{'text': correction_content}]}], 'generation_config': {'temperature': creativity}}
                            headers = {'Content-Type': 'application/json'}
                        elif model_key.startswith('replicate'):
                            URL = 'https://api.replicate.com/v1/predictions'
                            body = {'version': version, 'input': {'prompt': correction_content, 'temperature': creativity, 'max_new_tokens': 200000}}
                            headers = {'Authorization': 'Token '+api_key, 'Content-Type': 'application/json'}
                        elif model_key.startswith('deepinfra'):
                            URL = 'https://api.deepinfra.com/v1/openai/chat/completions'
                            body = {'model': model_route, 'messages': [{'role': 'user', 'content': correction_content}]}
                            headers = {'Content-Type': 'application/json', 'Authorization': 'Bearer '+api_key}
                        else:
                            URL = 'http://localhost:11434/api/generate'
                            body = {'model': model_name, 'prompt': correction_content, 'stream': False}, 
                            headers = {'Content-Type': 'application/json'}
                        main_result, cost_of_correction, _ = self.executeModel(URL=URL, body=body, headers=headers)
                        _code_list = self.getCodeList(string=main_result, html=False)
                        for _code in _code_list:
                            execution_result = self.executePythonCode(string_code=_code)
                            if len(execution_result.strip()) < 1:
                                file_list = getFileList(files_path=file_directory, extensions=extensions)
                                for temporary_path in file_list:
                                    file_dictionary = self.fileToBase64(file_path=temporary_path)
                                    result['files'].append(file_dictionary)
                        total_cost += cost_of_correction
                self.deleteDirectory(directory_path=file_directory)
                if file_directory in result['answer']: result['answer'] = result['answer'].replace(file_directory, '')
            if url_path_lenght > 0 and url_path in result['answer'] and '/' in url_path: result['answer'] = result['answer'].replace(url_path.replace(url_path.split('/')[-1], ''), '')
            if len(file_data) > 0 and not return_in_base64:
                file_dictionary = self.fileToBase64(file_path=file_data, media_metadata=media_metadata)
                result['files'].append(file_dictionary)
            elif len(file_data) > 0:
                def copyAfterBase64(string=''):
                    substring = 'base64,'
                    index = string.find(substring)
                    if index != -1: return string[index+len(substring):]
                    else: return string
                def copyUntilSemicolon(string=''):
                    index = string.find(';')
                    if index != -1: return string[:index]
                    else: return string
                base64_string, _type = copyAfterBase64(string=file_data), str(_type).strip() if _type else copyUntilSemicolon(string=file_data).split('/')[-1]
                file_dictionary = {'base64_string': base64_string, 'type': _type}
                if len(media_metadata) > 0: file_dictionary = self.updateMediaMetadata(file_path='', file_dictionary=file_dictionary, metadata=media_metadata)
                result['files'].append(file_dictionary)
            if 'answer' not in result: result['answer'] = ''
            if 'answer_for_files' not in result: result['answer_for_files'] = ''
            if 'files' not in result: result['files'] = []
            if 'sources' not in result: result['sources'] = []
            if 'javascript_charts' not in result: result['javascript_charts'] = []
            if 'artifacts' not in result: result['artifacts'] = []
            if 'next_token' not in result: result['next_token'] = ''
            if 'str_cost' not in result: result['str_cost'] = '0.0000000000'
            if len(result['files']) > 0 or len(result['artifacts']) > 0 or len(result['javascript_charts']) > 0: result['answer_for_files'] = self.getAnswerForFiles(prompt=original_prompt, language=language if len(language) > 0 else None)
            if len(result['answer']) < 1 and len(result['answer_for_files']) < 1 and 'TEXT_SUMMARY_WITH_BULLET_POINTS' in task_names: result['answer'] = self.summary(string=self.formatPrompt(prompt=original_prompt, task_names=['TEXT_SUMMARY_WITH_BULLET_POINTS'], language=language), topics=True)
            elif len(result['answer']) < 1 and len(result['answer_for_files']) < 1 and 'TEXT_SUMMARY' in task_names: result['answer'] = self.summary(string=self.formatPrompt(prompt=original_prompt, task_names=['TEXT_SUMMARY'], language=language), topics=False)
            total_cost += total_response_cost
            result['str_cost'] = f'{total_cost:.10f}'.strip()
            return result
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.__executeTask: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'answer': '', 'answer_for_files': '', 'files': [], 'sources': [], 'javascript_charts': [], 'artifacts': [], 'next_token': '', 'str_cost': '0.3200000000'}
    def executeSapiensTask(self, system_instruction='', prompt='', messages=[], task_names=[], config=None, stream=True):
        try:
            from time import time
            start_time = time()
            user_code = self.getHashCode()
            self.__total_number_of_tokens[user_code] = 0
            system_instruction = str(system_instruction).strip()
            prompt = str(prompt).strip()
            messages = list(messages) if type(messages) in (tuple, list) else []
            task_names = list(task_names) if type(task_names) in (tuple, list) else []
            config = config if config is not None and type(config) == dict else None
            stream = bool(stream) if type(stream) in (bool, int, float) else True
            cost_per_execution = 0.0
            cost_per_token = 0.0
            cost_per_million_tokens = 0.0
            cost_per_second = 0.0
            if config and type(config) == dict:
                cost_per_execution = float(config.get('cost_per_execution', 0.0))
                cost_per_token = float(config.get('cost_per_token', 0.0))
                cost_per_million_tokens = float(config.get('cost_per_million_tokens', 0.0))
                cost_per_second = float(config.get('cost_per_second', 0.0))
            if stream:
                def get_generator():
                    result_generator = self.__executeSapiensTask(system_instruction=system_instruction, prompt=prompt, messages=messages, task_names=task_names, config=config, user_code=user_code)
                    for dictionary in result_generator:
                        if cost_per_token > 0: dictionary['str_cost'] = f'{float(cost_per_token*self.__total_number_of_tokens[user_code]):.10f}'
                        elif cost_per_million_tokens > 0: dictionary['str_cost'] = f'{float((cost_per_million_tokens/1000000)*self.__total_number_of_tokens[user_code]):.10f}'
                        elif cost_per_second > 0: dictionary['str_cost'] = f'{float(abs(time()-start_time)*cost_per_second):.10f}'
                        elif cost_per_execution > 0: dictionary['str_cost'] = f'{cost_per_execution:.10f}'
                        yield dictionary
                    del self.__total_number_of_tokens[user_code]
                return get_generator()
            else:
                result_dictionary = {'answer': '', 'answer_for_files': '', 'files': [], 'sources': [], 'javascript_charts': [], 'artifacts': [], 'next_token': '', 'str_cost': '0.0000000000'}
                result_generator = self.__executeSapiensTask(system_instruction=system_instruction, prompt=prompt, messages=messages, task_names=task_names, config=config, user_code=user_code)
                for dictionary in result_generator:
                    if cost_per_token > 0: dictionary['str_cost'] = f'{float(cost_per_token*self.__total_number_of_tokens[user_code]):.10f}'
                    elif cost_per_million_tokens > 0: dictionary['str_cost'] = f'{float((cost_per_million_tokens/1000000)*self.__total_number_of_tokens[user_code]):.10f}'
                    result_dictionary = dictionary
                if cost_per_second > 0: result_dictionary['str_cost'] = f'{float(abs(time()-start_time)*cost_per_second):.10f}'
                elif cost_per_execution > 0: result_dictionary['str_cost'] = f'{cost_per_execution:.10f}'
                self.__total_number_of_tokens[user_code] = 0
                del self.__total_number_of_tokens[user_code]
                return result_dictionary
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.executeSapiensTask: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'answer': '', 'answer_for_files': '', 'files': [], 'sources': [], 'javascript_charts': [], 'artifacts': [], 'next_token': '', 'str_cost': '0.3200000000'}
    def executeTask(self, system_instruction='', prompt='', messages=[], task_names=[], config=None, stream=True):
        try:
            system_instruction = str(system_instruction).strip()
            prompt = str(prompt).strip()
            messages = list(messages) if type(messages) in (tuple, list) else []
            task_names = list(task_names) if type(task_names) in (tuple, list) else []
            config = config if config is not None and type(config) == dict else None
            stream = bool(stream) if type(stream) in (bool, int, float) else True
            if stream:
                result_dictionary = self.__executeTask(system_instruction=system_instruction, prompt=prompt, messages=messages, task_names=task_names, config=config)
                answer = str(result_dictionary.get('answer', '')).strip()
                answer_for_files = str(result_dictionary.get('answer_for_files', '')).strip()
                key_name, tokens = 'answer', []
                if answer_for_files: key_name, tokens = 'answer_for_files', self.stringToTokens(string=answer_for_files)
                else: key_name, tokens = 'answer', self.stringToTokens(string=answer)
                result_dictionary[key_name], result_dictionary['next_token'] = '', ''
                for token in tokens:
                    result_dictionary[key_name] += token
                    result_dictionary['next_token'] = token
                    yield result_dictionary
                return
            else: return self.__executeTask(system_instruction=system_instruction, prompt=prompt, messages=messages, task_names=task_names, config=config)
        except Exception as error:
            if self.__show_errors:
                error_message = 'ERROR in UtilitiesNLP.executeTask: '+str(error)
                print(error_message)
                try: self.__print_exc() if self.__display_error_point else None
                except: pass
            return {'answer': '', 'answer_for_files': '', 'files': [], 'sources': [], 'javascript_charts': [], 'artifacts': [], 'next_token': '', 'str_cost': '0.3200000000'}            
# This is a library of utility codes with features to facilitate the development and programming of language model algorithms from Sapiens Technology®.
# All code here is the intellectual property of Sapiens Technology®, and any public mention, distribution, modification, customization, or unauthorized sharing of this or other codes from Sapiens Technology® will result in the author being legally punished by our legal team.
# --------------------------> A SAPIENS TECHNOLOGY®️ PRODUCTION) <--------------------------
