# OMTX Python SDK

Lightweight helper for the OM Gateway. Queue diligence jobs, poll for results, and unlock/stream private data sets.

## Installation

```bash
pip install omtx
```

## Quick start

```python
from omtx import OMTXClient

client = OMTXClient()  # picks up OMTX_API_KEY

# 1) Generate claims (returns 202 + job_id)
claims_job = client.diligence_generate_claims(
    target="BRAF",
    prompt="Summarize known inhibitors"
)

# 2) Check status immediately (returns metadata + payload when ready)
job_record = client.jobs.status(claims_job["job_id"])
print(job_record["status"])

# 3) Optionally wait for completion (defaults to a 1 hour timeout)
claims_result = client.jobs.wait(
    claims_job["job_id"],
    result_endpoint="/v2/jobs/generateClaims/{job_id}",
)
print("total claims:", claims_result["total_claims"])

# 4) Kick off report synthesis (returns job metadata)
report_job = client.diligence.synthesize_report(gene_key="acad8")
report = client.jobs.wait(
    report_job["job_id"],
    result_endpoint="/v2/jobs/synthesizeReport/{job_id}",
)
print(report["sections"][0]["title"])

# 5) Deep research job (poll with jobs.wait when ready)
deep_job = client.diligence.deep_research(
    query="CRISPR applications in cancer therapy",
)
deep = client.jobs.wait(
    deep_job["job_id"],
    result_endpoint="/v2/jobs/deep-research/{job_id}",
)
print(deep["final_report"][:400])
```

## Setup

### Get an API key

1. Sign up at [https://omtx.ai](https://omtx.ai)
2. Generate an API key from the dashboard

### Provide the API key

```bash
export OMTX_API_KEY="your-api-key"
```

The SDK defaults to the hosted gateway at `https://api-gateway-129153908223.us-central1.run.app`. If you need a different deployment, pass `base_url` explicitly (or set `OMTX_BASE_URL`).

Or pass both API key and base URL when constructing the client:

```python
from omtx import OMTXClient

client = OMTXClient(
    api_key="your-api-key",
    base_url="https://api-gateway-129153908223.us-central1.run.app",
)
```

## Usage examples

### Error handling and context manager

```python
from omtx import OMTXClient, InsufficientCreditsError, OMTXError

with OMTXClient() as client:
    try:
        job = client.diligence.synthesize_report(gene_key="acad8")
        report = client.jobs.wait(
            job["job_id"], result_endpoint="/v2/jobs/synthesizeReport/{job_id}"
        )
    except InsufficientCreditsError:
        print("Add credits before running resynthesis jobs.")
    except OMTXError as exc:
        print(f"Gateway call failed: {exc}")
```

### Selective data access

```python
from omtx import OMTXClient

client = OMTXClient()

# 1) Unlock a dataset (consumes one Access Credit)
client.binders.unlock(protein_uuid="aa11bb22", gene_name="KRAS")

# 2) Stream the private dataset
stream = client.binders.selectivity.stream(
    protein_uuid="aa11bb22",
    limit=100_000,
)

df = stream.to_dataframe()
print("Rows:", len(df))
```

### Gene key discovery

```python
from omtx import OMTXClient

client = OMTXClient()
gene_keys = client.diligence.list_gene_keys()
print(gene_keys["items"][:5])
```

### Retries & idempotency

```python
from omtx import OMTXClient

client = OMTXClient()

# Every POST call receives a timestamp-based idempotency key automatically.
claims = client.diligence.generate_claims(
    target="BRAF",
    prompt="Summarize inhibitors",
)
```

## Available helper methods

- `diligence.generate_claims(target, prompt)`
- `diligence.synthesize_report(gene_key)`
- `diligence.deep_research(query)`
- `diligence.list_gene_keys()` (fetches all keys)
- `jobs.history(...)`, `jobs.status(job_id)`, `jobs.wait(job_id, …)`
- `binders.unlock(protein_uuid, gene_name=None)`, `binders.list_unlocks()`
- `binders.selectivity.stream|stats|cost_estimate(...)`
- `binders.public/private/community/decoys.stream|cost_estimate(...)`
- `pricing.manifest()`, `pricing.cost_estimate(...)`
- `gateway.status()`
- `users.profile()`

## Requirements

- Python 3.9 or higher
- An OMTX API key

## Support

- Email: support@omtx.ai
- Docs: https://docs.omtx.ai
- Issues: https://github.com/omtx/python-sdk/issues

## License

MIT License – see `LICENSE` for details.
