# SPDX-License-Identifier: Apache-2.0

import re

import jax

from tpu_commons.logger import init_logger

logger = init_logger(__name__)

# Below are tuned block sizes.

# key:
#    - tpu_version
#    - batch_size
#    - n_output_features
#    - n_input_features
#    - activation_dtype
#    - quantize_activation
# value:
#    - batch_block_size
#    - out_block_size
#    - in_block_size
TUNED_BLOCK_SIZES = {
    # go/keep-sorted start
    (6, 1024, 1024, 4096, 'bfloat16', True): (1024, 256, 4096),
    (6, 1024, 1024, 8192, 'bfloat16', True): (1024, 256, 8192),
    (6, 1024, 128, 8192, 'bfloat16', True): (512, 128, 8192),
    (6, 1024, 1280, 8192, 'bfloat16', True): (1024, 256, 8192),
    (6, 1024, 13824, 5120, 'bfloat16', True): (1024, 768, 5120),
    (6, 1024, 14336, 4096, 'bfloat16', True): (1024, 1024, 4096),
    (6, 1024, 1792, 5120, 'bfloat16', True): (1024, 256, 5120),
    (6, 1024, 28672, 4096, 'bfloat16', True): (1024, 2048, 4096),
    (6, 1024, 3584, 18944, 'bfloat16', True): (1024, 3584, 512),
    (6, 1024, 3584, 3584, 'bfloat16', True): (1024, 512, 3584),
    (6, 1024, 3584, 8192, 'bfloat16', True): (1024, 512, 8192),
    (6, 1024, 37888, 3584, 'bfloat16', True): (1024, 1024, 3584),
    (6, 1024, 4096, 14336, 'bfloat16', True): (1024, 256, 14336),
    (6, 1024, 4096, 4096, 'bfloat16', True): (1024, 512, 4096),
    (6, 1024, 4608, 3584, 'bfloat16', True): (1024, 768, 3584),
    (6, 1024, 5120, 1280, 'bfloat16', True): (1024, 1280, 1280),
    (6, 1024, 5120, 3456, 'bfloat16', True): (1024, 1024, 3456),
    (6, 1024, 5120, 640, 'bfloat16', True): (256, 5120, 640),
    (6, 1024, 5120, 6912, 'bfloat16', True): (1024, 512, 6912),
    (6, 1024, 6144, 4096, 'bfloat16', True): (1024, 768, 4096),
    (6, 1024, 6912, 5120, 'bfloat16', True): (1024, 768, 5120),
    (6, 1024, 7168, 8192, 'bfloat16', True): (1024, 512, 8192),
    (6, 1024, 8192, 1024, 'bfloat16', True): (1024, 4096, 1024),
    (6, 1024, 8192, 3584, 'bfloat16', True): (1024, 1024, 3584),
    (6, 1024, 896, 5120, 'bfloat16', True): (1024, 896, 2560),
    (6, 128, 1024, 4096, 'bfloat16', True): (128, 512, 4096),
    (6, 128, 1024, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 128, 128, 8192, 'bfloat16', True): (128, 128, 8192),
    (6, 128, 1280, 8192, 'bfloat16', True): (128, 1280, 2048),
    (6, 128, 13824, 5120, 'bfloat16', True): (128, 512, 5120),
    (6, 128, 14336, 4096, 'bfloat16', True): (128, 512, 4096),
    (6, 128, 1792, 5120, 'bfloat16', True): (128, 1792, 1280),
    (6, 128, 28672, 4096, 'bfloat16', True): (128, 28672, 256),
    (6, 128, 3584, 18944, 'bfloat16', True): (128, 256, 18944),
    (6, 128, 3584, 3584, 'bfloat16', True): (128, 3584, 896),
    (6, 128, 3584, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 128, 37888, 3584, 'bfloat16', True): (128, 1024, 3584),
    (6, 128, 4096, 14336, 'bfloat16', True): (128, 4096, 896),
    (6, 128, 4096, 4096, 'bfloat16', True): (128, 512, 4096),
    (6, 128, 4608, 3584, 'bfloat16', True): (128, 768, 3584),
    (6, 128, 5120, 1280, 'bfloat16', True): (128, 1280, 1280),
    (6, 128, 5120, 3456, 'bfloat16', True): (128, 640, 3456),
    (6, 128, 5120, 640, 'bfloat16', True): (128, 2560, 640),
    (6, 128, 5120, 6912, 'bfloat16', True): (128, 2560, 1152),
    (6, 128, 6144, 4096, 'bfloat16', True): (128, 768, 4096),
    (6, 128, 6912, 5120, 'bfloat16', True): (128, 1152, 2560),
    (6, 128, 7168, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 128, 8192, 1024, 'bfloat16', True): (128, 2048, 1024),
    (6, 128, 8192, 3584, 'bfloat16', True): (128, 8192, 512),
    (6, 128, 896, 5120, 'bfloat16', True): (128, 896, 2560),
    (6, 16, 1024, 4096, 'bfloat16', True): (128, 512, 4096),
    (6, 16, 1024, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 16, 128, 8192, 'bfloat16', True): (128, 128, 8192),
    (6, 16, 1280, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 16, 13824, 5120, 'bfloat16', True): (128, 512, 5120),
    (6, 16, 14336, 4096, 'bfloat16', True): (128, 896, 4096),
    (6, 16, 1792, 5120, 'bfloat16', True): (128, 896, 2560),
    (6, 16, 28672, 4096, 'bfloat16', True): (128, 28672, 256),
    (6, 16, 3584, 18944, 'bfloat16', True): (128, 256, 18944),
    (6, 16, 3584, 3584, 'bfloat16', True): (128, 896, 3584),
    (6, 16, 3584, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 16, 37888, 3584, 'bfloat16', True): (128, 1024, 3584),
    (6, 16, 4096, 14336, 'bfloat16', True): (128, 4096, 896),
    (6, 16, 4096, 4096, 'bfloat16', True): (128, 512, 4096),
    (6, 16, 4608, 3584, 'bfloat16', True): (128, 768, 3584),
    (6, 16, 5120, 1280, 'bfloat16', True): (128, 1280, 1280),
    (6, 16, 5120, 3456, 'bfloat16', True): (128, 640, 3456),
    (6, 16, 5120, 640, 'bfloat16', True): (128, 2560, 640),
    (6, 16, 5120, 6912, 'bfloat16', True): (128, 1280, 2304),
    (6, 16, 6144, 4096, 'bfloat16', True): (128, 768, 4096),
    (6, 16, 6912, 5120, 'bfloat16', True): (128, 1152, 2560),
    (6, 16, 7168, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 16, 8192, 1024, 'bfloat16', True): (128, 2048, 1024),
    (6, 16, 8192, 3584, 'bfloat16', True): (128, 1024, 3584),
    (6, 16, 896, 5120, 'bfloat16', True): (128, 896, 2560),
    (6, 16384, 13824, 5120, 'bfloat16', True): (2048, 1536, 5120),
    (6, 16384, 1792, 5120, 'bfloat16', True): (1024, 1792, 5120),
    (6, 16384, 3584, 18944, 'bfloat16', True): (256, 3584, 18944),
    (6, 16384, 3584, 3584, 'bfloat16', True): (512, 3584, 3584),
    (6, 16384, 37888, 3584, 'bfloat16', True): (4096, 512, 3584),
    (6, 16384, 4608, 3584, 'bfloat16', True): (512, 4608, 3584),
    (6, 16384, 5120, 1280, 'bfloat16', True): (512, 5120, 1280),
    (6, 16384, 5120, 3456, 'bfloat16', True): (512, 5120, 3456),
    (6, 16384, 5120, 640, 'bfloat16', True): (512, 5120, 640),
    (6, 16384, 5120, 6912, 'bfloat16', True): (512, 5120, 6912),
    (6, 16384, 6912, 5120, 'bfloat16', True): (512, 6912, 5120),
    (6, 16384, 896, 5120, 'bfloat16', True): (1024, 896, 5120),
    (6, 2048, 1024, 4096, 'bfloat16', True): (2048, 256, 4096),
    (6, 2048, 1024, 8192, 'bfloat16', True): (2048, 256, 8192),
    (6, 2048, 128, 8192, 'bfloat16', True): (1024, 128, 8192),
    (6, 2048, 1280, 8192, 'bfloat16', True): (2048, 256, 8192),
    (6, 2048, 13824, 5120, 'bfloat16', True): (2048, 768, 5120),
    (6, 2048, 14336, 4096, 'bfloat16', True): (2048, 1024, 4096),
    (6, 2048, 1792, 5120, 'bfloat16', True): (2048, 256, 5120),
    (6, 2048, 28672, 4096, 'bfloat16', True): (2048, 1024, 4096),
    (6, 2048, 3584, 18944, 'bfloat16', True): (2048, 3584, 512),
    (6, 2048, 3584, 3584, 'bfloat16', True): (2048, 512, 3584),
    (6, 2048, 3584, 8192, 'bfloat16', True): (2048, 256, 8192),
    (6, 2048, 37888, 3584, 'bfloat16', True): (2048, 1024, 3584),
    (6, 2048, 4096, 14336, 'bfloat16', True): (2048, 4096, 512),
    (6, 2048, 4096, 4096, 'bfloat16', True): (2048, 512, 4096),
    (6, 2048, 4608, 3584, 'bfloat16', True): (2048, 512, 3584),
    (6, 2048, 5120, 1280, 'bfloat16', True): (256, 5120, 1280),
    (6, 2048, 5120, 3456, 'bfloat16', True): (2048, 512, 3456),
    (6, 2048, 5120, 640, 'bfloat16', True): (256, 5120, 640),
    (6, 2048, 5120, 6912, 'bfloat16', True): (2048, 512, 6912),
    (6, 2048, 6144, 4096, 'bfloat16', True): (2048, 512, 4096),
    (6, 2048, 6912, 5120, 'bfloat16', True): (2048, 768, 5120),
    (6, 2048, 7168, 8192, 'bfloat16', True): (2048, 256, 8192),
    (6, 2048, 8192, 1024, 'bfloat16', True): (256, 8192, 1024),
    (6, 2048, 8192, 3584, 'bfloat16', True): (2048, 512, 3584),
    (6, 2048, 896, 5120, 'bfloat16', True): (1024, 896, 5120),
    (6, 256, 1024, 4096, 'bfloat16', True): (256, 512, 4096),
    (6, 256, 1024, 8192, 'bfloat16', True): (256, 512, 8192),
    (6, 256, 128, 8192, 'bfloat16', True): (256, 128, 8192),
    (6, 256, 1280, 8192, 'bfloat16', True): (256, 256, 8192),
    (6, 256, 13824, 5120, 'bfloat16', True): (256, 512, 5120),
    (6, 256, 14336, 4096, 'bfloat16', True): (256, 3584, 1024),
    (6, 256, 1792, 5120, 'bfloat16', True): (256, 1792, 1280),
    (6, 256, 28672, 4096, 'bfloat16', True): (256, 2048, 4096),
    (6, 256, 3584, 18944, 'bfloat16', True): (256, 256, 18944),
    (6, 256, 3584, 3584, 'bfloat16', True): (256, 896, 3584),
    (6, 256, 3584, 8192, 'bfloat16', True): (256, 3584, 1024),
    (6, 256, 37888, 3584, 'bfloat16', True): (256, 4736, 896),
    (6, 256, 4096, 14336, 'bfloat16', True): (256, 4096, 512),
    (6, 256, 4096, 4096, 'bfloat16', True): (256, 512, 4096),
    (6, 256, 4608, 3584, 'bfloat16', True): (256, 768, 3584),
    (6, 256, 5120, 1280, 'bfloat16', True): (256, 2560, 1280),
    (6, 256, 5120, 3456, 'bfloat16', True): (256, 1024, 3456),
    (6, 256, 5120, 640, 'bfloat16', True): (256, 2560, 640),
    (6, 256, 5120, 6912, 'bfloat16', True): (256, 5120, 768),
    (6, 256, 6144, 4096, 'bfloat16', True): (256, 512, 4096),
    (6, 256, 6912, 5120, 'bfloat16', True): (256, 6912, 512),
    (6, 256, 7168, 8192, 'bfloat16', True): (256, 512, 8192),
    (6, 256, 8192, 1024, 'bfloat16', True): (256, 2048, 1024),
    (6, 256, 8192, 3584, 'bfloat16', True): (256, 8192, 512),
    (6, 256, 896, 5120, 'bfloat16', True): (256, 896, 2560),
    (6, 32, 1024, 4096, 'bfloat16', True): (128, 512, 4096),
    (6, 32, 1024, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 32, 128, 8192, 'bfloat16', True): (128, 128, 8192),
    (6, 32, 1280, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 32, 13824, 5120, 'bfloat16', True): (128, 512, 5120),
    (6, 32, 14336, 4096, 'bfloat16', True): (128, 1024, 4096),
    (6, 32, 1792, 5120, 'bfloat16', True): (128, 896, 2560),
    (6, 32, 28672, 4096, 'bfloat16', True): (128, 28672, 256),
    (6, 32, 3584, 18944, 'bfloat16', True): (128, 128, 18944),
    (6, 32, 3584, 3584, 'bfloat16', True): (128, 896, 3584),
    (6, 32, 3584, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 32, 37888, 3584, 'bfloat16', True): (128, 1024, 3584),
    (6, 32, 4096, 14336, 'bfloat16', True): (128, 4096, 896),
    (6, 32, 4096, 4096, 'bfloat16', True): (128, 512, 4096),
    (6, 32, 4608, 3584, 'bfloat16', True): (128, 768, 3584),
    (6, 32, 5120, 1280, 'bfloat16', True): (128, 1280, 1280),
    (6, 32, 5120, 3456, 'bfloat16', True): (128, 640, 3456),
    (6, 32, 5120, 640, 'bfloat16', True): (128, 2560, 640),
    (6, 32, 5120, 6912, 'bfloat16', True): (128, 1280, 2304),
    (6, 32, 6144, 4096, 'bfloat16', True): (128, 768, 4096),
    (6, 32, 6912, 5120, 'bfloat16', True): (128, 2304, 1280),
    (6, 32, 7168, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 32, 8192, 1024, 'bfloat16', True): (128, 2048, 1024),
    (6, 32, 8192, 3584, 'bfloat16', True): (128, 1024, 3584),
    (6, 32, 896, 5120, 'bfloat16', True): (128, 896, 2560),
    (6, 4096, 13824, 5120, 'bfloat16', True): (2048, 1536, 5120),
    (6, 4096, 1792, 5120, 'bfloat16', True): (512, 1792, 5120),
    (6, 4096, 3584, 18944, 'bfloat16', True): (2048, 3584, 512),
    (6, 4096, 3584, 3584, 'bfloat16', True): (4096, 256, 3584),
    (6, 4096, 37888, 3584, 'bfloat16', True): (4096, 512, 3584),
    (6, 4096, 4608, 3584, 'bfloat16', True): (4096, 512, 3584),
    (6, 4096, 5120, 1280, 'bfloat16', True): (256, 5120, 1280),
    (6, 4096, 5120, 3456, 'bfloat16', True): (4096, 512, 3456),
    (6, 4096, 5120, 640, 'bfloat16', True): (256, 5120, 640),
    (6, 4096, 5120, 6912, 'bfloat16', True): (256, 5120, 6912),
    (6, 4096, 6912, 5120, 'bfloat16', True): (256, 6912, 5120),
    (6, 4096, 896, 5120, 'bfloat16', True): (256, 896, 5120),
    (6, 512, 1024, 4096, 'bfloat16', True): (512, 512, 4096),
    (6, 512, 1024, 8192, 'bfloat16', True): (512, 256, 8192),
    (6, 512, 128, 8192, 'bfloat16', True): (512, 128, 8192),
    (6, 512, 1280, 8192, 'bfloat16', True): (512, 256, 8192),
    (6, 512, 13824, 5120, 'bfloat16', True): (512, 13824, 512),
    (6, 512, 14336, 4096, 'bfloat16', True): (512, 1792, 4096),
    (6, 512, 1792, 5120, 'bfloat16', True): (512, 1792, 1280),
    (6, 512, 28672, 4096, 'bfloat16', True): (512, 2048, 4096),
    (6, 512, 3584, 18944, 'bfloat16', True): (512, 256, 18944),
    (6, 512, 3584, 3584, 'bfloat16', True): (512, 1792, 3584),
    (6, 512, 3584, 8192, 'bfloat16', True): (512, 512, 8192),
    (6, 512, 37888, 3584, 'bfloat16', True): (512, 18944, 512),
    (6, 512, 4096, 14336, 'bfloat16', True): (512, 256, 14336),
    (6, 512, 4096, 4096, 'bfloat16', True): (512, 1024, 4096),
    (6, 512, 4608, 3584, 'bfloat16', True): (512, 768, 3584),
    (6, 512, 5120, 1280, 'bfloat16', True): (512, 2560, 1280),
    (6, 512, 5120, 3456, 'bfloat16', True): (512, 1280, 3456),
    (6, 512, 5120, 640, 'bfloat16', True): (512, 2560, 640),
    (6, 512, 5120, 6912, 'bfloat16', True): (512, 512, 6912),
    (6, 512, 6144, 4096, 'bfloat16', True): (512, 1024, 4096),
    (6, 512, 6912, 5120, 'bfloat16', True): (512, 768, 5120),
    (6, 512, 7168, 8192, 'bfloat16', True): (512, 512, 8192),
    (6, 512, 8192, 1024, 'bfloat16', True): (512, 4096, 1024),
    (6, 512, 8192, 3584, 'bfloat16', True): (512, 2048, 3584),
    (6, 512, 896, 5120, 'bfloat16', True): (512, 896, 2560),
    (6, 64, 1024, 4096, 'bfloat16', True): (128, 512, 4096),
    (6, 64, 1024, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 64, 128, 8192, 'bfloat16', True): (128, 128, 8192),
    (6, 64, 1280, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 64, 13824, 5120, 'bfloat16', True): (128, 512, 5120),
    (6, 64, 14336, 4096, 'bfloat16', True): (128, 512, 4096),
    (6, 64, 1792, 5120, 'bfloat16', True): (128, 896, 2560),
    (6, 64, 28672, 4096, 'bfloat16', True): (128, 28672, 256),
    (6, 64, 3584, 18944, 'bfloat16', True): (128, 256, 18944),
    (6, 64, 3584, 3584, 'bfloat16', True): (128, 896, 3584),
    (6, 64, 3584, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 64, 37888, 3584, 'bfloat16', True): (128, 1024, 3584),
    (6, 64, 4096, 14336, 'bfloat16', True): (128, 4096, 896),
    (6, 64, 4096, 4096, 'bfloat16', True): (128, 512, 4096),
    (6, 64, 4608, 3584, 'bfloat16', True): (128, 768, 3584),
    (6, 64, 5120, 1280, 'bfloat16', True): (128, 1280, 1280),
    (6, 64, 5120, 3456, 'bfloat16', True): (128, 1024, 3456),
    (6, 64, 5120, 640, 'bfloat16', True): (128, 2560, 640),
    (6, 64, 5120, 6912, 'bfloat16', True): (128, 1280, 2304),
    (6, 64, 6144, 4096, 'bfloat16', True): (128, 768, 4096),
    (6, 64, 6912, 5120, 'bfloat16', True): (128, 2304, 1280),
    (6, 64, 7168, 8192, 'bfloat16', True): (128, 256, 8192),
    (6, 64, 8192, 1024, 'bfloat16', True): (128, 2048, 1024),
    (6, 64, 8192, 3584, 'bfloat16', True): (128, 1024, 3584),
    (6, 64, 896, 5120, 'bfloat16', True): (128, 896, 2560),
    (6, 8192, 13824, 5120, 'bfloat16', True): (2048, 1536, 5120),
    (6, 8192, 1792, 5120, 'bfloat16', True): (512, 1792, 5120),
    (6, 8192, 3584, 18944, 'bfloat16', True): (2048, 3584, 512),
    (6, 8192, 3584, 3584, 'bfloat16', True): (4096, 512, 3584),
    (6, 8192, 37888, 3584, 'bfloat16', True): (4096, 1024, 3584),
    (6, 8192, 4608, 3584, 'bfloat16', True): (4096, 512, 3584),
    (6, 8192, 5120, 1280, 'bfloat16', True): (256, 5120, 1280),
    (6, 8192, 5120, 3456, 'bfloat16', True): (512, 5120, 3456),
    (6, 8192, 5120, 640, 'bfloat16', True): (512, 5120, 640),
    (6, 8192, 5120, 6912, 'bfloat16', True): (512, 5120, 6912),
    (6, 8192, 6912, 5120, 'bfloat16', True): (512, 6912, 5120),
    (6, 8192, 896, 5120, 'bfloat16', True): (512, 896, 5120),
    # go/keep-sorted end
}


def get_tpu_version() -> int:
    """Returns the numeric version of the TPU, or -1 if not on TPU."""
    kind = jax.devices()[0].device_kind
    match = re.match(r'^TPU[^\d]*(\d+)', kind)
    if match is None:
        return -1
    return int(match.group(1))


def get_key(
    batch_size,
    n_output_features,
    n_input_features,
    activation_dtype,
    quantize_activation,
):
    """Returns the key for the given parameters."""
    return (
        get_tpu_version(),
        batch_size,
        n_output_features,
        n_input_features,
        activation_dtype,
        quantize_activation,
    )


def get_tuned_block_sizes(
    batch_size,
    n_output_features,
    n_input_features,
    activation_dtype,
    quantize_activation,
):
    """Retrieve the tuned block sizes for the given parameters.

  Args:
      batch_size (int): The batch size.
      n_output_features (int): The number of output features.
      n_input_features (int): The number of input features.
      activation_dtype (str): The data type of the activation ('bfloat16' or
        'float32').
      quantize_activation (bool): Whether to quantize the activation.

  Returns:
      tuple: A tuple containing the batch_block_size, out_block_size, and
      in_block_size.
  """
    key = get_key(
        batch_size,
        n_output_features,
        n_input_features,
        activation_dtype,
        quantize_activation,
    )
    tuned_sizes = TUNED_BLOCK_SIZES.get(key)
    if tuned_sizes is None:
        logger.warning(
            "Couldn't find tuned sizes for the quantized matmul kernel with %s",
            key)
        return (128, 128, 128)
    else:
        return tuned_sizes
