"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsIsoServiceAutoscaler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
/**
 * Creates a EcsIsoServiceAutoscaler construct. This construct allows you to scale an ECS service in an ISO
 * region where classic ECS Autoscaling may not be available.
 *
 * @param scope The parent creating construct (usually `this`).
 * @param id The construct's name.
 * @param this A `EcsIsoServiceAutoscalerthis` interface.
 */
class EcsIsoServiceAutoscaler extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { minimumTaskCount = 1, maximumTaskCount = 10, scaleOutIncrement = 1, scaleInIncrement = 1, scaleOutCooldown = aws_cdk_lib_1.Duration.seconds(60), scaleInCooldown = aws_cdk_lib_1.Duration.seconds(60), } = props;
        this.ecsScalingManagerFunction = new aws_lambda_1.Function(this, `${id}-EcsServiceScalingManager`, {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../../../resources/constructs/ecsIsoServiceAutoscaler')),
            handler: 'ecs_scaling_manager.handler',
            runtime: aws_lambda_1.Runtime.PYTHON_3_7,
            role: props.role || undefined,
            environment: {
                ECS_CLUSTER_NAME: props.ecsCluster.clusterName,
                ECS_SERVICE_NAME: props.ecsService.serviceName,
                MINIMUM_TASK_COUNT: minimumTaskCount.toString(),
                MAXIMUM_TASK_COUNT: maximumTaskCount.toString(),
                SCALE_ALARM_NAME: props.scaleAlarm.alarmName,
                SCALE_OUT_INCREMENT: scaleOutIncrement.toString(),
                SCALE_OUT_COOLDOWN: scaleOutCooldown.toSeconds().toString(),
                SCALE_IN_INCREMENT: scaleInIncrement.toString(),
                SCALE_IN_COOLDOWN: scaleInCooldown.toSeconds().toString(),
            },
        });
        new aws_events_1.Rule(this, `${id}-EcsScalingManagerSchedule`, {
            description: `Kicks off Lambda to adjust ECS scaling for service: ${props.ecsService.serviceName}`,
            enabled: true,
            schedule: aws_events_1.Schedule.rate(aws_cdk_lib_1.Duration.minutes(1)),
            targets: [new aws_events_targets_1.LambdaFunction(this.ecsScalingManagerFunction)],
        });
        if (!props.role) {
            // Set permissions for ecsScalingManagerFunction role
            this.ecsScalingManagerFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ['cloudwatch:DescribeAlarms'],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [props.scaleAlarm.alarmArn],
            }));
            this.ecsScalingManagerFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ['ecs:DescribeServices', 'ecs:UpdateService'],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [props.ecsService.serviceArn],
                conditions: {
                    StringEquals: {
                        'ecs:cluster': props.ecsCluster.clusterArn,
                    },
                },
            }));
        }
    }
}
exports.EcsIsoServiceAutoscaler = EcsIsoServiceAutoscaler;
_a = JSII_RTTI_SYMBOL_1;
EcsIsoServiceAutoscaler[_a] = { fqn: "@cdklabs/cdk-enterprise-iac.EcsIsoServiceAutoscaler", version: "0.0.79" };
//# sourceMappingURL=data:application/json;base64,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