"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SplitVpcEvenly = exports.PopulateWithConfig = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const constructs_1 = require("constructs");
/**
 * Populate a provided VPC with subnets based on a provided configuration
 *
 * @example
 * const mySubnetConfig: SubnetConfig[] = [
      {
        groupName: 'app',
        cidrRange: '172.31.0.0/27',
        availabilityZone: 'a',
        subnetType: subnetType.PUBLIC,
      },
      {
        groupName: 'app',
        cidrRange: '172.31.0.32/27',
        availabilityZone: 'b',
        subnetType: subnetType.PUBLIC,
      },
      {
        groupName: 'db',
        cidrRange: '172.31.0.64/27',
        availabilityZone: 'a',
        subnetType: subnetType.PRIVATE_WITH_EGRESS,
      },
      {
        groupName: 'db',
        cidrRange: '172.31.0.96/27',
        availabilityZone: 'b',
        subnetType: subnetType.PRIVATE_WITH_EGRESS,
      },
      {
        groupName: 'iso',
        cidrRange: '172.31.0.128/26',
        availabilityZone: 'a',
        subnetType: subnetType.PRIVATE_ISOLATED,
      },
      {
        groupName: 'iso',
        cidrRange: '172.31.0.196/26',
        availabilityZone: 'b',
        subnetType: subnetType.PRIVATE_ISOLATED,
      },
    ];
 * new PopulateWithConfig(this, "vpcPopulater", {
 *   vpcId: 'vpc-abcdefg1234567',
 *   privateRouteTableId: 'rt-abcdefg123456',
 *   localRouteTableId: 'rt-123456abcdefg',
 *   subnetConfig: mySubnetConfig,
 * })
 */
class PopulateWithConfig extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._vpcId = props.vpcId;
        this._privateRouteTableId = props.privateRouteTableId;
        this._localRouteTableId = props.localRouteTableId;
        this._subnetConfig = props.subnetConfig;
        for (const key in this._subnetConfig) {
            if (Object.prototype.hasOwnProperty.call(this._subnetConfig, key)) {
                const subnet = this._subnetConfig[key];
                const sub = new aws_ec2_1.CfnSubnet(this, `Subnet${subnet.groupName}-${subnet.availabilityZone}`, {
                    availabilityZone: `${aws_cdk_lib_1.Stack.of(this).region}${subnet.availabilityZone}`,
                    vpcId: this._vpcId,
                    cidrBlock: subnet.cidrRange,
                });
                aws_cdk_lib_1.Tags.of(sub).add('aws-cdk:subnet-type', subnet.subnetType);
                aws_cdk_lib_1.Tags.of(sub).add('Name', `subnet-${subnet.groupName}-${subnet.availabilityZone}`);
                new aws_ec2_1.CfnSubnetRouteTableAssociation(this, `SubnetReAssoc${subnet.groupName}-${subnet.availabilityZone}`, {
                    routeTableId: subnet.subnetType == 'Isolated'
                        ? this._privateRouteTableId
                        : this._localRouteTableId,
                    subnetId: sub.ref,
                });
            }
        }
    }
}
exports.PopulateWithConfig = PopulateWithConfig;
_a = JSII_RTTI_SYMBOL_1;
PopulateWithConfig[_a] = { fqn: "@cdklabs/cdk-enterprise-iac.PopulateWithConfig", version: "0.0.79" };
/**
 * Splits a VPC evenly between a provided number of AZs (3 if not defined), and attaches a provided route table to each, and labels
 *
 * @example
 * new SplitVpcEvenly(this, 'evenSplitVpc', {
 *   vpcId: 'vpc-abcdefg123456',
 *   vpcCidr: '172.16.0.0/24',
 *   routeTableId: 'rt-abcdefgh123456',
 * });
 * @example
 * // with more specific properties
 * new SplitVpcEvenly(this, 'evenSplitVpc', {
 *   vpcId: 'vpc-abcdefg123456',
 *   vpcCidr: '172.16.0.0/16',
 *   routeTableId: 'rt-abcdefgh123456',
 *   cidrBits: '10',
 *   numberOfAzs: 4,
 *   subnetType: subnetType.PRIVATE_ISOLATED,
 * });
 */
class SplitVpcEvenly extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._vpcId = props.vpcId;
        this._vpcCidr = props.vpcCidr;
        this._routeTableId = props.routeTableId;
        this._cidrBits = props.cidrBits || '6';
        this._numberOfAzs = props.numberOfAzs || 3;
        this._subnetType = props.subnetType || aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS;
        // Throw error if > 6 AZs
        if (this._numberOfAzs < 2 || this._numberOfAzs > 6) {
            throw new Error('numberOfAzs must be between 2 and 6');
        }
        // based on number of az, create an array of az strings
        const azs = [];
        for (let i = 0; i < this._numberOfAzs; i++) {
            azs.push(String.fromCharCode(97 + i));
        }
        azs.forEach((val, i) => {
            const sub = new aws_ec2_1.CfnSubnet(this, `TgwSubnet${val}`, {
                availabilityZone: aws_cdk_lib_1.Fn.select(i, aws_cdk_lib_1.Fn.getAzs()),
                vpcId: this._vpcId,
                cidrBlock: aws_cdk_lib_1.Fn.select(i, aws_cdk_lib_1.Fn.cidr(this._vpcCidr, this._numberOfAzs, this._cidrBits)),
            });
            aws_cdk_lib_1.Tags.of(sub).add('aws-cdk:subnet-type', this._subnetType);
            aws_cdk_lib_1.Tags.of(sub).add('Name', `subnet-${val}`);
            new aws_ec2_1.CfnSubnetRouteTableAssociation(this, `SubnetRtAssoc${val}`, {
                routeTableId: this._routeTableId,
                subnetId: sub.ref,
            });
        });
    }
}
exports.SplitVpcEvenly = SplitVpcEvenly;
_b = JSII_RTTI_SYMBOL_1;
SplitVpcEvenly[_b] = { fqn: "@cdklabs/cdk-enterprise-iac.SplitVpcEvenly", version: "0.0.79" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicG9wdWxhdGVQcm92aWRlZFZwYy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9jb25zdHJ1Y3RzL3BvcHVsYXRlUHJvdmlkZWRWcGMvcG9wdWxhdGVQcm92aWRlZFZwYy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBOzs7RUFHRTtBQUNGLDZDQUE4QztBQUM5QyxpREFJNkI7QUFDN0IsMkNBQXVDO0FBcUR2Qzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBZ0RHO0FBQ0gsTUFBYSxrQkFBbUIsU0FBUSxzQkFBUztJQU0vQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQThCO1FBQ3RFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDO1FBQzFCLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxLQUFLLENBQUMsbUJBQW1CLENBQUM7UUFDdEQsSUFBSSxDQUFDLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQztRQUNsRCxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUM7UUFFeEMsS0FBSyxNQUFNLEdBQUcsSUFBSSxJQUFJLENBQUMsYUFBYSxFQUFFO1lBQ3BDLElBQUksTUFBTSxDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsR0FBRyxDQUFDLEVBQUU7Z0JBQ2pFLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBRXZDLE1BQU0sR0FBRyxHQUFHLElBQUksbUJBQVMsQ0FDdkIsSUFBSSxFQUNKLFNBQVMsTUFBTSxDQUFDLFNBQVMsSUFBSSxNQUFNLENBQUMsZ0JBQWdCLEVBQUUsRUFDdEQ7b0JBQ0UsZ0JBQWdCLEVBQUUsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNLEdBQ3hDLE1BQU0sQ0FBQyxnQkFDVCxFQUFFO29CQUNGLEtBQUssRUFBRSxJQUFJLENBQUMsTUFBTTtvQkFDbEIsU0FBUyxFQUFFLE1BQU0sQ0FBQyxTQUFTO2lCQUM1QixDQUNGLENBQUM7Z0JBQ0Ysa0JBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLHFCQUFxQixFQUFFLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQztnQkFDM0Qsa0JBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxDQUNkLE1BQU0sRUFDTixVQUFVLE1BQU0sQ0FBQyxTQUFTLElBQUksTUFBTSxDQUFDLGdCQUFnQixFQUFFLENBQ3hELENBQUM7Z0JBRUYsSUFBSSx3Q0FBOEIsQ0FDaEMsSUFBSSxFQUNKLGdCQUFnQixNQUFNLENBQUMsU0FBUyxJQUFJLE1BQU0sQ0FBQyxnQkFBZ0IsRUFBRSxFQUM3RDtvQkFDRSxZQUFZLEVBQ1YsTUFBTSxDQUFDLFVBQVUsSUFBSSxVQUFVO3dCQUM3QixDQUFDLENBQUMsSUFBSSxDQUFDLG9CQUFvQjt3QkFDM0IsQ0FBQyxDQUFDLElBQUksQ0FBQyxrQkFBa0I7b0JBQzdCLFFBQVEsRUFBRSxHQUFHLENBQUMsR0FBRztpQkFDbEIsQ0FDRixDQUFDO2FBQ0g7U0FDRjtJQUNILENBQUM7O0FBaERILGdEQWlEQzs7O0FBZ0NEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBbUJHO0FBQ0gsTUFBYSxjQUFlLFNBQVEsc0JBQVM7SUFRM0MsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUEwQjtRQUNsRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQztRQUMxQixJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDOUIsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO1FBQ3hDLElBQUksQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDLFFBQVEsSUFBSSxHQUFHLENBQUM7UUFDdkMsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsV0FBVyxJQUFJLENBQUMsQ0FBQztRQUMzQyxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxVQUFVLElBQUksb0JBQVUsQ0FBQyxtQkFBbUIsQ0FBQztRQUV0RSx5QkFBeUI7UUFDekIsSUFBSSxJQUFJLENBQUMsWUFBWSxHQUFHLENBQUMsSUFBSSxJQUFJLENBQUMsWUFBWSxHQUFHLENBQUMsRUFBRTtZQUNsRCxNQUFNLElBQUksS0FBSyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7U0FDeEQ7UUFFRCx1REFBdUQ7UUFDdkQsTUFBTSxHQUFHLEdBQWEsRUFBRSxDQUFDO1FBQ3pCLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUMsRUFBRSxFQUFFO1lBQzFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUN2QztRQUVELEdBQUcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxFQUFFLEVBQUU7WUFDckIsTUFBTSxHQUFHLEdBQUcsSUFBSSxtQkFBUyxDQUFDLElBQUksRUFBRSxZQUFZLEdBQUcsRUFBRSxFQUFFO2dCQUNqRCxnQkFBZ0IsRUFBRSxnQkFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsZ0JBQUUsQ0FBQyxNQUFNLEVBQUUsQ0FBQztnQkFDM0MsS0FBSyxFQUFFLElBQUksQ0FBQyxNQUFNO2dCQUNsQixTQUFTLEVBQUUsZ0JBQUUsQ0FBQyxNQUFNLENBQ2xCLENBQUMsRUFDRCxnQkFBRSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxZQUFhLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUMzRDthQUNGLENBQUMsQ0FBQztZQUNILGtCQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsRUFBRSxJQUFJLENBQUMsV0FBWSxDQUFDLENBQUM7WUFDM0Qsa0JBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxVQUFVLEdBQUcsRUFBRSxDQUFDLENBQUM7WUFFMUMsSUFBSSx3Q0FBOEIsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEdBQUcsRUFBRSxFQUFFO2dCQUM5RCxZQUFZLEVBQUUsSUFBSSxDQUFDLGFBQWE7Z0JBQ2hDLFFBQVEsRUFBRSxHQUFHLENBQUMsR0FBRzthQUNsQixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7O0FBOUNILHdDQStDQyIsInNvdXJjZXNDb250ZW50IjpbIi8qXG5Db3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4qL1xuaW1wb3J0IHsgRm4sIFN0YWNrLCBUYWdzIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHtcbiAgQ2ZuU3VibmV0LFxuICBDZm5TdWJuZXRSb3V0ZVRhYmxlQXNzb2NpYXRpb24sXG4gIFN1Ym5ldFR5cGUsXG59IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lYzInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgU3VibmV0Q29uZmlnIHtcbiAgLyoqXG4gICAqIExvZ2ljYWwgZ3JvdXAgbmFtZSBvZiBhIHN1Ym5ldFxuICAgKiBAZXhhbXBsZSBhcHBcbiAgICogQGV4YW1wbGUgZGJcbiAgICovXG4gIHJlYWRvbmx5IGdyb3VwTmFtZTogc3RyaW5nO1xuICAvKipcbiAgICogV2hpY2ggYXZhaWxhYmlsaXR5IHpvbmUgdGhlIHN1Ym5ldCBzaG91bGQgYmUgaW5cbiAgICpcbiAgICovXG4gIHJlYWRvbmx5IGF2YWlsYWJpbGl0eVpvbmU6IHN0cmluZztcbiAgLyoqXG4gICAqIENpZHIgcmFuZ2Ugb2YgdGhlIHN1Ym5ldCB0byBjcmVhdGVcbiAgICovXG4gIHJlYWRvbmx5IGNpZHJSYW5nZTogc3RyaW5nO1xuICAvKipcbiAgICogV2hhdCBbU3VibmV0VHlwZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nkay9hcGkvdjIvZG9jcy9hd3MtY2RrLWxpYi5hd3NfZWMyLlN1Ym5ldFR5cGUuaHRtbCkgdG8gdXNlXG4gICAqXG4gICAqIFRoaXMgd2lsbCBnb3Zlcm4gdGhlIGBhd3MtY2RrOnN1Ym5ldC10eXBlYCB0YWcgb24gdGhlIHN1Ym5ldFxuICAgKlxuICAgKiBTdWJuZXRUeXBlIHwgYGF3cy1jZGs6c3VibmV0LXR5cGVgIHRhZyB2YWx1ZVxuICAgKiAtLS0gfCAtLS1cbiAgICogYFBSSVZBVEVfSVNPTEFURURgIHwgJ0lzb2xhdGVkJ1xuICAgKiBgUFJJVkFURV9XSVRIX0VHUkVTU2AgfCAnUHJpdmF0ZSdcbiAgICogYFBVQkxJQ2AgfCAnUHVibGljJ1xuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0VHlwZTogU3VibmV0VHlwZTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBQb3B1bGF0ZVdpdGhDb25maWdQcm9wcyB7XG4gIC8qKlxuICAgKiBJRCBvZiB0aGUgVlBDIHByb3ZpZGVkIHRoYXQgbmVlZHMgdG8gYmUgcG9wdWxhdGVkXG4gICAqL1xuICByZWFkb25seSB2cGNJZDogc3RyaW5nO1xuICAvKipcbiAgICogUm91dGUgdGFibGUgSUQgZm9yIGEgcHJvdmlkZWQgcm91dGUgdGFibGUgd2l0aCByb3V0ZXMgdG8gZW50ZXJwcmlzZSBuZXR3b3JrXG4gICAqXG4gICAqIEJvdGggc3VibmV0VHlwZS5QVUJMSUMgYW5kIHN1Ym5ldFR5cGUuUFJJVkFURV9XSVRIX0VHUkVTUyB3aWxsIHVzZSB0aGlzIHByb3BlcnR5XG4gICAqL1xuICByZWFkb25seSBwcml2YXRlUm91dGVUYWJsZUlkOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBMb2NhbCByb3V0ZSB0YWJsZSBJRCwgd2l0aCByb3V0ZXMgb25seSB0byBsb2NhbCBWUENcbiAgICovXG4gIHJlYWRvbmx5IGxvY2FsUm91dGVUYWJsZUlkOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBMaXN0IG9mIFN1Ym5ldCBjb25maWdzIHRvIHByb3Zpc2lvbiB0byBwcm92aXNpb25cbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldENvbmZpZzogU3VibmV0Q29uZmlnW107XG59XG5cbi8qKlxuICogUG9wdWxhdGUgYSBwcm92aWRlZCBWUEMgd2l0aCBzdWJuZXRzIGJhc2VkIG9uIGEgcHJvdmlkZWQgY29uZmlndXJhdGlvblxuICpcbiAqIEBleGFtcGxlXG4gKiBjb25zdCBteVN1Ym5ldENvbmZpZzogU3VibmV0Q29uZmlnW10gPSBbXG4gICAgICB7XG4gICAgICAgIGdyb3VwTmFtZTogJ2FwcCcsXG4gICAgICAgIGNpZHJSYW5nZTogJzE3Mi4zMS4wLjAvMjcnLFxuICAgICAgICBhdmFpbGFiaWxpdHlab25lOiAnYScsXG4gICAgICAgIHN1Ym5ldFR5cGU6IHN1Ym5ldFR5cGUuUFVCTElDLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgZ3JvdXBOYW1lOiAnYXBwJyxcbiAgICAgICAgY2lkclJhbmdlOiAnMTcyLjMxLjAuMzIvMjcnLFxuICAgICAgICBhdmFpbGFiaWxpdHlab25lOiAnYicsXG4gICAgICAgIHN1Ym5ldFR5cGU6IHN1Ym5ldFR5cGUuUFVCTElDLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgZ3JvdXBOYW1lOiAnZGInLFxuICAgICAgICBjaWRyUmFuZ2U6ICcxNzIuMzEuMC42NC8yNycsXG4gICAgICAgIGF2YWlsYWJpbGl0eVpvbmU6ICdhJyxcbiAgICAgICAgc3VibmV0VHlwZTogc3VibmV0VHlwZS5QUklWQVRFX1dJVEhfRUdSRVNTLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgZ3JvdXBOYW1lOiAnZGInLFxuICAgICAgICBjaWRyUmFuZ2U6ICcxNzIuMzEuMC45Ni8yNycsXG4gICAgICAgIGF2YWlsYWJpbGl0eVpvbmU6ICdiJyxcbiAgICAgICAgc3VibmV0VHlwZTogc3VibmV0VHlwZS5QUklWQVRFX1dJVEhfRUdSRVNTLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgZ3JvdXBOYW1lOiAnaXNvJyxcbiAgICAgICAgY2lkclJhbmdlOiAnMTcyLjMxLjAuMTI4LzI2JyxcbiAgICAgICAgYXZhaWxhYmlsaXR5Wm9uZTogJ2EnLFxuICAgICAgICBzdWJuZXRUeXBlOiBzdWJuZXRUeXBlLlBSSVZBVEVfSVNPTEFURUQsXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICBncm91cE5hbWU6ICdpc28nLFxuICAgICAgICBjaWRyUmFuZ2U6ICcxNzIuMzEuMC4xOTYvMjYnLFxuICAgICAgICBhdmFpbGFiaWxpdHlab25lOiAnYicsXG4gICAgICAgIHN1Ym5ldFR5cGU6IHN1Ym5ldFR5cGUuUFJJVkFURV9JU09MQVRFRCxcbiAgICAgIH0sXG4gICAgXTtcbiAqIG5ldyBQb3B1bGF0ZVdpdGhDb25maWcodGhpcywgXCJ2cGNQb3B1bGF0ZXJcIiwge1xuICogICB2cGNJZDogJ3ZwYy1hYmNkZWZnMTIzNDU2NycsXG4gKiAgIHByaXZhdGVSb3V0ZVRhYmxlSWQ6ICdydC1hYmNkZWZnMTIzNDU2JyxcbiAqICAgbG9jYWxSb3V0ZVRhYmxlSWQ6ICdydC0xMjM0NTZhYmNkZWZnJyxcbiAqICAgc3VibmV0Q29uZmlnOiBteVN1Ym5ldENvbmZpZyxcbiAqIH0pXG4gKi9cbmV4cG9ydCBjbGFzcyBQb3B1bGF0ZVdpdGhDb25maWcgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICBwcml2YXRlIHJlYWRvbmx5IF92cGNJZDogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IF9wcml2YXRlUm91dGVUYWJsZUlkOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgX2xvY2FsUm91dGVUYWJsZUlkOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgX3N1Ym5ldENvbmZpZzogU3VibmV0Q29uZmlnW107XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFBvcHVsYXRlV2l0aENvbmZpZ1Byb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMuX3ZwY0lkID0gcHJvcHMudnBjSWQ7XG4gICAgdGhpcy5fcHJpdmF0ZVJvdXRlVGFibGVJZCA9IHByb3BzLnByaXZhdGVSb3V0ZVRhYmxlSWQ7XG4gICAgdGhpcy5fbG9jYWxSb3V0ZVRhYmxlSWQgPSBwcm9wcy5sb2NhbFJvdXRlVGFibGVJZDtcbiAgICB0aGlzLl9zdWJuZXRDb25maWcgPSBwcm9wcy5zdWJuZXRDb25maWc7XG5cbiAgICBmb3IgKGNvbnN0IGtleSBpbiB0aGlzLl9zdWJuZXRDb25maWcpIHtcbiAgICAgIGlmIChPYmplY3QucHJvdG90eXBlLmhhc093blByb3BlcnR5LmNhbGwodGhpcy5fc3VibmV0Q29uZmlnLCBrZXkpKSB7XG4gICAgICAgIGNvbnN0IHN1Ym5ldCA9IHRoaXMuX3N1Ym5ldENvbmZpZ1trZXldO1xuXG4gICAgICAgIGNvbnN0IHN1YiA9IG5ldyBDZm5TdWJuZXQoXG4gICAgICAgICAgdGhpcyxcbiAgICAgICAgICBgU3VibmV0JHtzdWJuZXQuZ3JvdXBOYW1lfS0ke3N1Ym5ldC5hdmFpbGFiaWxpdHlab25lfWAsXG4gICAgICAgICAge1xuICAgICAgICAgICAgYXZhaWxhYmlsaXR5Wm9uZTogYCR7U3RhY2sub2YodGhpcykucmVnaW9ufSR7XG4gICAgICAgICAgICAgIHN1Ym5ldC5hdmFpbGFiaWxpdHlab25lXG4gICAgICAgICAgICB9YCxcbiAgICAgICAgICAgIHZwY0lkOiB0aGlzLl92cGNJZCxcbiAgICAgICAgICAgIGNpZHJCbG9jazogc3VibmV0LmNpZHJSYW5nZSxcbiAgICAgICAgICB9XG4gICAgICAgICk7XG4gICAgICAgIFRhZ3Mub2Yoc3ViKS5hZGQoJ2F3cy1jZGs6c3VibmV0LXR5cGUnLCBzdWJuZXQuc3VibmV0VHlwZSk7XG4gICAgICAgIFRhZ3Mub2Yoc3ViKS5hZGQoXG4gICAgICAgICAgJ05hbWUnLFxuICAgICAgICAgIGBzdWJuZXQtJHtzdWJuZXQuZ3JvdXBOYW1lfS0ke3N1Ym5ldC5hdmFpbGFiaWxpdHlab25lfWBcbiAgICAgICAgKTtcblxuICAgICAgICBuZXcgQ2ZuU3VibmV0Um91dGVUYWJsZUFzc29jaWF0aW9uKFxuICAgICAgICAgIHRoaXMsXG4gICAgICAgICAgYFN1Ym5ldFJlQXNzb2Mke3N1Ym5ldC5ncm91cE5hbWV9LSR7c3VibmV0LmF2YWlsYWJpbGl0eVpvbmV9YCxcbiAgICAgICAgICB7XG4gICAgICAgICAgICByb3V0ZVRhYmxlSWQ6XG4gICAgICAgICAgICAgIHN1Ym5ldC5zdWJuZXRUeXBlID09ICdJc29sYXRlZCdcbiAgICAgICAgICAgICAgICA/IHRoaXMuX3ByaXZhdGVSb3V0ZVRhYmxlSWRcbiAgICAgICAgICAgICAgICA6IHRoaXMuX2xvY2FsUm91dGVUYWJsZUlkLFxuICAgICAgICAgICAgc3VibmV0SWQ6IHN1Yi5yZWYsXG4gICAgICAgICAgfVxuICAgICAgICApO1xuICAgICAgfVxuICAgIH1cbiAgfVxufVxuXG5leHBvcnQgaW50ZXJmYWNlIFNwbGl0VnBjRXZlbmx5UHJvcHMge1xuICAvKipcbiAgICogSUQgb2YgdGhlIGV4aXN0aW5nIFZQQyB5b3UncmUgdHJ5aW5nIHRvIHBvcHVsYXRlXG4gICAqL1xuICByZWFkb25seSB2cGNJZDogc3RyaW5nO1xuICAvKipcbiAgICogQ0lEUiByYW5nZSBvZiB0aGUgVlBDIHlvdSdyZSBwb3B1bGF0aW5nXG4gICAqL1xuICByZWFkb25seSB2cGNDaWRyOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBSb3V0ZSBUYWJsZSBJRCB0aGF0IHdpbGwgYmUgYXR0YWNoZWQgdG8gZWFjaCBzdWJuZXQgY3JlYXRlZFxuICAgKi9cbiAgcmVhZG9ubHkgcm91dGVUYWJsZUlkOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBgY2lkckJpdHNgIGFyZ3VtZW50IGZvciB0aGUgW2BGbjo6Q2lkcmBdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2ludHJpbnNpYy1mdW5jdGlvbi1yZWZlcmVuY2UtY2lkci5odG1sKSBDbG91ZGZvcm1hdGlvbiBpbnRyaW5zaWMgZnVuY3Rpb25cbiAgICpcbiAgICogQGRlZmF1bHQgJzYnXG4gICAqL1xuICByZWFkb25seSBjaWRyQml0cz86IHN0cmluZztcbiAgLyoqXG4gICAqIE51bWJlciBvZiBBWnMgdG8gZXZlbmx5IHNwbGl0IGludG9cbiAgICpcbiAgICogQGRlZmF1bHQgM1xuICAgKi9cbiAgcmVhZG9ubHkgbnVtYmVyT2ZBenM/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBAZGVmYXVsdCBzdWJuZXRUeXBlLlBSSVZBVEVcbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldFR5cGU/OiBTdWJuZXRUeXBlO1xufVxuLyoqXG4gKiBTcGxpdHMgYSBWUEMgZXZlbmx5IGJldHdlZW4gYSBwcm92aWRlZCBudW1iZXIgb2YgQVpzICgzIGlmIG5vdCBkZWZpbmVkKSwgYW5kIGF0dGFjaGVzIGEgcHJvdmlkZWQgcm91dGUgdGFibGUgdG8gZWFjaCwgYW5kIGxhYmVsc1xuICpcbiAqIEBleGFtcGxlXG4gKiBuZXcgU3BsaXRWcGNFdmVubHkodGhpcywgJ2V2ZW5TcGxpdFZwYycsIHtcbiAqICAgdnBjSWQ6ICd2cGMtYWJjZGVmZzEyMzQ1NicsXG4gKiAgIHZwY0NpZHI6ICcxNzIuMTYuMC4wLzI0JyxcbiAqICAgcm91dGVUYWJsZUlkOiAncnQtYWJjZGVmZ2gxMjM0NTYnLFxuICogfSk7XG4gKiBAZXhhbXBsZVxuICogLy8gd2l0aCBtb3JlIHNwZWNpZmljIHByb3BlcnRpZXNcbiAqIG5ldyBTcGxpdFZwY0V2ZW5seSh0aGlzLCAnZXZlblNwbGl0VnBjJywge1xuICogICB2cGNJZDogJ3ZwYy1hYmNkZWZnMTIzNDU2JyxcbiAqICAgdnBjQ2lkcjogJzE3Mi4xNi4wLjAvMTYnLFxuICogICByb3V0ZVRhYmxlSWQ6ICdydC1hYmNkZWZnaDEyMzQ1NicsXG4gKiAgIGNpZHJCaXRzOiAnMTAnLFxuICogICBudW1iZXJPZkF6czogNCxcbiAqICAgc3VibmV0VHlwZTogc3VibmV0VHlwZS5QUklWQVRFX0lTT0xBVEVELFxuICogfSk7XG4gKi9cbmV4cG9ydCBjbGFzcyBTcGxpdFZwY0V2ZW5seSBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHByaXZhdGUgcmVhZG9ubHkgX3ZwY0lkOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgX3ZwY0NpZHI6IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSBfcm91dGVUYWJsZUlkOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgX2NpZHJCaXRzPzogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IF9udW1iZXJPZkF6cz86IG51bWJlcjtcbiAgcHJpdmF0ZSByZWFkb25seSBfc3VibmV0VHlwZT86IFN1Ym5ldFR5cGU7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFNwbGl0VnBjRXZlbmx5UHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5fdnBjSWQgPSBwcm9wcy52cGNJZDtcbiAgICB0aGlzLl92cGNDaWRyID0gcHJvcHMudnBjQ2lkcjtcbiAgICB0aGlzLl9yb3V0ZVRhYmxlSWQgPSBwcm9wcy5yb3V0ZVRhYmxlSWQ7XG4gICAgdGhpcy5fY2lkckJpdHMgPSBwcm9wcy5jaWRyQml0cyB8fCAnNic7XG4gICAgdGhpcy5fbnVtYmVyT2ZBenMgPSBwcm9wcy5udW1iZXJPZkF6cyB8fCAzO1xuICAgIHRoaXMuX3N1Ym5ldFR5cGUgPSBwcm9wcy5zdWJuZXRUeXBlIHx8IFN1Ym5ldFR5cGUuUFJJVkFURV9XSVRIX0VHUkVTUztcblxuICAgIC8vIFRocm93IGVycm9yIGlmID4gNiBBWnNcbiAgICBpZiAodGhpcy5fbnVtYmVyT2ZBenMgPCAyIHx8IHRoaXMuX251bWJlck9mQXpzID4gNikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdudW1iZXJPZkF6cyBtdXN0IGJlIGJldHdlZW4gMiBhbmQgNicpO1xuICAgIH1cblxuICAgIC8vIGJhc2VkIG9uIG51bWJlciBvZiBheiwgY3JlYXRlIGFuIGFycmF5IG9mIGF6IHN0cmluZ3NcbiAgICBjb25zdCBhenM6IHN0cmluZ1tdID0gW107XG4gICAgZm9yIChsZXQgaSA9IDA7IGkgPCB0aGlzLl9udW1iZXJPZkF6czsgaSsrKSB7XG4gICAgICBhenMucHVzaChTdHJpbmcuZnJvbUNoYXJDb2RlKDk3ICsgaSkpO1xuICAgIH1cblxuICAgIGF6cy5mb3JFYWNoKCh2YWwsIGkpID0+IHtcbiAgICAgIGNvbnN0IHN1YiA9IG5ldyBDZm5TdWJuZXQodGhpcywgYFRnd1N1Ym5ldCR7dmFsfWAsIHtcbiAgICAgICAgYXZhaWxhYmlsaXR5Wm9uZTogRm4uc2VsZWN0KGksIEZuLmdldEF6cygpKSxcbiAgICAgICAgdnBjSWQ6IHRoaXMuX3ZwY0lkLFxuICAgICAgICBjaWRyQmxvY2s6IEZuLnNlbGVjdChcbiAgICAgICAgICBpLFxuICAgICAgICAgIEZuLmNpZHIodGhpcy5fdnBjQ2lkciwgdGhpcy5fbnVtYmVyT2ZBenMhLCB0aGlzLl9jaWRyQml0cylcbiAgICAgICAgKSxcbiAgICAgIH0pO1xuICAgICAgVGFncy5vZihzdWIpLmFkZCgnYXdzLWNkazpzdWJuZXQtdHlwZScsIHRoaXMuX3N1Ym5ldFR5cGUhKTtcbiAgICAgIFRhZ3Mub2Yoc3ViKS5hZGQoJ05hbWUnLCBgc3VibmV0LSR7dmFsfWApO1xuXG4gICAgICBuZXcgQ2ZuU3VibmV0Um91dGVUYWJsZUFzc29jaWF0aW9uKHRoaXMsIGBTdWJuZXRSdEFzc29jJHt2YWx9YCwge1xuICAgICAgICByb3V0ZVRhYmxlSWQ6IHRoaXMuX3JvdXRlVGFibGVJZCxcbiAgICAgICAgc3VibmV0SWQ6IHN1Yi5yZWYsXG4gICAgICB9KTtcbiAgICB9KTtcbiAgfVxufVxuIl19