"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnStore = void 0;
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const cp = require("child_process");
const flattener_1 = require("./flattener");
const resourceExtractor_1 = require("./resourceExtractor");
class CfnStore {
    constructor(props) {
        this.templates = {};
        this.extractedStackExports = {};
        /** Save CloudFormation templates for future lookup */
        for (const item of props.stackArtifacts) {
            this.templates[item.stackName] = item.template;
        }
        this.flatTemplates = flattener_1.Flattener.flattenObject(this.templates);
        /** Save Value Map if we are using the `API_LOOKUP` sharing method */
        if (props.valueShareMethod == resourceExtractor_1.ResourceExtractorShareMethod.API_LOOKUP) {
            const stack = this.describeStack(props.extractedStackName, props.region);
            if (stack) {
                this.extractedStackExports = this.createExportMap(stack);
            }
        }
    }
    /**
     * Determine what the export value should be, for example if it should be a
     * `Ref` or `Fn::GetAtt`
     *
     * @param resource
     * @param flattendKey
     * @returns
     */
    determineExportValue(resource, flattendKey) {
        const splitKey = flattendKey.split('.');
        if (splitKey.slice(-1)[0] == 'Ref') {
            return resource.ref;
        }
        else if (splitKey.slice(-2)[0] == 'Fn::GetAtt') {
            const attToGet = flattener_1.Flattener.getValueByPath(this.templates, splitKey.slice(0, -1).concat(['1']).join('.'));
            return resource.getAtt(attToGet);
        }
        else if (splitKey.slice(-2)[0] == 'DependsOn') {
            return false;
        }
        else {
            throw new Error(`Can't determine export value for ${flattendKey}`);
        }
    }
    /**
     * Retrieve a Stack Name from a given Logical ID
     *
     * @param logicalId the Logical ID of the Resource to find
     * @returns the Stack Name that the Logical ID is found to be in
     */
    getStackNameFromLogicalId(logicalId) {
        const stackTypeKey = Object.keys(this.flatTemplates).filter((key) => key.indexOf(`${logicalId}.Type`) > -1)[0];
        const stackName = stackTypeKey.split('.')[0];
        return stackName;
    }
    /**
     * Retrieve a Resource Type from a given Logical ID
     *
     * @param logicalId the logical ID of the Resource to find
     * @returns the Resource Type of the provided Logical ID
     */
    getResourceTypeFromLogicalId(logicalId) {
        const typeKey = Object.keys(this.flatTemplates).filter((key) => key.indexOf(`${logicalId}.Type`) > -1)[0];
        const resourceType = this.flatTemplates[typeKey];
        return resourceType;
    }
    /**
     * Performs a Describe Stack API call with the AWS SDK to determine what
     * the CloudFormation Exports are for a given Stack Name.
     *
     * @param stackName the CloudFormation stack name to query against
     * @param region the AWS region to target
     * @returns CloudFormation Stack object
     */
    describeStack(stackName, region) {
        let stack;
        try {
            const output = cp.spawnSync('node', [
                '-e',
                `
            const sdk = require('aws-sdk');
            const cfn = new sdk.CloudFormation({
              apiVersion: '2016-11-15',
              region: '${region}'
            });
            cfn.describeStacks({StackName: '${stackName}'})
                .promise()
                .then((data) => {
                  const j = JSON.stringify(data);
                  console.log(j);
                }
            );
          `,
            ], { encoding: 'utf8' });
            const response = JSON.parse(output.stdout);
            const stacks = response.Stacks;
            stack = stacks[0];
        }
        catch { }
        return stack;
    }
    /**
     * Builds an Export lookup table from the provided AWS SDK CloudFormation
     * stack object. This will be in the form of `{ 'MyExport': 'foobar' }`.
     *
     * @param stack
     * @returns
     */
    createExportMap(stack) {
        const exports = {};
        const outputs = stack.Outputs ?? [];
        for (const output of outputs) {
            if (output.ExportName && output.OutputValue) {
                exports[output.ExportName] = output.OutputValue;
            }
        }
        return exports;
    }
}
exports.CfnStore = CfnStore;
//# sourceMappingURL=data:application/json;base64,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