"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceExtractor = exports.ResourceExtractorShareMethod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const cfnStore_1 = require("./cfnStore");
const flattener_1 = require("./flattener");
const resourceTransformer_1 = require("./resourceTransformer");
/**
 * The available value sharing methods to pass values from the extracted stack
 * onto the original stack(s).
 */
var ResourceExtractorShareMethod;
(function (ResourceExtractorShareMethod) {
    ResourceExtractorShareMethod["CFN_OUTPUT"] = "CFN_OUTPUT";
    ResourceExtractorShareMethod["SSM_PARAMETER"] = "SSM_PARAMETER";
    ResourceExtractorShareMethod["API_LOOKUP"] = "API_LOOKUP";
})(ResourceExtractorShareMethod = exports.ResourceExtractorShareMethod || (exports.ResourceExtractorShareMethod = {}));
/**
 * This Aspect takes a CDK application, all synthesized CloudFormationStackArtifact,
 * a value share method, and a list of Cloudformation resources that should be
 * pulled out of the main CDK application, which should be synthesized to a
 * cloudformation template that an external team (e.g. security team) to deploy,
 * and adjusting the CDK application to reference pre-created resources already pulled out
 *
 * @example
    const app = App()
    const stack = new Stack(app, 'MyStack');
    extractedStack = new Stack(app, 'ExtractedStack');
    const synthedApp = app.synth();

    Aspects.of(app).add(new ResourceExtractor({
      extractDestinationStack: extractedStack,
      stackArtifacts: synthedApp.stacks,
      valueShareMethod: ResourceExtractorShareMethod.CFN_OUTPUT,
      resourceTypesToExtract: [
        'AWS::IAM::Role',
        'AWS::IAM::Policy',
        'AWS::IAM::ManagedPolicy',
        'AWS::IAM::InstanceProfile',
      ],
    });
    app.synth({ force: true });
 */
class ResourceExtractor {
    constructor(props) {
        this.valueShareMethod = ResourceExtractorShareMethod.CFN_OUTPUT;
        /** Save props */
        this.extractDestinationStack = props.extractDestinationStack;
        this.resourceTypesToExtract = props.resourceTypesToExtract;
        if (props.valueShareMethod) {
            this.valueShareMethod = props.valueShareMethod;
        }
        aws_cdk_lib_1.Annotations.of(this.extractDestinationStack).addWarning('❗ ResourceExtractor is in experimental mode. \
      Please be sure to validate synthesized assets prior to deploy. \
      Please open any issues found at https://github.com/cdklabs/cdk-enterprise-iac/issues');
        /** Initialize CfnStore to save templates and mappings */
        this.cfn = new cfnStore_1.CfnStore({
            extractedStackName: props.extractDestinationStack.stackName,
            region: props.extractDestinationStack.region,
            stackArtifacts: props.stackArtifacts,
            valueShareMethod: this.valueShareMethod,
        });
        /** Initialize resource transformer */
        this.resourceTransformer = new resourceTransformer_1.ResourceTransformer({
            cfnStore: this.cfn,
            additionalTransforms: props.additionalTransforms,
        });
        /** Save resources that are not getting extracted */
        this.logicalIdsNotGettingExtracted =
            this.getLogicalIdsNotGettingExtracted();
    }
    /** Entrypoint */
    visit(node) {
        // Ignore the extracted stack
        const anyNode = node;
        if ('stack' in anyNode &&
            anyNode.stack.toString() === this.extractDestinationStack.stackName) {
            return;
        }
        // Process all resources
        if (node instanceof aws_cdk_lib_1.CfnResource) {
            if (this.resourceTypesToExtract.includes(node.cfnResourceType)) {
                this.processCfnResource(node);
            }
        }
    }
    /**
     * Takes in a CfnResource object and processes it by recreating it in
     * the Security Stack and deleting it from the stack in which it came
     * from.
     *
     * @param node the CfnResource to process
     */
    processCfnResource(node) {
        const stackName = node.stack.stackName;
        const logicalId = node.stack.resolve(node.logicalId);
        const props = this.cfn.templates[stackName].Resources[logicalId].Properties;
        /** Recreate resource in extracted stack and export value */
        if (!this.extractDestinationStack.node.tryFindChild(node.stack.resolve(logicalId))) {
            // Check if props include references to resources that _aren't_ being
            // extracted
            const flattenedProps = flattener_1.Flattener.flattenObject(props);
            const propsToAdjust = Object.keys(flattenedProps).filter((key) => Object.keys(this.logicalIdsNotGettingExtracted).includes(flattenedProps[key]));
            // If properties being extracted to another stack contain
            // references to resources in the original stack, we need to adjust
            let modifiedProps = {};
            if (propsToAdjust) {
                modifiedProps = this.modifyExtractedResourceProperties({
                    props,
                    propsToAdjust,
                });
            }
            // Add to extracted stack
            const res = new aws_cdk_lib_1.CfnResource(this.extractDestinationStack, logicalId, {
                type: node.cfnResourceType,
                properties: modifiedProps || props,
            });
            // Look in all existing stacks for any refs to the resource being
            // extracted
            const foundRefs = Object.keys(this.cfn.flatTemplates).filter((key) => this.cfn.flatTemplates[key] === logicalId);
            // loop through resources in stack with Cfn intrinsic functions
            // referencing extracted resources
            for (const foundRef of foundRefs) {
                // See if the found ref is also getting extracted. Ignore if so.
                if (this.isRefAlsoGettingExtracted(foundRef))
                    continue;
                // Get the CfnResource that is referencing this extracted resource
                const foundRefNode = this.getResourceFromFoundRef(node, foundRef);
                // Figure out the pattern of how extracted resource is being referenced
                // e.g. using `Fn::GetAtt` or `Ref`
                const exportValue = this.cfn.determineExportValue(node, foundRef);
                if (exportValue) {
                    // Generate export in ExportedStack, and return the `Fn::ImportValue`
                    // method to use when referencing in the App stack
                    const importValue = this.exportValue(node.stack, this.extractDestinationStack.resolve(res.logicalId), exportValue);
                    // Override any ref to extracted resource
                    this.overrideFoundRefWithImportValue({
                        foundRefNode,
                        importValue,
                        flattenedKey: foundRef,
                    });
                }
                // Remove any DependsOn references
                this.deletionOverrideDependsOn({
                    foundRefNode,
                    flattenedKey: foundRef,
                });
            }
        }
        /** Delete from original stack */
        const removed = node.stack.node.tryRemoveChild(node.node.id);
        if (!removed) {
            const parent = node.node.scope?.node;
            parent?.tryRemoveChild(node.node.id);
        }
    }
    /**
     * This will adjust properties in extracted resources that have
     * CloudFormation intrinsic functions referencing resources in the original
     * stack.
     *
     * Most commonly found in IAM policies that scope IAM actions to specific
     * resources in the Stack the policy is being extracted from. In this case
     * it will take a look at the resource type and transform into a partial ARN.
     *
     * @param props modifyExtractedResourcePropertiesProps
     * @returns Json of modified properties, including partial wildcard matchers
     */
    modifyExtractedResourceProperties(props) {
        let modifiedProps = props.props;
        for (const key in props.propsToAdjust) {
            if (Object.prototype.hasOwnProperty.call(props.propsToAdjust, key)) {
                const resourceLogicalIdToReplace = flattener_1.Flattener.getValueByPath(props.props, props.propsToAdjust[key]);
                const partial = this.resourceTransformer.toPartial(resourceLogicalIdToReplace);
                const splitKey = props.propsToAdjust[key].split('.');
                if (splitKey.slice(-1)[0] == 'Ref') {
                    flattener_1.Flattener.setToValue(modifiedProps, splitKey.slice(0, -1).join('.'), partial);
                }
                else if (splitKey.slice(-2)[0] == 'Fn::GetAtt') {
                    flattener_1.Flattener.setToValue(modifiedProps, splitKey.slice(0, -2).join('.'), partial);
                }
            }
        }
        return modifiedProps;
    }
    /**
     * This takes the flattened key of the resource that's referencing an
     * extracted resource via Cfn intrinsic function, the CfnResource itself
     * (e.g. CfnFunction) as well as the import value (from
     * determineExportValue), and adds a Property override.
     *
     * @param props overrideFoundRefWithImportValueProps
     */
    overrideFoundRefWithImportValue(props) {
        // find property to override
        const splitKey = props.flattenedKey.split('.');
        let propertyOverridePath;
        if (splitKey.slice(-1)[0] == 'Ref') {
            propertyOverridePath = splitKey.slice(4, -1).join('.');
        }
        else if (splitKey.slice(-2)[0] == 'Fn::GetAtt') {
            propertyOverridePath = splitKey.slice(4, -2).join('.');
        }
        else {
            propertyOverridePath = 'notFound';
        }
        if (this.valueShareMethod == ResourceExtractorShareMethod.CFN_OUTPUT) {
            props.foundRefNode.addPropertyOverride(propertyOverridePath, props.foundRefNode.stack.resolve(props.importValue));
        }
        else if (this.valueShareMethod == ResourceExtractorShareMethod.SSM_PARAMETER) {
            props.foundRefNode.addPropertyOverride(propertyOverridePath, aws_ssm_1.StringParameter.valueFromLookup(props.foundRefNode.stack, props.importValue));
        }
        else if (this.valueShareMethod == ResourceExtractorShareMethod.API_LOOKUP) {
            const importValue = props.foundRefNode.stack.resolve(props.importValue)['Fn::ImportValue'];
            props.foundRefNode.addPropertyOverride(propertyOverridePath, this.cfn.extractedStackExports[importValue] ||
                `dummy-value-for-${importValue}`);
        }
    }
    /**
     * Remove a `DependsOn` reference when not needed
     *
     * @param props
     */
    deletionOverrideDependsOn(props) {
        const splitKey = props.flattenedKey.split('.');
        if (splitKey.slice(-2)[0] == 'DependsOn') {
            props.foundRefNode.addDeletionOverride(`DependsOn.${splitKey.slice(-1)[0]}`);
        }
    }
    /**
     * Finds the CDK resource for a given flattened key by looking it up in the
     * context of the provided node's stack.
     *
     * @param node
     * @param flattendKey
     * @returns CfnResource
     */
    getResourceFromFoundRef(node, flattendKey) {
        const foundRefLogicalId = flattendKey.split('.')[2];
        return node.stack.node.findAll().find((x) => {
            if ('stack' in x && 'logicalId' in x) {
                return x.stack.resolve(x.logicalId) == foundRefLogicalId;
            }
            else {
                return undefined;
            }
        });
    }
    /**
     * Given a flattened key, determine if the resouce is also getting
     * extracted so we know whether or not to adjust CloudFormation Intrinsic
     * functions.
     *
     * @param flattendKey
     * @returns boolean
     */
    isRefAlsoGettingExtracted(flattendKey) {
        const splitKey = flattendKey.split('.');
        const pathToType = flattener_1.Flattener.getValueByPath(this.cfn.templates, splitKey.slice(0, 3).concat(['Type']).join('.'));
        return this.resourceTypesToExtract.includes(pathToType);
    }
    /**
     * Get a list of logical ids of resources _not_ getting exported.
     *
     * @returns flat json in the form of `{ LogicalId: 'ResourceType' }`
     */
    getLogicalIdsNotGettingExtracted() {
        const logicalIds = {};
        for (const key in this.cfn.flatTemplates) {
            if (Object.prototype.hasOwnProperty.call(this.cfn.flatTemplates, key)) {
                const splitKey = key.split('.');
                if (splitKey.slice(-1)[0] == 'Type' &&
                    splitKey.slice(1, 2)[0] == 'Resources') {
                    if (!this.resourceTypesToExtract.includes(this.cfn.flatTemplates[key])) {
                        logicalIds[splitKey.slice(-2)[0]] = this.cfn.flatTemplates[key];
                    }
                }
            }
        }
        return logicalIds;
    }
    /**
     * Exports a value using a consistent standard for the different value
     * share methods.
     *
     * @param stack - the CDK `Stack` object for the resource to import
     * @param name - the name of the export, normally the LogicalId of the
     * resource being exported
     * @param value - the value to export
     */
    exportValue(stack, name, value) {
        const stackName = stack.stackName;
        const shareName = `${stackName}:${name}`;
        if (this.valueShareMethod == ResourceExtractorShareMethod.SSM_PARAMETER) {
            const paramName = `/${shareName.replace(':', '/')}`;
            const paramLogicalId = `p${name}`;
            if (!this.extractDestinationStack.node.tryFindChild(paramLogicalId)) {
                new aws_cdk_lib_1.CfnResource(this.extractDestinationStack, paramLogicalId, {
                    type: 'AWS::SSM::Parameter',
                    properties: {
                        Name: paramName,
                        Type: 'String',
                        Value: stack.resolve(value),
                    },
                });
            }
            return paramName;
        }
        else {
            // CFN_OUTPUT and API_LOOKUP share the same method of exporting values
            if (!this.extractDestinationStack.node.tryFindChild(`Export${stackName}:${name}`)) {
                return this.extractDestinationStack.exportValue(stack.resolve(value), {
                    name: shareName,
                });
            }
            else {
                return aws_cdk_lib_1.Fn.importValue(shareName);
            }
        }
    }
}
exports.ResourceExtractor = ResourceExtractor;
_a = JSII_RTTI_SYMBOL_1;
ResourceExtractor[_a] = { fqn: "@cdklabs/cdk-enterprise-iac.ResourceExtractor", version: "0.0.79" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb3VyY2VFeHRyYWN0b3IuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvcGF0Y2hlcy9yZXNvdXJjZS1leHRyYWN0b3IvcmVzb3VyY2VFeHRyYWN0b3IudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7O0VBR0U7QUFDRiw2Q0FBMkU7QUFDM0UsaURBQXNEO0FBR3RELHlDQUFzQztBQUN0QywyQ0FBd0M7QUFDeEMsK0RBQTREO0FBbUI1RDs7O0dBR0c7QUFDSCxJQUFZLDRCQUlYO0FBSkQsV0FBWSw0QkFBNEI7SUFDdEMseURBQTJCLENBQUE7SUFDM0IsK0RBQWlDLENBQUE7SUFDakMseURBQTJCLENBQUE7QUFDN0IsQ0FBQyxFQUpXLDRCQUE0QixHQUE1QixvQ0FBNEIsS0FBNUIsb0NBQTRCLFFBSXZDO0FBbUJEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBeUJHO0FBQ0gsTUFBYSxpQkFBaUI7SUFTNUIsWUFBWSxLQUE2QjtRQU54QixxQkFBZ0IsR0FDL0IsNEJBQTRCLENBQUMsVUFBVSxDQUFDO1FBTXhDLGlCQUFpQjtRQUNqQixJQUFJLENBQUMsdUJBQXVCLEdBQUcsS0FBSyxDQUFDLHVCQUF1QixDQUFDO1FBQzdELElBQUksQ0FBQyxzQkFBc0IsR0FBRyxLQUFLLENBQUMsc0JBQXNCLENBQUM7UUFDM0QsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDMUIsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztTQUNoRDtRQUNELHlCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDLFVBQVUsQ0FDckQ7OzJGQUVxRixDQUN0RixDQUFDO1FBRUYseURBQXlEO1FBQ3pELElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxtQkFBUSxDQUFDO1lBQ3RCLGtCQUFrQixFQUFFLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxTQUFTO1lBQzNELE1BQU0sRUFBRSxLQUFLLENBQUMsdUJBQXVCLENBQUMsTUFBTTtZQUM1QyxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLGdCQUFnQjtTQUN4QyxDQUFDLENBQUM7UUFFSCxzQ0FBc0M7UUFDdEMsSUFBSSxDQUFDLG1CQUFtQixHQUFHLElBQUkseUNBQW1CLENBQUM7WUFDakQsUUFBUSxFQUFFLElBQUksQ0FBQyxHQUFHO1lBQ2xCLG9CQUFvQixFQUFFLEtBQUssQ0FBQyxvQkFBb0I7U0FDakQsQ0FBQyxDQUFDO1FBRUgsb0RBQW9EO1FBQ3BELElBQUksQ0FBQyw2QkFBNkI7WUFDaEMsSUFBSSxDQUFDLGdDQUFnQyxFQUFFLENBQUM7SUFDNUMsQ0FBQztJQUVELGlCQUFpQjtJQUNWLEtBQUssQ0FBQyxJQUFnQjtRQUMzQiw2QkFBNkI7UUFDN0IsTUFBTSxPQUFPLEdBQUcsSUFBVyxDQUFDO1FBQzVCLElBQ0UsT0FBTyxJQUFJLE9BQU87WUFDbEIsT0FBTyxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsS0FBSyxJQUFJLENBQUMsdUJBQXVCLENBQUMsU0FBUyxFQUNuRTtZQUNBLE9BQU87U0FDUjtRQUVELHdCQUF3QjtRQUN4QixJQUFJLElBQUksWUFBWSx5QkFBVyxFQUFFO1lBQy9CLElBQUksSUFBSSxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLEVBQUU7Z0JBQzlELElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUMvQjtTQUNGO0lBQ0gsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNLLGtCQUFrQixDQUFDLElBQWlCO1FBQzFDLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ3ZDLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNyRCxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUMsVUFBVSxDQUFDO1FBRTVFLDREQUE0RDtRQUM1RCxJQUNFLENBQUMsSUFBSSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxZQUFZLENBQzdDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUM5QixFQUNEO1lBQ0EscUVBQXFFO1lBQ3JFLFlBQVk7WUFDWixNQUFNLGNBQWMsR0FBRyxxQkFBUyxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUN0RCxNQUFNLGFBQWEsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQy9ELE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLDZCQUE2QixDQUFDLENBQUMsUUFBUSxDQUN0RCxjQUFjLENBQUMsR0FBRyxDQUFDLENBQ3BCLENBQ0YsQ0FBQztZQUVGLHlEQUF5RDtZQUN6RCxtRUFBbUU7WUFDbkUsSUFBSSxhQUFhLEdBQVMsRUFBRSxDQUFDO1lBQzdCLElBQUksYUFBYSxFQUFFO2dCQUNqQixhQUFhLEdBQUcsSUFBSSxDQUFDLGlDQUFpQyxDQUFDO29CQUNyRCxLQUFLO29CQUNMLGFBQWE7aUJBQ2QsQ0FBQyxDQUFDO2FBQ0o7WUFFRCx5QkFBeUI7WUFDekIsTUFBTSxHQUFHLEdBQUcsSUFBSSx5QkFBVyxDQUFDLElBQUksQ0FBQyx1QkFBdUIsRUFBRSxTQUFTLEVBQUU7Z0JBQ25FLElBQUksRUFBRSxJQUFJLENBQUMsZUFBZTtnQkFDMUIsVUFBVSxFQUFFLGFBQWEsSUFBSSxLQUFLO2FBQ25DLENBQUMsQ0FBQztZQUVILGlFQUFpRTtZQUNqRSxZQUFZO1lBQ1osTUFBTSxTQUFTLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxDQUFDLE1BQU0sQ0FDMUQsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxLQUFLLFNBQVMsQ0FDbkQsQ0FBQztZQUVGLCtEQUErRDtZQUMvRCxrQ0FBa0M7WUFDbEMsS0FBSyxNQUFNLFFBQVEsSUFBSSxTQUFTLEVBQUU7Z0JBQ2hDLGdFQUFnRTtnQkFDaEUsSUFBSSxJQUFJLENBQUMseUJBQXlCLENBQUMsUUFBUSxDQUFDO29CQUFFLFNBQVM7Z0JBRXZELGtFQUFrRTtnQkFDbEUsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUMvQyxJQUFJLEVBQ0osUUFBUSxDQUNNLENBQUM7Z0JBRWpCLHVFQUF1RTtnQkFDdkUsbUNBQW1DO2dCQUNuQyxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsQ0FBQztnQkFDbEUsSUFBSSxXQUFXLEVBQUU7b0JBQ2YscUVBQXFFO29CQUNyRSxrREFBa0Q7b0JBQ2xELE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQ2xDLElBQUksQ0FBQyxLQUFLLEVBQ1YsSUFBSSxDQUFDLHVCQUF1QixDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDLEVBQ25ELFdBQVcsQ0FDWixDQUFDO29CQUVGLHlDQUF5QztvQkFDekMsSUFBSSxDQUFDLCtCQUErQixDQUFDO3dCQUNuQyxZQUFZO3dCQUNaLFdBQVc7d0JBQ1gsWUFBWSxFQUFFLFFBQVE7cUJBQ3ZCLENBQUMsQ0FBQztpQkFDSjtnQkFFRCxrQ0FBa0M7Z0JBQ2xDLElBQUksQ0FBQyx5QkFBeUIsQ0FBQztvQkFDN0IsWUFBWTtvQkFDWixZQUFZLEVBQUUsUUFBUTtpQkFDdkIsQ0FBQyxDQUFDO2FBQ0o7U0FDRjtRQUVELGlDQUFpQztRQUNqQyxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUM3RCxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQ1osTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDO1lBQ3JDLE1BQU0sRUFBRSxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztTQUN0QztJQUNILENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNLLGlDQUFpQyxDQUN2QyxLQUE2QztRQUU3QyxJQUFJLGFBQWEsR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDO1FBQ2hDLEtBQUssTUFBTSxHQUFHLElBQUksS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUNyQyxJQUFJLE1BQU0sQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxFQUFFLEdBQUcsQ0FBQyxFQUFFO2dCQUNsRSxNQUFNLDBCQUEwQixHQUFHLHFCQUFTLENBQUMsY0FBYyxDQUN6RCxLQUFLLENBQUMsS0FBSyxFQUNYLEtBQUssQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLENBQ3pCLENBQUM7Z0JBQ0YsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFNBQVMsQ0FDaEQsMEJBQTBCLENBQzNCLENBQUM7Z0JBRUYsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQ3JELElBQUksUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLEtBQUssRUFBRTtvQkFDbEMscUJBQVMsQ0FBQyxVQUFVLENBQ2xCLGFBQWEsRUFDYixRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFDL0IsT0FBTyxDQUNSLENBQUM7aUJBQ0g7cUJBQU0sSUFBSSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksWUFBWSxFQUFFO29CQUNoRCxxQkFBUyxDQUFDLFVBQVUsQ0FDbEIsYUFBYSxFQUNiLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUMvQixPQUFPLENBQ1IsQ0FBQztpQkFDSDthQUNGO1NBQ0Y7UUFDRCxPQUFPLGFBQWEsQ0FBQztJQUN2QixDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNLLCtCQUErQixDQUNyQyxLQUEyQztRQUUzQyw0QkFBNEI7UUFDNUIsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDL0MsSUFBSSxvQkFBNEIsQ0FBQztRQUNqQyxJQUFJLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxLQUFLLEVBQUU7WUFDbEMsb0JBQW9CLEdBQUcsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDeEQ7YUFBTSxJQUFJLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxZQUFZLEVBQUU7WUFDaEQsb0JBQW9CLEdBQUcsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDeEQ7YUFBTTtZQUNMLG9CQUFvQixHQUFHLFVBQVUsQ0FBQztTQUNuQztRQUNELElBQUksSUFBSSxDQUFDLGdCQUFnQixJQUFJLDRCQUE0QixDQUFDLFVBQVUsRUFBRTtZQUNwRSxLQUFLLENBQUMsWUFBWSxDQUFDLG1CQUFtQixDQUNwQyxvQkFBb0IsRUFDcEIsS0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FDcEQsQ0FBQztTQUNIO2FBQU0sSUFDTCxJQUFJLENBQUMsZ0JBQWdCLElBQUksNEJBQTRCLENBQUMsYUFBYSxFQUNuRTtZQUNBLEtBQUssQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQ3BDLG9CQUFvQixFQUNwQix5QkFBZSxDQUFDLGVBQWUsQ0FDN0IsS0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQ3hCLEtBQUssQ0FBQyxXQUFXLENBQ2xCLENBQ0YsQ0FBQztTQUNIO2FBQU0sSUFDTCxJQUFJLENBQUMsZ0JBQWdCLElBQUksNEJBQTRCLENBQUMsVUFBVSxFQUNoRTtZQUNBLE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQ3JFLGlCQUFpQixDQUNsQixDQUFDO1lBQ0YsS0FBSyxDQUFDLFlBQVksQ0FBQyxtQkFBbUIsQ0FDcEMsb0JBQW9CLEVBQ3BCLElBQUksQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsV0FBVyxDQUFDO2dCQUN6QyxtQkFBbUIsV0FBVyxFQUFFLENBQ25DLENBQUM7U0FDSDtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0sseUJBQXlCLENBQy9CLEtBQXFDO1FBRXJDLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQy9DLElBQUksUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLFdBQVcsRUFBRTtZQUN4QyxLQUFLLENBQUMsWUFBWSxDQUFDLG1CQUFtQixDQUNwQyxhQUFhLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUNyQyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNLLHVCQUF1QixDQUM3QixJQUFpQixFQUNqQixXQUFtQjtRQUVuQixNQUFNLGlCQUFpQixHQUFHLFdBQVcsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDcEQsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFNLEVBQUUsRUFBRTtZQUMvQyxJQUFJLE9BQU8sSUFBSSxDQUFDLElBQUksV0FBVyxJQUFJLENBQUMsRUFBRTtnQkFDcEMsT0FBTyxDQUFDLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLElBQUksaUJBQWlCLENBQUM7YUFDMUQ7aUJBQU07Z0JBQ0wsT0FBTyxTQUFTLENBQUM7YUFDbEI7UUFDSCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7OztPQU9HO0lBQ0sseUJBQXlCLENBQUMsV0FBbUI7UUFDbkQsTUFBTSxRQUFRLEdBQUcsV0FBVyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUN4QyxNQUFNLFVBQVUsR0FBRyxxQkFBUyxDQUFDLGNBQWMsQ0FDekMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxTQUFTLEVBQ2xCLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUNoRCxDQUFDO1FBQ0YsT0FBTyxJQUFJLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssZ0NBQWdDO1FBQ3RDLE1BQU0sVUFBVSxHQUFhLEVBQUUsQ0FBQztRQUNoQyxLQUFLLE1BQU0sR0FBRyxJQUFJLElBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxFQUFFO1lBQ3hDLElBQUksTUFBTSxDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxFQUFFLEdBQUcsQ0FBQyxFQUFFO2dCQUNyRSxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO2dCQUNoQyxJQUNFLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxNQUFNO29CQUMvQixRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxXQUFXLEVBQ3RDO29CQUNBLElBQ0UsQ0FBQyxJQUFJLENBQUMsc0JBQXNCLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQ2xFO3dCQUNBLFVBQVUsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQztxQkFDakU7aUJBQ0Y7YUFDRjtTQUNGO1FBQ0QsT0FBTyxVQUFVLENBQUM7SUFDcEIsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ssV0FBVyxDQUFDLEtBQVksRUFBRSxJQUFZLEVBQUUsS0FBVTtRQUN4RCxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ2xDLE1BQU0sU0FBUyxHQUFHLEdBQUcsU0FBUyxJQUFJLElBQUksRUFBRSxDQUFDO1FBRXpDLElBQUksSUFBSSxDQUFDLGdCQUFnQixJQUFJLDRCQUE0QixDQUFDLGFBQWEsRUFBRTtZQUN2RSxNQUFNLFNBQVMsR0FBRyxJQUFJLFNBQVMsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUM7WUFDcEQsTUFBTSxjQUFjLEdBQUcsSUFBSSxJQUFJLEVBQUUsQ0FBQztZQUNsQyxJQUFJLENBQUMsSUFBSSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLEVBQUU7Z0JBQ25FLElBQUkseUJBQVcsQ0FBQyxJQUFJLENBQUMsdUJBQXVCLEVBQUUsY0FBYyxFQUFFO29CQUM1RCxJQUFJLEVBQUUscUJBQXFCO29CQUMzQixVQUFVLEVBQUU7d0JBQ1YsSUFBSSxFQUFFLFNBQVM7d0JBQ2YsSUFBSSxFQUFFLFFBQVE7d0JBQ2QsS0FBSyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDO3FCQUM1QjtpQkFDRixDQUFDLENBQUM7YUFDSjtZQUNELE9BQU8sU0FBUyxDQUFDO1NBQ2xCO2FBQU07WUFDTCxzRUFBc0U7WUFDdEUsSUFDRSxDQUFDLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUM3QyxTQUFTLFNBQVMsSUFBSSxJQUFJLEVBQUUsQ0FDN0IsRUFDRDtnQkFDQSxPQUFPLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRTtvQkFDcEUsSUFBSSxFQUFFLFNBQVM7aUJBQ2hCLENBQUMsQ0FBQzthQUNKO2lCQUFNO2dCQUNMLE9BQU8sZ0JBQUUsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLENBQUM7YUFDbEM7U0FDRjtJQUNILENBQUM7O0FBclhILDhDQXNYQyIsInNvdXJjZXNDb250ZW50IjpbIi8qXG5Db3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4qL1xuaW1wb3J0IHsgQW5ub3RhdGlvbnMsIENmblJlc291cmNlLCBGbiwgSUFzcGVjdCwgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBTdHJpbmdQYXJhbWV0ZXIgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3NtJztcbmltcG9ydCB7IENsb3VkRm9ybWF0aW9uU3RhY2tBcnRpZmFjdCB9IGZyb20gJ2F3cy1jZGstbGliL2N4LWFwaSc7XG5pbXBvcnQgeyBJQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDZm5TdG9yZSB9IGZyb20gJy4vY2ZuU3RvcmUnO1xuaW1wb3J0IHsgRmxhdHRlbmVyIH0gZnJvbSAnLi9mbGF0dGVuZXInO1xuaW1wb3J0IHsgUmVzb3VyY2VUcmFuc2Zvcm1lciB9IGZyb20gJy4vcmVzb3VyY2VUcmFuc2Zvcm1lcic7XG5pbXBvcnQgeyBGbGF0SnNvbiwgSnNvbiB9IGZyb20gJy4vdHlwZXMnO1xuXG5pbnRlcmZhY2Ugb3ZlcnJpZGVGb3VuZFJlZldpdGhJbXBvcnRWYWx1ZVByb3BzIHtcbiAgcmVhZG9ubHkgZm91bmRSZWZOb2RlOiBDZm5SZXNvdXJjZTtcbiAgcmVhZG9ubHkgaW1wb3J0VmFsdWU6IHN0cmluZztcbiAgcmVhZG9ubHkgZmxhdHRlbmVkS2V5OiBzdHJpbmc7XG59XG5cbmludGVyZmFjZSBkZWxldGlvbk92ZXJyaWRlRGVwZW5kc09uUHJvcHMge1xuICByZWFkb25seSBmb3VuZFJlZk5vZGU6IENmblJlc291cmNlO1xuICByZWFkb25seSBmbGF0dGVuZWRLZXk6IHN0cmluZztcbn1cblxuaW50ZXJmYWNlIG1vZGlmeUV4dHJhY3RlZFJlc291cmNlUHJvcGVydGllc1Byb3BzIHtcbiAgcmVhZG9ubHkgcHJvcHM6IEpzb247XG4gIHJlYWRvbmx5IHByb3BzVG9BZGp1c3Q6IHN0cmluZ1tdO1xufVxuXG4vKipcbiAqIFRoZSBhdmFpbGFibGUgdmFsdWUgc2hhcmluZyBtZXRob2RzIHRvIHBhc3MgdmFsdWVzIGZyb20gdGhlIGV4dHJhY3RlZCBzdGFja1xuICogb250byB0aGUgb3JpZ2luYWwgc3RhY2socykuXG4gKi9cbmV4cG9ydCBlbnVtIFJlc291cmNlRXh0cmFjdG9yU2hhcmVNZXRob2Qge1xuICAnQ0ZOX09VVFBVVCcgPSAnQ0ZOX09VVFBVVCcsXG4gICdTU01fUEFSQU1FVEVSJyA9ICdTU01fUEFSQU1FVEVSJyxcbiAgJ0FQSV9MT09LVVAnID0gJ0FQSV9MT09LVVAnLFxufVxuXG5leHBvcnQgaW50ZXJmYWNlIFJlc291cmNlRXh0cmFjdG9yUHJvcHMge1xuICAvKiogU3RhY2sgdG8gbW92ZSBmb3VuZCBleHRyYWN0ZWQgcmVzb3VyY2VzIGludG8uICovXG4gIHJlYWRvbmx5IGV4dHJhY3REZXN0aW5hdGlvblN0YWNrOiBTdGFjaztcblxuICAvKiogU3ludGhlZCBzdGFjayBhcnRpZmFjdHMgZnJvbSB5b3VyIENESyBhcHAuICovXG4gIHJlYWRvbmx5IHN0YWNrQXJ0aWZhY3RzOiBDbG91ZEZvcm1hdGlvblN0YWNrQXJ0aWZhY3RbXTtcblxuICAvKiogVGhlIHNoYXJpbmcgbWV0aG9kIHRvIHVzZSB3aGVuIHBhc3NpbmcgZXhwb3J0ZWQgcmVzb3VyY2VzIGZyb20gdGhlIFwiRXh0cmFjdGVkIFN0YWNrXCIgaW50byB0aGUgb3JpZ2luYWwgc3RhY2socykuICovXG4gIHJlYWRvbmx5IHZhbHVlU2hhcmVNZXRob2Q/OiBSZXNvdXJjZUV4dHJhY3RvclNoYXJlTWV0aG9kO1xuXG4gIC8qKiBMaXN0IG9mIHJlc291cmNlIHR5cGVzIHRvIGV4dHJhY3QsIGV4OiBgQVdTOjpJQU06OlJvbGVgLiAqL1xuICByZWFkb25seSByZXNvdXJjZVR5cGVzVG9FeHRyYWN0OiBzdHJpbmdbXTtcblxuICAvKiogQWRkaXRpb25hbCByZXNvdXJjZSB0cmFuc2Zvcm1hdGlvbnMuICovXG4gIHJlYWRvbmx5IGFkZGl0aW9uYWxUcmFuc2Zvcm1zPzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcbn1cblxuLyoqXG4gKiBUaGlzIEFzcGVjdCB0YWtlcyBhIENESyBhcHBsaWNhdGlvbiwgYWxsIHN5bnRoZXNpemVkIENsb3VkRm9ybWF0aW9uU3RhY2tBcnRpZmFjdCxcbiAqIGEgdmFsdWUgc2hhcmUgbWV0aG9kLCBhbmQgYSBsaXN0IG9mIENsb3VkZm9ybWF0aW9uIHJlc291cmNlcyB0aGF0IHNob3VsZCBiZVxuICogcHVsbGVkIG91dCBvZiB0aGUgbWFpbiBDREsgYXBwbGljYXRpb24sIHdoaWNoIHNob3VsZCBiZSBzeW50aGVzaXplZCB0byBhXG4gKiBjbG91ZGZvcm1hdGlvbiB0ZW1wbGF0ZSB0aGF0IGFuIGV4dGVybmFsIHRlYW0gKGUuZy4gc2VjdXJpdHkgdGVhbSkgdG8gZGVwbG95LFxuICogYW5kIGFkanVzdGluZyB0aGUgQ0RLIGFwcGxpY2F0aW9uIHRvIHJlZmVyZW5jZSBwcmUtY3JlYXRlZCByZXNvdXJjZXMgYWxyZWFkeSBwdWxsZWQgb3V0XG4gKlxuICogQGV4YW1wbGVcbiAgICBjb25zdCBhcHAgPSBBcHAoKVxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ015U3RhY2snKTtcbiAgICBleHRyYWN0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdFeHRyYWN0ZWRTdGFjaycpO1xuICAgIGNvbnN0IHN5bnRoZWRBcHAgPSBhcHAuc3ludGgoKTtcblxuICAgIEFzcGVjdHMub2YoYXBwKS5hZGQobmV3IFJlc291cmNlRXh0cmFjdG9yKHtcbiAgICAgIGV4dHJhY3REZXN0aW5hdGlvblN0YWNrOiBleHRyYWN0ZWRTdGFjayxcbiAgICAgIHN0YWNrQXJ0aWZhY3RzOiBzeW50aGVkQXBwLnN0YWNrcyxcbiAgICAgIHZhbHVlU2hhcmVNZXRob2Q6IFJlc291cmNlRXh0cmFjdG9yU2hhcmVNZXRob2QuQ0ZOX09VVFBVVCxcbiAgICAgIHJlc291cmNlVHlwZXNUb0V4dHJhY3Q6IFtcbiAgICAgICAgJ0FXUzo6SUFNOjpSb2xlJyxcbiAgICAgICAgJ0FXUzo6SUFNOjpQb2xpY3knLFxuICAgICAgICAnQVdTOjpJQU06Ok1hbmFnZWRQb2xpY3knLFxuICAgICAgICAnQVdTOjpJQU06Okluc3RhbmNlUHJvZmlsZScsXG4gICAgICBdLFxuICAgIH0pO1xuICAgIGFwcC5zeW50aCh7IGZvcmNlOiB0cnVlIH0pO1xuICovXG5leHBvcnQgY2xhc3MgUmVzb3VyY2VFeHRyYWN0b3IgaW1wbGVtZW50cyBJQXNwZWN0IHtcbiAgcHJpdmF0ZSByZWFkb25seSBleHRyYWN0RGVzdGluYXRpb25TdGFjazogU3RhY2s7XG4gIHByaXZhdGUgcmVhZG9ubHkgcmVzb3VyY2VUeXBlc1RvRXh0cmFjdDogc3RyaW5nW107XG4gIHByaXZhdGUgcmVhZG9ubHkgdmFsdWVTaGFyZU1ldGhvZDogUmVzb3VyY2VFeHRyYWN0b3JTaGFyZU1ldGhvZCA9XG4gICAgUmVzb3VyY2VFeHRyYWN0b3JTaGFyZU1ldGhvZC5DRk5fT1VUUFVUO1xuICBwcml2YXRlIHJlYWRvbmx5IGxvZ2ljYWxJZHNOb3RHZXR0aW5nRXh0cmFjdGVkOiBGbGF0SnNvbjtcbiAgcHJpdmF0ZSByZWFkb25seSBjZm46IENmblN0b3JlO1xuICBwcml2YXRlIHJlYWRvbmx5IHJlc291cmNlVHJhbnNmb3JtZXI6IFJlc291cmNlVHJhbnNmb3JtZXI7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IFJlc291cmNlRXh0cmFjdG9yUHJvcHMpIHtcbiAgICAvKiogU2F2ZSBwcm9wcyAqL1xuICAgIHRoaXMuZXh0cmFjdERlc3RpbmF0aW9uU3RhY2sgPSBwcm9wcy5leHRyYWN0RGVzdGluYXRpb25TdGFjaztcbiAgICB0aGlzLnJlc291cmNlVHlwZXNUb0V4dHJhY3QgPSBwcm9wcy5yZXNvdXJjZVR5cGVzVG9FeHRyYWN0O1xuICAgIGlmIChwcm9wcy52YWx1ZVNoYXJlTWV0aG9kKSB7XG4gICAgICB0aGlzLnZhbHVlU2hhcmVNZXRob2QgPSBwcm9wcy52YWx1ZVNoYXJlTWV0aG9kO1xuICAgIH1cbiAgICBBbm5vdGF0aW9ucy5vZih0aGlzLmV4dHJhY3REZXN0aW5hdGlvblN0YWNrKS5hZGRXYXJuaW5nKFxuICAgICAgJ+KdlyBSZXNvdXJjZUV4dHJhY3RvciBpcyBpbiBleHBlcmltZW50YWwgbW9kZS4gXFxcbiAgICAgIFBsZWFzZSBiZSBzdXJlIHRvIHZhbGlkYXRlIHN5bnRoZXNpemVkIGFzc2V0cyBwcmlvciB0byBkZXBsb3kuIFxcXG4gICAgICBQbGVhc2Ugb3BlbiBhbnkgaXNzdWVzIGZvdW5kIGF0IGh0dHBzOi8vZ2l0aHViLmNvbS9jZGtsYWJzL2Nkay1lbnRlcnByaXNlLWlhYy9pc3N1ZXMnXG4gICAgKTtcblxuICAgIC8qKiBJbml0aWFsaXplIENmblN0b3JlIHRvIHNhdmUgdGVtcGxhdGVzIGFuZCBtYXBwaW5ncyAqL1xuICAgIHRoaXMuY2ZuID0gbmV3IENmblN0b3JlKHtcbiAgICAgIGV4dHJhY3RlZFN0YWNrTmFtZTogcHJvcHMuZXh0cmFjdERlc3RpbmF0aW9uU3RhY2suc3RhY2tOYW1lLFxuICAgICAgcmVnaW9uOiBwcm9wcy5leHRyYWN0RGVzdGluYXRpb25TdGFjay5yZWdpb24sXG4gICAgICBzdGFja0FydGlmYWN0czogcHJvcHMuc3RhY2tBcnRpZmFjdHMsXG4gICAgICB2YWx1ZVNoYXJlTWV0aG9kOiB0aGlzLnZhbHVlU2hhcmVNZXRob2QsXG4gICAgfSk7XG5cbiAgICAvKiogSW5pdGlhbGl6ZSByZXNvdXJjZSB0cmFuc2Zvcm1lciAqL1xuICAgIHRoaXMucmVzb3VyY2VUcmFuc2Zvcm1lciA9IG5ldyBSZXNvdXJjZVRyYW5zZm9ybWVyKHtcbiAgICAgIGNmblN0b3JlOiB0aGlzLmNmbixcbiAgICAgIGFkZGl0aW9uYWxUcmFuc2Zvcm1zOiBwcm9wcy5hZGRpdGlvbmFsVHJhbnNmb3JtcyxcbiAgICB9KTtcblxuICAgIC8qKiBTYXZlIHJlc291cmNlcyB0aGF0IGFyZSBub3QgZ2V0dGluZyBleHRyYWN0ZWQgKi9cbiAgICB0aGlzLmxvZ2ljYWxJZHNOb3RHZXR0aW5nRXh0cmFjdGVkID1cbiAgICAgIHRoaXMuZ2V0TG9naWNhbElkc05vdEdldHRpbmdFeHRyYWN0ZWQoKTtcbiAgfVxuXG4gIC8qKiBFbnRyeXBvaW50ICovXG4gIHB1YmxpYyB2aXNpdChub2RlOiBJQ29uc3RydWN0KTogdm9pZCB7XG4gICAgLy8gSWdub3JlIHRoZSBleHRyYWN0ZWQgc3RhY2tcbiAgICBjb25zdCBhbnlOb2RlID0gbm9kZSBhcyBhbnk7XG4gICAgaWYgKFxuICAgICAgJ3N0YWNrJyBpbiBhbnlOb2RlICYmXG4gICAgICBhbnlOb2RlLnN0YWNrLnRvU3RyaW5nKCkgPT09IHRoaXMuZXh0cmFjdERlc3RpbmF0aW9uU3RhY2suc3RhY2tOYW1lXG4gICAgKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgLy8gUHJvY2VzcyBhbGwgcmVzb3VyY2VzXG4gICAgaWYgKG5vZGUgaW5zdGFuY2VvZiBDZm5SZXNvdXJjZSkge1xuICAgICAgaWYgKHRoaXMucmVzb3VyY2VUeXBlc1RvRXh0cmFjdC5pbmNsdWRlcyhub2RlLmNmblJlc291cmNlVHlwZSkpIHtcbiAgICAgICAgdGhpcy5wcm9jZXNzQ2ZuUmVzb3VyY2Uobm9kZSk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFRha2VzIGluIGEgQ2ZuUmVzb3VyY2Ugb2JqZWN0IGFuZCBwcm9jZXNzZXMgaXQgYnkgcmVjcmVhdGluZyBpdCBpblxuICAgKiB0aGUgU2VjdXJpdHkgU3RhY2sgYW5kIGRlbGV0aW5nIGl0IGZyb20gdGhlIHN0YWNrIGluIHdoaWNoIGl0IGNhbWVcbiAgICogZnJvbS5cbiAgICpcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIHByb2Nlc3NcbiAgICovXG4gIHByaXZhdGUgcHJvY2Vzc0NmblJlc291cmNlKG5vZGU6IENmblJlc291cmNlKTogdm9pZCB7XG4gICAgY29uc3Qgc3RhY2tOYW1lID0gbm9kZS5zdGFjay5zdGFja05hbWU7XG4gICAgY29uc3QgbG9naWNhbElkID0gbm9kZS5zdGFjay5yZXNvbHZlKG5vZGUubG9naWNhbElkKTtcbiAgICBjb25zdCBwcm9wcyA9IHRoaXMuY2ZuLnRlbXBsYXRlc1tzdGFja05hbWVdLlJlc291cmNlc1tsb2dpY2FsSWRdLlByb3BlcnRpZXM7XG5cbiAgICAvKiogUmVjcmVhdGUgcmVzb3VyY2UgaW4gZXh0cmFjdGVkIHN0YWNrIGFuZCBleHBvcnQgdmFsdWUgKi9cbiAgICBpZiAoXG4gICAgICAhdGhpcy5leHRyYWN0RGVzdGluYXRpb25TdGFjay5ub2RlLnRyeUZpbmRDaGlsZChcbiAgICAgICAgbm9kZS5zdGFjay5yZXNvbHZlKGxvZ2ljYWxJZClcbiAgICAgIClcbiAgICApIHtcbiAgICAgIC8vIENoZWNrIGlmIHByb3BzIGluY2x1ZGUgcmVmZXJlbmNlcyB0byByZXNvdXJjZXMgdGhhdCBfYXJlbid0XyBiZWluZ1xuICAgICAgLy8gZXh0cmFjdGVkXG4gICAgICBjb25zdCBmbGF0dGVuZWRQcm9wcyA9IEZsYXR0ZW5lci5mbGF0dGVuT2JqZWN0KHByb3BzKTtcbiAgICAgIGNvbnN0IHByb3BzVG9BZGp1c3QgPSBPYmplY3Qua2V5cyhmbGF0dGVuZWRQcm9wcykuZmlsdGVyKChrZXkpID0+XG4gICAgICAgIE9iamVjdC5rZXlzKHRoaXMubG9naWNhbElkc05vdEdldHRpbmdFeHRyYWN0ZWQpLmluY2x1ZGVzKFxuICAgICAgICAgIGZsYXR0ZW5lZFByb3BzW2tleV1cbiAgICAgICAgKVxuICAgICAgKTtcblxuICAgICAgLy8gSWYgcHJvcGVydGllcyBiZWluZyBleHRyYWN0ZWQgdG8gYW5vdGhlciBzdGFjayBjb250YWluXG4gICAgICAvLyByZWZlcmVuY2VzIHRvIHJlc291cmNlcyBpbiB0aGUgb3JpZ2luYWwgc3RhY2ssIHdlIG5lZWQgdG8gYWRqdXN0XG4gICAgICBsZXQgbW9kaWZpZWRQcm9wczogSnNvbiA9IHt9O1xuICAgICAgaWYgKHByb3BzVG9BZGp1c3QpIHtcbiAgICAgICAgbW9kaWZpZWRQcm9wcyA9IHRoaXMubW9kaWZ5RXh0cmFjdGVkUmVzb3VyY2VQcm9wZXJ0aWVzKHtcbiAgICAgICAgICBwcm9wcyxcbiAgICAgICAgICBwcm9wc1RvQWRqdXN0LFxuICAgICAgICB9KTtcbiAgICAgIH1cblxuICAgICAgLy8gQWRkIHRvIGV4dHJhY3RlZCBzdGFja1xuICAgICAgY29uc3QgcmVzID0gbmV3IENmblJlc291cmNlKHRoaXMuZXh0cmFjdERlc3RpbmF0aW9uU3RhY2ssIGxvZ2ljYWxJZCwge1xuICAgICAgICB0eXBlOiBub2RlLmNmblJlc291cmNlVHlwZSxcbiAgICAgICAgcHJvcGVydGllczogbW9kaWZpZWRQcm9wcyB8fCBwcm9wcyxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBMb29rIGluIGFsbCBleGlzdGluZyBzdGFja3MgZm9yIGFueSByZWZzIHRvIHRoZSByZXNvdXJjZSBiZWluZ1xuICAgICAgLy8gZXh0cmFjdGVkXG4gICAgICBjb25zdCBmb3VuZFJlZnMgPSBPYmplY3Qua2V5cyh0aGlzLmNmbi5mbGF0VGVtcGxhdGVzKS5maWx0ZXIoXG4gICAgICAgIChrZXkpID0+IHRoaXMuY2ZuLmZsYXRUZW1wbGF0ZXNba2V5XSA9PT0gbG9naWNhbElkXG4gICAgICApO1xuXG4gICAgICAvLyBsb29wIHRocm91Z2ggcmVzb3VyY2VzIGluIHN0YWNrIHdpdGggQ2ZuIGludHJpbnNpYyBmdW5jdGlvbnNcbiAgICAgIC8vIHJlZmVyZW5jaW5nIGV4dHJhY3RlZCByZXNvdXJjZXNcbiAgICAgIGZvciAoY29uc3QgZm91bmRSZWYgb2YgZm91bmRSZWZzKSB7XG4gICAgICAgIC8vIFNlZSBpZiB0aGUgZm91bmQgcmVmIGlzIGFsc28gZ2V0dGluZyBleHRyYWN0ZWQuIElnbm9yZSBpZiBzby5cbiAgICAgICAgaWYgKHRoaXMuaXNSZWZBbHNvR2V0dGluZ0V4dHJhY3RlZChmb3VuZFJlZikpIGNvbnRpbnVlO1xuXG4gICAgICAgIC8vIEdldCB0aGUgQ2ZuUmVzb3VyY2UgdGhhdCBpcyByZWZlcmVuY2luZyB0aGlzIGV4dHJhY3RlZCByZXNvdXJjZVxuICAgICAgICBjb25zdCBmb3VuZFJlZk5vZGUgPSB0aGlzLmdldFJlc291cmNlRnJvbUZvdW5kUmVmKFxuICAgICAgICAgIG5vZGUsXG4gICAgICAgICAgZm91bmRSZWZcbiAgICAgICAgKSBhcyBDZm5SZXNvdXJjZTtcblxuICAgICAgICAvLyBGaWd1cmUgb3V0IHRoZSBwYXR0ZXJuIG9mIGhvdyBleHRyYWN0ZWQgcmVzb3VyY2UgaXMgYmVpbmcgcmVmZXJlbmNlZFxuICAgICAgICAvLyBlLmcuIHVzaW5nIGBGbjo6R2V0QXR0YCBvciBgUmVmYFxuICAgICAgICBjb25zdCBleHBvcnRWYWx1ZSA9IHRoaXMuY2ZuLmRldGVybWluZUV4cG9ydFZhbHVlKG5vZGUsIGZvdW5kUmVmKTtcbiAgICAgICAgaWYgKGV4cG9ydFZhbHVlKSB7XG4gICAgICAgICAgLy8gR2VuZXJhdGUgZXhwb3J0IGluIEV4cG9ydGVkU3RhY2ssIGFuZCByZXR1cm4gdGhlIGBGbjo6SW1wb3J0VmFsdWVgXG4gICAgICAgICAgLy8gbWV0aG9kIHRvIHVzZSB3aGVuIHJlZmVyZW5jaW5nIGluIHRoZSBBcHAgc3RhY2tcbiAgICAgICAgICBjb25zdCBpbXBvcnRWYWx1ZSA9IHRoaXMuZXhwb3J0VmFsdWUoXG4gICAgICAgICAgICBub2RlLnN0YWNrLFxuICAgICAgICAgICAgdGhpcy5leHRyYWN0RGVzdGluYXRpb25TdGFjay5yZXNvbHZlKHJlcy5sb2dpY2FsSWQpLFxuICAgICAgICAgICAgZXhwb3J0VmFsdWVcbiAgICAgICAgICApO1xuXG4gICAgICAgICAgLy8gT3ZlcnJpZGUgYW55IHJlZiB0byBleHRyYWN0ZWQgcmVzb3VyY2VcbiAgICAgICAgICB0aGlzLm92ZXJyaWRlRm91bmRSZWZXaXRoSW1wb3J0VmFsdWUoe1xuICAgICAgICAgICAgZm91bmRSZWZOb2RlLFxuICAgICAgICAgICAgaW1wb3J0VmFsdWUsXG4gICAgICAgICAgICBmbGF0dGVuZWRLZXk6IGZvdW5kUmVmLFxuICAgICAgICAgIH0pO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gUmVtb3ZlIGFueSBEZXBlbmRzT24gcmVmZXJlbmNlc1xuICAgICAgICB0aGlzLmRlbGV0aW9uT3ZlcnJpZGVEZXBlbmRzT24oe1xuICAgICAgICAgIGZvdW5kUmVmTm9kZSxcbiAgICAgICAgICBmbGF0dGVuZWRLZXk6IGZvdW5kUmVmLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICAvKiogRGVsZXRlIGZyb20gb3JpZ2luYWwgc3RhY2sgKi9cbiAgICBjb25zdCByZW1vdmVkID0gbm9kZS5zdGFjay5ub2RlLnRyeVJlbW92ZUNoaWxkKG5vZGUubm9kZS5pZCk7XG4gICAgaWYgKCFyZW1vdmVkKSB7XG4gICAgICBjb25zdCBwYXJlbnQgPSBub2RlLm5vZGUuc2NvcGU/Lm5vZGU7XG4gICAgICBwYXJlbnQ/LnRyeVJlbW92ZUNoaWxkKG5vZGUubm9kZS5pZCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFRoaXMgd2lsbCBhZGp1c3QgcHJvcGVydGllcyBpbiBleHRyYWN0ZWQgcmVzb3VyY2VzIHRoYXQgaGF2ZVxuICAgKiBDbG91ZEZvcm1hdGlvbiBpbnRyaW5zaWMgZnVuY3Rpb25zIHJlZmVyZW5jaW5nIHJlc291cmNlcyBpbiB0aGUgb3JpZ2luYWxcbiAgICogc3RhY2suXG4gICAqXG4gICAqIE1vc3QgY29tbW9ubHkgZm91bmQgaW4gSUFNIHBvbGljaWVzIHRoYXQgc2NvcGUgSUFNIGFjdGlvbnMgdG8gc3BlY2lmaWNcbiAgICogcmVzb3VyY2VzIGluIHRoZSBTdGFjayB0aGUgcG9saWN5IGlzIGJlaW5nIGV4dHJhY3RlZCBmcm9tLiBJbiB0aGlzIGNhc2VcbiAgICogaXQgd2lsbCB0YWtlIGEgbG9vayBhdCB0aGUgcmVzb3VyY2UgdHlwZSBhbmQgdHJhbnNmb3JtIGludG8gYSBwYXJ0aWFsIEFSTi5cbiAgICpcbiAgICogQHBhcmFtIHByb3BzIG1vZGlmeUV4dHJhY3RlZFJlc291cmNlUHJvcGVydGllc1Byb3BzXG4gICAqIEByZXR1cm5zIEpzb24gb2YgbW9kaWZpZWQgcHJvcGVydGllcywgaW5jbHVkaW5nIHBhcnRpYWwgd2lsZGNhcmQgbWF0Y2hlcnNcbiAgICovXG4gIHByaXZhdGUgbW9kaWZ5RXh0cmFjdGVkUmVzb3VyY2VQcm9wZXJ0aWVzKFxuICAgIHByb3BzOiBtb2RpZnlFeHRyYWN0ZWRSZXNvdXJjZVByb3BlcnRpZXNQcm9wc1xuICApOiBKc29uIHtcbiAgICBsZXQgbW9kaWZpZWRQcm9wcyA9IHByb3BzLnByb3BzO1xuICAgIGZvciAoY29uc3Qga2V5IGluIHByb3BzLnByb3BzVG9BZGp1c3QpIHtcbiAgICAgIGlmIChPYmplY3QucHJvdG90eXBlLmhhc093blByb3BlcnR5LmNhbGwocHJvcHMucHJvcHNUb0FkanVzdCwga2V5KSkge1xuICAgICAgICBjb25zdCByZXNvdXJjZUxvZ2ljYWxJZFRvUmVwbGFjZSA9IEZsYXR0ZW5lci5nZXRWYWx1ZUJ5UGF0aChcbiAgICAgICAgICBwcm9wcy5wcm9wcyxcbiAgICAgICAgICBwcm9wcy5wcm9wc1RvQWRqdXN0W2tleV1cbiAgICAgICAgKTtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IHRoaXMucmVzb3VyY2VUcmFuc2Zvcm1lci50b1BhcnRpYWwoXG4gICAgICAgICAgcmVzb3VyY2VMb2dpY2FsSWRUb1JlcGxhY2VcbiAgICAgICAgKTtcblxuICAgICAgICBjb25zdCBzcGxpdEtleSA9IHByb3BzLnByb3BzVG9BZGp1c3Rba2V5XS5zcGxpdCgnLicpO1xuICAgICAgICBpZiAoc3BsaXRLZXkuc2xpY2UoLTEpWzBdID09ICdSZWYnKSB7XG4gICAgICAgICAgRmxhdHRlbmVyLnNldFRvVmFsdWUoXG4gICAgICAgICAgICBtb2RpZmllZFByb3BzLFxuICAgICAgICAgICAgc3BsaXRLZXkuc2xpY2UoMCwgLTEpLmpvaW4oJy4nKSxcbiAgICAgICAgICAgIHBhcnRpYWxcbiAgICAgICAgICApO1xuICAgICAgICB9IGVsc2UgaWYgKHNwbGl0S2V5LnNsaWNlKC0yKVswXSA9PSAnRm46OkdldEF0dCcpIHtcbiAgICAgICAgICBGbGF0dGVuZXIuc2V0VG9WYWx1ZShcbiAgICAgICAgICAgIG1vZGlmaWVkUHJvcHMsXG4gICAgICAgICAgICBzcGxpdEtleS5zbGljZSgwLCAtMikuam9pbignLicpLFxuICAgICAgICAgICAgcGFydGlhbFxuICAgICAgICAgICk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIG1vZGlmaWVkUHJvcHM7XG4gIH1cblxuICAvKipcbiAgICogVGhpcyB0YWtlcyB0aGUgZmxhdHRlbmVkIGtleSBvZiB0aGUgcmVzb3VyY2UgdGhhdCdzIHJlZmVyZW5jaW5nIGFuXG4gICAqIGV4dHJhY3RlZCByZXNvdXJjZSB2aWEgQ2ZuIGludHJpbnNpYyBmdW5jdGlvbiwgdGhlIENmblJlc291cmNlIGl0c2VsZlxuICAgKiAoZS5nLiBDZm5GdW5jdGlvbikgYXMgd2VsbCBhcyB0aGUgaW1wb3J0IHZhbHVlIChmcm9tXG4gICAqIGRldGVybWluZUV4cG9ydFZhbHVlKSwgYW5kIGFkZHMgYSBQcm9wZXJ0eSBvdmVycmlkZS5cbiAgICpcbiAgICogQHBhcmFtIHByb3BzIG92ZXJyaWRlRm91bmRSZWZXaXRoSW1wb3J0VmFsdWVQcm9wc1xuICAgKi9cbiAgcHJpdmF0ZSBvdmVycmlkZUZvdW5kUmVmV2l0aEltcG9ydFZhbHVlKFxuICAgIHByb3BzOiBvdmVycmlkZUZvdW5kUmVmV2l0aEltcG9ydFZhbHVlUHJvcHNcbiAgKTogdm9pZCB7XG4gICAgLy8gZmluZCBwcm9wZXJ0eSB0byBvdmVycmlkZVxuICAgIGNvbnN0IHNwbGl0S2V5ID0gcHJvcHMuZmxhdHRlbmVkS2V5LnNwbGl0KCcuJyk7XG4gICAgbGV0IHByb3BlcnR5T3ZlcnJpZGVQYXRoOiBzdHJpbmc7XG4gICAgaWYgKHNwbGl0S2V5LnNsaWNlKC0xKVswXSA9PSAnUmVmJykge1xuICAgICAgcHJvcGVydHlPdmVycmlkZVBhdGggPSBzcGxpdEtleS5zbGljZSg0LCAtMSkuam9pbignLicpO1xuICAgIH0gZWxzZSBpZiAoc3BsaXRLZXkuc2xpY2UoLTIpWzBdID09ICdGbjo6R2V0QXR0Jykge1xuICAgICAgcHJvcGVydHlPdmVycmlkZVBhdGggPSBzcGxpdEtleS5zbGljZSg0LCAtMikuam9pbignLicpO1xuICAgIH0gZWxzZSB7XG4gICAgICBwcm9wZXJ0eU92ZXJyaWRlUGF0aCA9ICdub3RGb3VuZCc7XG4gICAgfVxuICAgIGlmICh0aGlzLnZhbHVlU2hhcmVNZXRob2QgPT0gUmVzb3VyY2VFeHRyYWN0b3JTaGFyZU1ldGhvZC5DRk5fT1VUUFVUKSB7XG4gICAgICBwcm9wcy5mb3VuZFJlZk5vZGUuYWRkUHJvcGVydHlPdmVycmlkZShcbiAgICAgICAgcHJvcGVydHlPdmVycmlkZVBhdGgsXG4gICAgICAgIHByb3BzLmZvdW5kUmVmTm9kZS5zdGFjay5yZXNvbHZlKHByb3BzLmltcG9ydFZhbHVlKVxuICAgICAgKTtcbiAgICB9IGVsc2UgaWYgKFxuICAgICAgdGhpcy52YWx1ZVNoYXJlTWV0aG9kID09IFJlc291cmNlRXh0cmFjdG9yU2hhcmVNZXRob2QuU1NNX1BBUkFNRVRFUlxuICAgICkge1xuICAgICAgcHJvcHMuZm91bmRSZWZOb2RlLmFkZFByb3BlcnR5T3ZlcnJpZGUoXG4gICAgICAgIHByb3BlcnR5T3ZlcnJpZGVQYXRoLFxuICAgICAgICBTdHJpbmdQYXJhbWV0ZXIudmFsdWVGcm9tTG9va3VwKFxuICAgICAgICAgIHByb3BzLmZvdW5kUmVmTm9kZS5zdGFjayxcbiAgICAgICAgICBwcm9wcy5pbXBvcnRWYWx1ZVxuICAgICAgICApXG4gICAgICApO1xuICAgIH0gZWxzZSBpZiAoXG4gICAgICB0aGlzLnZhbHVlU2hhcmVNZXRob2QgPT0gUmVzb3VyY2VFeHRyYWN0b3JTaGFyZU1ldGhvZC5BUElfTE9PS1VQXG4gICAgKSB7XG4gICAgICBjb25zdCBpbXBvcnRWYWx1ZSA9IHByb3BzLmZvdW5kUmVmTm9kZS5zdGFjay5yZXNvbHZlKHByb3BzLmltcG9ydFZhbHVlKVtcbiAgICAgICAgJ0ZuOjpJbXBvcnRWYWx1ZSdcbiAgICAgIF07XG4gICAgICBwcm9wcy5mb3VuZFJlZk5vZGUuYWRkUHJvcGVydHlPdmVycmlkZShcbiAgICAgICAgcHJvcGVydHlPdmVycmlkZVBhdGgsXG4gICAgICAgIHRoaXMuY2ZuLmV4dHJhY3RlZFN0YWNrRXhwb3J0c1tpbXBvcnRWYWx1ZV0gfHxcbiAgICAgICAgICBgZHVtbXktdmFsdWUtZm9yLSR7aW1wb3J0VmFsdWV9YFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogUmVtb3ZlIGEgYERlcGVuZHNPbmAgcmVmZXJlbmNlIHdoZW4gbm90IG5lZWRlZFxuICAgKlxuICAgKiBAcGFyYW0gcHJvcHNcbiAgICovXG4gIHByaXZhdGUgZGVsZXRpb25PdmVycmlkZURlcGVuZHNPbihcbiAgICBwcm9wczogZGVsZXRpb25PdmVycmlkZURlcGVuZHNPblByb3BzXG4gICk6IHZvaWQge1xuICAgIGNvbnN0IHNwbGl0S2V5ID0gcHJvcHMuZmxhdHRlbmVkS2V5LnNwbGl0KCcuJyk7XG4gICAgaWYgKHNwbGl0S2V5LnNsaWNlKC0yKVswXSA9PSAnRGVwZW5kc09uJykge1xuICAgICAgcHJvcHMuZm91bmRSZWZOb2RlLmFkZERlbGV0aW9uT3ZlcnJpZGUoXG4gICAgICAgIGBEZXBlbmRzT24uJHtzcGxpdEtleS5zbGljZSgtMSlbMF19YFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogRmluZHMgdGhlIENESyByZXNvdXJjZSBmb3IgYSBnaXZlbiBmbGF0dGVuZWQga2V5IGJ5IGxvb2tpbmcgaXQgdXAgaW4gdGhlXG4gICAqIGNvbnRleHQgb2YgdGhlIHByb3ZpZGVkIG5vZGUncyBzdGFjay5cbiAgICpcbiAgICogQHBhcmFtIG5vZGVcbiAgICogQHBhcmFtIGZsYXR0ZW5kS2V5XG4gICAqIEByZXR1cm5zIENmblJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGdldFJlc291cmNlRnJvbUZvdW5kUmVmKFxuICAgIG5vZGU6IENmblJlc291cmNlLFxuICAgIGZsYXR0ZW5kS2V5OiBzdHJpbmdcbiAgKTogSUNvbnN0cnVjdCB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3QgZm91bmRSZWZMb2dpY2FsSWQgPSBmbGF0dGVuZEtleS5zcGxpdCgnLicpWzJdO1xuICAgIHJldHVybiBub2RlLnN0YWNrLm5vZGUuZmluZEFsbCgpLmZpbmQoKHg6IGFueSkgPT4ge1xuICAgICAgaWYgKCdzdGFjaycgaW4geCAmJiAnbG9naWNhbElkJyBpbiB4KSB7XG4gICAgICAgIHJldHVybiB4LnN0YWNrLnJlc29sdmUoeC5sb2dpY2FsSWQpID09IGZvdW5kUmVmTG9naWNhbElkO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICAgIH1cbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHaXZlbiBhIGZsYXR0ZW5lZCBrZXksIGRldGVybWluZSBpZiB0aGUgcmVzb3VjZSBpcyBhbHNvIGdldHRpbmdcbiAgICogZXh0cmFjdGVkIHNvIHdlIGtub3cgd2hldGhlciBvciBub3QgdG8gYWRqdXN0IENsb3VkRm9ybWF0aW9uIEludHJpbnNpY1xuICAgKiBmdW5jdGlvbnMuXG4gICAqXG4gICAqIEBwYXJhbSBmbGF0dGVuZEtleVxuICAgKiBAcmV0dXJucyBib29sZWFuXG4gICAqL1xuICBwcml2YXRlIGlzUmVmQWxzb0dldHRpbmdFeHRyYWN0ZWQoZmxhdHRlbmRLZXk6IHN0cmluZyk6IGJvb2xlYW4ge1xuICAgIGNvbnN0IHNwbGl0S2V5ID0gZmxhdHRlbmRLZXkuc3BsaXQoJy4nKTtcbiAgICBjb25zdCBwYXRoVG9UeXBlID0gRmxhdHRlbmVyLmdldFZhbHVlQnlQYXRoKFxuICAgICAgdGhpcy5jZm4udGVtcGxhdGVzLFxuICAgICAgc3BsaXRLZXkuc2xpY2UoMCwgMykuY29uY2F0KFsnVHlwZSddKS5qb2luKCcuJylcbiAgICApO1xuICAgIHJldHVybiB0aGlzLnJlc291cmNlVHlwZXNUb0V4dHJhY3QuaW5jbHVkZXMocGF0aFRvVHlwZSk7XG4gIH1cblxuICAvKipcbiAgICogR2V0IGEgbGlzdCBvZiBsb2dpY2FsIGlkcyBvZiByZXNvdXJjZXMgX25vdF8gZ2V0dGluZyBleHBvcnRlZC5cbiAgICpcbiAgICogQHJldHVybnMgZmxhdCBqc29uIGluIHRoZSBmb3JtIG9mIGB7IExvZ2ljYWxJZDogJ1Jlc291cmNlVHlwZScgfWBcbiAgICovXG4gIHByaXZhdGUgZ2V0TG9naWNhbElkc05vdEdldHRpbmdFeHRyYWN0ZWQoKTogRmxhdEpzb24ge1xuICAgIGNvbnN0IGxvZ2ljYWxJZHM6IEZsYXRKc29uID0ge307XG4gICAgZm9yIChjb25zdCBrZXkgaW4gdGhpcy5jZm4uZmxhdFRlbXBsYXRlcykge1xuICAgICAgaWYgKE9iamVjdC5wcm90b3R5cGUuaGFzT3duUHJvcGVydHkuY2FsbCh0aGlzLmNmbi5mbGF0VGVtcGxhdGVzLCBrZXkpKSB7XG4gICAgICAgIGNvbnN0IHNwbGl0S2V5ID0ga2V5LnNwbGl0KCcuJyk7XG4gICAgICAgIGlmIChcbiAgICAgICAgICBzcGxpdEtleS5zbGljZSgtMSlbMF0gPT0gJ1R5cGUnICYmXG4gICAgICAgICAgc3BsaXRLZXkuc2xpY2UoMSwgMilbMF0gPT0gJ1Jlc291cmNlcydcbiAgICAgICAgKSB7XG4gICAgICAgICAgaWYgKFxuICAgICAgICAgICAgIXRoaXMucmVzb3VyY2VUeXBlc1RvRXh0cmFjdC5pbmNsdWRlcyh0aGlzLmNmbi5mbGF0VGVtcGxhdGVzW2tleV0pXG4gICAgICAgICAgKSB7XG4gICAgICAgICAgICBsb2dpY2FsSWRzW3NwbGl0S2V5LnNsaWNlKC0yKVswXV0gPSB0aGlzLmNmbi5mbGF0VGVtcGxhdGVzW2tleV07XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiBsb2dpY2FsSWRzO1xuICB9XG5cbiAgLyoqXG4gICAqIEV4cG9ydHMgYSB2YWx1ZSB1c2luZyBhIGNvbnNpc3RlbnQgc3RhbmRhcmQgZm9yIHRoZSBkaWZmZXJlbnQgdmFsdWVcbiAgICogc2hhcmUgbWV0aG9kcy5cbiAgICpcbiAgICogQHBhcmFtIHN0YWNrIC0gdGhlIENESyBgU3RhY2tgIG9iamVjdCBmb3IgdGhlIHJlc291cmNlIHRvIGltcG9ydFxuICAgKiBAcGFyYW0gbmFtZSAtIHRoZSBuYW1lIG9mIHRoZSBleHBvcnQsIG5vcm1hbGx5IHRoZSBMb2dpY2FsSWQgb2YgdGhlXG4gICAqIHJlc291cmNlIGJlaW5nIGV4cG9ydGVkXG4gICAqIEBwYXJhbSB2YWx1ZSAtIHRoZSB2YWx1ZSB0byBleHBvcnRcbiAgICovXG4gIHByaXZhdGUgZXhwb3J0VmFsdWUoc3RhY2s6IFN0YWNrLCBuYW1lOiBzdHJpbmcsIHZhbHVlOiBhbnkpOiBzdHJpbmcge1xuICAgIGNvbnN0IHN0YWNrTmFtZSA9IHN0YWNrLnN0YWNrTmFtZTtcbiAgICBjb25zdCBzaGFyZU5hbWUgPSBgJHtzdGFja05hbWV9OiR7bmFtZX1gO1xuXG4gICAgaWYgKHRoaXMudmFsdWVTaGFyZU1ldGhvZCA9PSBSZXNvdXJjZUV4dHJhY3RvclNoYXJlTWV0aG9kLlNTTV9QQVJBTUVURVIpIHtcbiAgICAgIGNvbnN0IHBhcmFtTmFtZSA9IGAvJHtzaGFyZU5hbWUucmVwbGFjZSgnOicsICcvJyl9YDtcbiAgICAgIGNvbnN0IHBhcmFtTG9naWNhbElkID0gYHAke25hbWV9YDtcbiAgICAgIGlmICghdGhpcy5leHRyYWN0RGVzdGluYXRpb25TdGFjay5ub2RlLnRyeUZpbmRDaGlsZChwYXJhbUxvZ2ljYWxJZCkpIHtcbiAgICAgICAgbmV3IENmblJlc291cmNlKHRoaXMuZXh0cmFjdERlc3RpbmF0aW9uU3RhY2ssIHBhcmFtTG9naWNhbElkLCB7XG4gICAgICAgICAgdHlwZTogJ0FXUzo6U1NNOjpQYXJhbWV0ZXInLFxuICAgICAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgIE5hbWU6IHBhcmFtTmFtZSxcbiAgICAgICAgICAgIFR5cGU6ICdTdHJpbmcnLFxuICAgICAgICAgICAgVmFsdWU6IHN0YWNrLnJlc29sdmUodmFsdWUpLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgICAgcmV0dXJuIHBhcmFtTmFtZTtcbiAgICB9IGVsc2Uge1xuICAgICAgLy8gQ0ZOX09VVFBVVCBhbmQgQVBJX0xPT0tVUCBzaGFyZSB0aGUgc2FtZSBtZXRob2Qgb2YgZXhwb3J0aW5nIHZhbHVlc1xuICAgICAgaWYgKFxuICAgICAgICAhdGhpcy5leHRyYWN0RGVzdGluYXRpb25TdGFjay5ub2RlLnRyeUZpbmRDaGlsZChcbiAgICAgICAgICBgRXhwb3J0JHtzdGFja05hbWV9OiR7bmFtZX1gXG4gICAgICAgIClcbiAgICAgICkge1xuICAgICAgICByZXR1cm4gdGhpcy5leHRyYWN0RGVzdGluYXRpb25TdGFjay5leHBvcnRWYWx1ZShzdGFjay5yZXNvbHZlKHZhbHVlKSwge1xuICAgICAgICAgIG5hbWU6IHNoYXJlTmFtZSxcbiAgICAgICAgfSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICByZXR1cm4gRm4uaW1wb3J0VmFsdWUoc2hhcmVOYW1lKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbn1cbiJdfQ==