"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceTransformer = exports.ResourceTransform = exports.MissingTransformError = void 0;
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_dynamodb_1 = require("aws-cdk-lib/aws-dynamodb");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_elasticsearch_1 = require("aws-cdk-lib/aws-elasticsearch");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_opensearchservice_1 = require("aws-cdk-lib/aws-opensearchservice");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_sns_1 = require("aws-cdk-lib/aws-sns");
const aws_sqs_1 = require("aws-cdk-lib/aws-sqs");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
class MissingTransformError extends Error {
    constructor(resourceType) {
        super(`
    Unable to find transform for resource type: ${resourceType}.
    You can provide an additional transformation to the Aspect to resolve
    this and/or open a GitHub Issue to get the resource transformation
    added at https://github.com/cdklabs/cdk-enterprise-iac/issues.
    `);
    }
}
exports.MissingTransformError = MissingTransformError;
var ResourceTransform;
(function (ResourceTransform) {
    ResourceTransform["STACK_NAME"] = "ResourceTransform.STACK_NAME";
    ResourceTransform["LOGICAL_ID"] = "ResourceTransform.LOGICAL_ID";
})(ResourceTransform = exports.ResourceTransform || (exports.ResourceTransform = {}));
class ResourceTransformer {
    constructor(props) {
        this.cfn = props.cfnStore;
        this.defaultTransforms = this.createDefaultTransforms();
        this.additionalTransforms = props.additionalTransforms;
    }
    /**
     * Helper function that generates the beginning portion of an AWS Arn.
     */
    generateArnPreamble(service, includeRegion = true, includeAccount = true) {
        const region = includeRegion ? `${aws_cdk_lib_1.Aws.REGION}` : '';
        const account = includeAccount ? `${aws_cdk_lib_1.Aws.ACCOUNT_ID}` : '';
        return `arn:${aws_cdk_lib_1.Aws.PARTITION}:${service}:${region}:${account}`;
    }
    /**
     * Creates the Default Transformation function table that contains transform
     * functions for each CloudFormation Resource Type.
     *
     * This should be updated to support additional resource types.
     * In addition, it may need to be fixed if some of the transformations are
     * found to be incorrect or inconsistent.
     *
     * @returns object in the form of { 'AWS::S3::Bucket': transform function }
     */
    createDefaultTransforms() {
        return {
            /** Standard */
            [aws_dynamodb_1.CfnTable.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId) => {
                const partial = `${stackName}-${logicalId}*`;
                const preamble = this.generateArnPreamble('dynamodb');
                return `${preamble}:table/${partial}`;
            },
            [aws_elasticsearch_1.CfnDomain.CFN_RESOURCE_TYPE_NAME]: (_, logicalId) => {
                const partial = `${logicalId.substring(0, 15).toLowerCase()}*`;
                const preamble = this.generateArnPreamble('es');
                return `${preamble}:domain/${partial}`;
            },
            [aws_opensearchservice_1.CfnDomain.CFN_RESOURCE_TYPE_NAME]: (_, logicalId) => {
                const partial = `${logicalId.substring(0, 15).toLowerCase()}*`;
                const preamble = this.generateArnPreamble('es');
                return `${preamble}:domain/${partial}`;
            },
            [aws_ecs_1.CfnTaskDefinition.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId) => {
                const partial = `${stackName}${logicalId}*:*`.replace('-', '');
                const preamble = this.generateArnPreamble('ecs');
                return `${preamble}:task-definition/${partial}`;
            },
            [aws_ecs_1.CfnCluster.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId) => {
                const partial = `${stackName}-${logicalId}*`;
                const preamble = this.generateArnPreamble('ecs');
                return `${preamble}:cluster/${partial}`;
            },
            /** Colon-resource name grouping */
            [aws_logs_1.CfnLogGroup.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId) => {
                const partial = `${stackName}-${logicalId}*:*`;
                const preamble = this.generateArnPreamble('logs');
                return `${preamble}:log-group:${partial}`;
            },
            [aws_lambda_1.CfnFunction.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId) => {
                const partial = `${stackName}-${logicalId}`.substring(0, 50) + '*';
                const preamble = this.generateArnPreamble('lambda');
                return `${preamble}:function:${partial}`;
            },
            [aws_stepfunctions_1.CfnStateMachine.CFN_RESOURCE_TYPE_NAME]: (_, logicalId) => {
                const partial = `${logicalId}*`;
                const preamble = this.generateArnPreamble('states');
                return `${preamble}:stateMachine:${partial}`;
            },
            /** No resource name grouping */
            [aws_sqs_1.CfnQueue.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId) => {
                const partial = `${stackName}-${logicalId}*`;
                const preamble = this.generateArnPreamble('sqs');
                return `${preamble}:${partial}`;
            },
            [aws_s3_1.CfnBucket.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId) => {
                const partial = `${stackName}-${logicalId}*`.toLowerCase();
                const preamble = this.generateArnPreamble('s3', false, false);
                return `${preamble}:${partial}`;
            },
            [aws_sns_1.CfnTopic.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId) => {
                const partial = `${stackName}-${logicalId}*`;
                const preamble = this.generateArnPreamble('sns');
                return `${preamble}:${partial}`;
            },
            /** Non-standard */
            [aws_ssm_1.CfnParameter.CFN_RESOURCE_TYPE_NAME]: (_, logicalId) => {
                const partial = `CFN-${logicalId}*`;
                return partial;
            },
            [aws_apigateway_1.CfnApiKey.CFN_RESOURCE_TYPE_NAME]: (stackName, logicalId) => {
                const partial = `${stackName.substring(0, 6)}-${logicalId.substring(0, 5)}*`;
                return partial;
            },
        };
    }
    /**
     * Transforms resource names to partial values (primarily ARNs) using
     * wildcards.
     *
     * Takes in a generated resource name from CDK and transforms it to a
     * partial value that is used to replace resource references that use
     * `Fn::GetAtt` from the generated resource name. This is mainly used to
     * avoid cyclical dependencies within CDK and ensure that Policies can be
     * correctly created without knowing the future value of a resource. This
     * relies on the assumption that the developer does NOT input the `name`
     * of the resource they are creating. In other words, you must let CDK
     * generate the resource name.
     *
     * @param logicalId the Logical ID of the Resource generated by CDK
     * @returns string of partial match value to use in IAM Policies
     */
    toPartial(logicalId) {
        const stackName = this.cfn.getStackNameFromLogicalId(logicalId);
        const resourceType = this.cfn.getResourceTypeFromLogicalId(logicalId);
        if (this.additionalTransforms &&
            resourceType in this.additionalTransforms) {
            return this.additionalTransforms[resourceType]
                .replace(ResourceTransform.STACK_NAME, stackName)
                .replace(ResourceTransform.LOGICAL_ID, logicalId);
        }
        else if (resourceType in this.defaultTransforms) {
            const transformResourceToPartial = this.defaultTransforms[resourceType];
            return transformResourceToPartial(stackName, logicalId);
        }
        else {
            throw new MissingTransformError(resourceType);
        }
    }
}
exports.ResourceTransformer = ResourceTransformer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb3VyY2VUcmFuc2Zvcm1lci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9wYXRjaGVzL3Jlc291cmNlLWV4dHJhY3Rvci9yZXNvdXJjZVRyYW5zZm9ybWVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBOzs7RUFHRTtBQUNGLDZDQUFrQztBQUNsQywrREFBdUQ7QUFDdkQsMkRBQW9EO0FBQ3BELGlEQUFvRTtBQUNwRSxxRUFBMEU7QUFDMUUsdURBQXFEO0FBQ3JELG1EQUFtRDtBQUNuRCw2RUFBOEU7QUFDOUUsK0NBQStDO0FBQy9DLGlEQUErQztBQUMvQyxpREFBK0M7QUFDL0MsaURBQW1EO0FBQ25ELHFFQUFnRTtBQU1oRSxNQUFhLHFCQUFzQixTQUFRLEtBQUs7SUFDOUMsWUFBWSxZQUFvQjtRQUM5QixLQUFLLENBQUM7a0RBQ3dDLFlBQVk7Ozs7S0FJekQsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBVEQsc0RBU0M7QUFFRCxJQUFZLGlCQUdYO0FBSEQsV0FBWSxpQkFBaUI7SUFDM0IsZ0VBQTJDLENBQUE7SUFDM0MsZ0VBQTJDLENBQUE7QUFDN0MsQ0FBQyxFQUhXLGlCQUFpQixHQUFqQix5QkFBaUIsS0FBakIseUJBQWlCLFFBRzVCO0FBT0QsTUFBYSxtQkFBbUI7SUFLOUIsWUFBWSxLQUErQjtRQUN6QyxJQUFJLENBQUMsR0FBRyxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDMUIsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQyx1QkFBdUIsRUFBRSxDQUFDO1FBQ3hELElBQUksQ0FBQyxvQkFBb0IsR0FBRyxLQUFLLENBQUMsb0JBQW9CLENBQUM7SUFDekQsQ0FBQztJQUVEOztPQUVHO0lBQ0ssbUJBQW1CLENBQ3pCLE9BQWUsRUFDZixnQkFBeUIsSUFBSSxFQUM3QixpQkFBMEIsSUFBSTtRQUU5QixNQUFNLE1BQU0sR0FBRyxhQUFhLENBQUMsQ0FBQyxDQUFDLEdBQUcsaUJBQUcsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBQ3BELE1BQU0sT0FBTyxHQUFHLGNBQWMsQ0FBQyxDQUFDLENBQUMsR0FBRyxpQkFBRyxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFFMUQsT0FBTyxPQUFPLGlCQUFHLENBQUMsU0FBUyxJQUFJLE9BQU8sSUFBSSxNQUFNLElBQUksT0FBTyxFQUFFLENBQUM7SUFDaEUsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNLLHVCQUF1QjtRQUM3QixPQUFPO1lBQ0wsZUFBZTtZQUNmLENBQUMsdUJBQVEsQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRSxFQUFFO2dCQUMxRCxNQUFNLE9BQU8sR0FBRyxHQUFHLFNBQVMsSUFBSSxTQUFTLEdBQUcsQ0FBQztnQkFDN0MsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFVBQVUsQ0FBQyxDQUFDO2dCQUN0RCxPQUFPLEdBQUcsUUFBUSxVQUFVLE9BQU8sRUFBRSxDQUFDO1lBQ3hDLENBQUM7WUFDRCxDQUFDLDZCQUFZLENBQUMsc0JBQXNCLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxTQUFTLEVBQUUsRUFBRTtnQkFDdEQsTUFBTSxPQUFPLEdBQUcsR0FBRyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxXQUFXLEVBQUUsR0FBRyxDQUFDO2dCQUMvRCxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ2hELE9BQU8sR0FBRyxRQUFRLFdBQVcsT0FBTyxFQUFFLENBQUM7WUFDekMsQ0FBQztZQUNELENBQUMsaUNBQVksQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLFNBQVMsRUFBRSxFQUFFO2dCQUN0RCxNQUFNLE9BQU8sR0FBRyxHQUFHLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLFdBQVcsRUFBRSxHQUFHLENBQUM7Z0JBQy9ELE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDaEQsT0FBTyxHQUFHLFFBQVEsV0FBVyxPQUFPLEVBQUUsQ0FBQztZQUN6QyxDQUFDO1lBQ0QsQ0FBQywyQkFBaUIsQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRSxFQUFFO2dCQUNuRSxNQUFNLE9BQU8sR0FBRyxHQUFHLFNBQVMsR0FBRyxTQUFTLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxDQUFDO2dCQUMvRCxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBQ2pELE9BQU8sR0FBRyxRQUFRLG9CQUFvQixPQUFPLEVBQUUsQ0FBQztZQUNsRCxDQUFDO1lBQ0QsQ0FBQyxvQkFBVSxDQUFDLHNCQUFzQixDQUFDLEVBQUUsQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFLEVBQUU7Z0JBQzVELE1BQU0sT0FBTyxHQUFHLEdBQUcsU0FBUyxJQUFJLFNBQVMsR0FBRyxDQUFDO2dCQUM3QyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBQ2pELE9BQU8sR0FBRyxRQUFRLFlBQVksT0FBTyxFQUFFLENBQUM7WUFDMUMsQ0FBQztZQUNELG1DQUFtQztZQUNuQyxDQUFDLHNCQUFXLENBQUMsc0JBQXNCLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUUsRUFBRTtnQkFDN0QsTUFBTSxPQUFPLEdBQUcsR0FBRyxTQUFTLElBQUksU0FBUyxLQUFLLENBQUM7Z0JBQy9DLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLENBQUMsQ0FBQztnQkFDbEQsT0FBTyxHQUFHLFFBQVEsY0FBYyxPQUFPLEVBQUUsQ0FBQztZQUM1QyxDQUFDO1lBQ0QsQ0FBQyx3QkFBVyxDQUFDLHNCQUFzQixDQUFDLEVBQUUsQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFLEVBQUU7Z0JBQzdELE1BQU0sT0FBTyxHQUFHLEdBQUcsU0FBUyxJQUFJLFNBQVMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLEdBQUcsR0FBRyxDQUFDO2dCQUNuRSxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsUUFBUSxDQUFDLENBQUM7Z0JBQ3BELE9BQU8sR0FBRyxRQUFRLGFBQWEsT0FBTyxFQUFFLENBQUM7WUFDM0MsQ0FBQztZQUNELENBQUMsbUNBQWUsQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLFNBQVMsRUFBRSxFQUFFO2dCQUN6RCxNQUFNLE9BQU8sR0FBRyxHQUFHLFNBQVMsR0FBRyxDQUFDO2dCQUNoQyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsUUFBUSxDQUFDLENBQUM7Z0JBQ3BELE9BQU8sR0FBRyxRQUFRLGlCQUFpQixPQUFPLEVBQUUsQ0FBQztZQUMvQyxDQUFDO1lBQ0QsZ0NBQWdDO1lBQ2hDLENBQUMsa0JBQVEsQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRSxFQUFFO2dCQUMxRCxNQUFNLE9BQU8sR0FBRyxHQUFHLFNBQVMsSUFBSSxTQUFTLEdBQUcsQ0FBQztnQkFDN0MsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUNqRCxPQUFPLEdBQUcsUUFBUSxJQUFJLE9BQU8sRUFBRSxDQUFDO1lBQ2xDLENBQUM7WUFDRCxDQUFDLGtCQUFTLENBQUMsc0JBQXNCLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUUsRUFBRTtnQkFDM0QsTUFBTSxPQUFPLEdBQUcsR0FBRyxTQUFTLElBQUksU0FBUyxHQUFHLENBQUMsV0FBVyxFQUFFLENBQUM7Z0JBQzNELE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO2dCQUM5RCxPQUFPLEdBQUcsUUFBUSxJQUFJLE9BQU8sRUFBRSxDQUFDO1lBQ2xDLENBQUM7WUFDRCxDQUFDLGtCQUFRLENBQUMsc0JBQXNCLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUUsRUFBRTtnQkFDMUQsTUFBTSxPQUFPLEdBQUcsR0FBRyxTQUFTLElBQUksU0FBUyxHQUFHLENBQUM7Z0JBQzdDLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDakQsT0FBTyxHQUFHLFFBQVEsSUFBSSxPQUFPLEVBQUUsQ0FBQztZQUNsQyxDQUFDO1lBQ0QsbUJBQW1CO1lBQ25CLENBQUMsc0JBQVksQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLFNBQVMsRUFBRSxFQUFFO2dCQUN0RCxNQUFNLE9BQU8sR0FBRyxPQUFPLFNBQVMsR0FBRyxDQUFDO2dCQUNwQyxPQUFPLE9BQU8sQ0FBQztZQUNqQixDQUFDO1lBQ0QsQ0FBQywwQkFBUyxDQUFDLHNCQUFzQixDQUFDLEVBQUUsQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFLEVBQUU7Z0JBQzNELE1BQU0sT0FBTyxHQUFHLEdBQUcsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksU0FBUyxDQUFDLFNBQVMsQ0FDakUsQ0FBQyxFQUNELENBQUMsQ0FDRixHQUFHLENBQUM7Z0JBQ0wsT0FBTyxPQUFPLENBQUM7WUFDakIsQ0FBQztTQUNGLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksU0FBUyxDQUFDLFNBQWlCO1FBQ2hDLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDaEUsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUV0RSxJQUNFLElBQUksQ0FBQyxvQkFBb0I7WUFDekIsWUFBWSxJQUFJLElBQUksQ0FBQyxvQkFBb0IsRUFDekM7WUFDQSxPQUFPLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxZQUFZLENBQUM7aUJBQzNDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxVQUFVLEVBQUUsU0FBUyxDQUFDO2lCQUNoRCxPQUFPLENBQUMsaUJBQWlCLENBQUMsVUFBVSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1NBQ3JEO2FBQU0sSUFBSSxZQUFZLElBQUksSUFBSSxDQUFDLGlCQUFpQixFQUFFO1lBQ2pELE1BQU0sMEJBQTBCLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFlBQVksQ0FBQyxDQUFDO1lBQ3hFLE9BQU8sMEJBQTBCLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1NBQ3pEO2FBQU07WUFDTCxNQUFNLElBQUkscUJBQXFCLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDL0M7SUFDSCxDQUFDO0NBQ0Y7QUFoSkQsa0RBZ0pDIiwic291cmNlc0NvbnRlbnQiOlsiLypcbkNvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiovXG5pbXBvcnQgeyBBd3MgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDZm5BcGlLZXkgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtYXBpZ2F0ZXdheSc7XG5pbXBvcnQgeyBDZm5UYWJsZSB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1keW5hbW9kYic7XG5pbXBvcnQgeyBDZm5DbHVzdGVyLCBDZm5UYXNrRGVmaW5pdGlvbiB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lY3MnO1xuaW1wb3J0IHsgQ2ZuRG9tYWluIGFzIENmbkRvbWFpbkVzcyB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lbGFzdGljc2VhcmNoJztcbmltcG9ydCB7IENmbkZ1bmN0aW9uIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBDZm5Mb2dHcm91cCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1sb2dzJztcbmltcG9ydCB7IENmbkRvbWFpbiBhcyBDZm5Eb21haW5Pc3MgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtb3BlbnNlYXJjaHNlcnZpY2UnO1xuaW1wb3J0IHsgQ2ZuQnVja2V0IH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzJztcbmltcG9ydCB7IENmblRvcGljIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLXNucyc7XG5pbXBvcnQgeyBDZm5RdWV1ZSB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zcXMnO1xuaW1wb3J0IHsgQ2ZuUGFyYW1ldGVyIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLXNzbSc7XG5pbXBvcnQgeyBDZm5TdGF0ZU1hY2hpbmUgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgeyBDZm5TdG9yZSB9IGZyb20gJy4vY2ZuU3RvcmUnO1xuaW1wb3J0IHsgRmxhdEpzb24gfSBmcm9tICcuL3R5cGVzJztcblxudHlwZSBUcmFuc2Zvcm1lciA9IChzdGFja05hbWU6IHN0cmluZywgbG9naWNhbElkOiBzdHJpbmcpID0+IHN0cmluZztcblxuZXhwb3J0IGNsYXNzIE1pc3NpbmdUcmFuc2Zvcm1FcnJvciBleHRlbmRzIEVycm9yIHtcbiAgY29uc3RydWN0b3IocmVzb3VyY2VUeXBlOiBzdHJpbmcpIHtcbiAgICBzdXBlcihgXG4gICAgVW5hYmxlIHRvIGZpbmQgdHJhbnNmb3JtIGZvciByZXNvdXJjZSB0eXBlOiAke3Jlc291cmNlVHlwZX0uXG4gICAgWW91IGNhbiBwcm92aWRlIGFuIGFkZGl0aW9uYWwgdHJhbnNmb3JtYXRpb24gdG8gdGhlIEFzcGVjdCB0byByZXNvbHZlXG4gICAgdGhpcyBhbmQvb3Igb3BlbiBhIEdpdEh1YiBJc3N1ZSB0byBnZXQgdGhlIHJlc291cmNlIHRyYW5zZm9ybWF0aW9uXG4gICAgYWRkZWQgYXQgaHR0cHM6Ly9naXRodWIuY29tL2Nka2xhYnMvY2RrLWVudGVycHJpc2UtaWFjL2lzc3Vlcy5cbiAgICBgKTtcbiAgfVxufVxuXG5leHBvcnQgZW51bSBSZXNvdXJjZVRyYW5zZm9ybSB7XG4gIFNUQUNLX05BTUUgPSAnUmVzb3VyY2VUcmFuc2Zvcm0uU1RBQ0tfTkFNRScsXG4gIExPR0lDQUxfSUQgPSAnUmVzb3VyY2VUcmFuc2Zvcm0uTE9HSUNBTF9JRCcsXG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgUmVzb3VyY2VUcmFuc2Zvcm1lclByb3BzIHtcbiAgcmVhZG9ubHkgY2ZuU3RvcmU6IENmblN0b3JlO1xuICByZWFkb25seSBhZGRpdGlvbmFsVHJhbnNmb3Jtcz86IEZsYXRKc29uO1xufVxuXG5leHBvcnQgY2xhc3MgUmVzb3VyY2VUcmFuc2Zvcm1lciB7XG4gIHByaXZhdGUgcmVhZG9ubHkgY2ZuOiBDZm5TdG9yZTtcbiAgcHJpdmF0ZSByZWFkb25seSBkZWZhdWx0VHJhbnNmb3JtczogeyBba2V5OiBzdHJpbmddOiBUcmFuc2Zvcm1lciB9O1xuICBwcml2YXRlIHJlYWRvbmx5IGFkZGl0aW9uYWxUcmFuc2Zvcm1zPzogRmxhdEpzb247XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IFJlc291cmNlVHJhbnNmb3JtZXJQcm9wcykge1xuICAgIHRoaXMuY2ZuID0gcHJvcHMuY2ZuU3RvcmU7XG4gICAgdGhpcy5kZWZhdWx0VHJhbnNmb3JtcyA9IHRoaXMuY3JlYXRlRGVmYXVsdFRyYW5zZm9ybXMoKTtcbiAgICB0aGlzLmFkZGl0aW9uYWxUcmFuc2Zvcm1zID0gcHJvcHMuYWRkaXRpb25hbFRyYW5zZm9ybXM7XG4gIH1cblxuICAvKipcbiAgICogSGVscGVyIGZ1bmN0aW9uIHRoYXQgZ2VuZXJhdGVzIHRoZSBiZWdpbm5pbmcgcG9ydGlvbiBvZiBhbiBBV1MgQXJuLlxuICAgKi9cbiAgcHJpdmF0ZSBnZW5lcmF0ZUFyblByZWFtYmxlKFxuICAgIHNlcnZpY2U6IHN0cmluZyxcbiAgICBpbmNsdWRlUmVnaW9uOiBib29sZWFuID0gdHJ1ZSxcbiAgICBpbmNsdWRlQWNjb3VudDogYm9vbGVhbiA9IHRydWVcbiAgKTogc3RyaW5nIHtcbiAgICBjb25zdCByZWdpb24gPSBpbmNsdWRlUmVnaW9uID8gYCR7QXdzLlJFR0lPTn1gIDogJyc7XG4gICAgY29uc3QgYWNjb3VudCA9IGluY2x1ZGVBY2NvdW50ID8gYCR7QXdzLkFDQ09VTlRfSUR9YCA6ICcnO1xuXG4gICAgcmV0dXJuIGBhcm46JHtBd3MuUEFSVElUSU9OfToke3NlcnZpY2V9OiR7cmVnaW9ufToke2FjY291bnR9YDtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIHRoZSBEZWZhdWx0IFRyYW5zZm9ybWF0aW9uIGZ1bmN0aW9uIHRhYmxlIHRoYXQgY29udGFpbnMgdHJhbnNmb3JtXG4gICAqIGZ1bmN0aW9ucyBmb3IgZWFjaCBDbG91ZEZvcm1hdGlvbiBSZXNvdXJjZSBUeXBlLlxuICAgKlxuICAgKiBUaGlzIHNob3VsZCBiZSB1cGRhdGVkIHRvIHN1cHBvcnQgYWRkaXRpb25hbCByZXNvdXJjZSB0eXBlcy5cbiAgICogSW4gYWRkaXRpb24sIGl0IG1heSBuZWVkIHRvIGJlIGZpeGVkIGlmIHNvbWUgb2YgdGhlIHRyYW5zZm9ybWF0aW9ucyBhcmVcbiAgICogZm91bmQgdG8gYmUgaW5jb3JyZWN0IG9yIGluY29uc2lzdGVudC5cbiAgICpcbiAgICogQHJldHVybnMgb2JqZWN0IGluIHRoZSBmb3JtIG9mIHsgJ0FXUzo6UzM6OkJ1Y2tldCc6IHRyYW5zZm9ybSBmdW5jdGlvbiB9XG4gICAqL1xuICBwcml2YXRlIGNyZWF0ZURlZmF1bHRUcmFuc2Zvcm1zKCk6IHsgW2tleTogc3RyaW5nXTogVHJhbnNmb3JtZXIgfSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIC8qKiBTdGFuZGFyZCAqL1xuICAgICAgW0NmblRhYmxlLkNGTl9SRVNPVVJDRV9UWVBFX05BTUVdOiAoc3RhY2tOYW1lLCBsb2dpY2FsSWQpID0+IHtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IGAke3N0YWNrTmFtZX0tJHtsb2dpY2FsSWR9KmA7XG4gICAgICAgIGNvbnN0IHByZWFtYmxlID0gdGhpcy5nZW5lcmF0ZUFyblByZWFtYmxlKCdkeW5hbW9kYicpO1xuICAgICAgICByZXR1cm4gYCR7cHJlYW1ibGV9OnRhYmxlLyR7cGFydGlhbH1gO1xuICAgICAgfSxcbiAgICAgIFtDZm5Eb21haW5Fc3MuQ0ZOX1JFU09VUkNFX1RZUEVfTkFNRV06IChfLCBsb2dpY2FsSWQpID0+IHtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IGAke2xvZ2ljYWxJZC5zdWJzdHJpbmcoMCwgMTUpLnRvTG93ZXJDYXNlKCl9KmA7XG4gICAgICAgIGNvbnN0IHByZWFtYmxlID0gdGhpcy5nZW5lcmF0ZUFyblByZWFtYmxlKCdlcycpO1xuICAgICAgICByZXR1cm4gYCR7cHJlYW1ibGV9OmRvbWFpbi8ke3BhcnRpYWx9YDtcbiAgICAgIH0sXG4gICAgICBbQ2ZuRG9tYWluT3NzLkNGTl9SRVNPVVJDRV9UWVBFX05BTUVdOiAoXywgbG9naWNhbElkKSA9PiB7XG4gICAgICAgIGNvbnN0IHBhcnRpYWwgPSBgJHtsb2dpY2FsSWQuc3Vic3RyaW5nKDAsIDE1KS50b0xvd2VyQ2FzZSgpfSpgO1xuICAgICAgICBjb25zdCBwcmVhbWJsZSA9IHRoaXMuZ2VuZXJhdGVBcm5QcmVhbWJsZSgnZXMnKTtcbiAgICAgICAgcmV0dXJuIGAke3ByZWFtYmxlfTpkb21haW4vJHtwYXJ0aWFsfWA7XG4gICAgICB9LFxuICAgICAgW0NmblRhc2tEZWZpbml0aW9uLkNGTl9SRVNPVVJDRV9UWVBFX05BTUVdOiAoc3RhY2tOYW1lLCBsb2dpY2FsSWQpID0+IHtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IGAke3N0YWNrTmFtZX0ke2xvZ2ljYWxJZH0qOipgLnJlcGxhY2UoJy0nLCAnJyk7XG4gICAgICAgIGNvbnN0IHByZWFtYmxlID0gdGhpcy5nZW5lcmF0ZUFyblByZWFtYmxlKCdlY3MnKTtcbiAgICAgICAgcmV0dXJuIGAke3ByZWFtYmxlfTp0YXNrLWRlZmluaXRpb24vJHtwYXJ0aWFsfWA7XG4gICAgICB9LFxuICAgICAgW0NmbkNsdXN0ZXIuQ0ZOX1JFU09VUkNFX1RZUEVfTkFNRV06IChzdGFja05hbWUsIGxvZ2ljYWxJZCkgPT4ge1xuICAgICAgICBjb25zdCBwYXJ0aWFsID0gYCR7c3RhY2tOYW1lfS0ke2xvZ2ljYWxJZH0qYDtcbiAgICAgICAgY29uc3QgcHJlYW1ibGUgPSB0aGlzLmdlbmVyYXRlQXJuUHJlYW1ibGUoJ2VjcycpO1xuICAgICAgICByZXR1cm4gYCR7cHJlYW1ibGV9OmNsdXN0ZXIvJHtwYXJ0aWFsfWA7XG4gICAgICB9LFxuICAgICAgLyoqIENvbG9uLXJlc291cmNlIG5hbWUgZ3JvdXBpbmcgKi9cbiAgICAgIFtDZm5Mb2dHcm91cC5DRk5fUkVTT1VSQ0VfVFlQRV9OQU1FXTogKHN0YWNrTmFtZSwgbG9naWNhbElkKSA9PiB7XG4gICAgICAgIGNvbnN0IHBhcnRpYWwgPSBgJHtzdGFja05hbWV9LSR7bG9naWNhbElkfSo6KmA7XG4gICAgICAgIGNvbnN0IHByZWFtYmxlID0gdGhpcy5nZW5lcmF0ZUFyblByZWFtYmxlKCdsb2dzJyk7XG4gICAgICAgIHJldHVybiBgJHtwcmVhbWJsZX06bG9nLWdyb3VwOiR7cGFydGlhbH1gO1xuICAgICAgfSxcbiAgICAgIFtDZm5GdW5jdGlvbi5DRk5fUkVTT1VSQ0VfVFlQRV9OQU1FXTogKHN0YWNrTmFtZSwgbG9naWNhbElkKSA9PiB7XG4gICAgICAgIGNvbnN0IHBhcnRpYWwgPSBgJHtzdGFja05hbWV9LSR7bG9naWNhbElkfWAuc3Vic3RyaW5nKDAsIDUwKSArICcqJztcbiAgICAgICAgY29uc3QgcHJlYW1ibGUgPSB0aGlzLmdlbmVyYXRlQXJuUHJlYW1ibGUoJ2xhbWJkYScpO1xuICAgICAgICByZXR1cm4gYCR7cHJlYW1ibGV9OmZ1bmN0aW9uOiR7cGFydGlhbH1gO1xuICAgICAgfSxcbiAgICAgIFtDZm5TdGF0ZU1hY2hpbmUuQ0ZOX1JFU09VUkNFX1RZUEVfTkFNRV06IChfLCBsb2dpY2FsSWQpID0+IHtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IGAke2xvZ2ljYWxJZH0qYDtcbiAgICAgICAgY29uc3QgcHJlYW1ibGUgPSB0aGlzLmdlbmVyYXRlQXJuUHJlYW1ibGUoJ3N0YXRlcycpO1xuICAgICAgICByZXR1cm4gYCR7cHJlYW1ibGV9OnN0YXRlTWFjaGluZToke3BhcnRpYWx9YDtcbiAgICAgIH0sXG4gICAgICAvKiogTm8gcmVzb3VyY2UgbmFtZSBncm91cGluZyAqL1xuICAgICAgW0NmblF1ZXVlLkNGTl9SRVNPVVJDRV9UWVBFX05BTUVdOiAoc3RhY2tOYW1lLCBsb2dpY2FsSWQpID0+IHtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IGAke3N0YWNrTmFtZX0tJHtsb2dpY2FsSWR9KmA7XG4gICAgICAgIGNvbnN0IHByZWFtYmxlID0gdGhpcy5nZW5lcmF0ZUFyblByZWFtYmxlKCdzcXMnKTtcbiAgICAgICAgcmV0dXJuIGAke3ByZWFtYmxlfToke3BhcnRpYWx9YDtcbiAgICAgIH0sXG4gICAgICBbQ2ZuQnVja2V0LkNGTl9SRVNPVVJDRV9UWVBFX05BTUVdOiAoc3RhY2tOYW1lLCBsb2dpY2FsSWQpID0+IHtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IGAke3N0YWNrTmFtZX0tJHtsb2dpY2FsSWR9KmAudG9Mb3dlckNhc2UoKTtcbiAgICAgICAgY29uc3QgcHJlYW1ibGUgPSB0aGlzLmdlbmVyYXRlQXJuUHJlYW1ibGUoJ3MzJywgZmFsc2UsIGZhbHNlKTtcbiAgICAgICAgcmV0dXJuIGAke3ByZWFtYmxlfToke3BhcnRpYWx9YDtcbiAgICAgIH0sXG4gICAgICBbQ2ZuVG9waWMuQ0ZOX1JFU09VUkNFX1RZUEVfTkFNRV06IChzdGFja05hbWUsIGxvZ2ljYWxJZCkgPT4ge1xuICAgICAgICBjb25zdCBwYXJ0aWFsID0gYCR7c3RhY2tOYW1lfS0ke2xvZ2ljYWxJZH0qYDtcbiAgICAgICAgY29uc3QgcHJlYW1ibGUgPSB0aGlzLmdlbmVyYXRlQXJuUHJlYW1ibGUoJ3NucycpO1xuICAgICAgICByZXR1cm4gYCR7cHJlYW1ibGV9OiR7cGFydGlhbH1gO1xuICAgICAgfSxcbiAgICAgIC8qKiBOb24tc3RhbmRhcmQgKi9cbiAgICAgIFtDZm5QYXJhbWV0ZXIuQ0ZOX1JFU09VUkNFX1RZUEVfTkFNRV06IChfLCBsb2dpY2FsSWQpID0+IHtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IGBDRk4tJHtsb2dpY2FsSWR9KmA7XG4gICAgICAgIHJldHVybiBwYXJ0aWFsO1xuICAgICAgfSxcbiAgICAgIFtDZm5BcGlLZXkuQ0ZOX1JFU09VUkNFX1RZUEVfTkFNRV06IChzdGFja05hbWUsIGxvZ2ljYWxJZCkgPT4ge1xuICAgICAgICBjb25zdCBwYXJ0aWFsID0gYCR7c3RhY2tOYW1lLnN1YnN0cmluZygwLCA2KX0tJHtsb2dpY2FsSWQuc3Vic3RyaW5nKFxuICAgICAgICAgIDAsXG4gICAgICAgICAgNVxuICAgICAgICApfSpgO1xuICAgICAgICByZXR1cm4gcGFydGlhbDtcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUcmFuc2Zvcm1zIHJlc291cmNlIG5hbWVzIHRvIHBhcnRpYWwgdmFsdWVzIChwcmltYXJpbHkgQVJOcykgdXNpbmdcbiAgICogd2lsZGNhcmRzLlxuICAgKlxuICAgKiBUYWtlcyBpbiBhIGdlbmVyYXRlZCByZXNvdXJjZSBuYW1lIGZyb20gQ0RLIGFuZCB0cmFuc2Zvcm1zIGl0IHRvIGFcbiAgICogcGFydGlhbCB2YWx1ZSB0aGF0IGlzIHVzZWQgdG8gcmVwbGFjZSByZXNvdXJjZSByZWZlcmVuY2VzIHRoYXQgdXNlXG4gICAqIGBGbjo6R2V0QXR0YCBmcm9tIHRoZSBnZW5lcmF0ZWQgcmVzb3VyY2UgbmFtZS4gVGhpcyBpcyBtYWlubHkgdXNlZCB0b1xuICAgKiBhdm9pZCBjeWNsaWNhbCBkZXBlbmRlbmNpZXMgd2l0aGluIENESyBhbmQgZW5zdXJlIHRoYXQgUG9saWNpZXMgY2FuIGJlXG4gICAqIGNvcnJlY3RseSBjcmVhdGVkIHdpdGhvdXQga25vd2luZyB0aGUgZnV0dXJlIHZhbHVlIG9mIGEgcmVzb3VyY2UuIFRoaXNcbiAgICogcmVsaWVzIG9uIHRoZSBhc3N1bXB0aW9uIHRoYXQgdGhlIGRldmVsb3BlciBkb2VzIE5PVCBpbnB1dCB0aGUgYG5hbWVgXG4gICAqIG9mIHRoZSByZXNvdXJjZSB0aGV5IGFyZSBjcmVhdGluZy4gSW4gb3RoZXIgd29yZHMsIHlvdSBtdXN0IGxldCBDREtcbiAgICogZ2VuZXJhdGUgdGhlIHJlc291cmNlIG5hbWUuXG4gICAqXG4gICAqIEBwYXJhbSBsb2dpY2FsSWQgdGhlIExvZ2ljYWwgSUQgb2YgdGhlIFJlc291cmNlIGdlbmVyYXRlZCBieSBDREtcbiAgICogQHJldHVybnMgc3RyaW5nIG9mIHBhcnRpYWwgbWF0Y2ggdmFsdWUgdG8gdXNlIGluIElBTSBQb2xpY2llc1xuICAgKi9cbiAgcHVibGljIHRvUGFydGlhbChsb2dpY2FsSWQ6IHN0cmluZykge1xuICAgIGNvbnN0IHN0YWNrTmFtZSA9IHRoaXMuY2ZuLmdldFN0YWNrTmFtZUZyb21Mb2dpY2FsSWQobG9naWNhbElkKTtcbiAgICBjb25zdCByZXNvdXJjZVR5cGUgPSB0aGlzLmNmbi5nZXRSZXNvdXJjZVR5cGVGcm9tTG9naWNhbElkKGxvZ2ljYWxJZCk7XG5cbiAgICBpZiAoXG4gICAgICB0aGlzLmFkZGl0aW9uYWxUcmFuc2Zvcm1zICYmXG4gICAgICByZXNvdXJjZVR5cGUgaW4gdGhpcy5hZGRpdGlvbmFsVHJhbnNmb3Jtc1xuICAgICkge1xuICAgICAgcmV0dXJuIHRoaXMuYWRkaXRpb25hbFRyYW5zZm9ybXNbcmVzb3VyY2VUeXBlXVxuICAgICAgICAucmVwbGFjZShSZXNvdXJjZVRyYW5zZm9ybS5TVEFDS19OQU1FLCBzdGFja05hbWUpXG4gICAgICAgIC5yZXBsYWNlKFJlc291cmNlVHJhbnNmb3JtLkxPR0lDQUxfSUQsIGxvZ2ljYWxJZCk7XG4gICAgfSBlbHNlIGlmIChyZXNvdXJjZVR5cGUgaW4gdGhpcy5kZWZhdWx0VHJhbnNmb3Jtcykge1xuICAgICAgY29uc3QgdHJhbnNmb3JtUmVzb3VyY2VUb1BhcnRpYWwgPSB0aGlzLmRlZmF1bHRUcmFuc2Zvcm1zW3Jlc291cmNlVHlwZV07XG4gICAgICByZXR1cm4gdHJhbnNmb3JtUmVzb3VyY2VUb1BhcnRpYWwoc3RhY2tOYW1lLCBsb2dpY2FsSWQpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgTWlzc2luZ1RyYW5zZm9ybUVycm9yKHJlc291cmNlVHlwZSk7XG4gICAgfVxuICB9XG59XG4iXX0=