#!/usr/bin/python
#
# Copyright 2020 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

library = """
->(left:, right:) = {arg: left, value: right};
`=`(left:, right:) = right :- left == right;

Arrow(left, right) = arrow :-
  left == arrow.arg,
  right == arrow.value;

PrintToConsole(message) :- 1 == SqlExpr("PrintToConsole({message})", {message:});

ArgMin(arr) = SqlExpr(
    "argmin({a}, {v})", {a:, v:}) :- Arrow(a, v) == arr;

ArgMax(arr) = SqlExpr(
    "argmax({a}, {v})", {a:, v:}) :- Arrow(a, v) == arr;

ArgMaxK(a, l) = SqlExpr(
  "(array_agg({arg_1} order by {value_1} desc))[1:{lim}]",
  {arg_1: a.arg, value_1: a.value, lim: l});

ArgMinK(a, l) = SqlExpr(
  "(array_agg({arg_1} order by {value_1}))[1:{lim}]",
  {arg_1: a.arg, value_1: a.value, lim: l});

Array(a) = SqlExpr(
  "ARRAY_AGG({value} order by {arg})",
  {arg: a.arg, value: a.value});

RecordAsJson(r) = SqlExpr(
  "ROW_TO_JSON({r})", {r:});

Fingerprint(s) = NaturalHash(s);

ReadFile(filename) = SqlExpr("pg_read_file({filename})", {filename:});

Chr(x) = SqlExpr("Chr(cast({x} as integer))", {x:});
Ord(x) = SqlExpr("Ord({x})", {x:});

Num(a) = a;
Str(a) = a;

Epoch(a) = epoch :-
  epoch = SqlExpr("epoch_ns({a})", {a:}) / 1000000000,
  a ~ Time, 
  epoch ~ Num;
TimeDiffSeconds(a, b) = Epoch(SqlExpr("{a} - {b}", {a:, b:}));
ToTime(a) = SqlExpr("cast({a} as timestamp)", {a:});

NaturalHash(x) = ToInt64(SqlExpr("hash(cast({x} as string)) // cast(2 as ubigint)", {x:}));

# This is unsafe to use because due to the way Logica compiles this number
# will be unique for each use of the variable, which can be a pain to debug.
# It is OK to use it as long as you undertand and are OK with the difficulty.
UnsafeToUseUniqueNumber() = SqlExpr("nextval('eternal_logical_sequence')", {});

# Danger is immanent to life.
UniqueNumber() = SqlExpr("nextval('eternal_logical_sequence')", {});

# Aggregation that concatenates list.
# Doing via SqlExpr as Logica for now prohibits list of lists.
# TODO: We should allow list of lists in DuckDB.
MergeList(e) = SqlExpr("flatten(array_agg({e}))", {e:});

# Functional predicate for toy examples of solving
# NP-complete problems.
ProverChoice(slot, options:) = options[i] :-
  i = NaturalHash("ProverChoice-" ++
                  ToString(UniqueNumber())) % Size(options);

#######################
# Clingo support.
#

Clingo(p, m) = SqlExpr("Clingo({p}, {m})", {p:, m:}) :-
  m ~ [{predicate: Str, args: [Str]}];

RunClingo(p) = SqlExpr("RunClingo({p})", {p:});
RunClingoFile(p) = SqlExpr("RunClingoFile({p})", {p:});
RunClingoTemplate(p, a) = SqlExpr("RunClingoTemplate({p}, {a})", {p:, a:});
RunClingoFileTemplate(p, a) = SqlExpr("RunClingoFileTemplate({p}, {a})", {p:, a:});

RenderClingoArgs(args) = (
  if Size(args) == 0 then
    "()"
  else
    "(" ++ Join(args, ", ") ++ ")"
);

RenderClingoFact(predicate, args) =  predicate ++ RenderClingoArgs(args);

QuoteIt(x) = Chr(34) ++ x ++ Chr(34);
ClingoFact(predicate, args) = {predicate:,
                               args: List{QuoteIt(a) :- a in args}};

ExtractClingoCall(a, b, c, d, e, f, g, h,
                  predicate:, model_id:) = models :-
  model in models,
  model_id = model.model_id,
  entry in model.model,
  entry.predicate = predicate,
  args = entry.args,
  a = args[0], b = args[1], c = args[2],
  d = args[3], e = args[4], f = args[5],
  g = args[6], h = args[7];

JoinOrEmpty(x, s) = Coalesce(Join(x, s), "");

RenderClingoModel(model, sep) = JoinOrEmpty(
    List{RenderClingoFact(fact.predicate, fact.args) :-
         fact in model}, sep);

# Indexed sum, that Clingo needs.
ISum(x) = SqlExpr("SUM({x})", {x:}) :- Error("ISum is to be used only in Clingo.") = true;
"""
