"""Context manager utilities for tracking cache dependencies per thread."""

import threading
from types import TracebackType

from general_manager.cache.dependency_index import (
    Dependency,
    filter_type,
    general_manager_name,
)

# Thread-local storage for tracking dependencies
_dependency_storage = threading.local()


class DependencyTracker:
    def __enter__(
        self,
    ) -> set[Dependency]:
        """
        Enter a dependency tracking context and return the collector set.

        Returns:
            set[Dependency]: Mutable set capturing dependencies discovered inside the context.
        """
        if not hasattr(_dependency_storage, "dependencies"):
            _dependency_storage._depth = 0
            _dependency_storage.dependencies = list()
        else:
            _dependency_storage._depth += 1
        _dependency_storage.dependencies.append(set())
        return _dependency_storage.dependencies[_dependency_storage._depth]

    def __exit__(
        self,
        exc_type: type[BaseException] | None,
        exc_val: BaseException | None,
        exc_tb: TracebackType | None,
    ) -> None:
        """
        Leave the dependency tracking context and perform nested-scope cleanup.

        Parameters:
            exc_type: Exception type raised within the context, if any.
            exc_val: Exception instance raised within the context, if any.
            exc_tb: Traceback generated by the exception, if any.

        Returns:
            None
        """
        if hasattr(_dependency_storage, "dependencies"):
            if _dependency_storage._depth == 0:
                self.reset_thread_local_storage()

            else:
                # For nested contexts only reduce depth and pop one level.
                _dependency_storage._depth -= 1
                _dependency_storage.dependencies.pop()

    @staticmethod
    def track(
        class_name: general_manager_name,
        operation: filter_type,
        identifier: str,
    ) -> None:
        """
        Record a dependency tuple in the active tracking scopes.

        Parameters:
            class_name (str): Name of the GeneralManager subclass.
            operation (filter_type): Operation being tracked, such as `filter` or `exclude`.
            identifier (str): String representation of the lookup parameters.

        Returns:
            None
        """
        if hasattr(_dependency_storage, "dependencies"):
            for dep_set in _dependency_storage.dependencies[
                : _dependency_storage._depth + 1
            ]:
                dep_set.add((class_name, operation, identifier))

    @staticmethod
    def reset_thread_local_storage() -> None:
        """
        Clear all dependency tracking data from thread-local storage.

        Returns:
            None
        """
        if hasattr(_dependency_storage, "dependencies"):
            del _dependency_storage.dependencies
            del _dependency_storage._depth
