"""Interface for integrating existing Django models with GeneralManager."""

from __future__ import annotations

from typing import Any, ClassVar, TypeVar, cast

from django.apps import apps
from django.db import models

from simple_history import register  # type: ignore

from general_manager.factory.auto_factory import AutoFactory
from general_manager.interface.base_interface import (
    attributes,
    classPostCreationMethod,
    classPreCreationMethod,
    generalManagerClassName,
    interfaceBaseClass,
    newlyCreatedGeneralManagerClass,
    newlyCreatedInterfaceClass,
    relatedClass,
)
from general_manager.interface.database_based_interface import (
    WritableDBBasedInterface,
)
from general_manager.interface.models import (
    GeneralManagerBasisModel,
    get_full_clean_methode,
)

ExistingModelT = TypeVar("ExistingModelT", bound=models.Model)


class MissingModelConfigurationError(ValueError):
    """Raised when an ExistingModelInterface does not define a model to manage."""

    def __init__(self, interface_name: str) -> None:
        """
        Initialize the MissingModelConfigurationError for a specific interface that did not define a required `model` attribute.

        Parameters:
            interface_name (str): The name of the interface used to construct the error message.
        """
        super().__init__(f"{interface_name} must define a 'model' attribute.")


class InvalidModelReferenceError(TypeError):
    """Raised when the model attribute is neither a Django model class nor a resolvable label."""

    def __init__(self, reference: object) -> None:
        """
        Initialize the InvalidModelReferenceError with the offending model reference.

        Parameters:
            reference (object): The model reference that could not be resolved; used in the error message.
        """
        super().__init__(f"Invalid model reference '{reference}'.")


class ExistingModelInterface(WritableDBBasedInterface[ExistingModelT]):
    """Interface that reuses an existing Django model instead of generating a new one."""

    _interface_type: ClassVar[str] = "existing"
    model: ClassVar[type[models.Model] | str | None] = None

    @classmethod
    def _resolve_model_class(cls) -> type[models.Model]:
        """
        Resolve the configured `model` attribute to a concrete Django model class.

        If `cls.model` is a string, attempt to resolve it via Django's app registry; if it is already a Django model class, use it directly. The resolved class is cached on `cls._model` and `cls.model`.

        Returns:
            type[~django.db.models.Model]: The resolved Django model class.

        Raises:
            MissingModelConfigurationError: If the interface did not define a `model` attribute.
            InvalidModelReferenceError: If `model` is neither a Django model class nor a resolvable app label.
        """
        model_reference = getattr(cls, "model", None)
        # if model_reference is None:
        #     model_reference = getattr(cls, "_model", None)
        if model_reference is None:
            raise MissingModelConfigurationError(cls.__name__)
        if isinstance(model_reference, str):
            try:
                model = apps.get_model(model_reference)
            except LookupError as error:
                raise InvalidModelReferenceError(model_reference) from error
        elif isinstance(model_reference, type) and issubclass(
            model_reference, models.Model
        ):
            model = model_reference
        else:
            raise InvalidModelReferenceError(model_reference)
        cls._model = cast(type[ExistingModelT], model)
        cls.model = model
        return cast(type[models.Model], model)

    @staticmethod
    def _ensure_history(model: type[models.Model]) -> None:
        """
        Attach django-simple-history tracking to a Django model if it isn't already registered.

        Parameters:
            model (type[models.Model]): The Django model class to register for history tracking.

        Notes:
            If the model is already registered for simple-history, this function is a no-op.
        """
        if hasattr(model._meta, "simple_history_manager_attribute"):
            return
        register(model)

    @classmethod
    def _apply_rules_to_model(cls, model: type[models.Model]) -> None:
        """
        Attach interface-defined validation rules to the provided Django model and replace its `full_clean` with a validating implementation.

        If the interface defines no rules, the model is left unchanged. When rules are present, they are appended to the model's `_meta.rules` and the model's `full_clean` method is replaced with a generated validating method.

        Parameters:
            model (type[models.Model]): The Django model class to modify.
        """
        meta_class = getattr(cls, "Meta", None)
        rules = getattr(meta_class, "rules", None) if meta_class else None
        if not rules:
            return
        combined_rules: list[Any] = []
        existing_rules = getattr(model._meta, "rules", None)
        if existing_rules:
            combined_rules.extend(existing_rules)
        combined_rules.extend(rules)
        model._meta.rules = combined_rules  # type: ignore[attr-defined]
        model.full_clean = get_full_clean_methode(model)  # type: ignore[assignment]

    @classmethod
    def _build_factory(
        cls,
        name: generalManagerClassName,
        interface_cls: type["ExistingModelInterface"],
        model: type[ExistingModelT],
        factory_definition: type | None = None,
    ) -> type[AutoFactory]:
        """
        Create a new AutoFactory subclass configured to produce instances of the given Django model.

        Parameters:
            name (str): Base name used to name the generated factory class (the factory will be "<name>Factory").
            interface_cls (type[ExistingModelInterface]): Interface class that the factory will reference via its `interface` attribute.
            model (type[models.Model]): Django model class that the factory's inner `Meta.model` will point to.
            factory_definition (type | None): Optional existing Factory class whose non-dunder attributes will be copied into the generated factory.

        Returns:
            type[AutoFactory]: A dynamically created AutoFactory subclass bound to `model`, with copied attributes, an `interface` attribute set to `interface_cls`, and an inner `Meta` class referencing `model`.
        """
        factory_definition = factory_definition or getattr(cls, "Factory", None)
        factory_attributes: dict[str, Any] = {}
        if factory_definition:
            for attr_name, attr_value in factory_definition.__dict__.items():
                if not attr_name.startswith("__"):
                    factory_attributes[attr_name] = attr_value
        factory_attributes["interface"] = interface_cls
        factory_attributes["Meta"] = type("Meta", (), {"model": model})
        return type(f"{name}Factory", (AutoFactory,), factory_attributes)

    @staticmethod
    def _pre_create(
        name: generalManagerClassName,
        attrs: attributes,
        interface: interfaceBaseClass,
        base_model_class: type[GeneralManagerBasisModel] = GeneralManagerBasisModel,
    ) -> tuple[attributes, interfaceBaseClass, relatedClass]:
        """
        Prepare an interface for GeneralManager creation by resolving and attaching the existing Django model, ensuring history and rules are applied, and wiring a bound Factory.

        Parameters:
            name: The name to use for the generated manager class (used when building the Factory).
            attrs: The attribute dict that will be used to create the manager class; this function mutates and returns it.
            interface: The interface class object that declares `model` (class or app label); a concrete interface subclass bound to the resolved model is produced.
            base_model_class: Ignored hook parameter kept for compatibility with the creation pipeline.

        Returns:
            A tuple (attrs, concrete_interface, model) where `attrs` is the possibly-modified attribute dict, `concrete_interface` is the new interface class bound to the resolved model, and `model` is the resolved Django model class.
        """
        _ = base_model_class
        interface_cls = cast(type["ExistingModelInterface"], interface)
        model = interface_cls._resolve_model_class()
        interface_cls._ensure_history(model)
        interface_cls._apply_rules_to_model(model)

        concrete_interface = cast(
            type["ExistingModelInterface"],
            type(interface.__name__, (interface,), {}),
        )
        concrete_interface._model = cast(type[ExistingModelT], model)
        concrete_interface.model = model

        manager_factory = cast(type | None, attrs.pop("Factory", None))
        attrs["_interface_type"] = interface_cls._interface_type
        attrs["Interface"] = concrete_interface
        attrs["Factory"] = interface_cls._build_factory(
            name, concrete_interface, model, manager_factory
        )

        return attrs, concrete_interface, model

    @staticmethod
    def _post_create(
        new_class: newlyCreatedGeneralManagerClass,
        interface_class: newlyCreatedInterfaceClass,
        model: relatedClass,
    ) -> None:
        """
        Link the created GeneralManager subclass with its interface and the resolved Django model.

        Sets the interface's internal parent reference to the newly created manager class and, if a model is provided, assigns that manager class to the model's `_general_manager_class` attribute.

        Parameters:
            new_class: The newly created GeneralManager subclass to be linked as the parent.
            interface_class: The interface class instance that should reference `new_class` as its parent.
            model: The Django model class managed by the interface; if not None, its `_general_manager_class` will be set.
        """
        interface_class._parent_class = new_class
        if model is not None:
            model._general_manager_class = new_class  # type: ignore[attr-defined]

    @classmethod
    def handle_interface(cls) -> tuple[classPreCreationMethod, classPostCreationMethod]:
        """
        Provide the pre- and post-creation hooks used by GeneralManagerMeta.

        Returns:
            tuple[classPreCreationMethod, classPostCreationMethod]: A pair where the first element is the pre-creation hook (called before class creation) and the second element is the post-creation hook (called after class creation).
        """
        return cls._pre_create, cls._post_create

    @classmethod
    def get_field_type(cls, field_name: str) -> type:
        """
        Get the Python type for a field on the wrapped model, resolving the configured model first if not already resolved.

        Parameters:
            field_name (str): Name of the field on the underlying Django model.

        Returns:
            type: The Python type corresponding to the specified model field.
        """
        if not hasattr(cls, "_model"):
            cls._resolve_model_class()
        return super().get_field_type(field_name)
