from strif import atomic_output_file

from kash.config.logger import get_logger
from kash.exec import kash_action
from kash.exec.preconditions import has_html_compatible_body
from kash.kits.docs.doc_formats.simple_html_to_docx import SimpleHtmlToDocx
from kash.model import FileExt, Format, Item, ItemType
from kash.utils.errors import InvalidInput
from kash.workspaces import current_ws

log = get_logger(__name__)


@kash_action(
    precondition=has_html_compatible_body,
    mcp_tool=True,
)
def create_docx(item: Item) -> Item:
    """
    Create a .docx file from basic HTML or Markdown.
    Intended to handle Markdown or HTML generated by Markdown, not generic styled HTML.
    """
    if not item.body:
        raise InvalidInput(f"Item must have a body: {item}")
    assert item.store_path

    docx_item = item.derived_copy(type=ItemType.export, format=Format.docx, file_ext=FileExt.docx)
    target_docx_path = current_ws().target_path_for(docx_item)

    content_html = item.body_as_html()

    # Trying html4docx:
    # This approach has issues with extra linebreaks if there are paragraphs inside items.
    # parser = HtmlToDocx()
    # parser.table_style = "Table Grid"  # Add borders to tables
    # docx = parser.parse_html_string(content_html)

    # Using our own simpler converter instead and write directly to the store.
    docx = SimpleHtmlToDocx().convert_html_string(content_html)
    with atomic_output_file(target_docx_path, make_parents=True) as f:
        docx.save(str(f))

    # Indicate that we've already saved the file.
    docx_item.external_path = str(target_docx_path)

    return docx_item
