import logging
import os
import platform
from datetime import datetime
from logging import ERROR
from pathlib import Path
from textwrap import dedent

from strif import atomic_output_file

from kash.config.settings import APP_NAME
from kash.web_gen.template_render import additional_template_dirs, render_web_template

log = logging.getLogger(__name__)

templates_dir = Path(__file__).parent / "templates"


def weasyprint_setup() -> None:
    is_macos = platform.system() == "Darwin"
    if is_macos:
        dylib = os.getenv("DYLD_LIBRARY_PATH")
        homebrew_lib = "/opt/homebrew/lib"
        # Manually append to DYLD_LIBRARY_PATH.
        new_dylib = f"{dylib}:{homebrew_lib}" if dylib else homebrew_lib
        # print("Setting DYLD_LIBRARY_PATH to: %s", new_dylib)
        os.environ["DYLD_LIBRARY_PATH"] = new_dylib

    try:
        import weasyprint  # noqa: F401
    except ImportError:
        if is_macos:
            log.exception(
                dedent(
                    """Failed to import weasyprint. Installing it with `brew` may help. See:
                    https://github.com/Kozea/WeasyPrint/issues/1448
                    https://github.com/astral-sh/uv/issues/6971
                    """
                )
            )
        raise

    # Reduce weasyprint noisiness.
    log_levels = {
        "weasyprint": ERROR,
        "weasyprint.progress": ERROR,
    }
    for logger_name, level in log_levels.items():
        logger = logging.getLogger(logger_name)
        logger.setLevel(level)
        logger.propagate = True


def html_to_pdf(
    html_content: str,
    pdf_out_path: Path,
    title: str | None = None,
    html_out_path: Path | None = None,
) -> None:
    """
    Converts an HTML doc to a nicely formatted PDF file. Optionally also saves
    the HTML template.
    """

    today = datetime.now().strftime("%Y-%m-%d")
    footer = f"Generated by {APP_NAME} on {today}"

    title = title or ""

    # Scale the content down a bit.
    scaled_html = f"""
        <style>
            body {{
                transform: scale(0.9);
                transform-origin: top left;
            }}
        </style>
        {html_content}
        """

    with additional_template_dirs(templates_dir):
        extra_css = render_web_template(
            "pdf_styles.css.jinja",
            {
                "title": title,
                "footer": footer,
            },
        )
        full_html = render_web_template(
            "base_webpage.html.jinja",
            {
                "title": title,
                "content": scaled_html,  # Use the scaled content
                "extra_css": extra_css,
            },
        )

    if html_out_path:
        with atomic_output_file(html_out_path, make_parents=True) as temp_file_path:
            temp_file_path.write_text(full_html)

    # Create PDF.
    weasyprint_setup()
    import weasyprint

    weasy_html = weasyprint.HTML(string=full_html)
    with atomic_output_file(pdf_out_path, make_parents=True) as temp_file_path:
        weasy_html.write_pdf(temp_file_path)
