# A Bug/Leakage Prevention and File Management System.

A lightweight, dependency-free Python module for managing structured text-based data with validation, backup, and recovery features. Ideal for small-scale data storage, prototypes, or personal tools that need structure without the overhead of a full database.

---

## 🚀 Features

- ✅ Safe read/write with structure validation (1D & 2D lists)
- ➕ Append support with consistency checks
- ❌ Granular delete with filters (cutoff, keep, reverse, size)
- 💾 Backup and 📸 Snapshot support
- 🧹 Debug and auto-clean corrupted files
- 📚 Built-in mini-guide
- 🎗️ Supports Multiple Console usage
- 🗂️ Clean folder/files organization.
- 💪 Strong anti-corruption and Tamper mechanism

---

## Why Use This Package?

* perfect for quick prototyping, automation, or lightweight apps.
* Zero setup overhead — no SQL, no migrations, just plain text files.
* Built-in concurrency and validation — reduce bugs and data corruption.
* Auto-debugging — self-healing files to avoid downtime.
* Ideal for web scraping, automation pipelines, and small apps where a full database is overkill.

## Installation

Copy the package folder into your project or install via your preferred method (if you package it for PyPI or other).

No external dependencies beyond standard Python libraries are required.


## 📦 Core Functions

### `w(txt_name: str, write_list: list) -> None`

Write (or reset) the contents of a file, validating structure before saving to all backup locations.

### `r(txt_name: str, set_new: list | None = [], notify_new: bool = True) -> list | None`

Read file contents. If the file is missing, return `set_new` and optionally notify user of new file creation.

### `a(txt_name: str, append_list: list) -> list`

Append new rows to an existing file after validating structure. Returns the updated list.

### `d(txt_name: str, ...) -> tuple[int, list]`

Delete matching rows with flexible options:
- `del_list`: values to delete
- `index`: column index for 2D deletion
- `cutoff`: max deletions per value
- `keep`: retain only N per value
- `reverse`: delete from end
- `size`: trim to max N items

### `backup(txt_name: str, display=True)`

Create a manual backup of a file or all (`txt_name="*"`).

### `snapshot(txt_name: str, unit, gap, begin=0, display=True)`

Take time-based snapshots if eligible. Supports:
- `unit`: `'minute'`, `'hour'`, `'day'`, `'month'`, etc.
- `gap`: how much time must pass
- `begin`: used for daily-based triggers

### `debug(txt_name, is_2D=None, clean=None, length=None, display=True)`

Scan and optionally auto-clean a file that fails validation. Great for corrupted data recovery.

### `help()`

Opens the interactive mini-guide documentation tool.

---

## 📁 File Organization

Each file is saved as a `.txt` in a structured folder. All backups and snapshots are handled automatically.

Other folders:
- `Backup 💾/` – Manual backups
- `Snapshot 📸/` – A timed backup

Validation files:
- `*_validation.txt` – Schema registry per file

---

# --- Usage Examples ---

# Write (w): Overwrite data
w("students", ["Alice", "Bob"])                    # 1D list
w("scores", [[1, "Math", 80], [2, "Science", 90]]) # 2D list

# Read (r): Read data, or set new if missing
r("students")                          # Read normally
r("new_file", [], notify_new=True)     # Handle missing file gracefully

# Append (a): Add new entries
a("students", ["Charlie"])                              # Append to 1D
a("scores", [[3, "English", 85], [4, "Math", 75]])      # Append to 2D

# Delete (d): Various modes

Delete by value (1D)

d("students", ["Bob"])

# Multi-row delete for ID

d("students", ["Bob", "Charlie"])

Delete by value (2D)

# Delete where row[index] == value (2D)
d("scores", [2], index=0)                          # Delete rows with row[0] == 2

# Delete limited occurrences
d("scores", ["Math"], index=1, cutoff=1)           # Delete only 1 occurrence of "Math"

# Keep only certain occurrences
d("scores", ["Math"], index=1, keep=1)             # Keep only 1 row with "Math"

# Trim list size after deletion
d("scores", size=2)                                # Keep only last 2 entries

# Reverse order deletion
d("scores", ["English"], index=1, reverse=True)    # Delete in reverse order

# Multi-index delete (like SQL WHERE col1=val1 AND col2=val2)
d("scores", [[2, "Science"]], index=[0, 1])              # Match row[0]==2 AND row[1]=="Science"
d("scores", [[1, "Math", 80]], index=[0, 1, 2] or index = "*")           # Match entire row explicitly

# OR condition: multiple rows with multi-index matching
d("scores", [[1, 80], [4, 75]], index=[0, 2])            # Delete rows where (row[0]==1 AND row[2]==80) OR (row[0]==4 AND row[2]==75)


# Backup & Snapshot
backup("students")                    # Manual backup of one file
backup("*")                           # Backup all files

snapshot("students", "day", 1)        # Snapshot once per day if gap passed
snapshot("*", "hour", 6)              # Snapshot all files every 6 hours

# Debugging
debug("students", is_2D=False, clean=True)  # Debug and clean 1D list
debug("scores", is_2D=True, length=3)       # Debug 2D list with expected 3 columns

# Help
help()  # Show usage guide




