from __future__ import annotations

import time
import logging

from .page import GridPage
from ..event import GridUIKeyEvent

from dataclasses import dataclass
from typing import TYPE_CHECKING, Callable
if TYPE_CHECKING:
    from ..ui import GridUI
    from ...grid import Grid

logger = logging.getLogger(__name__)


class GridUIControl:
    def __init__(self, grid, x: int, y: int, mode: str = None, state: int = 0, handler: Callable = None):
        """
        Represents a key on the Monome Grid.

        Args:
            grid (Grid): The grid object.
            x (int): The x position of the key, where 0 == left.
            y (int): The y position of the key, where 0 == top.
            mode (str): The mode of the key. Can be "toggle", "momentary", or "radio".
            state (int): The state of the key. 0 or 1.
            handler (Callable): The function to call when the key is pressed.
        """
        self.grid = grid
        self.x = x
        self.y = y
        self.mode = mode
        self.state = state
        self.handler = handler
        self.group = None
    

@dataclass
class GridUIControlGroup:
    index: int
    controls: list[GridUIControl]

    def __repr__(self):
        return f"GridUIControlGroup(index={self.index})"

class GridUIKeyRadioEvent(GridUIKeyEvent):
    def __init__(self, page: GridPage, x: int, y: int, down: bool, group: GridUIControlGroup, selected_index: int):
        """
        Event generated by a key up/down on Monome Grid.

        Args:
            x (int): The x position of the key, where 0 == left.
            y (int): The y position of the key, where 0 == top.
            down (bool): Whether the key is pressed down (True) or released (False).
            group (GridUIControlGroup): The group of controls this key belongs to.
            selected_index (int): The index of the selected control in the group.
        """
        super().__init__(page, x, y, down)
        self.group = group
        self.selected_index = selected_index
    
    def __repr__(self):
        return f"GridUIKeyRadioEvent(page={self.page}, x={self.x}, y={self.y}, down={self.down}, group={self.group}, selected_index={self.selected_index})"


class GridPageFreeform (GridPage):
    def __init__(self,
                 grid: GridUI):
        super().__init__(grid)

        self.keys = [[GridUIControl(grid, x, y) for x in range(grid.width)] for y in range(grid.height)]
        self.control_groups = []
    
    def _handle_grid_key(self, x: int, y: int, down: int):
        from ..event import GridUIKeyEvent
        
        logger.debug("Grid key: %d, %d, %s" % (x, y, down))
        key = self.keys[y][x]

        if key.mode is None:
            return
        elif key.mode == "toggle":
            if down:
                key.state = 1 - key.state
                self.grid.led_level_set(x, y, self.grid.led_intensity_high if key.state else self.grid.led_intensity_low)
                if key.handler:
                    event = GridUIKeyEvent(self, x, y, key.state)
                    key.handler(event)
        elif key.mode == "momentary":
            if down:
                self.grid.led_level_set(x, y, self.grid.led_intensity_high)
            else:
                self.grid.led_level_set(x, y, self.grid.led_intensity_low)
            if key.handler:
                event = GridUIKeyEvent(self, x, y, down)
                key.handler(event)
        elif key.mode == "radio":
            if down:
                group = key.group
                for other_key in group.controls:
                    if other_key.state:
                        other_key.state = 0
                        self.grid.led_level_set(other_key.x, other_key.y, self.grid.led_intensity_low)
                key.state = 1
                self.grid.led_level_set(key.x, key.y, self.grid.led_intensity_high)
                selected_index = group.controls.index(key)
                if key.handler:
                    event = GridUIKeyRadioEvent(self, x, y, down, group=group, selected_index=selected_index)
                    key.handler(event)
    
    def add_control(self, mode: str, x: int, y: int, handler: Callable, group: GridUIControlGroup = None):
        key = self.keys[y][x]
        if key.mode is not None:
            raise ValueError("Key already has a control assigned (%d, %d)" % (x, y))
        key.mode = mode
        key.handler = handler
        key.group = group
        if group:
            group.controls.append(key)
            if len(group.controls) == 1:
                key.state = 1
        self.draw()
    
    def remove_control(self, x: int, y: int):
        key = self.keys[y][x]
        key.mode = None
        key.handler = None
        key.group = None
        self.draw()

    # To enable @control_for_key decorator
    def control_for_key(self, mode: str, x: int, y: int):
        def wrapper(handler):
            self.add_control_for_key(mode, x, y, handler)
        return wrapper
    
    def draw(self):
        for y, row in enumerate(self.keys):
            for x, key in enumerate(row):
                if key.mode is None:
                    pass
                elif key.mode == "momentary":
                    self.grid.led_level_set(key.x, key.y, self.grid.led_intensity_low)
                elif key.mode == "toggle":
                    self.grid.led_level_set(key.x, key.y, self.grid.led_intensity_high if key.state else self.grid.led_intensity_low)
                elif key.mode == "radio":
                    self.grid.led_level_set(key.x, key.y, self.grid.led_intensity_high if key.state else self.grid.led_intensity_low)

    
    def add_control_group(self):
        control_group = GridUIControlGroup(len(self.control_groups), [])
        self.control_groups.append(control_group)
        return control_group

if __name__ == "__main__":
    from ..ui import GridUI

    gridui = GridUI()
    page = gridui.add_page("freeform")

    page.add_control("toggle", x=0, y=0, handler=print)
    page.add_control("momentary", x=2, y=0, handler=print)
    page.add_control("momentary", x=3, y=0, handler=print)
    page.add_control("momentary", x=4, y=0, handler=print)

    radio_group = page.add_control_group()
    page.add_control("radio", 0, 2, handler=print, group=radio_group)
    page.add_control("radio", 1, 2, handler=print, group=radio_group)
    page.add_control("radio", 2, 2, handler=print, group=radio_group)
    
    while True:
        time.sleep(1)