"""
Example usage of the Maybank For Me API (uses httpx)
"""

import httpx


# Example 1: Process a single PDF file
def process_single_pdf():
    url = "http://localhost:8000/process"

    with open("Statement_20231215.pdf", "rb") as pdf_file:
        files = {"files": ("Statement_20231215.pdf", pdf_file, "application/pdf")}
        data = {"password": ""}  # Add password if PDF is encrypted

        response = httpx.post(url, files=files, data=data, timeout=60.0)

    if response.status_code == 200:
        # Save CSV response
        with open("transactions.csv", "wb") as f:
            f.write(response.content)
        print("Success! Transactions saved to transactions.csv")
    else:
        print(f"Error: {response.status_code}")
        print(response.json())


# Example 2: Process multiple PDF files
def process_multiple_pdfs():
    url = "http://localhost:8000/process"

    # Open all files using context managers
    with (
        open("Statement_20231215.pdf", "rb") as f1,
        open("Statement_20240115.pdf", "rb") as f2,
        open("Statement_20240215.pdf", "rb") as f3,
    ):
        files = [
            ("files", ("Statement_20231215.pdf", f1, "application/pdf")),
            ("files", ("Statement_20240115.pdf", f2, "application/pdf")),
            ("files", ("Statement_20240215.pdf", f3, "application/pdf")),
        ]
        data = {"password": "my-secure-password"}

        response = httpx.post(url, files=files, data=data, timeout=120.0)

        if response.status_code == 200:
            with open("all_transactions.csv", "wb") as f:
                f.write(response.content)
            print("Success! All transactions saved to all_transactions.csv")
        else:
            print(f"Error: {response.status_code}")
            try:
                print(response.json())
            except Exception:
                print("Non-JSON error response:")
                print(response.text[:500])


# Example 3: Using curl (from command line)
"""
# Single file
curl -X POST "http://localhost:8000/process" \
  -F "files=@Statement_20231215.pdf" \
  -F "password=" \
  -o transactions.csv

# Multiple files
curl -X POST "http://localhost:8000/process" \
  -F "files=@Statement_20231215.pdf" \
  -F "files=@Statement_20240115.pdf" \
  -F "password=my-secure-password" \
  -o all_transactions.csv
"""


# Example 4: Check API health
def check_health():
    response = httpx.get("http://localhost:8000/health", timeout=10.0)
    print(response.json())  # {'status': 'healthy'}


if __name__ == "__main__":
    # Uncomment the example you want to run
    # process_single_pdf()
    # process_multiple_pdfs()
    check_health()
