"""CLI tool for processing PDF files and converting to CSV.

This module provides the command-line interface for the maybankforme tool.
"""

import argparse
import os

from .process_transaction import process_transaction


def main() -> None:
    """Main entry point for the CLI application.

    Parses command-line arguments and processes PDF files to extract
    transaction data into a CSV file.

    Command-line Arguments:
        input_folder: Folder containing PDF files
        output_file: CSV file to save transactions
        --password: Password to open PDF files (optional)
        --dataset_folder: Folder containing dataset (default: 'dataset')

    Environment Variables:
        PDF_PASSWORD: Alternative way to provide PDF password

    Examples:
        $ maybankforme /data/pdfs output.csv --password=secret
        $ PDF_PASSWORD=secret maybankforme /data/pdfs output.csv
    """
    parser = argparse.ArgumentParser(
        description="Process Maybank credit card statement PDF files to CSV"
    )
    parser.add_argument("input_folder", type=str, help="Folder containing PDF files")
    parser.add_argument("output_file", type=str, help="CSV file to save transactions")
    parser.add_argument(
        "--password", type=str, help="Password to open PDF files", default=""
    )
    parser.add_argument(
        "--dataset_folder",
        type=str,
        help="Folder containing dataset",
        default="dataset",
    )
    args = parser.parse_args()
    password = os.environ.get("PDF_PASSWORD", args.password)
    process_transaction(
        args.input_folder, args.output_file, password, args.dataset_folder
    )


if __name__ == "__main__":
    main()
