"""Tests for text to CSV conversion functions."""

from pathlib import Path

import pytest

from maybankforme.common.txt_convert_csv import (
    regex_match_credit,
    regex_match_transaction,
    txt_to_csv,
)


def test_regex_match_transaction_valid() -> None:
    """Test regex_match_transaction with valid transaction line."""
    line = "12/01 13/01 PURCHASE AT STORE 100.50"
    result = regex_match_transaction(line)
    assert result is not None
    assert result == ("12/01", "13/01", " PURCHASE AT STORE", "100.50")


def test_regex_match_transaction_invalid() -> None:
    """Test regex_match_transaction with invalid line."""
    line = "This is not a transaction"
    result = regex_match_transaction(line)
    assert result is None


def test_regex_match_transaction_with_comma_amount() -> None:
    """Test regex_match_transaction with comma-separated amount."""
    line = "12/01 13/01 BIG PURCHASE 1,234.56"
    result = regex_match_transaction(line)
    assert result is not None
    assert result[3] == "1,234.56"


def test_regex_match_credit_valid() -> None:
    """Test regex_match_credit with valid credit line."""
    line = "12/01 13/01 PAYMENT CR"
    result = regex_match_credit(line)
    assert result is not None
    assert result == ("12/01", "13/01", "PAYMENT ")


def test_regex_match_credit_invalid() -> None:
    """Test regex_match_credit with non-credit line."""
    line = "12/01 13/01 PURCHASE 100.50"
    result = regex_match_credit(line)
    assert result is None


def test_txt_to_csv_creates_file(
    temp_dir: Path, sample_txt_content: str, expected_csv_rows: list[list[str]]
) -> None:
    """Test that txt_to_csv creates a CSV file."""
    txt_path = temp_dir / "test.txt"
    csv_path = temp_dir / "test.csv"

    # Write sample text file
    txt_path.write_text(sample_txt_content, encoding="utf-8")

    # Convert to CSV
    txt_to_csv(str(txt_path), str(csv_path))

    # Verify CSV was created
    assert csv_path.exists()

    # Read and verify content
    content = csv_path.read_text(encoding="utf-8")
    lines = content.strip().split("\n")

    # Should have header plus data rows
    assert len(lines) == len(expected_csv_rows) + 1
    assert "Posting Date" in lines[0]


def test_txt_to_csv_filters_credit_transactions(temp_dir: Path) -> None:
    """Test that credit transactions are filtered out."""
    txt_path = temp_dir / "test.txt"
    csv_path = temp_dir / "test.csv"

    # Text with credit transaction
    content = "12/01 13/01 PAYMENT CR\n14/01 15/01 PURCHASE 100.00\n"
    txt_path.write_text(content, encoding="utf-8")

    txt_to_csv(str(txt_path), str(csv_path))

    # Read CSV - should only have purchase, not credit
    csv_content = csv_path.read_text(encoding="utf-8")
    lines = csv_content.strip().split("\n")

    # Header + 1 data row (credit filtered out)
    assert len(lines) == 2
    assert "PAYMENT" not in csv_content


def test_txt_to_csv_filters_management_charges(temp_dir: Path) -> None:
    """Test that management charges are filtered out."""
    txt_path = temp_dir / "test.txt"
    csv_path = temp_dir / "test.csv"

    content = "12/01 13/01 RTL MGMT CHRG RATE 5.00\n14/01 15/01 PURCHASE 100.00\n"
    txt_path.write_text(content, encoding="utf-8")

    txt_to_csv(str(txt_path), str(csv_path))

    csv_content = csv_path.read_text(encoding="utf-8")
    lines = csv_content.strip().split("\n")

    # Header + 1 data row (mgmt charge filtered out)
    assert len(lines) == 2
    assert "MGMT" not in csv_content


def test_txt_to_csv_file_not_found(temp_dir: Path) -> None:
    """Test that txt_to_csv raises error for missing file."""
    txt_path = temp_dir / "nonexistent.txt"
    csv_path = temp_dir / "output.csv"

    with pytest.raises(FileNotFoundError):
        txt_to_csv(str(txt_path), str(csv_path))
