"""Tests for utility functions and classes."""

import structlog

from maybankforme.common.utils import DockerSafeLogger, configure_logging, get_logger


def test_configure_logging() -> None:
    """Test that logging configuration works correctly."""
    configure_logging()
    assert structlog.is_configured()


def test_get_logger() -> None:
    """Test that get_logger returns a structlog logger."""
    configure_logging()
    log = get_logger("test")
    assert log is not None
    # structlog may return a proxy or BoundLogger
    assert hasattr(log, "info") and hasattr(log, "error")


def test_docker_safe_logger_initialization() -> None:
    """Test that DockerSafeLogger initializes correctly."""
    logger = DockerSafeLogger()
    assert logger.logger is not None
    # structlog returns BoundLogger or proxy, not stdlib Logger
    assert isinstance(logger.logger, structlog.stdlib.BoundLogger) or hasattr(
        logger.logger, "_logger"
    )


def test_docker_safe_logger_delegation() -> None:
    """Test that logger methods are delegated correctly."""
    logger = DockerSafeLogger()
    # Should be able to call standard logging methods
    assert callable(logger.info)
    assert callable(logger.error)
    assert callable(logger.warning)
    assert callable(logger.debug)


def test_docker_safe_logger_compatibility() -> None:
    """Test that DockerSafeLogger maintains backward compatibility."""
    logger = DockerSafeLogger()
    # Should be able to log without errors
    logger.info("test message")
    logger.debug("debug message")
    logger.warning("warning message")
    logger.error("error message")
