"""Pytest configuration and shared fixtures for tests."""

import tempfile
from collections.abc import Generator
from pathlib import Path

import pytest


@pytest.fixture
def temp_dir() -> Generator[Path, None, None]:
    """Create a temporary directory for tests.

    Yields:
        Path to temporary directory that is cleaned up after test
    """
    with tempfile.TemporaryDirectory() as tmp_dir:
        yield Path(tmp_dir)


@pytest.fixture
def sample_pdf_path(temp_dir: Path) -> Path:
    """Create a sample PDF file path for testing.

    Args:
        temp_dir: Temporary directory fixture

    Returns:
        Path to a sample PDF file location
    """
    return temp_dir / "Statement_20240131.pdf"


@pytest.fixture
def sample_txt_content() -> str:
    """Sample transaction text content for testing.

    Returns:
        String containing sample transaction data
    """
    return """12/01 13/01 PURCHASE AT STORE 100.50
14/01 14/01 ONLINE SHOPPING 250.00
15/01 15/01 RTL MGMT CHRG RATE 5.00
16/01 17/01 PAYMENT CR 500.00
18/01 19/01 GROCERY SHOPPING 75.25
"""


@pytest.fixture
def expected_csv_rows() -> list[list[str]]:
    """Expected CSV rows after processing sample text.

    Returns:
        List of expected transaction rows
    """
    return [
        ["12/01", "13/01", "PURCHASE AT STORE", "100.50"],
        ["14/01", "14/01", "ONLINE SHOPPING", "250.00"],
        ["18/01", "19/01", "GROCERY SHOPPING", "75.25"],
    ]
