"""Tests for API endpoints."""

from io import BytesIO

from fastapi.testclient import TestClient

from maybankforme.api import app

client = TestClient(app)


def test_root_endpoint() -> None:
    """Test the root endpoint returns API information."""
    response = client.get("/")
    assert response.status_code == 200
    data = response.json()
    assert "name" in data
    assert "version" in data
    assert data["name"] == "Maybank For Me API"


def test_health_endpoint() -> None:
    """Test the health check endpoint."""
    response = client.get("/health")
    assert response.status_code == 200
    data = response.json()
    assert data["status"] == "healthy"


def test_process_endpoint_no_files() -> None:
    """Test process endpoint with no files returns error."""
    response = client.post("/process", data={"password": ""})
    assert response.status_code == 422  # Unprocessable Entity


def test_process_endpoint_invalid_file_type() -> None:
    """Test process endpoint with non-PDF file returns error."""
    files = {"files": ("test.txt", BytesIO(b"content"), "text/plain")}
    response = client.post("/process", files=files, data={"password": ""})
    assert response.status_code == 400
    assert "PDF files" in response.json()["detail"]


def test_process_endpoint_missing_filename() -> None:
    """Test process endpoint validates filename presence."""
    # Create file upload without filename - FastAPI returns 422 for validation errors
    files = {"files": (None, BytesIO(b"content"), "application/pdf")}
    response = client.post("/process", files=files, data={"password": ""})
    # FastAPI returns 422 for validation errors (Unprocessable Entity)
    assert response.status_code == 422
