from scipy import stats
from ecdfgof import kstest
from ecdfgof import adtest
from ecdfgof import cvmtest
import conversions
from typing import Dict, Any


def utests(sample: conversions.VectorLike) -> Dict[str, Any]:
    """
    Perform multiple goodness-of-fit tests for uniformity on a sample.

    This function tests the null hypothesis that the input sample was
    generated by a standard uniform distribution. It returns p-values for
    several tests: Kolmogorov-Smirnov, Anderson-Darling,
    Cramer-von Mises, and Cook-Gelman-Rubin.

    Parameters
    ----------
    sample : VectorLike
        1-D vector-like sample to be tested for uniformity. 

    Returns
    -------
    dict of str to float
        Dictionary with p-values for each test. If the sample contains
        values outside [0, 1], all p-values are None and a message is
        included.
    """
    sample_np = conversions.vlToArray(sample)

    if min(sample_np) < 0.0 or max(sample_np) > 1.0:
        return {"cook_gelman_rubin_pval": None,
                "kolmogorov_smirnov_pval": None,
                "anderson_darling_pval": None,
                "cramer_von_mises_pval": None,
                "message": "items in sample were outside the unit interval"}

    # KS & AD tests
    ks_pval = kstest(sample_np, stats.uniform())[1]
    ad_pval = adtest(sample_np, stats.uniform())[1]
    cvm_pval = cvmtest(sample_np, stats.uniform())[1]

    # chi-squared-based test (Cook, Gelman, Rubin)
    N = len(sample_np)
    stat = sum(map(lambda x: stats.norm().ppf(x)**2.0, sample_np))
    cgr_pval = 1.0 - stats.chi2.cdf(stat, N)

    return {"cook_gelman_rubin_pval": cgr_pval,
            "kolmogorov_smirnov_pval": ks_pval,
            "anderson_darling_pval": ad_pval,
            "cramer_von_mises_pval": cvm_pval,}