/**
 * BioLM API Client
 */
import { BioLMModel, BioLMAPIModel, APIError } from './types';

// Proxy endpoints
const MODELS_ENDPOINT_PROXY = '/biolm/api/models';
const TEST_CONNECTION_ENDPOINT_PROXY = '/biolm/api/test-connection';
const CONFIG_ENDPOINT = '/biolm/api/config';

// Default values (will be overridden by config)
const DEFAULT_CACHE_TTL = 5 * 60 * 1000; // 5 minutes
const DEFAULT_API_BASE_URL = 'https://biolm.ai/api/ui/community-api-models/';

interface CacheEntry {
  data: BioLMModel[];
  timestamp: number;
}

interface APIConfig {
  useProxy: boolean;
  apiBaseUrl: string | null;
  requestTimeout: number;
  cacheTtl: number;
}

let cache: CacheEntry | null = null;
let config: APIConfig | null = null;
let configPromise: Promise<APIConfig> | null = null;

/**
 * Fetch configuration from server
 */
async function getConfig(): Promise<APIConfig> {
  // Return cached config if available
  if (config) {
    return config;
  }
  
  // Return existing promise if one is in flight
  if (configPromise) {
    return configPromise;
  }
  
  // Fetch config from server
  configPromise = (async () => {
    try {
      const response = await fetch(CONFIG_ENDPOINT, {
        method: 'GET',
        credentials: 'same-origin',
      });
      
      if (response.ok) {
        const serverConfig = await response.json();
        const useProxy = serverConfig.useProxy ?? true;
        // If not using proxy, apiBaseUrl should be provided; otherwise use default
        const apiBaseUrl = useProxy 
          ? DEFAULT_API_BASE_URL  // Not used when proxy is enabled, but set for consistency
          : (serverConfig.apiBaseUrl || DEFAULT_API_BASE_URL);
        config = {
          useProxy: useProxy,
          apiBaseUrl: apiBaseUrl,
          requestTimeout: serverConfig.requestTimeout ?? 10,
          cacheTtl: serverConfig.cacheTtl ?? DEFAULT_CACHE_TTL,
        };
        console.log('[BioLM] Config loaded:', config);
        return config;
      } else {
        // If config endpoint doesn't exist (404), server extension likely not loaded
        // Fall back to direct API calls instead of proxy
        if (response.status === 404) {
          console.warn('[BioLM] Config endpoint not found (404) - server extension may not be loaded. Using direct API calls as fallback.');
          config = {
            useProxy: false,  // Try direct instead of proxy when extension not available
            apiBaseUrl: DEFAULT_API_BASE_URL,
            requestTimeout: 10,
            cacheTtl: DEFAULT_CACHE_TTL,
          };
        } else {
          // Other errors, default to proxy (extension might be loaded but having issues)
          console.warn('[BioLM] Failed to fetch config (status:', response.status, '), using proxy defaults');
          config = {
            useProxy: true,
            apiBaseUrl: DEFAULT_API_BASE_URL,
            requestTimeout: 10,
            cacheTtl: DEFAULT_CACHE_TTL,
          };
        }
        return config;
      }
    } catch (error) {
      // Network errors or fetch failures - likely means server extension not available
      // Fall back to direct API calls
      console.warn('[BioLM] Error fetching config (network/fetch error), using direct API calls as fallback:', error);
      config = {
        useProxy: false,  // Try direct instead of proxy when extension not available
        apiBaseUrl: DEFAULT_API_BASE_URL,
        requestTimeout: 10,
        cacheTtl: DEFAULT_CACHE_TTL,
      };
      return config;
    } finally {
      configPromise = null;
    }
  })();
  
  return configPromise;
}

/**
 * Convert API model to internal model format
 */
function convertAPIModel(apiModel: BioLMAPIModel): BioLMModel {
  // Derive actions from boolean flags
  const actions: string[] = [];
  if (apiModel.predictor) actions.push('predict');
  if (apiModel.encoder) actions.push('encode');
  if (apiModel.generator) actions.push('generate');
  if (apiModel.classifier) actions.push('classify');
  if (apiModel.similarity) actions.push('similarity');
  if (apiModel.explainer) actions.push('explain');
  if (apiModel.transformer) actions.push('transform');
  
  // If no specific actions, default to predict
  if (actions.length === 0) {
    actions.push('predict');
  }

  return {
    id: apiModel.model_slug,
    name: apiModel.model_name,
    description: apiModel.description,
    tags: apiModel.tags || [],
    apiVersion: apiModel.api_version,
    communityModel: apiModel.community_model,
    companyModel: apiModel.company_model,
    actions: actions,
    apiDocsLink: apiModel.api_docs_link,
  };
}

/**
 * Fetch models from BioLM API
 */
export async function fetchModels(apiKey?: string): Promise<BioLMModel[]> {
  const apiConfig = await getConfig();
  
  // Check cache first
  if (cache && Date.now() - cache.timestamp < apiConfig.cacheTtl) {
    return cache.data;
  }

  let url: string;
  const headers: HeadersInit = {
    'Content-Type': 'application/json',
  };
  
  if (apiKey) {
    headers['Authorization'] = `Bearer ${apiKey}`;
  }

  if (apiConfig.useProxy) {
    // Use server proxy to avoid CORS issues
    console.log('[BioLM] Fetching models via server proxy...');
    url = MODELS_ENDPOINT_PROXY;
    if (apiKey) {
      url += `?api_key=${encodeURIComponent(apiKey)}`;
    }
  } else {
    // Direct API call
    console.log('[BioLM] Fetching models via direct API call...');
    url = apiConfig.apiBaseUrl || DEFAULT_API_BASE_URL;
  }

  try {
    let response = await fetch(url, {
      method: 'GET',
      headers: headers,
      credentials: apiConfig.useProxy ? 'same-origin' : 'omit',
    });
    
    console.log('[BioLM] Response received:', response.status);
    console.log('[BioLM] Response status:', response.status, response.statusText);
    console.log('[BioLM] Response URL:', response.url);
    console.log('[BioLM] Response ok:', response.ok);

    if (!response.ok) {
      // Try to get error details from response
      let errorMessage = `Failed to fetch models: ${response.status} ${response.statusText}`;
      let errorDetails: any = null;
      try {
        const text = await response.text();
        console.log('[BioLM] Error response body:', text);
        try {
          errorDetails = JSON.parse(text);
          if (errorDetails.message) {
            errorMessage = errorDetails.message;
          } else if (errorDetails.error) {
            errorMessage = errorDetails.error;
          }
        } catch {
          // If not JSON, use the text
          if (text) {
            errorMessage = `${errorMessage}: ${text.substring(0, 200)}`;
          }
        }
      } catch (e) {
        console.error('[BioLM] Error reading response:', e);
      }
      
      console.error('[BioLM] API Error:', errorMessage, errorDetails);
      const error: APIError = {
        message: errorMessage,
        status: response.status,
      };
      throw error;
    }

    const data: BioLMAPIModel[] = await response.json();
    
    // Convert API models to internal format
    const models: BioLMModel[] = Array.isArray(data)
      ? data.map(convertAPIModel)
      : [];

    // Update cache
    cache = {
      data: models,
      timestamp: Date.now(),
    };

    return models;
  } catch (error) {
    console.error('[BioLM] Fetch failed:', error);
    
    // If we have cached data, return it even if stale
    if (cache) {
      console.warn('Using cached models due to fetch error:', error);
      return cache.data;
    }
    
    // Re-throw with more helpful error message
    if (error instanceof Error) {
      let errorMessage = error.message;
      if (error.message.includes('fetch') || error.message.includes('CORS') || error.message.includes('Failed to fetch')) {
        if (apiConfig.useProxy) {
          errorMessage = 'Unable to connect to BioLM API via server proxy. Please check your internet connection and ensure the server extension is loaded.';
        } else {
          errorMessage = 'Unable to connect to BioLM API. Please check your internet connection and CORS configuration.';
        }
      }
      
      const apiError: APIError = {
        message: errorMessage,
      };
      throw apiError;
    }
    throw error;
  }
}

/**
 * Clear the models cache
 */
export function clearCache(): void {
  cache = null;
}

/**
 * Fetch detailed model information including code examples
 */
export async function fetchModelDetails(
  modelSlug: string,
  apiKey?: string
): Promise<BioLMAPIModel> {
  const apiConfig = await getConfig();
  
  let url: string;
  const headers: HeadersInit = {
    'Content-Type': 'application/json',
  };
  
  if (apiKey) {
    headers['Authorization'] = `Bearer ${apiKey}`;
  }

  if (apiConfig.useProxy) {
    // Use server proxy to avoid CORS issues
    console.log('[BioLM] Fetching model details via server proxy...');
    url = `/biolm/api/models/${modelSlug}`;
    if (apiKey) {
      url += `?api_key=${encodeURIComponent(apiKey)}`;
    }
  } else {
    // Direct API call
    console.log('[BioLM] Fetching model details via direct API call...');
    url = `${apiConfig.apiBaseUrl || DEFAULT_API_BASE_URL}${modelSlug}/?code_examples=true&exclude_docs_html=true`;
  }

  try {
    const response = await fetch(url, {
      method: 'GET',
      headers: headers,
      credentials: apiConfig.useProxy ? 'same-origin' : 'omit',
    });
    
    console.log('[BioLM] Response received:', response.status);

    if (!response.ok) {
      let errorMessage = `Failed to fetch model details: ${response.status} ${response.statusText}`;
      try {
        const text = await response.text();
        try {
          const errorDetails = JSON.parse(text);
          if (errorDetails.message) {
            errorMessage = errorDetails.message;
          } else if (errorDetails.error) {
            errorMessage = errorDetails.error;
          }
        } catch {
          if (text) {
            errorMessage = `${errorMessage}: ${text.substring(0, 200)}`;
          }
        }
      } catch (e) {
        console.error('[BioLM] Error reading response:', e);
      }
      
      const error: APIError = {
        message: errorMessage,
        status: response.status,
      };
      throw error;
    }

    const data: BioLMAPIModel = await response.json();
    return data;
  } catch (error) {
    console.error('[BioLM] Fetch failed:', error);
    
    if (error instanceof Error) {
      const apiError: APIError = {
        message: error.message,
      };
      throw apiError;
    }
    throw error;
  }
}

/**
 * Test API connection with authentication
 */
export async function testConnection(apiKey: string): Promise<{ valid: boolean; message?: string; accountInfo?: any }> {
  const apiConfig = await getConfig();
  
  let url: string;
  const headers: HeadersInit = {
    'Content-Type': 'application/json',
    'Authorization': `Bearer ${apiKey}`,
  };

  if (apiConfig.useProxy) {
    // Use server proxy to avoid CORS issues
    console.log('[BioLM] Testing connection via server proxy...');
    url = TEST_CONNECTION_ENDPOINT_PROXY;
  } else {
    // Direct API call
    console.log('[BioLM] Testing connection via direct API call...');
    url = apiConfig.apiBaseUrl || DEFAULT_API_BASE_URL;
  }

  try {
    const response = await fetch(url, {
      method: apiConfig.useProxy ? 'POST' : 'GET',
      headers: headers,
      credentials: apiConfig.useProxy ? 'same-origin' : 'omit',
      body: apiConfig.useProxy ? JSON.stringify({ api_key: apiKey }) : undefined,
    });

    if (response.ok) {
      let result: any;
      if (apiConfig.useProxy) {
        result = await response.json();
      } else {
        // For direct API calls, any 200 response means success
        result = {
          valid: true,
          message: 'Connection successful',
        };
      }
      return {
        valid: result.valid !== false,
        message: result.message || 'Connection test completed',
        accountInfo: result.accountInfo,
      };
    } else {
      let errorMessage = `HTTP ${response.status}: ${response.statusText}`;
      try {
        const error = await response.json().catch(() => ({}));
        if (error.message) {
          errorMessage = error.message;
        }
      } catch (e) {
        // Ignore JSON parse errors
      }
      return {
        valid: false,
        message: errorMessage,
      };
    }
  } catch (error) {
    return {
      valid: false,
      message: error instanceof Error ? error.message : 'Connection failed',
    };
  }
}
