from selenium.webdriver.common.by import By
from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.remote.webelement import WebElement
from robo_appian.utils.ComponentUtils import ComponentUtils


class ButtonUtils:
    """
    Utility class for interacting with button components in Appian UI.
    Usage Example:
        # Click a button by its label
        from robo_appian.components.ButtonUtils import ButtonUtils
        ButtonUtils.clickByLabelText(wait, "Submit")
    """

    @staticmethod
    def __click(wait: WebDriverWait, component: WebElement):
        wait.until(EC.element_to_be_clickable(component))
        component.click()

    @staticmethod
    def _findByPartialLabelText(wait: WebDriverWait, label: str):
        """
        Finds a button by its label.

        Parameters:
            wait: Selenium WebDriverWait instance.
            label: The label of the button to find.
            label: The label of the button to find.

        Returns:
            WebElement representing the button.

        Example:
            component = ButtonUtils._findByPartialLabelText(wait, "Submit")
        """
        xpath = f".//button[./span[contains(translate(normalize-space(.), '\u00a0', ' '), '{label}')]]"
        try:
            component = ComponentUtils.findVisibleComponentByXpath(wait, xpath)
        except Exception as e:
            raise RuntimeError(f"Button with label '{label}' not found or not clickable.") from e
        return component

    @staticmethod
    def __findByLabelText(wait: WebDriverWait, label: str):
        xpath = f".//button[./span[normalize-space(.)='{label}']]"
        try:
            component = ComponentUtils.findVisibleComponentByXpath(wait, xpath)
        except Exception as e:
            raise RuntimeError(f"Button with label '{label}' not found.") from e
        return component

    @staticmethod
    def clickByPartialLabelText(wait: WebDriverWait, label: str):
        """Finds a button by its partial label and clicks it.

        Parameters:
            wait: Selenium WebDriverWait instance.
            label: The partial label of the button to click.
            Example:
                ButtonUtils.clickByPartialLabelText(wait, "Button Label")
        """
        component = ButtonUtils._findByPartialLabelText(wait, label)
        ButtonUtils.__click(wait, component)

    @staticmethod
    def clickByLabelText(wait: WebDriverWait, label: str):
        """Finds a button by its label and clicks it.

        Parameters:
            wait: Selenium WebDriverWait instance.
            label: The label of the button to click.
            Example:
                ButtonUtils.clickByLabelText(wait, "Button Label")
        """
        component = ButtonUtils.__findByLabelText(wait, label)
        ButtonUtils.__click(wait, component)

    @staticmethod
    def clickById(wait: WebDriverWait, id: str):
        """
        Finds and clicks an input button by its HTML id attribute.

        Parameters:
            wait: Selenium WebDriverWait instance.
            id: The HTML id of the input button.

        Example:
            ButtonUtils.clickById(wait, "button_id")

        """
        try:
            component = wait.until(EC.element_to_be_clickable((By.ID, id)))
        except Exception as e:
            raise RuntimeError(f"Input button with id '{id}' not found or not clickable.") from e

        ButtonUtils.__click(wait, component)

    @staticmethod
    def checkButtonExistsByLabelText(wait: WebDriverWait, label: str):
        try:
            ButtonUtils.__findByLabelText(wait, label)
        except Exception:
            return False
        return True

    @staticmethod
    def checkButtonExistsByPartialLabelText(wait: WebDriverWait, label: str):
        try:
            ButtonUtils.__findByLabelText(wait, label)
        except Exception:
            return False
        return True
