import numpy as np
from numpy.random import RandomState

from flatland.core.grid.rail_env_grid import RailEnvTransitions
from flatland.envs.agent_utils import EnvAgent
from flatland.envs.distance_map import DistanceMap
from flatland.envs.rail_grid_transition_map import RailGridTransitionMap
from flatland.envs.rail_trainrun_data_structures import Waypoint
from flatland.envs.timetable_generators import timetable_generator


def test_timetable_generator():
    env_agents = [
        EnvAgent(handle=0, initial_position=(11, 40), initial_direction=3, target=(39, 8), direction=3, waypoints_latest_arrival=None,
                 waypoints_earliest_departure=None),
        EnvAgent(handle=1, initial_position=(38, 8), initial_direction=1, target=(10, 40), direction=1, waypoints_latest_arrival=None,
                 waypoints_earliest_departure=None),
        EnvAgent(handle=2, initial_position=(17, 5), initial_direction=3, target=(42, 22), direction=3, waypoints_latest_arrival=None,
                 waypoints_earliest_departure=None),
        EnvAgent(handle=3, initial_position=(41, 22), initial_direction=3, target=(18, 5), direction=3, waypoints_latest_arrival=None,
                 waypoints_earliest_departure=None),
        EnvAgent(handle=4, initial_position=(11, 40), initial_direction=1, target=(39, 8), direction=1, waypoints_latest_arrival=None,
                 waypoints_earliest_departure=None),
        EnvAgent(handle=5, initial_position=(38, 8), initial_direction=3, target=(12, 40), direction=3, waypoints_latest_arrival=None,
                 waypoints_earliest_departure=None),
        EnvAgent(handle=6, initial_position=(38, 8), initial_direction=1, target=(31, 27), direction=1, waypoints_latest_arrival=None,
                 waypoints_earliest_departure=None),
        EnvAgent(handle=7, initial_position=(31, 26), initial_direction=0, target=(39, 8), direction=0, waypoints_latest_arrival=None,
                 waypoints_earliest_departure=None),
        EnvAgent(handle=8, initial_position=(41, 22), initial_direction=1, target=(8, 27), direction=1, waypoints_latest_arrival=None,
                 waypoints_earliest_departure=None),
        EnvAgent(handle=9, initial_position=(9, 27), initial_direction=3, target=(44, 22), direction=3, waypoints_latest_arrival=None,
                 waypoints_earliest_departure=None),
    ]
    agents_hints = {
        'city_positions': [(31, 26), (10, 40), (44, 34), (7, 27), (17, 5), (27, 41), (20, 24), (38, 8), (8, 16), (42, 22)],
        'train_stations': [[((31, 26), 0), ((31, 27), 1)], [((9, 40), 0), ((10, 40), 1), ((11, 40), 2), ((12, 40), 3)],
                           [((44, 34), 0), ((45, 34), 1)], [((7, 27), 0), ((8, 27), 1), ((9, 27), 2), ((10, 27), 3)], [((17, 5), 0), ((18, 5), 1)],
                           [((27, 40), 0), ((27, 41), 1), ((27, 42), 2), ((27, 43), 3)],
                           [((20, 24), 0), ((20, 25), 1), ((20, 26), 2), ((20, 27), 3)], [((38, 8), 0), ((39, 8), 1)],
                           [((7, 16), 0), ((8, 16), 1), ((9, 16), 2), ((10, 16), 3)], [((41, 22), 0), ((42, 22), 1), ((43, 22), 2), ((44, 22), 3)]],
        'city_orientations': [0, 3, 3, 3,
                              1, 0, 2, 1,
                              1, 1]
    }
    rail = RailGridTransitionMap(50, 50, RailEnvTransitions())
    rail.grid = np.array(
        [[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16386, 1025, 4608, 0, 0, 0, 16386, 1025, 1025, 1025, 1025, 17411, 1025, 5633, 1025, 1025, 1025, 1025,
          1025, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16386, 1025, 1025, 5633, 17411, 3089, 1025, 1097, 5633, 17411, 1025, 3089, 1025, 1025, 5633, 1025, 3089, 1025, 1097,
          1025, 17411, 1025, 1025, 1025, 38505, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 16386, 1025, 1097, 3089, 5633, 1025, 17411, 1097, 3089, 1025, 1025, 17411, 1025, 1097, 1025, 5633, 1025, 17411,
          1025, 3089, 1025, 5633, 4608, 72, 37408, 0, 0, 0, 16386, 1025, 4608, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 72, 1025, 2064, 0, 0, 0, 0, 32800, 0, 0, 0, 72, 1025, 2064, 0, 0, 0, 72, 1097, 1025, 1097, 1025,
          5633, 17411, 3089, 1025, 1097, 5633, 17411, 1025, 4608, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 3089, 5633, 1025,
          17411, 1097, 2064, 0, 32800, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 5633, 5633, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 1025, 2064,
          0, 0, 0, 32800, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49186, 34864, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 72, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          0, 0, 32800, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 72, 4608, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          0, 0, 32800, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 4608, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          16386, 17411, 2064, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          16386, 2064, 32800, 0, 0, 0, 0, 0],
         [16386, 1025, 5633, 17411, 1025, 1025, 1025, 5633, 17411, 1025, 3089, 2064, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32872, 37408, 0, 0, 0, 0, 0, 0,
          0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 16386, 34864, 0, 0, 0, 0, 0],
         [32800, 0, 72, 3089, 1025, 1025, 1025, 1097, 2064, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          0, 0, 0, 16386, 33825, 33825, 2064, 0, 0, 0, 0, 0],
         [32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49186, 34864, 32872, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 32800, 0, 0, 0, 0, 0, 0],
         [32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 32800, 0, 0, 0, 0, 0, 0],
         [32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32872, 37408, 49186, 2064, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 32800, 0, 0, 0, 0, 0, 0],
         [72, 5633, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          49186, 2064, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 49186, 34864, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32872,
          37408, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 38505, 37408, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49186,
          34864, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 37408, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16386, 34864,
          32872, 4608, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800,
          32800, 32800, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32872, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 37408,
          49186, 2064, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49186, 34864, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32872,
          37408, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49186,
          34864, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16386, 2064,
          32800, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32872, 37408, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 16386,
          2064, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 72, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49186, 2064, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800,
          0, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0,
          0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32872, 1025, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 4608, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800,
          0, 0, 0, 0, 0, 0, 0, 0],
         [0, 72, 1025, 1097, 1025, 5633, 17411, 1025, 1025, 1025, 5633, 17411, 1025, 5633, 5633, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 72, 1025, 1025,
          1025, 1025, 1025, 1025, 1025, 1025, 4608, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 72, 3089, 1025, 1025, 1025, 1097, 2064, 0, 32800, 32800, 72, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72,
          1025, 4608, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800,
          32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 72, 4608, 72, 4608, 0, 0, 0, 16386, 1025, 4608, 0, 0, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 1025, 33825, 1025, 1097, 1025, 5633, 17411, 3089, 1025, 1097, 5633, 17411, 1025, 3089, 1025, 4608, 0, 0, 0,
          0, 0, 0, 0, 0, 0, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 1025, 1025, 1025, 1097, 3089, 5633, 1025, 17411, 1097, 3089, 1025, 5633, 1025, 37408, 0, 0, 0, 0, 0,
          0, 0, 0, 0, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 1025, 2064, 0, 0, 0, 72, 1025, 1097, 1025, 5633, 17411, 1025, 1025, 1025, 5633,
          17411, 1025, 3089, 3089, 2064, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 3089, 1025, 1025, 1025, 1097, 2064, 0, 0, 0, 0, 0, 0,
          0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          0]], dtype=np.uint16)

    random_state = ('MT19937', [1250951825, 3659883618, 1035830553, 1397161316, 766292927,
                                2661854372, 897315985, 4096412064, 3044635660, 1215956778,
                                1445464060, 473039202, 1052893923, 2866383412, 1166034620,
                                820694701, 2163754602, 896037377, 3925022808, 2281291903,
                                3768338789, 395623082, 3039639863, 3333350224, 2126207313,
                                1756562650, 2751698660, 3041198134, 2092866439, 2694110938,
                                3222498078, 4174202555, 2388674468, 3816099773, 2781726950,
                                2780033393, 500002616, 2888915983, 365836764, 1612132780,
                                3553880853, 3537869327, 312500739, 2990961898, 2596267918,
                                4267639090, 1769548463, 2316570584, 23276919, 2976277283,
                                2540301765, 588562926, 2567115676, 719766421, 866436388,
                                3186836431, 2989549022, 320931608, 1188497043, 3900492,
                                2184228392, 2936842895, 1043080908, 3728494110, 4038183085,
                                1336771798, 1121472796, 3065133537, 3793939648, 2281042727,
                                1092896585, 2966520469, 4236600082, 3569021941, 2785668202,
                                847751955, 3051168839, 1513549881, 2395582586, 2683957962,
                                2816915087, 1677427330, 2448747505, 1235581485, 2999141397,
                                67520790, 1209217076, 4107246097, 689264288, 1173847794,
                                2933690823, 2414163354, 965495194, 666371848, 2305009162,
                                3568890353, 875066137, 350838326, 990184617, 3722869193,
                                281303193, 3627753273, 3751059479, 3011373396, 1517702905,
                                241605618, 3016178188, 793427505, 4066383, 2368858070,
                                2654530660, 1749897508, 2320561651, 4100444266, 2468087843,
                                1303017573, 3109713152, 3321736715, 271703332, 3660934305,
                                2273571922, 3288787714, 3204714347, 2572936962, 848659135,
                                2272934796, 4236478046, 3282710008, 2566600961, 4237150010,
                                3491059062, 2248838686, 2498671209, 1026669232, 3131079418,
                                2712350604, 3048089917, 2748407880, 2189510480, 1966067424,
                                3746103820, 4025522931, 1249552296, 279928507, 2533735419,
                                3814155219, 396989127, 1364588536, 3100279524, 241961528,
                                2741920572, 3627198566, 1021730811, 2490484575, 3100268711,
                                1723282506, 3339748229, 1381703668, 683297727, 2120467646,
                                432484858, 1862673550, 3745817484, 1853652046, 4262212045,
                                1957345370, 2273479527, 3969759908, 858819099, 1269807037,
                                155471891, 2116725221, 1651306168, 3449849389, 3747291538,
                                2807637809, 282365877, 3578299352, 1515752215, 104569060,
                                2643971901, 1762475623, 4069544647, 3128591093, 1427363340,
                                930412331, 3821261353, 4286477181, 699523218, 3508087965,
                                2606166140, 161647716, 4261051440, 3212042157, 264508511,
                                3467856071, 1411759952, 2850853064, 523048748, 3004646774,
                                1873611928, 614653527, 1284102323, 2990519684, 2361891232,
                                1952257876, 2281602132, 1728409471, 1493224227, 2927024567,
                                274279705, 4084377255, 2524307319, 3003057090, 3801433112,
                                3781446869, 1469790685, 1089326689, 1672407173, 1511890972,
                                2157167077, 1266568448, 4101965669, 1750517686, 47822367,
                                4104110480, 3376284934, 3741957489, 4108734724, 3469047612,
                                2464027712, 1424359614, 1029802677, 1538601967, 3824417277,
                                3173351931, 3845971826, 689500000, 1990343332, 1318249652,
                                883499074, 85010029, 3240998242, 3514399906, 330796644,
                                1520064631, 594168302, 2941689579, 1709502539, 4240248307,
                                756308562, 3288872396, 2360196894, 940852000, 3128786208,
                                4261372782, 1016824165, 884056068, 899358739, 3085318012,
                                3667709204, 1561095566, 1460271903, 1766103582, 1356232307,
                                403879679, 3063249189, 2387632393, 1485554149, 282088368,
                                3705074112, 1905783953, 1985247346, 3442139412, 957766490,
                                3844016205, 3476514706, 1489997414, 204691117, 818868845,
                                1497749928, 1579163343, 3651452781, 3424554701, 794993496,
                                4088045569, 3101245711, 744464255, 4241115785, 855428069,
                                3150965756, 76157036, 689351805, 3048209691, 137708767,
                                3567639848, 416945448, 2351779809, 1969840055, 1902421516,
                                2931101573, 3655583917, 1058571340, 4188376084, 2793154821,
                                2308441164, 1905865973, 1748533067, 3756028349, 4080323721,
                                3563119189, 2426343629, 1871061205, 590388466, 412401969,
                                3665055315, 161477353, 1631484158, 3858486054, 2256781500,
                                1487921450, 2602296215, 3107125815, 2510253751, 3437418098,
                                3696843795, 1165179162, 3711635295, 3038731921, 1075270934,
                                4273507907, 2091975684, 1810435272, 3261173239, 942373598,
                                3686276169, 562423379, 4217384310, 934892478, 824195791,
                                2291909777, 2177023767, 694129350, 2524980215, 2111127690,
                                956314987, 2068975389, 2292409833, 1354436609, 2011265550,
                                1976078666, 3867712321, 3932405855, 1376359980, 2189123348,
                                3022877540, 1132678772, 2280649899, 1997036584, 410868018,
                                3034416667, 2442212289, 392347913, 1150972548, 460101094,
                                2649263999, 1430377948, 2689127849, 332467486, 225462544,
                                3896539226, 4063569984, 1829458839, 3989949270, 1068731197,
                                825451350, 1439026560, 2578400771, 1684245231, 420220866,
                                1874753787, 2791071980, 2310631655, 3184680478, 2646906481,
                                713551514, 1462612033, 2015954624, 3488587815, 461025154,
                                2467356252, 768961978, 1334361629, 3856652593, 3010763640,
                                1754589819, 2214570999, 1745594883, 3334343108, 255961026,
                                490126381, 1090815620, 831215310, 720448471, 1584324490,
                                3019256717, 905013987, 3253742283, 1657205040, 3242093236,
                                574453989, 2829770500, 3132054366, 1908702762, 4151718551,
                                1774675927, 2448263301, 1924160030, 2438056885, 4165466275,
                                2001708879, 183437428, 2332682170, 1859355266, 1330166833,
                                592975850, 3857388138, 2925723136, 3508011608, 936509048,
                                3542924806, 289224450, 978117298, 2389811176, 3532450376,
                                2090920288, 4140531812, 827820879, 3936742258, 3428876164,
                                355669227, 3432649803, 3523675811, 676851178, 2647250260,
                                3361520802, 1901837713, 4073157043, 2245564700, 3224414320,
                                486847917, 1782938771, 671915366, 1194104817, 987888340,
                                1531124389, 1989722466, 3865813111, 2614316853, 129642479,
                                2703334738, 675670308, 2059818877, 420182436, 3029598386,
                                1798381899, 134664325, 1007349356, 1602518955, 2862757002,
                                3075401853, 4020787254, 2244460617, 638903217, 1440055337,
                                2213807677, 1395812485, 1293275554, 1112168894, 1611242312,
                                3390782635, 2498142711, 2786328710, 952865129, 2771829536,
                                3215633096, 881709698, 3651017316, 2054309169, 1564837638,
                                254495944, 3697197992, 3563208868, 342253327, 4281060166,
                                4217084720, 3139421961, 2239594388, 2226717528, 2624029358,
                                1775380731, 1844980621, 341943395, 3757053009, 2928901079,
                                578457295, 2352128490, 704033454, 3959053192, 720173412,
                                1503409090, 1371713200, 3523696141, 3437706221, 990264391,
                                4009723794, 1794467791, 4245044269, 2884161469, 1477146805,
                                610664954, 3931866292, 2085434330, 2237840817, 2550911440,
                                1320041242, 759983127, 1808697150, 3750067275, 2107130482,
                                3733249626, 3763157416, 1798160664, 629481116, 3766399203,
                                2075241528, 3363648000, 1381920146, 1728352196, 2806526937,
                                1227465605, 2623194127, 1583967267, 497165590, 2410016493,
                                3519743224, 1610682756, 3140147857, 3507015698, 4097406564,
                                1702081770, 3243496802, 3817145169, 3825714729, 3675590830,
                                3910825413, 3976372579, 2870319890, 309574993, 2565097,
                                2504183384, 3824490725, 762810728, 1123075606, 4195694352,
                                1183710111, 3439232866, 1230420100, 285178253, 1269431436,
                                3569682661, 1039401826, 311470563, 1396473703, 1601495922,
                                2366635253, 706893154, 1738975404, 1192530907, 1225557225,
                                3904706411, 3799447969, 669730831, 1978053625, 93432443,
                                3145027406, 1730999427, 3707244774, 692752419, 2332272339,
                                691694098, 2937938791, 1585541636, 3420470914, 364855852,
                                798543783, 2306966589, 3330516943, 2766846753, 970122949,
                                3647107044, 2907678488, 923780366, 3156202426, 2735744337,
                                1362754624, 1442226188, 341042361, 1128104827, 3446857022,
                                2763988847, 3937882543, 3102282943, 4028362902, 563683267,
                                2194730687, 3837302547, 4226764320, 3241582087, 2342878883,
                                662135108, 1315371408, 4107775011, 2143230560], 93, 0, 0.0)
    np_random = RandomState()
    np_random.set_state(random_state)
    distance_map = DistanceMap(env_agents, 50, 50)
    distance_map.reset(env_agents, rail)
    timetable = timetable_generator(env_agents, distance_map, agents_hints, np_random)
    assert timetable.earliest_departures == [[14, None], [70, None], [94, None], [16, None], [17, None], [50, None], [130, None], [6, None], [27, None],
                                             [67, None]]
    assert timetable.latest_arrivals == [[None, 134], [None, 190], [None, 178], [None, 103], [None, 143], [None, 170], [None, 193], [None, 189], [None, 147],
                                         [None, 148]]

    assert len(timetable.earliest_departures) == 10
    assert len(timetable.latest_arrivals) == 10
    for eds, las, a in zip(timetable.earliest_departures, timetable.latest_arrivals, env_agents):
        assert len(eds) == 2
        assert len(las) == 2
        assert a.earliest_departure == eds[0]
        assert a.latest_arrival == las[-1]


def test_timetable_generator_intermediate():
    env_agents = [
        EnvAgent(handle=0, initial_position=(11, 40), initial_direction=3, target=(27, 41), direction=3,
                 waypoints=[Waypoint(position=(11, 40), direction=3), Waypoint(position=(38, 8), direction=1), Waypoint(position=(27, 41), direction=None)],
                 waypoints_latest_arrival=None, waypoints_earliest_departure=None),
        EnvAgent(handle=1, initial_position=(27, 40), initial_direction=0, target=(12, 40), direction=0,
                 waypoints=[Waypoint(position=(27, 40), direction=0), Waypoint(position=(38, 8), direction=3), Waypoint(position=(12, 40), direction=None)],
                 waypoints_latest_arrival=None, waypoints_earliest_departure=None),
        EnvAgent(handle=2, initial_position=(20, 26), initial_direction=2, target=(10, 40), direction=2,
                 waypoints=[Waypoint(position=(20, 26), direction=2), Waypoint(position=(9, 27), direction=3), Waypoint(position=(10, 40), direction=None)],
                 waypoints_latest_arrival=None, waypoints_earliest_departure=None),
        EnvAgent(handle=3, initial_position=(11, 40), initial_direction=1, target=(20, 27), direction=1,
                 waypoints=[Waypoint(position=(11, 40), direction=1), Waypoint(position=(9, 27), direction=3), Waypoint(position=(20, 27), direction=None)],
                 waypoints_latest_arrival=None, waypoints_earliest_departure=None),
        EnvAgent(handle=4, initial_position=(38, 8), initial_direction=1, target=(45, 34), direction=1,
                 waypoints=[Waypoint(position=(38, 8), direction=1), Waypoint(position=(31, 26), direction=0), Waypoint(position=(45, 34), direction=None)],
                 waypoints_latest_arrival=None, waypoints_earliest_departure=None),
        EnvAgent(handle=5, initial_position=(44, 34), initial_direction=1, target=(39, 8), direction=1,
                 waypoints=[Waypoint(position=(44, 34), direction=1), Waypoint(position=(31, 26), direction=0), Waypoint(position=(39, 8), direction=None)],
                 waypoints_latest_arrival=None, waypoints_earliest_departure=None),
        EnvAgent(handle=6, initial_position=(17, 5), initial_direction=1, target=(12, 40), direction=1,
                 waypoints=[Waypoint(position=(17, 5), direction=1), Waypoint(position=(41, 22), direction=3), Waypoint(position=(12, 40), direction=None)],
                 waypoints_latest_arrival=None, waypoints_earliest_departure=None),
        EnvAgent(handle=7, initial_position=(9, 40), initial_direction=1, target=(18, 5), direction=1,
                 waypoints=[Waypoint(position=(9, 40), direction=1), Waypoint(position=(41, 22), direction=3), Waypoint(position=(18, 5), direction=None)],
                 waypoints_latest_arrival=None, waypoints_earliest_departure=None),
        EnvAgent(handle=8, initial_position=(38, 8), initial_direction=1, target=(20, 25), direction=1,
                 waypoints=[Waypoint(position=(38, 8), direction=1), Waypoint(position=(41, 22), direction=1), Waypoint(position=(20, 25), direction=None)],
                 waypoints_latest_arrival=None, waypoints_earliest_departure=None),
        EnvAgent(handle=9, initial_position=(20, 24), initial_direction=0, target=(39, 8), direction=0,
                 waypoints=[Waypoint(position=(20, 24), direction=0), Waypoint(position=(41, 22), direction=3), Waypoint(position=(39, 8), direction=None)],
                 waypoints_latest_arrival=None, waypoints_earliest_departure=None),
    ]
    for a in env_agents:
        print(
            f"EnvAgent(handle={a.handle}, initial_position={a.initial_position}, initial_direction={a.initial_direction}, target={a.target}, direction={a.direction}, waypoints={a.waypoints}, waypoints_latest_arrival=None, waypoints_earliest_departure=None),")
    agents_hints = {
        'city_positions': [(31, 26), (10, 40), (44, 34), (7, 27), (17, 5), (27, 41), (20, 24), (38, 8), (8, 16), (42, 22)],
        'train_stations': [[((31, 26), 0), ((31, 27), 1)], [((9, 40), 0), ((10, 40), 1), ((11, 40), 2), ((12, 40), 3)],
                           [((44, 34), 0), ((45, 34), 1)], [((7, 27), 0), ((8, 27), 1), ((9, 27), 2), ((10, 27), 3)], [((17, 5), 0), ((18, 5), 1)],
                           [((27, 40), 0), ((27, 41), 1), ((27, 42), 2), ((27, 43), 3)],
                           [((20, 24), 0), ((20, 25), 1), ((20, 26), 2), ((20, 27), 3)], [((38, 8), 0), ((39, 8), 1)],
                           [((7, 16), 0), ((8, 16), 1), ((9, 16), 2), ((10, 16), 3)], [((41, 22), 0), ((42, 22), 1), ((43, 22), 2), ((44, 22), 3)]],
        'city_orientations': [0, 3, 3, 3,
                              1, 0, 2, 1,
                              1, 1]
    }
    rail = RailGridTransitionMap(50, 50, RailEnvTransitions())
    rail.grid = np.array(
        [[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16386, 1025, 4608, 0, 0, 0, 16386, 1025, 1025, 1025, 1025, 17411, 1025, 5633, 1025, 1025, 1025, 1025,
          1025, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16386, 1025, 1025, 5633, 17411, 3089, 1025, 1097, 5633, 17411, 1025, 3089, 1025, 1025, 5633, 1025, 3089, 1025, 1097,
          1025, 17411, 1025, 1025, 1025, 38505, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 16386, 1025, 1097, 3089, 5633, 1025, 17411, 1097, 3089, 1025, 1025, 17411, 1025, 1097, 1025, 5633, 1025, 17411,
          1025, 3089, 1025, 5633, 4608, 72, 37408, 0, 0, 0, 16386, 1025, 4608, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 72, 1025, 2064, 0, 0, 0, 0, 32800, 0, 0, 0, 72, 1025, 2064, 0, 0, 0, 72, 1097, 1025, 1097, 1025,
          5633, 17411, 3089, 1025, 1097, 5633, 17411, 1025, 4608, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 3089, 5633, 1025,
          17411, 1097, 2064, 0, 32800, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 5633, 5633, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 1025, 2064,
          0, 0, 0, 32800, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49186, 34864, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 72, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          0, 0, 32800, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 72, 4608, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          0, 0, 32800, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 4608, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          16386, 17411, 2064, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          16386, 2064, 32800, 0, 0, 0, 0, 0],
         [16386, 1025, 5633, 17411, 1025, 1025, 1025, 5633, 17411, 1025, 3089, 2064, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32872, 37408, 0, 0, 0, 0, 0, 0,
          0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 16386, 34864, 0, 0, 0, 0, 0],
         [32800, 0, 72, 3089, 1025, 1025, 1025, 1097, 2064, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          0, 0, 0, 16386, 33825, 33825, 2064, 0, 0, 0, 0, 0],
         [32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49186, 34864, 32872, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 32800, 0, 0, 0, 0, 0, 0],
         [32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 32800, 0, 0, 0, 0, 0, 0],
         [32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32872, 37408, 49186, 2064, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 32800, 0, 0, 0, 0, 0, 0],
         [72, 5633, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          49186, 2064, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 49186, 34864, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32872,
          37408, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 38505, 37408, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49186,
          34864, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 37408, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16386, 34864,
          32872, 4608, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800,
          32800, 32800, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32872, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 37408,
          49186, 2064, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49186, 34864, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32872,
          37408, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49186,
          34864, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16386, 2064,
          32800, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32872, 37408, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 16386,
          2064, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 72, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49186, 2064, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800,
          0, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0,
          0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32872, 1025, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 32800, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 4608, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800,
          0, 0, 0, 0, 0, 0, 0, 0],
         [0, 72, 1025, 1097, 1025, 5633, 17411, 1025, 1025, 1025, 5633, 17411, 1025, 5633, 5633, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 72, 1025, 1025,
          1025, 1025, 1025, 1025, 1025, 1025, 4608, 0, 0, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 72, 3089, 1025, 1025, 1025, 1097, 2064, 0, 32800, 32800, 72, 4608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72,
          1025, 4608, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 32800,
          32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800, 72, 4608, 72, 4608, 0, 0, 0, 16386, 1025, 4608, 0, 0, 0, 32800, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32800,
          32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 1025, 33825, 1025, 1097, 1025, 5633, 17411, 3089, 1025, 1097, 5633, 17411, 1025, 3089, 1025, 4608, 0, 0, 0,
          0, 0, 0, 0, 0, 0, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 1025, 1025, 1025, 1097, 3089, 5633, 1025, 17411, 1097, 3089, 1025, 5633, 1025, 37408, 0, 0, 0, 0, 0,
          0, 0, 0, 0, 32800, 32800, 32800, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 1025, 2064, 0, 0, 0, 72, 1025, 1097, 1025, 5633, 17411, 1025, 1025, 1025, 5633,
          17411, 1025, 3089, 3089, 2064, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 3089, 1025, 1025, 1025, 1097, 2064, 0, 0, 0, 0, 0, 0,
          0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
         [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          0]], dtype=np.uint16)

    random_state = ('MT19937', [1250951825, 3659883618, 1035830553, 1397161316, 766292927,
                                2661854372, 897315985, 4096412064, 3044635660, 1215956778,
                                1445464060, 473039202, 1052893923, 2866383412, 1166034620,
                                820694701, 2163754602, 896037377, 3925022808, 2281291903,
                                3768338789, 395623082, 3039639863, 3333350224, 2126207313,
                                1756562650, 2751698660, 3041198134, 2092866439, 2694110938,
                                3222498078, 4174202555, 2388674468, 3816099773, 2781726950,
                                2780033393, 500002616, 2888915983, 365836764, 1612132780,
                                3553880853, 3537869327, 312500739, 2990961898, 2596267918,
                                4267639090, 1769548463, 2316570584, 23276919, 2976277283,
                                2540301765, 588562926, 2567115676, 719766421, 866436388,
                                3186836431, 2989549022, 320931608, 1188497043, 3900492,
                                2184228392, 2936842895, 1043080908, 3728494110, 4038183085,
                                1336771798, 1121472796, 3065133537, 3793939648, 2281042727,
                                1092896585, 2966520469, 4236600082, 3569021941, 2785668202,
                                847751955, 3051168839, 1513549881, 2395582586, 2683957962,
                                2816915087, 1677427330, 2448747505, 1235581485, 2999141397,
                                67520790, 1209217076, 4107246097, 689264288, 1173847794,
                                2933690823, 2414163354, 965495194, 666371848, 2305009162,
                                3568890353, 875066137, 350838326, 990184617, 3722869193,
                                281303193, 3627753273, 3751059479, 3011373396, 1517702905,
                                241605618, 3016178188, 793427505, 4066383, 2368858070,
                                2654530660, 1749897508, 2320561651, 4100444266, 2468087843,
                                1303017573, 3109713152, 3321736715, 271703332, 3660934305,
                                2273571922, 3288787714, 3204714347, 2572936962, 848659135,
                                2272934796, 4236478046, 3282710008, 2566600961, 4237150010,
                                3491059062, 2248838686, 2498671209, 1026669232, 3131079418,
                                2712350604, 3048089917, 2748407880, 2189510480, 1966067424,
                                3746103820, 4025522931, 1249552296, 279928507, 2533735419,
                                3814155219, 396989127, 1364588536, 3100279524, 241961528,
                                2741920572, 3627198566, 1021730811, 2490484575, 3100268711,
                                1723282506, 3339748229, 1381703668, 683297727, 2120467646,
                                432484858, 1862673550, 3745817484, 1853652046, 4262212045,
                                1957345370, 2273479527, 3969759908, 858819099, 1269807037,
                                155471891, 2116725221, 1651306168, 3449849389, 3747291538,
                                2807637809, 282365877, 3578299352, 1515752215, 104569060,
                                2643971901, 1762475623, 4069544647, 3128591093, 1427363340,
                                930412331, 3821261353, 4286477181, 699523218, 3508087965,
                                2606166140, 161647716, 4261051440, 3212042157, 264508511,
                                3467856071, 1411759952, 2850853064, 523048748, 3004646774,
                                1873611928, 614653527, 1284102323, 2990519684, 2361891232,
                                1952257876, 2281602132, 1728409471, 1493224227, 2927024567,
                                274279705, 4084377255, 2524307319, 3003057090, 3801433112,
                                3781446869, 1469790685, 1089326689, 1672407173, 1511890972,
                                2157167077, 1266568448, 4101965669, 1750517686, 47822367,
                                4104110480, 3376284934, 3741957489, 4108734724, 3469047612,
                                2464027712, 1424359614, 1029802677, 1538601967, 3824417277,
                                3173351931, 3845971826, 689500000, 1990343332, 1318249652,
                                883499074, 85010029, 3240998242, 3514399906, 330796644,
                                1520064631, 594168302, 2941689579, 1709502539, 4240248307,
                                756308562, 3288872396, 2360196894, 940852000, 3128786208,
                                4261372782, 1016824165, 884056068, 899358739, 3085318012,
                                3667709204, 1561095566, 1460271903, 1766103582, 1356232307,
                                403879679, 3063249189, 2387632393, 1485554149, 282088368,
                                3705074112, 1905783953, 1985247346, 3442139412, 957766490,
                                3844016205, 3476514706, 1489997414, 204691117, 818868845,
                                1497749928, 1579163343, 3651452781, 3424554701, 794993496,
                                4088045569, 3101245711, 744464255, 4241115785, 855428069,
                                3150965756, 76157036, 689351805, 3048209691, 137708767,
                                3567639848, 416945448, 2351779809, 1969840055, 1902421516,
                                2931101573, 3655583917, 1058571340, 4188376084, 2793154821,
                                2308441164, 1905865973, 1748533067, 3756028349, 4080323721,
                                3563119189, 2426343629, 1871061205, 590388466, 412401969,
                                3665055315, 161477353, 1631484158, 3858486054, 2256781500,
                                1487921450, 2602296215, 3107125815, 2510253751, 3437418098,
                                3696843795, 1165179162, 3711635295, 3038731921, 1075270934,
                                4273507907, 2091975684, 1810435272, 3261173239, 942373598,
                                3686276169, 562423379, 4217384310, 934892478, 824195791,
                                2291909777, 2177023767, 694129350, 2524980215, 2111127690,
                                956314987, 2068975389, 2292409833, 1354436609, 2011265550,
                                1976078666, 3867712321, 3932405855, 1376359980, 2189123348,
                                3022877540, 1132678772, 2280649899, 1997036584, 410868018,
                                3034416667, 2442212289, 392347913, 1150972548, 460101094,
                                2649263999, 1430377948, 2689127849, 332467486, 225462544,
                                3896539226, 4063569984, 1829458839, 3989949270, 1068731197,
                                825451350, 1439026560, 2578400771, 1684245231, 420220866,
                                1874753787, 2791071980, 2310631655, 3184680478, 2646906481,
                                713551514, 1462612033, 2015954624, 3488587815, 461025154,
                                2467356252, 768961978, 1334361629, 3856652593, 3010763640,
                                1754589819, 2214570999, 1745594883, 3334343108, 255961026,
                                490126381, 1090815620, 831215310, 720448471, 1584324490,
                                3019256717, 905013987, 3253742283, 1657205040, 3242093236,
                                574453989, 2829770500, 3132054366, 1908702762, 4151718551,
                                1774675927, 2448263301, 1924160030, 2438056885, 4165466275,
                                2001708879, 183437428, 2332682170, 1859355266, 1330166833,
                                592975850, 3857388138, 2925723136, 3508011608, 936509048,
                                3542924806, 289224450, 978117298, 2389811176, 3532450376,
                                2090920288, 4140531812, 827820879, 3936742258, 3428876164,
                                355669227, 3432649803, 3523675811, 676851178, 2647250260,
                                3361520802, 1901837713, 4073157043, 2245564700, 3224414320,
                                486847917, 1782938771, 671915366, 1194104817, 987888340,
                                1531124389, 1989722466, 3865813111, 2614316853, 129642479,
                                2703334738, 675670308, 2059818877, 420182436, 3029598386,
                                1798381899, 134664325, 1007349356, 1602518955, 2862757002,
                                3075401853, 4020787254, 2244460617, 638903217, 1440055337,
                                2213807677, 1395812485, 1293275554, 1112168894, 1611242312,
                                3390782635, 2498142711, 2786328710, 952865129, 2771829536,
                                3215633096, 881709698, 3651017316, 2054309169, 1564837638,
                                254495944, 3697197992, 3563208868, 342253327, 4281060166,
                                4217084720, 3139421961, 2239594388, 2226717528, 2624029358,
                                1775380731, 1844980621, 341943395, 3757053009, 2928901079,
                                578457295, 2352128490, 704033454, 3959053192, 720173412,
                                1503409090, 1371713200, 3523696141, 3437706221, 990264391,
                                4009723794, 1794467791, 4245044269, 2884161469, 1477146805,
                                610664954, 3931866292, 2085434330, 2237840817, 2550911440,
                                1320041242, 759983127, 1808697150, 3750067275, 2107130482,
                                3733249626, 3763157416, 1798160664, 629481116, 3766399203,
                                2075241528, 3363648000, 1381920146, 1728352196, 2806526937,
                                1227465605, 2623194127, 1583967267, 497165590, 2410016493,
                                3519743224, 1610682756, 3140147857, 3507015698, 4097406564,
                                1702081770, 3243496802, 3817145169, 3825714729, 3675590830,
                                3910825413, 3976372579, 2870319890, 309574993, 2565097,
                                2504183384, 3824490725, 762810728, 1123075606, 4195694352,
                                1183710111, 3439232866, 1230420100, 285178253, 1269431436,
                                3569682661, 1039401826, 311470563, 1396473703, 1601495922,
                                2366635253, 706893154, 1738975404, 1192530907, 1225557225,
                                3904706411, 3799447969, 669730831, 1978053625, 93432443,
                                3145027406, 1730999427, 3707244774, 692752419, 2332272339,
                                691694098, 2937938791, 1585541636, 3420470914, 364855852,
                                798543783, 2306966589, 3330516943, 2766846753, 970122949,
                                3647107044, 2907678488, 923780366, 3156202426, 2735744337,
                                1362754624, 1442226188, 341042361, 1128104827, 3446857022,
                                2763988847, 3937882543, 3102282943, 4028362902, 563683267,
                                2194730687, 3837302547, 4226764320, 3241582087, 2342878883,
                                662135108, 1315371408, 4107775011, 2143230560], 93, 0, 0.0)
    np_random = RandomState()
    np_random.set_state(random_state)
    distance_map = DistanceMap(env_agents, 50, 50)
    distance_map.reset(env_agents, rail)
    timetable = timetable_generator(env_agents, distance_map, agents_hints, np_random)
    assert len(timetable.earliest_departures) == len(env_agents)
    assert len(timetable.latest_arrivals) == len(env_agents)

    for eds in timetable.earliest_departures:
        assert len(eds) == 3
        assert eds[-1] is None
        cur = 0
        for ed in eds[:-1]:
            assert ed is not None
            assert ed >= cur
            assert ed < timetable.max_episode_steps
            cur = ed

    assert timetable.max_episode_steps >= 0
    assert timetable.max_episode_steps < np.inf

    for las in timetable.latest_arrivals:
        assert len(las) == 3
        assert las[0] is None
        assert las[-1] <= timetable.max_episode_steps
        cur = 0
        for la in las[1:]:
            assert la is not None
            assert la >= cur
            assert la <= timetable.max_episode_steps
            cur = la
    for eds, las, a in zip(timetable.earliest_departures, timetable.latest_arrivals, env_agents):
        assert a.earliest_departure == eds[0]
        assert a.latest_arrival == las[-1]
        for ed, la in zip(eds[1:-1], las[1:-1]):
            assert ed <= la

    assert timetable.earliest_departures == [[108, 188, None], [98, 204, None], [14, 125, None], [70, 218, None], [94, 130, None], [16, 48, None],
                                             [17, 127, None], [107, 174, None], [234, 254, None], [178, 276, None]]
    assert timetable.latest_arrivals == [[None, 261, 318], [None, 292, 373], [None, 233, 378], [None, 300, 322], [None, 201, 298], [None, 128, 257],
                                         [None, 221, 321], [None, 242, 297], [None, 301, 332], [None, 343, 364]]
