from multiprocessing import Pool, cpu_count
import numpy as np
from copy import copy
import pandas as pd
import itertools
from collections import OrderedDict
from tqdm import tqdm
from .trigger import Trigger

_bg_data = None
_sig_data = None

def _init_worker(bg, sig):
    """Initialize worker globals (avoids sending large data each time)."""
    global _bg_data, _sig_data
    _bg_data = bg
    _sig_data = sig

def _evaluate_single(args):
    """Evaluate one trigger (worker function)."""
    name, trigger, roi_eta_key, roi_phi_key, dR_cutoff, bins, do_counts, do_eff = args
    counts = eff = None
    if do_counts:
        counts = trigger.counts(_bg_data)
    if do_eff:
        eff = trigger.efficiency(_sig_data, roi_eta_key, roi_phi_key, dR_cutoff=dR_cutoff, bins=bins)
    return name, (trigger, counts, eff)

class DiscriminantCutRange:
    def __init__(self, cut_range, low_cutoff_range, high_cutoff_range):
        self.cut_range = cut_range
        self.low_cutoff_range = low_cutoff_range
        self.high_cutoff_range = high_cutoff_range

    def gen_cuts(self):
        """
    	Generate a DataFrame with all (cut, low, high) combinations.
    	"""
        combinations = itertools.product(
            self.cut_range,
            self.low_cutoff_range,
            self.high_cutoff_range
        )
    
        df = pd.DataFrame(combinations, columns=["cut", "low", "high"])
        return df

class TriggerGrid:
    """
    Manages a grid of triggers iof a specific type (e.g. eTAUs all with different parameters) generated by providing a trigger which cuts on at least one discriminant along with
    the ranges of the discriminants. Stores metadata (e.g. efficiency and rate) for each trigger
    """
    def __init__(self, trigger_class):
        self.trigger_class = trigger_class
        self.trigger_grid = OrderedDict()
        self.evaluated = None

    def register_trigger(self, trigger):
        assert isinstance(trigger, self.trigger_class), f"{trigger} is not an instance of {self.trigger_class}"
        self.trigger_grid[str(trigger)] = copy(trigger)

    def evaluate(self, bg_data, sig_data, roi_eta_key, roi_phi_key, dR_cutoff=0.2, bins=np.arange(0, 120000, 2000), n_jobs=None, do_counts=True, do_eff=True):
        """
        Evaluate all registered triggers in parallel.
        """

        if self.evaluated is not None:
            return self.evaluated

        n_jobs = n_jobs or max(1, cpu_count() - 1)
        args_list = [
            (name, trigger, roi_eta_key, roi_phi_key, dR_cutoff, bins, do_counts, do_eff)
            for name, trigger in self.trigger_grid.items()
        ]
        evaluation = OrderedDict()
        with Pool(processes=n_jobs, initializer=_init_worker, initargs=(bg_data, sig_data)) as pool:
            for name, result in tqdm(
                pool.imap_unordered(_evaluate_single, args_list),
                total=len(args_list),
                desc=f"Evaluating {len(args_list)} triggers"
            ):
                evaluation[name] = result

        self.evaluated = evaluation
        return evaluation

    def set_range(self, cut_name, cuts):
        """
        Register a range for a trigger discriminant. The parameter cuts is a list of triplets of (low, discriminant value, high) where
    	low and high are the cutoff values of the primary discriminant for this particular discriminant. For example, if the primary
        discriminant is energy of the seed, low==7000, high==20000, then this discriminant's cut will be applied only when the seed energy
        is between 7 and 20 GeV.
        """
        assert cut_name in self.trigger.get_cuts()["name"].values, f"{cut_name} is not one of the discriminants defined in the trigger {trigger}"
        self.scan_ranges[cut_name] = cuts

    def make_grid(self):
        for k,v in self.scan_ranges.items():
            trigger = self.trigger_class()
             

