"use strict";
const assert_1 = require("@aws-cdk/assert");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'distribution with custom origin adds custom origin'(test) {
        const stack = new cdk.Stack();
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    originHeaders: {
                        "X-Custom-Header": "somevalue",
                    },
                    customOriginSource: {
                        domainName: "myorigin.com",
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        }
                    ],
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "ForwardedValues": {
                                    "Cookies": {
                                        "Forward": "none"
                                    },
                                    "QueryString": false
                                },
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "redirect-to-https",
                                "Compress": true
                            },
                            "DefaultRootObject": "index.html",
                            "Enabled": true,
                            "HttpVersion": "http2",
                            "IPV6Enabled": true,
                            "Origins": [
                                {
                                    "CustomOriginConfig": {
                                        "HTTPPort": 80,
                                        "HTTPSPort": 443,
                                        "OriginKeepaliveTimeout": 5,
                                        "OriginProtocolPolicy": "https-only",
                                        "OriginReadTimeout": 30,
                                        "OriginSSLProtocols": [
                                            "TLSv1.2"
                                        ]
                                    },
                                    "DomainName": "myorigin.com",
                                    "Id": "origin1",
                                    "OriginCustomHeaders": [
                                        {
                                            "HeaderName": "X-Custom-Header",
                                            "HeaderValue": "somevalue"
                                        }
                                    ]
                                }
                            ],
                            "PriceClass": "PriceClass_100",
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            }
                        }
                    }
                }
            }
        });
        test.done();
    },
    'most basic distribution'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        }
                    ]
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Bucket83908E77": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                },
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "DefaultRootObject": "index.html",
                            "Origins": [
                                {
                                    "DomainName": {
                                        "Fn::GetAtt": [
                                            "Bucket83908E77",
                                            "RegionalDomainName"
                                        ]
                                    },
                                    "Id": "origin1",
                                    "S3OriginConfig": {}
                                }
                            ],
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            },
                            "PriceClass": "PriceClass_100",
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "redirect-to-https",
                                "ForwardedValues": {
                                    "QueryString": false,
                                    "Cookies": { "Forward": "none" }
                                },
                                "Compress": true
                            },
                            "Enabled": true,
                            "IPV6Enabled": true,
                            "HttpVersion": "http2"
                        }
                    }
                }
            }
        });
        test.done();
    },
    'distribution with trusted signers on default distribution'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                            trustedSigners: ["1234"],
                        },
                    ]
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Bucket83908E77": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                },
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "DefaultRootObject": "index.html",
                            "Origins": [
                                {
                                    "DomainName": {
                                        "Fn::GetAtt": [
                                            "Bucket83908E77",
                                            "RegionalDomainName"
                                        ]
                                    },
                                    "Id": "origin1",
                                    "S3OriginConfig": {}
                                }
                            ],
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            },
                            "PriceClass": "PriceClass_100",
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "redirect-to-https",
                                "ForwardedValues": {
                                    "QueryString": false,
                                    "Cookies": { "Forward": "none" }
                                },
                                "TrustedSigners": [
                                    "1234"
                                ],
                                "Compress": true
                            },
                            "Enabled": true,
                            "IPV6Enabled": true,
                            "HttpVersion": "http2"
                        }
                    }
                }
            }
        });
        test.done();
    },
    'distribution with ViewerProtocolPolicy set to a non-default value'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            viewerProtocolPolicy: lib_1.ViewerProtocolPolicy.ALLOW_ALL,
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        }
                    ]
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Bucket83908E77": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                },
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "DefaultRootObject": "index.html",
                            "Origins": [
                                {
                                    "DomainName": {
                                        "Fn::GetAtt": [
                                            "Bucket83908E77",
                                            "RegionalDomainName"
                                        ]
                                    },
                                    "Id": "origin1",
                                    "S3OriginConfig": {}
                                }
                            ],
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            },
                            "PriceClass": "PriceClass_100",
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "allow-all",
                                "ForwardedValues": {
                                    "QueryString": false,
                                    "Cookies": { "Forward": "none" }
                                },
                                "Compress": true
                            },
                            "Enabled": true,
                            "IPV6Enabled": true,
                            "HttpVersion": "http2",
                        }
                    }
                }
            }
        });
        test.done();
    },
    'distribution with disabled compression'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                            compress: false
                        }
                    ]
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Bucket83908E77": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                },
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "DefaultRootObject": "index.html",
                            "Origins": [
                                {
                                    "DomainName": {
                                        "Fn::GetAtt": [
                                            "Bucket83908E77",
                                            "RegionalDomainName"
                                        ]
                                    },
                                    "Id": "origin1",
                                    "S3OriginConfig": {}
                                }
                            ],
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            },
                            "PriceClass": "PriceClass_100",
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "redirect-to-https",
                                "ForwardedValues": {
                                    "QueryString": false,
                                    "Cookies": { "Forward": "none" }
                                },
                                "Compress": false
                            },
                            "Enabled": true,
                            "IPV6Enabled": true,
                            "HttpVersion": "http2"
                        }
                    }
                }
            }
        });
        test.done();
    },
    'distribution with resolvable lambda-association'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        const lambdaFunction = new lambda.SingletonFunction(stack, 'Lambda', {
            uuid: 'xxxx-xxxx-xxxx-xxxx',
            code: lambda.Code.inline('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10
        });
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                            lambdaFunctionAssociations: [{
                                    eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                                    lambdaFunction: lambdaFunction.latestVersion
                                }]
                        }
                    ]
                }
            ]
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
            "DistributionConfig": {
                "DefaultCacheBehavior": {
                    "LambdaFunctionAssociations": [
                        {
                            "EventType": "origin-request",
                            "LambdaFunctionARN": {
                                "Fn::Join": [
                                    "",
                                    [
                                        { "Fn::GetAtt": ["SingletonLambdaxxxxxxxxxxxxxxxx69D4268A", "Arn"] },
                                        ":$LATEST"
                                    ]
                                ]
                            }
                        }
                    ],
                },
            }
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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