// g++ -std=c++17 -pthread -static -Wl,--whole-archive -lpthread -Wl,--no-whole-archive -O3 -rdynamic distances.hpp thread_wrappers.hpp embeding.cpp -o embeding.exe
#include <map>
#include <vector>
#include <string>
#include <iostream>
#include <thread>
#include <mutex>
#include <fstream>
#include <sstream>
#include <algorithm>
#include <cmath>
#include "distances.hpp"
#include "trf_file.hpp"
#include <functional> 
#include "thread_wrappers.hpp"

static std::mutex barrier;

const float CUTOFF_TO_REPORT = 0.2;
const uint LANG_SIZE = 512;
const size_t SIZE_WHILE_COMPUTE_ALL_PAIRS = 10000;
const std::map<std::string, int> token2id = {
        {"AAAAA", 0},
        {"AAAAC", 1},
        {"AAAAG", 2},
        {"AAAAT", 3},
        {"AAACA", 4},
        {"AAACC", 5},
        {"AAACG", 6},
        {"AAACT", 7},
        {"AAAGA", 8},
        {"AAAGC", 9},
        {"AAAGG", 10},
        {"AAAGT", 11},
        {"AAATA", 12},
        {"AAATC", 13},
        {"AAATG", 14},
        {"AAATT", 15},
        {"AACAA", 16},
        {"AACAC", 17},
        {"AACAG", 18},
        {"AACAT", 19},
        {"AACCA", 20},
        {"AACCC", 21},
        {"AACCG", 22},
        {"AACCT", 23},
        {"AACGA", 24},
        {"AACGC", 25},
        {"AACGG", 26},
        {"AACGT", 27},
        {"AACTA", 28},
        {"AACTC", 29},
        {"AACTG", 30},
        {"AACTT", 31},
        {"AAGAA", 32},
        {"AAGAC", 33},
        {"AAGAG", 34},
        {"AAGAT", 35},
        {"AAGCA", 36},
        {"AAGCC", 37},
        {"AAGCG", 38},
        {"AAGCT", 39},
        {"AAGGA", 40},
        {"AAGGC", 41},
        {"AAGGG", 42},
        {"AAGGT", 43},
        {"AAGTA", 44},
        {"AAGTC", 45},
        {"AAGTG", 46},
        {"AAGTT", 31},
        {"AATAA", 47},
        {"AATAC", 48},
        {"AATAG", 49},
        {"AATAT", 50},
        {"AATCA", 51},
        {"AATCC", 52},
        {"AATCG", 53},
        {"AATCT", 54},
        {"AATGA", 55},
        {"AATGC", 56},
        {"AATGG", 57},
        {"AATGT", 58},
        {"AATTA", 59},
        {"AATTC", 60},
        {"AATTG", 61},
        {"AATTT", 15},
        {"ACAAA", 62},
        {"ACAAC", 63},
        {"ACAAG", 64},
        {"ACAAT", 65},
        {"ACACA", 66},
        {"ACACC", 67},
        {"ACACG", 68},
        {"ACACT", 69},
        {"ACAGA", 70},
        {"ACAGC", 71},
        {"ACAGG", 72},
        {"ACAGT", 73},
        {"ACATA", 74},
        {"ACATC", 75},
        {"ACATG", 76},
        {"ACATT", 58},
        {"ACCAA", 77},
        {"ACCAC", 78},
        {"ACCAG", 79},
        {"ACCAT", 80},
        {"ACCCA", 81},
        {"ACCCC", 82},
        {"ACCCG", 83},
        {"ACCCT", 84},
        {"ACCGA", 85},
        {"ACCGC", 86},
        {"ACCGG", 87},
        {"ACCGT", 88},
        {"ACCTA", 89},
        {"ACCTC", 90},
        {"ACCTG", 91},
        {"ACCTT", 43},
        {"ACGAA", 92},
        {"ACGAC", 93},
        {"ACGAG", 94},
        {"ACGAT", 95},
        {"ACGCA", 96},
        {"ACGCC", 97},
        {"ACGCG", 98},
        {"ACGCT", 99},
        {"ACGGA", 100},
        {"ACGGC", 101},
        {"ACGGG", 102},
        {"ACGGT", 88},
        {"ACGTA", 103},
        {"ACGTC", 104},
        {"ACGTG", 105},
        {"ACGTT", 27},
        {"ACTAA", 106},
        {"ACTAC", 107},
        {"ACTAG", 108},
        {"ACTAT", 109},
        {"ACTCA", 110},
        {"ACTCC", 111},
        {"ACTCG", 112},
        {"ACTCT", 113},
        {"ACTGA", 114},
        {"ACTGC", 115},
        {"ACTGG", 116},
        {"ACTGT", 73},
        {"ACTTA", 117},
        {"ACTTC", 118},
        {"ACTTG", 119},
        {"ACTTT", 11},
        {"AGAAA", 120},
        {"AGAAC", 121},
        {"AGAAG", 122},
        {"AGAAT", 123},
        {"AGACA", 124},
        {"AGACC", 125},
        {"AGACG", 126},
        {"AGACT", 127},
        {"AGAGA", 128},
        {"AGAGC", 129},
        {"AGAGG", 130},
        {"AGAGT", 113},
        {"AGATA", 131},
        {"AGATC", 132},
        {"AGATG", 133},
        {"AGATT", 54},
        {"AGCAA", 134},
        {"AGCAC", 135},
        {"AGCAG", 136},
        {"AGCAT", 137},
        {"AGCCA", 138},
        {"AGCCC", 139},
        {"AGCCG", 140},
        {"AGCCT", 141},
        {"AGCGA", 142},
        {"AGCGC", 143},
        {"AGCGG", 144},
        {"AGCGT", 99},
        {"AGCTA", 145},
        {"AGCTC", 146},
        {"AGCTG", 147},
        {"AGCTT", 39},
        {"AGGAA", 148},
        {"AGGAC", 149},
        {"AGGAG", 150},
        {"AGGAT", 151},
        {"AGGCA", 152},
        {"AGGCC", 153},
        {"AGGCG", 154},
        {"AGGCT", 141},
        {"AGGGA", 155},
        {"AGGGC", 156},
        {"AGGGG", 157},
        {"AGGGT", 84},
        {"AGGTA", 158},
        {"AGGTC", 159},
        {"AGGTG", 160},
        {"AGGTT", 23},
        {"AGTAA", 161},
        {"AGTAC", 162},
        {"AGTAG", 163},
        {"AGTAT", 164},
        {"AGTCA", 165},
        {"AGTCC", 166},
        {"AGTCG", 167},
        {"AGTCT", 127},
        {"AGTGA", 168},
        {"AGTGC", 169},
        {"AGTGG", 170},
        {"AGTGT", 69},
        {"AGTTA", 171},
        {"AGTTC", 172},
        {"AGTTG", 173},
        {"AGTTT", 7},
        {"ATAAA", 174},
        {"ATAAC", 175},
        {"ATAAG", 176},
        {"ATAAT", 177},
        {"ATACA", 178},
        {"ATACC", 179},
        {"ATACG", 180},
        {"ATACT", 164},
        {"ATAGA", 181},
        {"ATAGC", 182},
        {"ATAGG", 183},
        {"ATAGT", 109},
        {"ATATA", 184},
        {"ATATC", 185},
        {"ATATG", 186},
        {"ATATT", 50},
        {"ATCAA", 187},
        {"ATCAC", 188},
        {"ATCAG", 189},
        {"ATCAT", 190},
        {"ATCCA", 191},
        {"ATCCC", 192},
        {"ATCCG", 193},
        {"ATCCT", 151},
        {"ATCGA", 194},
        {"ATCGC", 195},
        {"ATCGG", 196},
        {"ATCGT", 95},
        {"ATCTA", 197},
        {"ATCTC", 198},
        {"ATCTG", 199},
        {"ATCTT", 35},
        {"ATGAA", 200},
        {"ATGAC", 201},
        {"ATGAG", 202},
        {"ATGAT", 190},
        {"ATGCA", 203},
        {"ATGCC", 204},
        {"ATGCG", 205},
        {"ATGCT", 137},
        {"ATGGA", 206},
        {"ATGGC", 207},
        {"ATGGG", 208},
        {"ATGGT", 80},
        {"ATGTA", 209},
        {"ATGTC", 210},
        {"ATGTG", 211},
        {"ATGTT", 19},
        {"ATTAA", 212},
        {"ATTAC", 213},
        {"ATTAG", 214},
        {"ATTAT", 177},
        {"ATTCA", 215},
        {"ATTCC", 216},
        {"ATTCG", 217},
        {"ATTCT", 123},
        {"ATTGA", 218},
        {"ATTGC", 219},
        {"ATTGG", 220},
        {"ATTGT", 65},
        {"ATTTA", 221},
        {"ATTTC", 222},
        {"ATTTG", 223},
        {"ATTTT", 3},
        {"CAAAA", 224},
        {"CAAAC", 225},
        {"CAAAG", 226},
        {"CAAAT", 223},
        {"CAACA", 227},
        {"CAACC", 228},
        {"CAACG", 229},
        {"CAACT", 173},
        {"CAAGA", 230},
        {"CAAGC", 231},
        {"CAAGG", 232},
        {"CAAGT", 119},
        {"CAATA", 233},
        {"CAATC", 234},
        {"CAATG", 235},
        {"CAATT", 61},
        {"CACAA", 236},
        {"CACAC", 237},
        {"CACAG", 238},
        {"CACAT", 211},
        {"CACCA", 239},
        {"CACCC", 240},
        {"CACCG", 241},
        {"CACCT", 160},
        {"CACGA", 242},
        {"CACGC", 243},
        {"CACGG", 244},
        {"CACGT", 105},
        {"CACTA", 245},
        {"CACTC", 246},
        {"CACTG", 247},
        {"CACTT", 46},
        {"CAGAA", 248},
        {"CAGAC", 249},
        {"CAGAG", 250},
        {"CAGAT", 199},
        {"CAGCA", 251},
        {"CAGCC", 252},
        {"CAGCG", 253},
        {"CAGCT", 147},
        {"CAGGA", 254},
        {"CAGGC", 255},
        {"CAGGG", 256},
        {"CAGGT", 91},
        {"CAGTA", 257},
        {"CAGTC", 258},
        {"CAGTG", 247},
        {"CAGTT", 30},
        {"CATAA", 259},
        {"CATAC", 260},
        {"CATAG", 261},
        {"CATAT", 186},
        {"CATCA", 262},
        {"CATCC", 263},
        {"CATCG", 264},
        {"CATCT", 133},
        {"CATGA", 265},
        {"CATGC", 266},
        {"CATGG", 267},
        {"CATGT", 76},
        {"CATTA", 268},
        {"CATTC", 269},
        {"CATTG", 235},
        {"CATTT", 14},
        {"CCAAA", 270},
        {"CCAAC", 271},
        {"CCAAG", 272},
        {"CCAAT", 220},
        {"CCACA", 273},
        {"CCACC", 274},
        {"CCACG", 275},
        {"CCACT", 170},
        {"CCAGA", 276},
        {"CCAGC", 277},
        {"CCAGG", 278},
        {"CCAGT", 116},
        {"CCATA", 279},
        {"CCATC", 280},
        {"CCATG", 267},
        {"CCATT", 57},
        {"CCCAA", 281},
        {"CCCAC", 282},
        {"CCCAG", 283},
        {"CCCAT", 208},
        {"CCCCA", 284},
        {"CCCCC", 285},
        {"CCCCG", 286},
        {"CCCCT", 157},
        {"CCCGA", 287},
        {"CCCGC", 288},
        {"CCCGG", 289},
        {"CCCGT", 102},
        {"CCCTA", 290},
        {"CCCTC", 291},
        {"CCCTG", 256},
        {"CCCTT", 42},
        {"CCGAA", 292},
        {"CCGAC", 293},
        {"CCGAG", 294},
        {"CCGAT", 196},
        {"CCGCA", 295},
        {"CCGCC", 296},
        {"CCGCG", 297},
        {"CCGCT", 144},
        {"CCGGA", 298},
        {"CCGGC", 299},
        {"CCGGG", 289},
        {"CCGGT", 87},
        {"CCGTA", 300},
        {"CCGTC", 301},
        {"CCGTG", 244},
        {"CCGTT", 26},
        {"CCTAA", 302},
        {"CCTAC", 303},
        {"CCTAG", 304},
        {"CCTAT", 183},
        {"CCTCA", 305},
        {"CCTCC", 306},
        {"CCTCG", 307},
        {"CCTCT", 130},
        {"CCTGA", 308},
        {"CCTGC", 309},
        {"CCTGG", 278},
        {"CCTGT", 72},
        {"CCTTA", 310},
        {"CCTTC", 311},
        {"CCTTG", 232},
        {"CCTTT", 10},
        {"CGAAA", 312},
        {"CGAAC", 313},
        {"CGAAG", 314},
        {"CGAAT", 217},
        {"CGACA", 315},
        {"CGACC", 316},
        {"CGACG", 317},
        {"CGACT", 167},
        {"CGAGA", 318},
        {"CGAGC", 319},
        {"CGAGG", 307},
        {"CGAGT", 112},
        {"CGATA", 320},
        {"CGATC", 321},
        {"CGATG", 264},
        {"CGATT", 53},
        {"CGCAA", 322},
        {"CGCAC", 323},
        {"CGCAG", 324},
        {"CGCAT", 205},
        {"CGCCA", 325},
        {"CGCCC", 326},
        {"CGCCG", 327},
        {"CGCCT", 154},
        {"CGCGA", 328},
        {"CGCGC", 329},
        {"CGCGG", 297},
        {"CGCGT", 98},
        {"CGCTA", 330},
        {"CGCTC", 331},
        {"CGCTG", 253},
        {"CGCTT", 38},
        {"CGGAA", 332},
        {"CGGAC", 333},
        {"CGGAG", 334},
        {"CGGAT", 193},
        {"CGGCA", 335},
        {"CGGCC", 336},
        {"CGGCG", 327},
        {"CGGCT", 140},
        {"CGGGA", 337},
        {"CGGGC", 338},
        {"CGGGG", 286},
        {"CGGGT", 83},
        {"CGGTA", 339},
        {"CGGTC", 340},
        {"CGGTG", 241},
        {"CGGTT", 22},
        {"CGTAA", 341},
        {"CGTAC", 342},
        {"CGTAG", 343},
        {"CGTAT", 180},
        {"CGTCA", 344},
        {"CGTCC", 345},
        {"CGTCG", 317},
        {"CGTCT", 126},
        {"CGTGA", 346},
        {"CGTGC", 347},
        {"CGTGG", 275},
        {"CGTGT", 68},
        {"CGTTA", 348},
        {"CGTTC", 349},
        {"CGTTG", 229},
        {"CGTTT", 6},
        {"CTAAA", 350},
        {"CTAAC", 351},
        {"CTAAG", 352},
        {"CTAAT", 214},
        {"CTACA", 353},
        {"CTACC", 354},
        {"CTACG", 343},
        {"CTACT", 163},
        {"CTAGA", 355},
        {"CTAGC", 356},
        {"CTAGG", 304},
        {"CTAGT", 108},
        {"CTATA", 357},
        {"CTATC", 358},
        {"CTATG", 261},
        {"CTATT", 49},
        {"CTCAA", 359},
        {"CTCAC", 360},
        {"CTCAG", 361},
        {"CTCAT", 202},
        {"CTCCA", 362},
        {"CTCCC", 363},
        {"CTCCG", 334},
        {"CTCCT", 150},
        {"CTCGA", 364},
        {"CTCGC", 365},
        {"CTCGG", 294},
        {"CTCGT", 94},
        {"CTCTA", 366},
        {"CTCTC", 367},
        {"CTCTG", 250},
        {"CTCTT", 34},
        {"CTGAA", 368},
        {"CTGAC", 369},
        {"CTGAG", 361},
        {"CTGAT", 189},
        {"CTGCA", 370},
        {"CTGCC", 371},
        {"CTGCG", 324},
        {"CTGCT", 136},
        {"CTGGA", 372},
        {"CTGGC", 373},
        {"CTGGG", 283},
        {"CTGGT", 79},
        {"CTGTA", 374},
        {"CTGTC", 375},
        {"CTGTG", 238},
        {"CTGTT", 18},
        {"CTTAA", 376},
        {"CTTAC", 377},
        {"CTTAG", 352},
        {"CTTAT", 176},
        {"CTTCA", 378},
        {"CTTCC", 379},
        {"CTTCG", 314},
        {"CTTCT", 122},
        {"CTTGA", 380},
        {"CTTGC", 381},
        {"CTTGG", 272},
        {"CTTGT", 64},
        {"CTTTA", 382},
        {"CTTTC", 383},
        {"CTTTG", 226},
        {"CTTTT", 2},
        {"GAAAA", 384},
        {"GAAAC", 385},
        {"GAAAG", 383},
        {"GAAAT", 222},
        {"GAACA", 386},
        {"GAACC", 387},
        {"GAACG", 349},
        {"GAACT", 172},
        {"GAAGA", 388},
        {"GAAGC", 389},
        {"GAAGG", 311},
        {"GAAGT", 118},
        {"GAATA", 390},
        {"GAATC", 391},
        {"GAATG", 269},
        {"GAATT", 60},
        {"GACAA", 392},
        {"GACAC", 393},
        {"GACAG", 375},
        {"GACAT", 210},
        {"GACCA", 394},
        {"GACCC", 395},
        {"GACCG", 340},
        {"GACCT", 159},
        {"GACGA", 396},
        {"GACGC", 397},
        {"GACGG", 301},
        {"GACGT", 104},
        {"GACTA", 398},
        {"GACTC", 399},
        {"GACTG", 258},
        {"GACTT", 45},
        {"GAGAA", 400},
        {"GAGAC", 401},
        {"GAGAG", 367},
        {"GAGAT", 198},
        {"GAGCA", 402},
        {"GAGCC", 403},
        {"GAGCG", 331},
        {"GAGCT", 146},
        {"GAGGA", 404},
        {"GAGGC", 405},
        {"GAGGG", 291},
        {"GAGGT", 90},
        {"GAGTA", 406},
        {"GAGTC", 399},
        {"GAGTG", 246},
        {"GAGTT", 29},
        {"GATAA", 407},
        {"GATAC", 408},
        {"GATAG", 358},
        {"GATAT", 185},
        {"GATCA", 409},
        {"GATCC", 410},
        {"GATCG", 321},
        {"GATCT", 132},
        {"GATGA", 411},
        {"GATGC", 412},
        {"GATGG", 280},
        {"GATGT", 75},
        {"GATTA", 413},
        {"GATTC", 391},
        {"GATTG", 234},
        {"GATTT", 13},
        {"GCAAA", 414},
        {"GCAAC", 415},
        {"GCAAG", 381},
        {"GCAAT", 219},
        {"GCACA", 416},
        {"GCACC", 417},
        {"GCACG", 347},
        {"GCACT", 169},
        {"GCAGA", 418},
        {"GCAGC", 419},
        {"GCAGG", 309},
        {"GCAGT", 115},
        {"GCATA", 420},
        {"GCATC", 412},
        {"GCATG", 266},
        {"GCATT", 56},
        {"GCCAA", 421},
        {"GCCAC", 422},
        {"GCCAG", 373},
        {"GCCAT", 207},
        {"GCCCA", 423},
        {"GCCCC", 424},
        {"GCCCG", 338},
        {"GCCCT", 156},
        {"GCCGA", 425},
        {"GCCGC", 426},
        {"GCCGG", 299},
        {"GCCGT", 101},
        {"GCCTA", 427},
        {"GCCTC", 405},
        {"GCCTG", 255},
        {"GCCTT", 41},
        {"GCGAA", 428},
        {"GCGAC", 429},
        {"GCGAG", 365},
        {"GCGAT", 195},
        {"GCGCA", 430},
        {"GCGCC", 431},
        {"GCGCG", 329},
        {"GCGCT", 143},
        {"GCGGA", 432},
        {"GCGGC", 426},
        {"GCGGG", 288},
        {"GCGGT", 86},
        {"GCGTA", 433},
        {"GCGTC", 397},
        {"GCGTG", 243},
        {"GCGTT", 25},
        {"GCTAA", 434},
        {"GCTAC", 435},
        {"GCTAG", 356},
        {"GCTAT", 182},
        {"GCTCA", 436},
        {"GCTCC", 437},
        {"GCTCG", 319},
        {"GCTCT", 129},
        {"GCTGA", 438},
        {"GCTGC", 419},
        {"GCTGG", 277},
        {"GCTGT", 71},
        {"GCTTA", 439},
        {"GCTTC", 389},
        {"GCTTG", 231},
        {"GCTTT", 9},
        {"GGAAA", 440},
        {"GGAAC", 441},
        {"GGAAG", 379},
        {"GGAAT", 216},
        {"GGACA", 442},
        {"GGACC", 443},
        {"GGACG", 345},
        {"GGACT", 166},
        {"GGAGA", 444},
        {"GGAGC", 437},
        {"GGAGG", 306},
        {"GGAGT", 111},
        {"GGATA", 445},
        {"GGATC", 410},
        {"GGATG", 263},
        {"GGATT", 52},
        {"GGCAA", 446},
        {"GGCAC", 447},
        {"GGCAG", 371},
        {"GGCAT", 204},
        {"GGCCA", 448},
        {"GGCCC", 449},
        {"GGCCG", 336},
        {"GGCCT", 153},
        {"GGCGA", 450},
        {"GGCGC", 431},
        {"GGCGG", 296},
        {"GGCGT", 97},
        {"GGCTA", 451},
        {"GGCTC", 403},
        {"GGCTG", 252},
        {"GGCTT", 37},
        {"GGGAA", 452},
        {"GGGAC", 453},
        {"GGGAG", 363},
        {"GGGAT", 192},
        {"GGGCA", 454},
        {"GGGCC", 449},
        {"GGGCG", 326},
        {"GGGCT", 139},
        {"GGGGA", 455},
        {"GGGGC", 424},
        {"GGGGG", 285},
        {"GGGGT", 82},
        {"GGGTA", 456},
        {"GGGTC", 395},
        {"GGGTG", 240},
        {"GGGTT", 21},
        {"GGTAA", 457},
        {"GGTAC", 458},
        {"GGTAG", 354},
        {"GGTAT", 179},
        {"GGTCA", 459},
        {"GGTCC", 443},
        {"GGTCG", 316},
        {"GGTCT", 125},
        {"GGTGA", 460},
        {"GGTGC", 417},
        {"GGTGG", 274},
        {"GGTGT", 67},
        {"GGTTA", 461},
        {"GGTTC", 387},
        {"GGTTG", 228},
        {"GGTTT", 5},
        {"GTAAA", 462},
        {"GTAAC", 463},
        {"GTAAG", 377},
        {"GTAAT", 213},
        {"GTACA", 464},
        {"GTACC", 458},
        {"GTACG", 342},
        {"GTACT", 162},
        {"GTAGA", 465},
        {"GTAGC", 435},
        {"GTAGG", 303},
        {"GTAGT", 107},
        {"GTATA", 466},
        {"GTATC", 408},
        {"GTATG", 260},
        {"GTATT", 48},
        {"GTCAA", 467},
        {"GTCAC", 468},
        {"GTCAG", 369},
        {"GTCAT", 201},
        {"GTCCA", 469},
        {"GTCCC", 453},
        {"GTCCG", 333},
        {"GTCCT", 149},
        {"GTCGA", 470},
        {"GTCGC", 429},
        {"GTCGG", 293},
        {"GTCGT", 93},
        {"GTCTA", 471},
        {"GTCTC", 401},
        {"GTCTG", 249},
        {"GTCTT", 33},
        {"GTGAA", 472},
        {"GTGAC", 468},
        {"GTGAG", 360},
        {"GTGAT", 188},
        {"GTGCA", 473},
        {"GTGCC", 447},
        {"GTGCG", 323},
        {"GTGCT", 135},
        {"GTGGA", 474},
        {"GTGGC", 422},
        {"GTGGG", 282},
        {"GTGGT", 78},
        {"GTGTA", 475},
        {"GTGTC", 393},
        {"GTGTG", 237},
        {"GTGTT", 17},
        {"GTTAA", 476},
        {"GTTAC", 463},
        {"GTTAG", 351},
        {"GTTAT", 175},
        {"GTTCA", 477},
        {"GTTCC", 441},
        {"GTTCG", 313},
        {"GTTCT", 121},
        {"GTTGA", 478},
        {"GTTGC", 415},
        {"GTTGG", 271},
        {"GTTGT", 63},
        {"GTTTA", 479},
        {"GTTTC", 385},
        {"GTTTG", 225},
        {"GTTTT", 1},
        {"TAAAA", 480},
        {"TAAAC", 479},
        {"TAAAG", 382},
        {"TAAAT", 221},
        {"TAACA", 481},
        {"TAACC", 461},
        {"TAACG", 348},
        {"TAACT", 171},
        {"TAAGA", 482},
        {"TAAGC", 439},
        {"TAAGG", 310},
        {"TAAGT", 117},
        {"TAATA", 483},
        {"TAATC", 413},
        {"TAATG", 268},
        {"TAATT", 59},
        {"TACAA", 484},
        {"TACAC", 475},
        {"TACAG", 374},
        {"TACAT", 209},
        {"TACCA", 485},
        {"TACCC", 456},
        {"TACCG", 339},
        {"TACCT", 158},
        {"TACGA", 486},
        {"TACGC", 433},
        {"TACGG", 300},
        {"TACGT", 103},
        {"TACTA", 487},
        {"TACTC", 406},
        {"TACTG", 257},
        {"TACTT", 44},
        {"TAGAA", 488},
        {"TAGAC", 471},
        {"TAGAG", 366},
        {"TAGAT", 197},
        {"TAGCA", 489},
        {"TAGCC", 451},
        {"TAGCG", 330},
        {"TAGCT", 145},
        {"TAGGA", 490},
        {"TAGGC", 427},
        {"TAGGG", 290},
        {"TAGGT", 89},
        {"TAGTA", 487},
        {"TAGTC", 398},
        {"TAGTG", 245},
        {"TAGTT", 28},
        {"TATAA", 491},
        {"TATAC", 466},
        {"TATAG", 357},
        {"TATAT", 184},
        {"TATCA", 492},
        {"TATCC", 445},
        {"TATCG", 320},
        {"TATCT", 131},
        {"TATGA", 493},
        {"TATGC", 420},
        {"TATGG", 279},
        {"TATGT", 74},
        {"TATTA", 483},
        {"TATTC", 390},
        {"TATTG", 233},
        {"TATTT", 12},
        {"TCAAA", 494},
        {"TCAAC", 478},
        {"TCAAG", 380},
        {"TCAAT", 218},
        {"TCACA", 495},
        {"TCACC", 460},
        {"TCACG", 346},
        {"TCACT", 168},
        {"TCAGA", 496},
        {"TCAGC", 438},
        {"TCAGG", 308},
        {"TCAGT", 114},
        {"TCATA", 493},
        {"TCATC", 411},
        {"TCATG", 265},
        {"TCATT", 55},
        {"TCCAA", 497},
        {"TCCAC", 474},
        {"TCCAG", 372},
        {"TCCAT", 206},
        {"TCCCA", 498},
        {"TCCCC", 455},
        {"TCCCG", 337},
        {"TCCCT", 155},
        {"TCCGA", 499},
        {"TCCGC", 432},
        {"TCCGG", 298},
        {"TCCGT", 100},
        {"TCCTA", 490},
        {"TCCTC", 404},
        {"TCCTG", 254},
        {"TCCTT", 40},
        {"TCGAA", 500},
        {"TCGAC", 470},
        {"TCGAG", 364},
        {"TCGAT", 194},
        {"TCGCA", 501},
        {"TCGCC", 450},
        {"TCGCG", 328},
        {"TCGCT", 142},
        {"TCGGA", 499},
        {"TCGGC", 425},
        {"TCGGG", 287},
        {"TCGGT", 85},
        {"TCGTA", 486},
        {"TCGTC", 396},
        {"TCGTG", 242},
        {"TCGTT", 24},
        {"TCTAA", 502},
        {"TCTAC", 465},
        {"TCTAG", 355},
        {"TCTAT", 181},
        {"TCTCA", 503},
        {"TCTCC", 444},
        {"TCTCG", 318},
        {"TCTCT", 128},
        {"TCTGA", 496},
        {"TCTGC", 418},
        {"TCTGG", 276},
        {"TCTGT", 70},
        {"TCTTA", 482},
        {"TCTTC", 388},
        {"TCTTG", 230},
        {"TCTTT", 8},
        {"TGAAA", 504},
        {"TGAAC", 477},
        {"TGAAG", 378},
        {"TGAAT", 215},
        {"TGACA", 505},
        {"TGACC", 459},
        {"TGACG", 344},
        {"TGACT", 165},
        {"TGAGA", 503},
        {"TGAGC", 436},
        {"TGAGG", 305},
        {"TGAGT", 110},
        {"TGATA", 492},
        {"TGATC", 409},
        {"TGATG", 262},
        {"TGATT", 51},
        {"TGCAA", 506},
        {"TGCAC", 473},
        {"TGCAG", 370},
        {"TGCAT", 203},
        {"TGCCA", 507},
        {"TGCCC", 454},
        {"TGCCG", 335},
        {"TGCCT", 152},
        {"TGCGA", 501},
        {"TGCGC", 430},
        {"TGCGG", 295},
        {"TGCGT", 96},
        {"TGCTA", 489},
        {"TGCTC", 402},
        {"TGCTG", 251},
        {"TGCTT", 36},
        {"TGGAA", 508},
        {"TGGAC", 469},
        {"TGGAG", 362},
        {"TGGAT", 191},
        {"TGGCA", 507},
        {"TGGCC", 448},
        {"TGGCG", 325},
        {"TGGCT", 138},
        {"TGGGA", 498},
        {"TGGGC", 423},
        {"TGGGG", 284},
        {"TGGGT", 81},
        {"TGGTA", 485},
        {"TGGTC", 394},
        {"TGGTG", 239},
        {"TGGTT", 20},
        {"TGTAA", 509},
        {"TGTAC", 464},
        {"TGTAG", 353},
        {"TGTAT", 178},
        {"TGTCA", 505},
        {"TGTCC", 442},
        {"TGTCG", 315},
        {"TGTCT", 124},
        {"TGTGA", 495},
        {"TGTGC", 416},
        {"TGTGG", 273},
        {"TGTGT", 66},
        {"TGTTA", 481},
        {"TGTTC", 386},
        {"TGTTG", 227},
        {"TGTTT", 4},
        {"TTAAA", 510},
        {"TTAAC", 476},
        {"TTAAG", 376},
        {"TTAAT", 212},
        {"TTACA", 509},
        {"TTACC", 457},
        {"TTACG", 341},
        {"TTACT", 161},
        {"TTAGA", 502},
        {"TTAGC", 434},
        {"TTAGG", 302},
        {"TTAGT", 106},
        {"TTATA", 491},
        {"TTATC", 407},
        {"TTATG", 259},
        {"TTATT", 47},
        {"TTCAA", 511},
        {"TTCAC", 472},
        {"TTCAG", 368},
        {"TTCAT", 200},
        {"TTCCA", 508},
        {"TTCCC", 452},
        {"TTCCG", 332},
        {"TTCCT", 148},
        {"TTCGA", 500},
        {"TTCGC", 428},
        {"TTCGG", 292},
        {"TTCGT", 92},
        {"TTCTA", 488},
        {"TTCTC", 400},
        {"TTCTG", 248},
        {"TTCTT", 32},
        {"TTGAA", 511},
        {"TTGAC", 467},
        {"TTGAG", 359},
        {"TTGAT", 187},
        {"TTGCA", 506},
        {"TTGCC", 446},
        {"TTGCG", 322},
        {"TTGCT", 134},
        {"TTGGA", 497},
        {"TTGGC", 421},
        {"TTGGG", 281},
        {"TTGGT", 77},
        {"TTGTA", 484},
        {"TTGTC", 392},
        {"TTGTG", 236},
        {"TTGTT", 16},
        {"TTTAA", 510},
        {"TTTAC", 462},
        {"TTTAG", 350},
        {"TTTAT", 174},
        {"TTTCA", 504},
        {"TTTCC", 440},
        {"TTTCG", 312},
        {"TTTCT", 120},
        {"TTTGA", 494},
        {"TTTGC", 414},
        {"TTTGG", 270},
        {"TTTGT", 62},
        {"TTTTA", 480},
        {"TTTTC", 384},
        {"TTTTG", 224},
        {"TTTTT", 0}
};

void worker_for_embending(
                        const size_t start,
                        const size_t end,
                        const size_t step,
                        std::vector<TR>& trs_vector,
                        const std::map<std::string, int>& token2id,
                        const uint k) {

    barrier.lock();
    std::cout << "Started from " << start << " to " << end << " " << std::endl;
    barrier.unlock();

    for (size_t tid = start; tid < end; tid++) {

        if (tid % 10000 == 0) {
            barrier.lock();
            std::cout << "\tcomputed: from  worker: " << step+1 << " and " << 100*(tid-start)/(end-start) << "%" << std::endl;
            barrier.unlock();
        }

        // init embeding
        for (size_t i=0; i < LANG_SIZE; i++) {
            trs_vector[tid].embeding.push_back(0);
            trs_vector[tid].norm_embeding.push_back(0);
        }
        // count embeding
        for (size_t i=0; i < trs_vector[tid].tr_array.size() - k + 1; i++) {
            std::string token = trs_vector[tid].tr_array.substr(i, k);

            if (token2id.find(token) != token2id.end()) {
                int token_id = token2id.at(token);
                trs_vector[tid].embeding[token_id] += 1.0;
            } else {
                continue;
            }   
        }
        // normalize embeding
        double sum = trs_vector[tid].tr_array.size() - k + 1;
        for (size_t i=0; i < trs_vector[tid].embeding.size(); i++) {
            trs_vector[tid].norm_embeding[i] = trs_vector[tid].embeding[i] / sum;
        }
    }
}

void worker_for_distances(
                        const size_t start_i,
                        const size_t end_i,
                        const size_t step,
                        const size_t point,
                        const std::vector<TR>& v,
                        std::vector<Distance>& distances
                    ) {

    barrier.lock();
    std::cout << "Started from " << start_i << " to " << end_i << " " << std::endl;
    barrier.unlock();
    size_t p = 0;
    for (size_t i = start_i; i < end_i; i++) {

        if (i % 1000 == 0) {
            barrier.lock();
            std::cout << "\tcomputed: from  worker: " << 100*(i-start_i)/(end_i-start_i) << "%" << std::endl;
            barrier.unlock();
        }

        for (size_t j = i+1; j < v.size(); j++) {
            // barrier.lock();
            // std::cout << start_i << " " << i << " " << j << " " << point+p << " " << distances.size() << std::endl;
            // barrier.unlock();

            double dist = cosine_distance(v[i].norm_embeding, v[j].norm_embeding);
            
            distances[point+p].tr_idA = i;
            distances[point+p].tr_idB = j;
            distances[point+p].distance = dist;
            p++;
        }
    }
}

void worker_for_distances_low_mem(
                        const size_t start_i,
                        const size_t end_i,
                        const size_t step,
                        const std::vector<TR> &v,
                        std::vector<std::vector<Distance>> &distances_vector
                    ) {

    barrier.lock();
    std::cout << "Started from " << start_i << " to " << end_i << " " << std::endl;
    barrier.unlock();
    for (size_t i = start_i; i < end_i; i++) {

        if (i % 1000 == 0) {
            barrier.lock();
            std::cout << "\tcomputed: from  worker (" << step << "): " << 100*(i-start_i)/(end_i-start_i) << "%" << std::endl;
            barrier.unlock();
        }

        for (size_t j = i+1; j < v.size(); j++) {
            
            double dist = cosine_distance(v[i].norm_embeding, v[j].norm_embeding);
            if (dist > CUTOFF_TO_REPORT) {
                continue;
            }


            Distance d;
            d.tr_idA = i;
            d.tr_idB = j;
            d.distance = dist;
            distances_vector[step].push_back(d);

        }
    }
}


int main(int argc, char** argv) {

    if (argc != 6) {
        std::cerr << "Embed TR arrays to the vectors and computed distances." << std::endl;
        std::cerr << "Expected arguments: " << argv[0]
        << " <trf_file> <output_file> <distance_output_file> <threads> <skip_distances (0 for default don't skip)>" << std::endl;
        std::terminate();
    }

    std::string trs_file = argv[1];
    std::string output_file = argv[2];
    std::string distance_output_file = argv[3];
    static size_t num_threads = atoi(argv[4]);
    static bool skip_distances = atoi(argv[5]);
    

    const uint k = 5;
    
    std::vector<TR> trs_vector;

    // Read trs file
    std::cout << "1. Read trs file..." << std::endl;
    size_t trs_n_items = read_trs_file_to_vector(trs_file, trs_vector);

    std::cout << "\titems: " << trs_n_items << std::endl;


    if (trs_n_items < num_threads) {
        num_threads = trs_n_items;
    } 
    
    run_function_with_threads<std::function<void(size_t,
                        size_t,
                        size_t,
                        std::vector<TR>&,
                        std::map<std::string, int>&,
                        uint)>, 
                        std::vector<TR> &, 
                        std::map<std::string, int>,
                        uint
                        >(trs_n_items, 
                              num_threads, 
                              worker_for_embending,
                              std::ref(trs_vector), 
                              std::ref(token2id), 
                              k);

    
    std::cout << "\tDone." << std::endl;

    std::vector<Distance> distances;
    size_t dist_n_items = (trs_vector.size()*(trs_vector.size()-1))/2;
    std::vector<std::vector<Distance>> distances_vector(num_threads+1);
    
    if (dist_n_items < num_threads) {
        num_threads = dist_n_items;
    } 
    
    if (trs_n_items < SIZE_WHILE_COMPUTE_ALL_PAIRS) {
        
        distances.resize(dist_n_items);

        std::cout << "4. Compute distances for " << dist_n_items << " items using ALL PAIRS..." <<  std::endl;
         
        run_function_with_threads2D<std::function<void(size_t,
                        size_t,
                        size_t,
                        size_t,
                        std::vector<TR>&,
                        std::vector<Distance>&
                        )>,
                        const std::vector<TR>&,
                        std::vector<TR>&, 
                        std::vector<Distance>&
                        >(dist_n_items, 
                            num_threads, 
                            worker_for_distances,
                            trs_vector,
                            std::ref(trs_vector), 
                            std::ref(distances)
                        );

        std::cout << "\tDone." << std::endl;

    } else if (!skip_distances) {

        std::cout << "4. Skip full distances computation, too many TRs (limit 10000). Compute only for distances less than 0.1" <<  std::endl;
        
        run_function_with_results_vector<std::function<void(size_t,
                        size_t,
                        size_t,
                        std::vector<TR>&,
                        std::vector<std::vector<Distance>>&
                        )>,
                        const std::vector<TR>&,
                        std::vector<TR>&, 
                        std::vector<std::vector<Distance>>&
                        >(dist_n_items, 
                            num_threads, 
                            worker_for_distances_low_mem,
                            trs_vector,
                            std::ref(trs_vector), 
                            std::ref(distances_vector)
                        );

        std::cout << "\tDone." << std::endl;

    }

    std::ofstream fout(output_file, std::ios::out);
    if (!fout.is_open()) {
        std::cerr << "Cannot open output file: " << output_file << std::endl;
        std::terminate();
    }

    
    std::cout << "4. Saving results to " << output_file << " ..." <<  std::endl;

    for (size_t i=0; i < trs_vector.size(); i++) {
        fout << trs_vector[i].tr_id << "\t";
        for (size_t j=0; j < trs_vector[i].embeding.size(); j++) {
            fout << trs_vector[i].embeding[j] << " ";
        }
        fout << "\t" << trs_vector[i].tr_array.size() << std::endl;
    }
    fout.close();

    std::ofstream fout_distance(distance_output_file, std::ios::out);
        if (!fout_distance.is_open()) {
            std::cerr << "Cannot open output file: " << distance_output_file << std::endl;
            std::terminate();
        }

    std::cout << "5. Saving distances to " << distance_output_file << " ..." <<  std::endl;

    size_t n_distances = 0;
    if (trs_n_items < 10000) {

        for (size_t i=0; i < distances.size(); i++) {
            n_distances += 1;
            fout_distance << trs_vector[distances[i].tr_idA].tr_id << "\t" << trs_vector[distances[i].tr_idB].tr_id << "\t" << distances[i].distance << std::endl;
        }

    } else if (!skip_distances) {
        for (size_t i=0; i < distances_vector.size(); i++) {
            for (size_t j=0; j < distances_vector[i].size(); j++) {
                n_distances += 1;
                fout_distance << trs_vector[distances_vector[i][j].tr_idA].tr_id << "\t" << trs_vector[distances_vector[i][j].tr_idB].tr_id << "\t" << distances_vector[i][j].distance << std::endl;
            }
        }
    }
    fout_distance.close();
    std::cout << "All done for " << trs_vector.size() << " repeats." << std::endl;
    std::cout << "All done for " << n_distances << " distances." << std::endl;

    return 0;
}