"""Comprehensive tests for cli.core.main_service module.

These tests provide extensive coverage for main service management including
Docker Compose orchestration, workspace validation, service lifecycle operations,
status monitoring, log retrieval, and cross-platform compatibility.
All tests are designed with RED phase compliance for TDD workflow.
"""

import subprocess
from pathlib import Path
from unittest.mock import Mock, patch

from cli.core.main_service import MainService


class TestMainServiceInitialization:
    """Test MainService class initialization and path handling."""

    def test_init_with_workspace_path(self, temp_workspace):
        """Test MainService initializes correctly with provided workspace path."""
        service = MainService(temp_workspace)

        # Use resolve() to normalize symlinks (macOS /var -> /private/var)
        assert service.workspace_path.resolve() == temp_workspace.resolve()

    def test_init_with_default_workspace(self):
        """Test MainService initializes with current directory when no path provided."""
        service = MainService()

        # The service should resolve to the current working directory
        expected_path = Path(".").resolve()
        assert service.workspace_path == expected_path

    def test_init_with_string_workspace_path(self, temp_workspace):
        """Test MainService handles string workspace paths correctly."""
        service = MainService(str(temp_workspace))

        # Use resolve() to normalize symlinks (macOS /var -> /private/var)
        assert service.workspace_path.resolve() == temp_workspace.resolve()

    def test_init_with_windows_style_path(self):
        """Test MainService handles Windows-style paths correctly."""
        with patch("pathlib.Path.resolve", return_value=Path("/converted/path")):
            service = MainService("C:\\tmp\\workspace")

            assert isinstance(service.workspace_path, Path)

    def test_init_with_resolve_not_implemented_error(self):
        """Test MainService handles resolve() NotImplementedError for cross-platform compatibility."""
        with patch("pathlib.Path.resolve", side_effect=NotImplementedError("Mock cross-platform issue")):
            service = MainService()

            # Should fallback to basic Path without resolve
            assert isinstance(service.workspace_path, Path)


class TestWorkspaceValidation:
    """Test workspace validation functionality."""

    def test_validate_workspace_with_docker_main_compose(self, temp_workspace):
        """Test workspace validation succeeds with docker/main/docker-compose.yml file."""
        # Create docker/main directory and compose file
        docker_main_dir = temp_workspace / "docker" / "main"
        docker_main_dir.mkdir(parents=True, exist_ok=True)
        (docker_main_dir / "docker-compose.yml").write_text("""
version: '3.8'
services:
  app:
    image: python:3.11
""")

        service = MainService(temp_workspace)
        result = service._validate_workspace(temp_workspace)

        assert result is True

    def test_validate_workspace_with_root_compose(self, temp_workspace):
        """Test workspace validation succeeds with root docker-compose.yml file."""
        # Use existing docker-compose.yml from temp_workspace fixture
        service = MainService(temp_workspace)
        result = service._validate_workspace(temp_workspace)

        assert result is True

    def test_validate_workspace_nonexistent_path(self):
        """Test workspace validation fails with nonexistent path."""
        nonexistent_path = Path("/nonexistent/workspace/path")
        service = MainService()

        result = service._validate_workspace(nonexistent_path)

        assert result is False

    def test_validate_workspace_path_is_file(self, temp_workspace):
        """Test workspace validation fails when path is a file instead of directory."""
        file_path = temp_workspace / "not_a_directory.txt"
        file_path.write_text("This is a file, not a directory")

        service = MainService(temp_workspace)
        result = service._validate_workspace(file_path)

        assert result is False

    def test_validate_workspace_missing_compose_files(self, temp_workspace):
        """Test workspace validation fails when no docker-compose.yml files exist."""
        # Remove the existing docker-compose.yml from fixture
        (temp_workspace / "docker-compose.yml").unlink()

        service = MainService(temp_workspace)
        result = service._validate_workspace(temp_workspace)

        assert result is False

    def test_validate_workspace_with_resolve_not_implemented(self, temp_workspace):
        """Test workspace validation handles Path.resolve() NotImplementedError."""
        with patch.object(Path, "resolve", side_effect=NotImplementedError("Cross-platform issue")):
            service = MainService(temp_workspace)
            result = service._validate_workspace(temp_workspace)

            # NotImplementedError from resolve() is caught by general Exception handler and returns False
            # This is the actual behavior - other methods handle NotImplementedError specifically
            assert result is False

    def test_validate_workspace_with_mocking_type_error(self, temp_workspace):
        """Test workspace validation handles mocking issues with TypeError."""
        # Simulate the specific mocking error mentioned in the code
        with patch.object(
            Path,
            "exists",
            side_effect=TypeError("exists_side_effect() missing 1 required positional argument: 'path_self'"),
        ):
            service = MainService(temp_workspace)
            result = service._validate_workspace(temp_workspace)

            # Should handle mocking issues gracefully and assume validation passes
            assert result is True

    def test_validate_workspace_with_attribute_error(self, temp_workspace):
        """Test workspace validation handles AttributeError from mocking issues."""
        with patch.object(Path, "exists", side_effect=AttributeError("Mock attribute error")):
            service = MainService(temp_workspace)
            result = service._validate_workspace(temp_workspace)

            # Should handle mocking issues gracefully
            assert result is True

    def test_validate_workspace_with_general_exception(self, temp_workspace):
        """Test workspace validation handles general exceptions gracefully."""
        with patch.object(Path, "exists", side_effect=RuntimeError("General error")):
            service = MainService(temp_workspace)
            result = service._validate_workspace(temp_workspace)

            # Should handle general exceptions by returning False
            assert result is False


class TestMainEnvironmentValidation:
    """Test main environment validation functionality."""

    def test_validate_main_environment_success(self, temp_workspace):
        """Test main environment validation succeeds with proper .env file."""
        service = MainService(temp_workspace)
        result = service._validate_main_environment(temp_workspace)

        # temp_workspace fixture includes .env file
        assert result is True

    def test_validate_main_environment_missing_env_file(self, temp_workspace):
        """Test main environment validation fails without .env file."""
        # Remove .env file from fixture
        (temp_workspace / ".env").unlink()

        service = MainService(temp_workspace)
        result = service._validate_main_environment(temp_workspace)

        assert result is False

    def test_validate_main_environment_with_resolve_not_implemented(self, temp_workspace):
        """Test main environment validation handles Path.resolve() NotImplementedError."""
        with patch("pathlib.Path.resolve", side_effect=NotImplementedError("Cross-platform issue")):
            service = MainService(temp_workspace)
            result = service._validate_main_environment(temp_workspace)

            # Should handle NotImplementedError and still validate
            assert result is True

    def test_validate_main_environment_with_mocking_issues(self, temp_workspace):
        """Test main environment validation handles mocking TypeError/AttributeError."""
        with patch.object(Path, "exists", side_effect=TypeError("Mock error")):
            service = MainService(temp_workspace)
            result = service._validate_main_environment(temp_workspace)

            # Should handle mocking issues gracefully
            assert result is True

    def test_validate_main_environment_with_os_error(self, temp_workspace):
        """Test main environment validation handles OSError gracefully."""
        with patch.object(Path, "exists", side_effect=OSError("File system error")):
            service = MainService(temp_workspace)
            result = service._validate_main_environment(temp_workspace)

            # Should handle OS errors by returning False
            assert result is False

    def test_validate_main_environment_with_permission_error(self, temp_workspace):
        """Test main environment validation handles PermissionError gracefully."""
        with patch.object(Path, "exists", side_effect=PermissionError("Access denied")):
            service = MainService(temp_workspace)
            result = service._validate_main_environment(temp_workspace)

            # Should handle permission errors by returning False
            assert result is False


class TestMainContainerSetup:
    """Test main container setup functionality."""

    @patch("subprocess.run")
    def test_setup_main_containers_success_with_docker_main_compose(self, mock_run, temp_workspace):
        """Test successful main container setup using docker/main/docker-compose.yml."""
        # Create docker/main compose file
        docker_main_dir = temp_workspace / "docker" / "main"
        docker_main_dir.mkdir(parents=True, exist_ok=True)
        compose_content = """
version: '3.8'
services:
  postgres:
    image: postgres:15
    ports:
      - "5532:5432"
  app:
    image: automagik-hive:latest
    ports:
      - "8886:8886"
"""
        (docker_main_dir / "docker-compose.yml").write_text(compose_content)

        # Mock successful subprocess run
        mock_run.return_value.returncode = 0
        mock_run.return_value.stderr = ""

        service = MainService(temp_workspace)
        result = service._setup_main_containers(str(temp_workspace))

        assert result is True
        # Implementation uses os.fspath() which resolves symlinks, so we need to resolve the expected path
        expected_compose_path = str((docker_main_dir / "docker-compose.yml").resolve())
        mock_run.assert_called_once_with(
            ["docker", "compose", "-f", expected_compose_path, "up", "-d"],
            check=False,
            capture_output=True,
            text=True,
            timeout=120,
        )

    @patch("subprocess.run")
    def test_setup_main_containers_success_with_root_compose(self, mock_run, temp_workspace):
        """Test successful main container setup using root docker-compose.yml."""
        # Mock successful subprocess run
        mock_run.return_value.returncode = 0
        mock_run.return_value.stderr = ""

        service = MainService(temp_workspace)
        result = service._setup_main_containers(str(temp_workspace))

        assert result is True
        # Implementation uses os.fspath() which resolves symlinks, so we need to resolve the expected path
        expected_compose_path = str((temp_workspace / "docker-compose.yml").resolve())
        mock_run.assert_called_once_with(
            ["docker", "compose", "-f", expected_compose_path, "up", "-d"],
            check=False,
            capture_output=True,
            text=True,
            timeout=120,
        )

    @patch("subprocess.run")
    def test_setup_main_containers_missing_compose_file(self, mock_run, temp_workspace):
        """Test main container setup fails when no compose file exists."""
        # Remove docker-compose.yml from fixture
        (temp_workspace / "docker-compose.yml").unlink()

        service = MainService(temp_workspace)
        result = service._setup_main_containers(str(temp_workspace))

        assert result is False
        mock_run.assert_not_called()

    @patch("subprocess.run")
    def test_setup_main_containers_docker_command_failure(self, mock_run, temp_workspace):
        """Test main container setup handles Docker command failure."""
        # Mock failed subprocess run
        mock_run.return_value.returncode = 1
        mock_run.return_value.stderr = "Docker daemon not running"

        service = MainService(temp_workspace)
        result = service._setup_main_containers(str(temp_workspace))

        assert result is False

    @patch("subprocess.run")
    def test_setup_main_containers_timeout_error(self, mock_run, temp_workspace):
        """Test main container setup handles subprocess timeout."""
        # Mock timeout exception
        mock_run.side_effect = subprocess.TimeoutExpired(["docker"], timeout=120)

        service = MainService(temp_workspace)
        result = service._setup_main_containers(str(temp_workspace))

        assert result is False

    @patch("subprocess.run")
    def test_setup_main_containers_file_not_found(self, mock_run, temp_workspace):
        """Test main container setup handles missing Docker executable."""
        # Mock FileNotFoundError for missing docker command
        mock_run.side_effect = FileNotFoundError("docker command not found")

        service = MainService(temp_workspace)
        result = service._setup_main_containers(str(temp_workspace))

        assert result is False

    @patch("subprocess.run")
    def test_setup_main_containers_creates_data_directories(self, mock_run, temp_workspace):
        """Test main container setup creates necessary data directories."""
        # Mock successful subprocess run
        mock_run.return_value.returncode = 0
        mock_run.return_value.stderr = ""

        service = MainService(temp_workspace)
        result = service._setup_main_containers(str(temp_workspace))

        assert result is True
        # Check that data directories were created
        assert (temp_workspace / "data").exists()
        assert (temp_workspace / "data" / "postgres").exists()

    @patch("subprocess.run")
    def test_setup_main_containers_with_resolve_not_implemented(self, mock_run, temp_workspace):
        """Test main container setup handles Path.resolve() NotImplementedError."""
        with patch("pathlib.Path.resolve", side_effect=NotImplementedError("Cross-platform issue")):
            # Mock successful subprocess run
            mock_run.return_value.returncode = 0
            mock_run.return_value.stderr = ""

            service = MainService(temp_workspace)
            result = service._setup_main_containers(str(temp_workspace))

            assert result is True


class TestMainServiceOperations:
    """Test main service lifecycle operations."""

    def test_install_main_environment_success(self, temp_workspace):
        """Test successful main environment installation."""
        with (
            patch.object(MainService, "_validate_workspace", return_value=True),
            patch.object(MainService, "_setup_main_containers", return_value=True),
        ):
            service = MainService(temp_workspace)
            result = service.install_main_environment(str(temp_workspace))

            assert result is True

    def test_install_main_environment_validation_failure(self, temp_workspace):
        """Test main environment installation fails on workspace validation."""
        with patch.object(MainService, "_validate_workspace", return_value=False):
            service = MainService(temp_workspace)
            result = service.install_main_environment(str(temp_workspace))

            assert result is False

    def test_install_main_environment_setup_failure(self, temp_workspace):
        """Test main environment installation fails on container setup."""
        with (
            patch.object(MainService, "_validate_workspace", return_value=True),
            patch.object(MainService, "_setup_main_containers", return_value=False),
        ):
            service = MainService(temp_workspace)
            result = service.install_main_environment(str(temp_workspace))

            assert result is False

    def test_serve_main_environment_validation_failure(self, temp_workspace):
        """Test serve main fails on environment validation."""
        with patch.object(MainService, "_validate_main_environment", return_value=False):
            service = MainService(temp_workspace)
            result = service.serve_main(str(temp_workspace))

            assert result is False

    def test_serve_main_already_running(self, temp_workspace):
        """Test serve main detects already running containers."""
        with (
            patch.object(MainService, "_validate_main_environment", return_value=True),
            patch.object(
                MainService, "get_main_status", return_value={"hive-postgres": "✅ Running", "hive-api": "✅ Running"}
            ),
        ):
            service = MainService(temp_workspace)
            result = service.serve_main(str(temp_workspace))

            assert result is True

    def test_serve_main_starts_containers(self, temp_workspace):
        """Test serve main starts containers when not running."""
        with (
            patch.object(MainService, "_validate_main_environment", return_value=True),
            patch.object(
                MainService, "get_main_status", return_value={"hive-postgres": "🛑 Stopped", "hive-api": "🛑 Stopped"}
            ),
            patch.object(MainService, "_setup_main_containers", return_value=True),
        ):
            service = MainService(temp_workspace)
            result = service.serve_main(str(temp_workspace))

            assert result is True


class TestMainServiceStopRestart:
    """Test main service stop and restart operations."""

    @patch("subprocess.run")
    def test_stop_main_success(self, mock_run, temp_workspace):
        """Test successful main service stop."""
        # Mock successful subprocess run
        mock_run.return_value.returncode = 0
        mock_run.return_value.stderr = ""

        service = MainService(temp_workspace)
        result = service.stop_main(str(temp_workspace))

        assert result is True
        # Implementation uses os.fspath() which resolves symlinks, so we need to resolve the expected path
        expected_compose_path = str((temp_workspace / "docker-compose.yml").resolve())
        mock_run.assert_called_once_with(
            ["docker", "compose", "-f", expected_compose_path, "stop"],
            check=False,
            capture_output=True,
            text=True,
            timeout=60,
        )

    @patch("subprocess.run")
    def test_stop_main_missing_compose_file(self, mock_run, temp_workspace):
        """Test stop main fails when no compose file exists."""
        # Remove docker-compose.yml from fixture
        (temp_workspace / "docker-compose.yml").unlink()

        service = MainService(temp_workspace)
        result = service.stop_main(str(temp_workspace))

        assert result is False
        mock_run.assert_not_called()

    @patch("subprocess.run")
    def test_stop_main_docker_command_failure(self, mock_run, temp_workspace):
        """Test stop main handles Docker command failure."""
        # Mock failed subprocess run
        mock_run.return_value.returncode = 1
        mock_run.return_value.stderr = "Container not found"

        service = MainService(temp_workspace)
        result = service.stop_main(str(temp_workspace))

        assert result is False

    @patch("subprocess.run")
    def test_stop_main_with_mocking_issues(self, mock_run, temp_workspace):
        """Test stop main handles mocking issues gracefully."""
        # Mock successful subprocess run - this needs to be set BEFORE the patch that causes errors
        mock_run.return_value.returncode = 0
        mock_run.return_value.stderr = ""

        # The problem is that stop_main has unprotected exists() calls at the beginning
        # We need to mock in a way that allows the method to find the compose file
        # but then have mocking issues in the protected exists() call

        # Let's create a more sophisticated mock that succeeds for the first few calls
        # but fails for the protected call
        call_count = 0

        def exists_side_effect(*args, **kwargs):
            nonlocal call_count
            call_count += 1
            # First two calls (docker_compose_main and docker_compose_root) - let docker_compose_root succeed
            if call_count == 1:  # docker_compose_main.exists()
                return False
            elif call_count == 2:  # docker_compose_root.exists()
                return True  # This will make compose_file = docker_compose_root
            else:  # The protected compose_file.exists() call
                raise TypeError("Mock error")

        with patch.object(Path, "exists", side_effect=exists_side_effect):
            service = MainService(temp_workspace)
            result = service.stop_main(str(temp_workspace))

            # Should handle mocking issues in the protected exists() call and continue
            assert result is True

    @patch("subprocess.run")
    def test_stop_main_with_general_exception(self, mock_run, temp_workspace):
        """Test stop main handles general exceptions gracefully."""
        # Mock exception during subprocess execution
        mock_run.side_effect = RuntimeError("General error")

        service = MainService(temp_workspace)
        result = service.stop_main(str(temp_workspace))

        assert result is False

    @patch("subprocess.run")
    def test_restart_main_success(self, mock_run, temp_workspace):
        """Test successful main service restart."""
        # Mock successful subprocess run
        mock_run.return_value.returncode = 0
        mock_run.return_value.stderr = ""

        service = MainService(temp_workspace)
        result = service.restart_main(str(temp_workspace))

        assert result is True
        # Implementation uses os.fspath() which resolves symlinks, so we need to resolve the expected path
        expected_compose_path = str((temp_workspace / "docker-compose.yml").resolve())
        mock_run.assert_called_once_with(
            ["docker", "compose", "-f", expected_compose_path, "restart"],
            check=False,
            capture_output=True,
            text=True,
            timeout=120,
        )

    @patch("subprocess.run")
    def test_restart_main_fallback_to_stop_start(self, mock_run, temp_workspace):
        """Test restart main falls back to stop-and-start on restart command failure."""
        # Mock restart command failure, then successful stop and start
        mock_run.side_effect = [
            # First call (restart) fails
            Mock(returncode=1, stderr="Restart failed"),
            # Second call (stop) succeeds
            Mock(returncode=0, stderr=""),
            # Third call would be from serve_main -> get_main_status -> subprocess calls
        ]

        with (
            patch.object(MainService, "stop_main", return_value=True),
            patch.object(MainService, "serve_main", return_value=True),
            patch("time.sleep"),
        ):
            service = MainService(temp_workspace)
            result = service.restart_main(str(temp_workspace))

            assert result is True

    @patch("subprocess.run")
    def test_restart_main_missing_compose_file(self, mock_run, temp_workspace):
        """Test restart main fails when no compose file exists."""
        # Remove docker-compose.yml from fixture
        (temp_workspace / "docker-compose.yml").unlink()

        service = MainService(temp_workspace)
        result = service.restart_main(str(temp_workspace))

        assert result is False
        mock_run.assert_not_called()

    @patch("subprocess.run")
    def test_restart_main_with_general_exception(self, mock_run, temp_workspace):
        """Test restart main handles general exceptions gracefully."""
        # Mock exception during subprocess execution
        mock_run.side_effect = RuntimeError("General error")

        service = MainService(temp_workspace)
        result = service.restart_main(str(temp_workspace))

        assert result is False


class TestMainServiceStatus:
    """Test main service status monitoring."""

    @patch("subprocess.run")
    def test_get_main_status_both_running(self, mock_run, temp_workspace):
        """Test get main status returns running for both services."""
        # Mock successful docker compose ps calls
        mock_run.side_effect = [
            # postgres service check
            Mock(returncode=0, stdout="container_id_postgres\n"),
            Mock(returncode=0, stdout="true\n"),  # inspect postgres
            # app service check
            Mock(returncode=0, stdout="container_id_app\n"),
            Mock(returncode=0, stdout="true\n"),  # inspect app
        ]

        service = MainService(temp_workspace)
        status = service.get_main_status(str(temp_workspace))

        assert status["hive-postgres"] == "✅ Running"
        assert status["hive-api"] == "✅ Running"

    @patch("subprocess.run")
    def test_get_main_status_both_stopped(self, mock_run, temp_workspace):
        """Test get main status returns stopped for both services."""
        # Mock docker compose ps returning no container IDs
        mock_run.side_effect = [
            Mock(returncode=1, stdout=""),  # postgres not found
            Mock(returncode=1, stdout=""),  # app not found
        ]

        service = MainService(temp_workspace)
        status = service.get_main_status(str(temp_workspace))

        assert status["hive-postgres"] == "🛑 Stopped"
        assert status["hive-api"] == "🛑 Stopped"

    @patch("subprocess.run")
    def test_get_main_status_containers_not_running(self, mock_run, temp_workspace):
        """Test get main status detects containers exist but not running."""
        # Mock docker compose ps finding containers but inspect shows not running
        mock_run.side_effect = [
            Mock(returncode=0, stdout="container_id_postgres\n"),
            Mock(returncode=0, stdout="false\n"),  # inspect postgres - not running
            Mock(returncode=0, stdout="container_id_app\n"),
            Mock(returncode=0, stdout="false\n"),  # inspect app - not running
        ]

        service = MainService(temp_workspace)
        status = service.get_main_status(str(temp_workspace))

        assert status["hive-postgres"] == "🛑 Stopped"
        assert status["hive-api"] == "🛑 Stopped"

    def test_get_main_status_no_compose_file(self, temp_workspace):
        """Test get main status handles missing compose file gracefully."""
        # Remove docker-compose.yml from fixture
        (temp_workspace / "docker-compose.yml").unlink()

        service = MainService(temp_workspace)
        status = service.get_main_status(str(temp_workspace))

        assert status["hive-postgres"] == "🛑 Stopped"
        assert status["hive-api"] == "🛑 Stopped"

    @patch("subprocess.run")
    def test_get_main_status_with_exceptions(self, mock_run, temp_workspace):
        """Test get main status handles subprocess exceptions gracefully."""
        # Mock subprocess exceptions
        mock_run.side_effect = subprocess.TimeoutExpired(["docker"], timeout=10)

        service = MainService(temp_workspace)
        status = service.get_main_status(str(temp_workspace))

        # Should fallback to stopped status
        assert status["hive-postgres"] == "🛑 Stopped"
        assert status["hive-api"] == "🛑 Stopped"

    @patch("subprocess.run")
    def test_get_main_status_uses_docker_main_compose(self, mock_run, temp_workspace):
        """Test get main status prioritizes docker/main/docker-compose.yml."""
        # Create docker/main compose file
        docker_main_dir = temp_workspace / "docker" / "main"
        docker_main_dir.mkdir(parents=True, exist_ok=True)
        (docker_main_dir / "docker-compose.yml").write_text("version: '3.8'\nservices:\n  test: {}")

        # Mock successful subprocess calls
        mock_run.side_effect = [
            Mock(returncode=0, stdout="container_postgres\n"),
            Mock(returncode=0, stdout="true\n"),
            Mock(returncode=0, stdout="container_app\n"),
            Mock(returncode=0, stdout="true\n"),
        ]

        service = MainService(temp_workspace)
        service.get_main_status(str(temp_workspace))

        # Verify it used the docker/main compose file
        expected_compose_file = str(docker_main_dir / "docker-compose.yml")
        compose_calls = [call for call in mock_run.call_args_list if expected_compose_file in str(call)]
        assert len(compose_calls) > 0


class TestMainServiceLogs:
    """Test main service log retrieval."""

    @patch("subprocess.run")
    def test_show_main_logs_success(self, mock_run, temp_workspace):
        """Test successful main service log retrieval."""
        # Mock successful log commands for both services
        mock_run.side_effect = [
            Mock(returncode=0, stdout="2024-01-01 10:00:00 PostgreSQL log entry\n"),
            Mock(returncode=0, stdout="2024-01-01 10:00:01 FastAPI log entry\n"),
        ]

        service = MainService(temp_workspace)
        result = service.show_main_logs(str(temp_workspace))

        assert result is True
        # Should have called logs command for both postgres and app services
        assert mock_run.call_count == 2

    @patch("subprocess.run")
    def test_show_main_logs_with_tail_limit(self, mock_run, temp_workspace):
        """Test main service log retrieval with tail limit."""
        # Mock successful log commands
        mock_run.side_effect = [
            Mock(returncode=0, stdout="Last 50 PostgreSQL lines\n"),
            Mock(returncode=0, stdout="Last 50 FastAPI lines\n"),
        ]

        service = MainService(temp_workspace)
        result = service.show_main_logs(str(temp_workspace), tail=50)

        assert result is True
        # Verify tail parameter was passed to docker compose logs
        for call_args in mock_run.call_args_list:
            args = call_args[0][0]
            assert "--tail" in args
            assert "50" in args

    @patch("subprocess.run")
    def test_show_main_logs_empty_output(self, mock_run, temp_workspace):
        """Test main service log retrieval handles empty log output."""
        # Mock empty log output
        mock_run.side_effect = [
            Mock(returncode=0, stdout=""),  # empty postgres logs
            Mock(returncode=0, stdout=""),  # empty app logs
        ]

        service = MainService(temp_workspace)
        result = service.show_main_logs(str(temp_workspace))

        assert result is True

    @patch("subprocess.run")
    def test_show_main_logs_command_failure(self, mock_run, temp_workspace):
        """Test main service log retrieval handles command failures."""
        # Mock failed log commands
        mock_run.side_effect = [
            Mock(returncode=1, stderr="Container not found"),
            Mock(returncode=1, stderr="Service unavailable"),
        ]

        service = MainService(temp_workspace)
        result = service.show_main_logs(str(temp_workspace))

        # Should still return True even if individual commands fail
        assert result is True

    def test_show_main_logs_missing_compose_file(self, temp_workspace):
        """Test main service log retrieval handles missing compose file."""
        # Remove docker-compose.yml from fixture
        (temp_workspace / "docker-compose.yml").unlink()

        service = MainService(temp_workspace)
        result = service.show_main_logs(str(temp_workspace))

        assert result is False

    @patch("subprocess.run")
    def test_show_main_logs_with_mocking_issues(self, mock_run, temp_workspace):
        """Test main service log retrieval handles mocking issues gracefully."""
        # Mock successful log commands first
        mock_run.side_effect = [
            Mock(returncode=0, stdout="Log content\n"),
            Mock(returncode=0, stdout="Log content\n"),
        ]

        # Similar sophisticated mocking as stop_main
        call_count = 0

        def exists_side_effect(*args, **kwargs):
            nonlocal call_count
            call_count += 1
            # First two calls (docker_compose_main and docker_compose_root) - let docker_compose_root succeed
            if call_count == 1:  # docker_compose_main.exists()
                return False
            elif call_count == 2:  # docker_compose_root.exists()
                return True  # This will make compose_file = docker_compose_root
            else:  # The protected compose_file.exists() call
                raise TypeError("Mock error")

        with patch.object(Path, "exists", side_effect=exists_side_effect):
            service = MainService(temp_workspace)
            result = service.show_main_logs(str(temp_workspace))

            # Should handle mocking issues in the protected exists() call and continue
            assert result is True

    @patch("subprocess.run")
    def test_show_main_logs_with_general_exception(self, mock_run, temp_workspace):
        """Test main service log retrieval handles general exceptions gracefully."""
        # Mock exception during subprocess execution
        mock_run.side_effect = RuntimeError("General error")

        service = MainService(temp_workspace)
        result = service.show_main_logs(str(temp_workspace))

        assert result is False


class TestMainServiceUninstall:
    """Test main service uninstall operations."""

    def test_uninstall_preserve_data_success(self, temp_workspace):
        """Test successful uninstall while preserving data."""
        with patch.object(MainService, "_cleanup_containers_only", return_value=True):
            service = MainService(temp_workspace)
            result = service.uninstall_preserve_data(str(temp_workspace))

            assert result is True

    def test_uninstall_preserve_data_cleanup_failure(self, temp_workspace):
        """Test uninstall preserve data continues even if cleanup has issues."""
        with patch.object(MainService, "_cleanup_containers_only", return_value=False):
            service = MainService(temp_workspace)
            result = service.uninstall_preserve_data(str(temp_workspace))

            # Should return True even if cleanup has issues
            assert result is True

    def test_uninstall_wipe_data_success(self, temp_workspace):
        """Test successful uninstall with data wiping."""
        with patch.object(MainService, "_cleanup_main_environment", return_value=True):
            service = MainService(temp_workspace)
            result = service.uninstall_wipe_data(str(temp_workspace))

            assert result is True

    def test_uninstall_wipe_data_cleanup_failure(self, temp_workspace):
        """Test uninstall wipe data continues even if cleanup has issues."""
        with patch.object(MainService, "_cleanup_main_environment", return_value=False):
            service = MainService(temp_workspace)
            result = service.uninstall_wipe_data(str(temp_workspace))

            # Should return True even if cleanup has issues
            assert result is True

    @patch("subprocess.run")
    def test_cleanup_containers_only_success(self, mock_run, temp_workspace):
        """Test successful container-only cleanup."""
        # Mock successful docker compose down
        mock_run.return_value.returncode = 0

        service = MainService(temp_workspace)
        result = service._cleanup_containers_only(str(temp_workspace))

        assert result is True
        # Implementation uses os.fspath() which resolves symlinks, so we need to resolve the expected path
        expected_compose_path = str((temp_workspace / "docker-compose.yml").resolve())
        mock_run.assert_called_once_with(
            ["docker", "compose", "-f", expected_compose_path, "down"], check=False, capture_output=True, timeout=60
        )

    @patch("subprocess.run")
    def test_cleanup_containers_only_no_compose_file(self, mock_run, temp_workspace):
        """Test container cleanup handles missing compose file gracefully."""
        # Remove docker-compose.yml from fixture
        (temp_workspace / "docker-compose.yml").unlink()

        service = MainService(temp_workspace)
        result = service._cleanup_containers_only(str(temp_workspace))

        # Should return True even without compose file
        assert result is True

    @patch("subprocess.run")
    def test_cleanup_containers_only_docker_exception(self, mock_run, temp_workspace):
        """Test container cleanup handles Docker exceptions gracefully."""
        # Mock docker command exception
        mock_run.side_effect = subprocess.SubprocessError("Docker error")

        service = MainService(temp_workspace)
        result = service._cleanup_containers_only(str(temp_workspace))

        # Should return True even on Docker exceptions
        assert result is True

    @patch("subprocess.run")
    @patch("shutil.rmtree")
    def test_cleanup_main_environment_with_data_wipe(self, mock_rmtree, mock_run, temp_workspace):
        """Test full environment cleanup including data wiping."""
        # Create data directory to be wiped
        postgres_data_dir = temp_workspace / "data" / "postgres"
        postgres_data_dir.mkdir(parents=True, exist_ok=True)
        (postgres_data_dir / "test_data.sql").write_text("test data")

        # Mock successful docker compose down
        mock_run.return_value.returncode = 0

        service = MainService(temp_workspace)
        result = service._cleanup_main_environment(str(temp_workspace))

        assert result is True
        # Implementation uses os.fspath() which resolves symlinks, so we need to resolve the expected path
        expected_compose_path = str((temp_workspace / "docker-compose.yml").resolve())
        # Should call docker compose down with -v flag for volumes
        mock_run.assert_called_once_with(
            ["docker", "compose", "-f", expected_compose_path, "down", "-v"],
            check=False,
            capture_output=True,
            timeout=60,
        )
        # Implementation resolves workspace path, so data_dir is also resolved - need to match that
        expected_data_dir = temp_workspace.resolve() / "data" / "postgres"
        # Should attempt to remove data directory
        mock_rmtree.assert_called_once_with(expected_data_dir)

    @patch("subprocess.run")
    @patch("shutil.rmtree")
    def test_cleanup_main_environment_data_removal_error(self, mock_rmtree, mock_run, temp_workspace):
        """Test full environment cleanup handles data removal errors gracefully."""
        # Create data directory
        postgres_data_dir = temp_workspace / "data" / "postgres"
        postgres_data_dir.mkdir(parents=True, exist_ok=True)

        # Mock successful docker command but failed data removal
        mock_run.return_value.returncode = 0
        mock_rmtree.side_effect = PermissionError("Cannot delete data")

        service = MainService(temp_workspace)
        result = service._cleanup_main_environment(str(temp_workspace))

        # Should return True even if data removal fails
        assert result is True

    @patch("subprocess.run")
    def test_cleanup_main_environment_with_mocking_issues(self, mock_run, temp_workspace):
        """Test cleanup handles mocking issues gracefully."""
        with patch.object(Path, "exists", side_effect=TypeError("Mock error")):
            # Mock successful docker command
            mock_run.return_value.returncode = 0

            service = MainService(temp_workspace)
            result = service._cleanup_main_environment(str(temp_workspace))

            # Should handle mocking issues and continue
            assert result is True

    @patch("subprocess.run")
    def test_cleanup_main_environment_general_exception(self, mock_run, temp_workspace):
        """Test cleanup handles general exceptions gracefully."""
        # Mock general exception
        mock_run.side_effect = RuntimeError("General error")

        service = MainService(temp_workspace)
        result = service._cleanup_main_environment(str(temp_workspace))

        # Should return True even on general exceptions (best-effort cleanup)
        assert result is True


class TestMainServiceEnvironmentValidation:
    """Test environment variable validation."""

    def test_validate_environment_stub(self, temp_workspace):
        """Test environment validation stub returns True."""
        service = MainService(temp_workspace)
        result = service._validate_environment()

        # Current stub implementation always returns True
        assert result is True


class TestMainServiceEdgeCases:
    """Test edge cases and error conditions."""

    def test_operations_with_invalid_workspace(self):
        """Test service operations handle invalid workspace paths gracefully."""
        invalid_path = Path("/nonexistent/workspace/path")
        service = MainService(invalid_path)

        # Operations should handle invalid paths gracefully
        assert service.install_main_environment(str(invalid_path)) is False
        assert service.serve_main(str(invalid_path)) is False
        assert service.stop_main(str(invalid_path)) is False
        assert service.restart_main(str(invalid_path)) is False
        assert service.show_main_logs(str(invalid_path)) is False
        assert isinstance(service.get_main_status(str(invalid_path)), dict)

    def test_cross_platform_path_handling(self):
        """Test service handles cross-platform path scenarios."""
        # Test Windows-style path
        windows_path = "C:\\Users\\test\\workspace"
        service = MainService(windows_path)

        assert isinstance(service.workspace_path, Path)

    def test_docker_compose_file_priority(self, temp_workspace):
        """Test docker compose file detection priority."""
        # Create both docker/main/docker-compose.yml and root docker-compose.yml
        docker_main_dir = temp_workspace / "docker" / "main"
        docker_main_dir.mkdir(parents=True, exist_ok=True)
        (docker_main_dir / "docker-compose.yml").write_text("# Main compose")

        service = MainService(temp_workspace)

        # Should prioritize docker/main/docker-compose.yml
        with patch.object(MainService, "_setup_main_containers") as mock_setup:
            mock_setup.return_value = True
            service.install_main_environment(str(temp_workspace))
            # Verify the docker/main compose was used (this test validates behavior indirectly)

    def test_concurrent_operations_safety(self, temp_workspace):
        """Test service handles potential concurrent operation scenarios."""
        service = MainService(temp_workspace)

        # Mock concurrent status checks
        with patch.object(
            MainService, "get_main_status", return_value={"hive-postgres": "✅ Running", "hive-api": "✅ Running"}
        ):
            # Multiple status checks should work
            status1 = service.get_main_status(str(temp_workspace))
            status2 = service.get_main_status(str(temp_workspace))

            assert status1 == status2

    def test_path_resolution_edge_cases(self, temp_workspace):
        """Test various path resolution edge cases."""
        # Test with relative path
        with patch("pathlib.Path.cwd", return_value=temp_workspace):
            service = MainService(".")
            assert service.workspace_path.is_absolute()

    def test_docker_compose_error_handling(self, temp_workspace):
        """Test comprehensive Docker Compose error handling scenarios."""
        service = MainService(temp_workspace)

        # Test individual error scenarios separately to avoid cross-contamination

        # Test TimeoutExpired
        error = subprocess.TimeoutExpired(["docker"], timeout=120)
        with patch("subprocess.run", side_effect=error):
            result = service.stop_main(str(temp_workspace))
            assert result is False, "stop_main should return False for TimeoutExpired"

        with (
            patch("subprocess.run", side_effect=error),
            patch.object(service, "stop_main", return_value=False),
            patch.object(service, "serve_main", return_value=False),
        ):
            result = service.restart_main(str(temp_workspace))
            assert result is False, "restart_main should return False for TimeoutExpired"

        with patch("subprocess.run", side_effect=error):
            result = service._setup_main_containers(str(temp_workspace))
            assert result is False, "_setup_main_containers should return False for TimeoutExpired"

        # Test SubprocessError
        error = subprocess.SubprocessError("Docker daemon error")
        with patch("subprocess.run", side_effect=error):
            result = service.stop_main(str(temp_workspace))
            assert result is False, "stop_main should return False for SubprocessError"

        with (
            patch("subprocess.run", side_effect=error),
            patch.object(service, "stop_main", return_value=False),
            patch.object(service, "serve_main", return_value=False),
        ):
            result = service.restart_main(str(temp_workspace))
            assert result is False, "restart_main should return False for SubprocessError"

        with patch("subprocess.run", side_effect=error):
            result = service._setup_main_containers(str(temp_workspace))
            assert result is False, "_setup_main_containers should return False for SubprocessError"

        # Test FileNotFoundError
        error = FileNotFoundError("docker command not found")
        with patch("subprocess.run", side_effect=error):
            result = service.stop_main(str(temp_workspace))
            assert result is False, "stop_main should return False for FileNotFoundError"

        with (
            patch("subprocess.run", side_effect=error),
            patch.object(service, "stop_main", return_value=False),
            patch.object(service, "serve_main", return_value=False),
        ):
            result = service.restart_main(str(temp_workspace))
            assert result is False, "restart_main should return False for FileNotFoundError"

        with patch("subprocess.run", side_effect=error):
            result = service._setup_main_containers(str(temp_workspace))
            assert result is False, "_setup_main_containers should return False for FileNotFoundError"

        # Test PermissionError - this requires special handling because _setup_main_containers
        # doesn't catch PermissionError, so it will propagate up
        error = PermissionError("Docker access denied")
        with patch("subprocess.run", side_effect=error):
            result = service.stop_main(str(temp_workspace))
            assert result is False, "stop_main should return False for PermissionError"

        with (
            patch("subprocess.run", side_effect=error),
            patch.object(service, "stop_main", return_value=False),
            patch.object(service, "serve_main", return_value=False),
        ):
            result = service.restart_main(str(temp_workspace))
            assert result is False, "restart_main should return False for PermissionError"

        # For _setup_main_containers, PermissionError is not caught by the specific exception handler
        # so we need to test that the exception propagates properly
        with patch("subprocess.run", side_effect=error):
            try:
                result = service._setup_main_containers(str(temp_workspace))
                raise AssertionError("_setup_main_containers should raise PermissionError, not return False")
            except PermissionError:
                pass  # This is expected behavior - PermissionError propagates up

        # Test OSError - similar to PermissionError, this is not caught by _setup_main_containers
        error = OSError("System error")
        with patch("subprocess.run", side_effect=error):
            result = service.stop_main(str(temp_workspace))
            assert result is False, "stop_main should return False for OSError"

        with (
            patch("subprocess.run", side_effect=error),
            patch.object(service, "stop_main", return_value=False),
            patch.object(service, "serve_main", return_value=False),
        ):
            result = service.restart_main(str(temp_workspace))
            assert result is False, "restart_main should return False for OSError"

        # For _setup_main_containers, OSError is not caught by the specific exception handler
        # so we need to test that the exception propagates properly
        with patch("subprocess.run", side_effect=error):
            try:
                result = service._setup_main_containers(str(temp_workspace))
                raise AssertionError("_setup_main_containers should raise OSError, not return False")
            except OSError:
                pass  # This is expected behavior - OSError propagates up


class TestMainServiceIntegration:
    """Test service integration with external dependencies."""

    @patch("subprocess.run")
    def test_integration_with_docker_compose_main_directory(self, mock_run, temp_workspace):
        """Test service integrates correctly with docker/main directory structure."""
        # Create docker/main directory structure
        docker_main_dir = temp_workspace / "docker" / "main"
        docker_main_dir.mkdir(parents=True, exist_ok=True)

        compose_content = """
version: '3.8'
services:
  postgres:
    image: postgres:15
    container_name: hive-postgres
    ports:
      - "5532:5432"
    environment:
      POSTGRES_DB: hive
      POSTGRES_USER: hive
      POSTGRES_PASSWORD: main_password
    volumes:
      - postgres_data:/var/lib/postgresql/data
    restart: unless-stopped

  app:
    image: automagik-hive:latest
    container_name: main-app
    ports:
      - "8886:8886"
    environment:
      - HIVE_API_PORT=8886
      - HIVE_DATABASE_URL=postgresql://hive:main_password@postgres:5432/hive
    depends_on:
      - postgres
    restart: unless-stopped

volumes:
  postgres_data:
"""
        (docker_main_dir / "docker-compose.yml").write_text(compose_content)

        # Mock successful Docker operations
        mock_run.return_value.returncode = 0
        mock_run.return_value.stderr = ""

        service = MainService(temp_workspace)

        # Test various operations use the correct compose file
        result = service._setup_main_containers(str(temp_workspace))
        assert result is True

        # Implementation uses os.fspath() which resolves symlinks, so we need to resolve the expected path
        expected_compose_file = str((docker_main_dir / "docker-compose.yml").resolve())
        mock_run.assert_called_with(
            ["docker", "compose", "-f", expected_compose_file, "up", "-d"],
            check=False,
            capture_output=True,
            text=True,
            timeout=120,
        )

    def test_integration_with_environment_configuration(self, temp_workspace):
        """Test service integrates with environment configuration properly."""
        # Update .env file with comprehensive configuration
        env_content = """
# Main Application Configuration
HIVE_API_PORT=8886
HIVE_DATABASE_URL=postgresql://hive:main_password@localhost:5532/hive
HIVE_API_KEY=main_production_key

# PostgreSQL Configuration
POSTGRES_PORT=5532
POSTGRES_DB=hive
POSTGRES_USER=hive
POSTGRES_PASSWORD=main_password

# Additional Configuration
HIVE_LOG_LEVEL=INFO
HIVE_ENVIRONMENT=production
"""
        (temp_workspace / ".env").write_text(env_content)

        service = MainService(temp_workspace)

        # Environment validation should pass
        result = service._validate_main_environment(temp_workspace)
        assert result is True

    def test_integration_with_data_persistence(self, temp_workspace):
        """Test service integrates with data persistence correctly."""
        with patch("subprocess.run") as mock_run:
            mock_run.return_value.returncode = 0
            mock_run.return_value.stderr = ""

            service = MainService(temp_workspace)
            result = service._setup_main_containers(str(temp_workspace))

            assert result is True

            # Verify data directories were created
            assert (temp_workspace / "data").exists()
            assert (temp_workspace / "data" / "postgres").exists()

    @patch("subprocess.run")
    def test_integration_with_logging_system(self, mock_run, temp_workspace):
        """Test service integrates with logging system for comprehensive output."""
        # Mock log outputs for both services
        postgres_logs = """
2024-01-01 10:00:00.000 UTC [1] LOG:  database system is ready to accept connections
2024-01-01 10:00:01.000 UTC [1] LOG:  autovacuum launcher started
"""
        app_logs = """
2024-01-01 10:00:02 INFO:     Uvicorn running on http://0.0.0.0:8886
2024-01-01 10:00:03 INFO:     Application startup complete
"""

        mock_run.side_effect = [
            Mock(returncode=0, stdout=postgres_logs),
            Mock(returncode=0, stdout=app_logs),
        ]

        service = MainService(temp_workspace)
        result = service.show_main_logs(str(temp_workspace))

        assert result is True

        # Verify both services were queried for logs
        assert mock_run.call_count == 2

        # Verify correct services were queried
        call_args_list = [str(call) for call in mock_run.call_args_list]
        assert any("postgres" in call for call in call_args_list)
        assert any("app" in call for call in call_args_list)
