# Copyright © 2023-2024 Apple Inc.
import logging
import typing
from typing import List, Tuple

import numpy as np

from pfl.context import CentralContext, UserContext
from pfl.metrics import Histogram, MetricName, MetricNamePostfix, Metrics, SkipSerialization, get_overall_value
from pfl.postprocessor.base import Postprocessor
from pfl.stats import TrainingStatistics

logger = logging.getLogger(name=__name__)


class SummaryMetrics(Postprocessor):
    """
    Given a name of an existing metric, accumulate a histogram of per-user
    values of that metric, then calculate aggregate statistics on the
    histogram.

    This is useful for metrics that cannot be processed using (weighted)
    average and for providing more details on metric distribution.
    For example, this class can be used to get an accuracy below which
    1% of users fall (i.e., 1st percentile).

    :param metric_name:
        The name of the metric to accumulate histogram for. Must previously
        have been generated by e.g. model evaluation.
    :param quantiles:
        A list of quantiles to calculate on the histogram. E.g. quantile 0.1
        will return the histogram bound such that 10% of users fall below
        this bound based on the aggregated histogram (using linear
        interpolation). That is why to accurately compute quantiles, there
        must be sufficiently many histogram bins (typically well over 100).
    :param min_bound:
        Minimum bound for histogram. Values less than bound will be ignored.
    :param max_bound:
        Maximum bound for histogram. Values larger than bound will be ignored.
    :param num_bins:
        Number of bins for the histogram. The number of bins affects the
        fidelity of the quantiles and other summary statistics computed on the
        specified source metric.
        There is no point in picking a smaller value ``<100`` for lower
        fidelity unless it is only for visualization purposes (the histograms
        generated by this postprocessor can be visualized with e.g.
        :class:`~pfl.callback.tensorboard.TensorBoardCallback`).

        TODO: Add support for non-equally-spaced bins upon request.
          rdar://103595866 (histogram with non-equally-spaced bins)
    :param frequency:
        Compute summary of metric every ``frequency`` central iterations. This
        frequency should be the same as the frequency at which the metric
        being summarised is generated.
    :param stddev:
        Calculate standard deviation of the histogram.

    :example:

      This example shows summary metrics on accuracy. Since we know accuracy
      is bounded ``[0,1]``, we set ``min_bound`` and ``max_bound`` to those
      values. With ``num_bins=1000``, the range of each bin will be 0.1%
      (a value of 0.001) of the total range specified by the bounds.

      .. code-block:: python

        SummaryMetrics(
            metric_name=TrainMetricName('accuracy',
                                        Population.TRAIN,
                                        after_training=False),
            quantiles=[0.01, 0.02, 0.98, 0.99],
            min_bound=0.0,
            max_bound=1.0,
            num_bins=1000)

    :example:

      This example show summary metrics on loss. We know cross-entropy loss
      is bounded by ``[0,inf)``, so we need to pick a ``max_bound`` that suits
      the current experiment. The loss on iteration ``0`` for Reddit dataset
      was ``9.3``, so a good pick is slightly above ``9.3`` in this case.

      .. code-block:: python

        SummaryMetrics(
            metric_name=TrainMetricName('loss',
                                        Population.TRAIN,
                                        after_training=False),
            quantiles=[0.01, 0.02, 0.98, 0.99],
            min_bound=0.0,
            max_bound=10.0,
            num_bins=1000)

    """

    def __init__(self,
                 metric_name: MetricName,
                 quantiles: List[float],
                 min_bound: float,
                 max_bound: float,
                 num_bins: int = 100,
                 frequency: int = 1,
                 stddev: bool = False):

        if len(quantiles) and num_bins <= 100:
            logger.warning(
                f'You only have {num_bins} bins for histogram, '
                'increase to >100 for more accurate calculation of quantiles')
        self._metric_name = metric_name
        self._min_bound = min_bound
        self._max_bound = max_bound
        self._num_bins = num_bins
        self._quantiles = quantiles
        self._frequency = frequency
        self._stddev = stddev
        self._hist_name = SkipSerialization(
            MetricNamePostfix(self._metric_name, 'histogram'))

    def postprocess_one_user(
            self, *, stats: TrainingStatistics,
            user_context: UserContext) -> Tuple[TrainingStatistics, Metrics]:
        """
        Creates histogram metric of a single user's metrics.

        Code continues without action if metric to be used for histogram is
        not in user metrics passed via the user context.

        :param stats:
            Statistics returned from the local training procedure of this user.
            It is just passed through to output.
        :param user_context:
            Additional information about the current user. This includes the
            values of metrics to use in summary statistics.
        :return:
            A tuple `(stats, metrics)`, where `stats` is same as the input, and
            `metrics` contains the histogram metric for one user.
        """
        if self._metric_name not in user_context.metrics:
            return stats, Metrics()

        user_value = get_overall_value(user_context.metrics[self._metric_name])
        hist = Histogram.from_values_range([user_value],
                                           num_bins=self._num_bins,
                                           min_bound=self._min_bound,
                                           max_bound=self._max_bound)
        return stats, Metrics([(self._hist_name, hist)])

    def _quantile(self, counts, bins, quantile_target):
        cdf = np.cumsum(counts)
        cdf = cdf / cdf[-1]
        ix_right = len(cdf[cdf < quantile_target])
        ix_left = ix_right - 1
        quantile_left = cdf[ix_left]
        quantile_right = cdf[ix_right]

        # Estimate quantile with linear interpolation.
        return bins[ix_left] + (bins[ix_right] - bins[ix_left]) * (
            quantile_target - quantile_left) / (quantile_right - quantile_left)

    def postprocess_server(
            self, *, stats: TrainingStatistics,
            central_context: CentralContext,
            aggregate_metrics: Metrics) -> Tuple[TrainingStatistics, Metrics]:
        """
        Derive aggregate statistics from the aggregated histogram.

        Executes every ``freqency`` number of central iterations, and if the
        metric to be summarised is in ``aggregate_metrics``.

        :param stats:
            The aggregated statistics. Not used.
        :param central_context:
            Information about aggregation and other useful server-side
            properties.
            Should contain aggregated histogram metric.
        :return:
            A tuple `(stats, metrics)`, where `stats`
            is same as input, and `metrics` contains aggregate statistics
            about the histogram of interest.
        """
        if central_context.current_central_iteration % self._frequency != 0:
            return stats, Metrics()

        if self._hist_name not in aggregate_metrics:
            return stats, Metrics()

        metrics = Metrics()
        hist = aggregate_metrics[self._hist_name]
        hist = typing.cast(Histogram, hist)
        bins, counts = hist.bins, hist.bin_counts
        for q in self._quantiles:
            metrics[MetricNamePostfix(self._metric_name,
                                      f'quantile={q}')] = self._quantile(
                                          counts, bins, q)

        mids = bins[:-1] + (bins[1:] - bins[:-1]) / 2
        mean = np.average(mids, weights=counts)
        if self._stddev:
            metrics[MetricNamePostfix(self._metric_name, 'stddev')] = np.sqrt(
                np.average((mids - mean)**2, weights=counts))

        return stats, metrics
