"""honeybee radiance schedule command."""
import click
import sys
import logging
import os

from ladybug.epw import EPW


_logger = logging.getLogger(__name__)


@click.group(help='Commands to create and modify schedules.')
def schedule():
    pass


@schedule.command('epw-to-daylight-hours')
@click.argument(
    'epw', type=click.Path(exists=True, dir_okay=False, resolve_path=True))
@click.option('--folder', '-f', help='Output folder.', default='.', show_default=True)
@click.option(
    '--name', '-n', help='Output file name for schedule.',
    type=click.STRING, default='daylight_hours', show_default=True
)
def epw_to_daylight_hours(epw, folder, name):
    """Convert EPW to EN 17037 schedule as a CSV file.

    This command generates a valid schedule for EN 17037, also known as daylight hours.
    Rather than a typical occupancy schedule, the daylight hours is half the year with
    the largest quantity of daylight. This is generated by sorting the EPW file by
    diffuse horizontal illuminance.

    \b
    Args:
        epw: Path to epw file.
    """
    try:
        epw_data = EPW(epw)
        diffuse_horizontal_illuminance = epw_data.diffuse_horizontal_illuminance
        _illuminance, indices = diffuse_horizontal_illuminance.highest_values(4380)
        values = ['1' if i in indices else '0' for i in range(8760)]

        if not os.path.isdir(folder):
            os.makedirs(folder)

        # write csv
        file_path = os.path.join(folder, '%s.csv' % name)
        with open(file_path, 'w') as fp:
            fp.write('\n'.join(values))

    except Exception:
        _logger.exception('Failed to generate daylight hours schedule.')
        sys.exit(1)
    else:
        sys.exit(0)
