# Broadie A2A Registry Server

A simple, lightweight registry server for Agent-to-Agent (A2A) communication in the Broadie framework.

## Features

- **Agent Registration**: Agents can register themselves with identity and capability information
- **Heartbeat Monitoring**: Tracks agent health with configurable timeout
- **Agent Discovery**: Provides endpoints for agents to discover peers
- **RESTful API**: Simple HTTP API for all operations
- **Auto Cleanup**: Automatically removes inactive agents
- **Basic Authentication**: Optional API key authentication

## Quick Start

### 1. Run the Registry Server

```bash
# Using Python directly
python registry_server.py --port 8001 --host 0.0.0.0

# Or with custom settings
python registry_server.py --port 8001 --timeout 120 --log-level DEBUG
```

### 2. Configure Your Agents

Update your agent's `.env` file:

```bash
# Enable A2A communication
A2A_ENABLED=true
A2A_AGENT_ID=my-agent-001
A2A_AGENT_NAME=My Agent
A2A_REGISTRY_URL=http://localhost:8001
```

### 3. Start Your Agents

```bash
# Your agents will automatically register with the registry
broadie serve agent.json
```

## API Endpoints

### Agent Registration

```bash
POST /agents/{agent_id}/register
```

Register a new agent with the registry.

**Request Body:**
```json
{
  "agent_id": "my-agent-001",
  "agent_name": "My Agent",
  "address": "http://localhost:8000",
  "timestamp": 1703097600,
  "status": "online",
  "capabilities": {
    "streaming": true,
    "tools": ["search", "analyze"]
  }
}
```

### Heartbeat

```bash
POST /agents/{agent_id}/heartbeat
```

Send heartbeat to maintain agent registration.

**Request Body:**
```json
{
  "agent_id": "my-agent-001",
  "timestamp": 1703097600,
  "status": "online"
}
```

### Discovery

```bash
GET /agents
```

List all active agents.

**Response:**
```json
{
  "agents": [
    {
      "agent_id": "my-agent-001",
      "agent_name": "My Agent",
      "address": "http://localhost:8000",
      "status": "online",
      "capabilities": {...}
    }
  ],
  "total": 1,
  "timestamp": 1703097600
}
```

### Agent Information

```bash
GET /agents/{agent_id}
```

Get detailed information about a specific agent.

### Statistics

```bash
GET /stats
```

Get registry statistics and health information.

## Configuration

### Command Line Options

- `--host`: Host to bind to (default: 0.0.0.0)
- `--port`: Port to bind to (default: 8001)
- `--timeout`: Heartbeat timeout in seconds (default: 120)
- `--log-level`: Log level (default: INFO)

### Environment Variables

```bash
REGISTRY_HOST=0.0.0.0
REGISTRY_PORT=8001
REGISTRY_TIMEOUT=120
REGISTRY_LOG_LEVEL=INFO
REGISTRY_API_KEY=your-secret-key  # Optional
```

## Docker Deployment

### Build Docker Image

```dockerfile
FROM python:3.11-slim

WORKDIR /app
COPY requirements.txt .
RUN pip install -r requirements.txt

COPY registry_server.py .
EXPOSE 8001

CMD ["python", "registry_server.py", "--host", "0.0.0.0", "--port", "8001"]
```

### Run with Docker

```bash
# Build the image
docker build -t broadie-registry .

# Run the container
docker run -d -p 8001:8001 --name broadie-registry broadie-registry

# With custom configuration
docker run -d -p 8001:8001 \
  -e REGISTRY_TIMEOUT=180 \
  -e REGISTRY_LOG_LEVEL=DEBUG \
  --name broadie-registry broadie-registry
```

## Docker Compose

```yaml
version: '3.8'
services:
  registry:
    build: .
    ports:
      - "8001:8001"
    environment:
      - REGISTRY_TIMEOUT=120
      - REGISTRY_LOG_LEVEL=INFO
    restart: unless-stopped
    
  agent1:
    build: ../my-agent
    environment:
      - A2A_ENABLED=true
      - A2A_AGENT_ID=agent-001
      - A2A_REGISTRY_URL=http://registry:8001
    depends_on:
      - registry
```

## Production Deployment

### Security Considerations

1. **API Authentication**: Enable API key authentication in production
2. **Network Security**: Use HTTPS and restrict access to trusted networks
3. **Rate Limiting**: Implement rate limiting for registration endpoints
4. **Monitoring**: Set up proper logging and monitoring

### High Availability

1. **Load Balancer**: Use a load balancer for multiple registry instances
2. **Database**: Replace in-memory storage with persistent database
3. **Backup**: Implement regular backup of agent registrations
4. **Health Checks**: Configure proper health checks

### Monitoring

```bash
# Check registry health
curl http://localhost:8001/health

# Get statistics
curl http://localhost:8001/stats

# List active agents
curl http://localhost:8001/agents
```

## Development

### Running Tests

```bash
# Install test dependencies
pip install pytest httpx pytest-asyncio

# Run tests
pytest test_registry.py
```

### API Testing

```bash
# Register an agent
curl -X POST http://localhost:8001/agents/test-001/register \
  -H "Content-Type: application/json" \
  -d '{
    "agent_id": "test-001",
    "agent_name": "Test Agent",
    "address": "http://localhost:8000",
    "timestamp": 1703097600,
    "status": "online"
  }'

# Send heartbeat
curl -X POST http://localhost:8001/agents/test-001/heartbeat \
  -H "Content-Type: application/json" \
  -d '{
    "agent_id": "test-001",
    "timestamp": 1703097600,
    "status": "online"
  }'

# List agents
curl http://localhost:8001/agents
```

## Troubleshooting

### Common Issues

1. **Port Already in Use**: Change the port using `--port` option
2. **Agents Not Registering**: Check A2A_REGISTRY_URL configuration
3. **Heartbeat Timeouts**: Adjust timeout with `--timeout` option
4. **Network Issues**: Ensure firewall allows traffic on the registry port

### Logs

Check the registry logs for debugging information:

```bash
python registry_server.py --log-level DEBUG
```

### Health Check

The registry provides health check endpoints:

```bash
# Basic health check
curl http://localhost:8001/health

# Detailed status
curl http://localhost:8001/
```

## License

This registry server template is part of the Broadie framework and follows the same MIT license.