"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const dev_env_1 = require("../dev-env");
const file_1 = require("../file");
const gitpod_1 = require("../gitpod");
const logging = require("../logging");
const util_1 = require("./util");
// This is duplicated vs exported
const GITPOD_FILE = '.gitpod.yml';
const DEVCONTAINER_FILE = '.devcontainer.json';
logging.disable();
describe('dev environment constructor', () => {
    test('for gitpod', () => {
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: false,
        });
        const task = project.addTask('dummy-task', { exec: 'echo hello' });
        // WHEN
        new gitpod_1.Gitpod(project, {
            dockerImage: dev_env_1.DevEnvironmentDockerImage.fromImage('ubuntu:latest'),
            prebuilds: {
                pullRequests: true,
            },
            tasks: [task],
            ports: ['3000'],
            vscodeExtensions: ['dbaeumer.vscode-eslint@2.1.13:5sYlSD6wJi5s3xqD8hupUw=='],
        });
        project.synth();
        // THEN
        const outdir = util_1.synthSnapshot(project);
        expect(outdir[GITPOD_FILE]).toMatchSnapshot();
    });
});
describe('dev environment enable/disable', () => {
    test('given gitpod and devContainer are false', () => {
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: false,
            devContainer: false,
        });
        // WHEN
        project.synth();
        // THEN
        const gitpodFilePath = path.join(project.outdir, GITPOD_FILE);
        const devContainerFilePath = path.join(project.outdir, DEVCONTAINER_FILE);
        expect(fs.existsSync(gitpodFilePath)).toBeFalsy();
        expect(fs.existsSync(devContainerFilePath)).toBeFalsy();
    });
    test('no gitpod/devcontainer files if they are empty', () => {
        // WHEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // THEN
        project.synth();
        const gitpodFilePath = path.join(project.outdir, GITPOD_FILE);
        const devContainerFilePath = path.join(project.outdir, DEVCONTAINER_FILE);
        expect(fs.existsSync(gitpodFilePath)).toBeFalsy();
        expect(fs.existsSync(devContainerFilePath)).toBeFalsy();
    });
    test('given gitpod and devContainer are true', () => {
        var _a, _b;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addDockerImage({ image: 'foo' });
        (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addPorts('1234');
        // THEN
        project.synth();
        const gitpodFilePath = path.join(project.outdir, GITPOD_FILE);
        const devContainerFilePath = path.join(project.outdir, DEVCONTAINER_FILE);
        expect(fs.existsSync(gitpodFilePath)).toBeTruthy();
        expect(fs.existsSync(devContainerFilePath)).toBeTruthy();
    });
});
describe('dev environment docker options', () => {
    test('given an image', () => {
        var _a, _b;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addDockerImage(dev_env_1.DevEnvironmentDockerImage.fromImage('jsii/superchain'));
        (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addDockerImage(dev_env_1.DevEnvironmentDockerImage.fromImage('jsii/uberchain'));
        // THEN
        const outdir = util_1.synthSnapshot(project);
        const gitpodSnapshot = outdir[GITPOD_FILE];
        expect(gitpodSnapshot).toContain('image: jsii/superchain');
        const devContainerSnapshot = outdir[DEVCONTAINER_FILE];
        expect(devContainerSnapshot).toStrictEqual({
            '//': file_1.FileBase.PROJEN_MARKER,
            'image': 'jsii/uberchain',
        });
    });
    test('given a docker file dep', () => {
        var _a, _b;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addDockerImage(dev_env_1.DevEnvironmentDockerImage.fromFile('.gitpod.Dockerfile'));
        (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addDockerImage(dev_env_1.DevEnvironmentDockerImage.fromFile('Dockerfile'));
        // THEN
        const outdir = util_1.synthSnapshot(project);
        const gitpodSnapshot = outdir[GITPOD_FILE];
        expect(gitpodSnapshot).toContain('image:');
        expect(gitpodSnapshot).toContain('file: .gitpod.Dockerfile');
        const devContainerSnapshot = outdir[DEVCONTAINER_FILE];
        expect(devContainerSnapshot).toStrictEqual({
            '//': file_1.FileBase.PROJEN_MARKER,
            'build': { dockerfile: 'Dockerfile' },
        });
    });
});
describe('dev environment tasks', () => {
    test('given custom task', () => {
        var _a, _b;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        const task = project.addTask('gitpod-test', { exec: 'text' });
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addTasks(task);
        (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addTasks(task);
        // THEN
        const outdir = util_1.synthSnapshot(project);
        const gitpodSnapshot = outdir[GITPOD_FILE];
        expect(gitpodSnapshot).toContain('command');
        expect(gitpodSnapshot).toContain('gitpod-test');
        const devContainerSnapshot = outdir[DEVCONTAINER_FILE];
        expect(devContainerSnapshot.postCreateCommand).toContain('gitpod-test');
    });
    test('given gitpod task options', () => {
        var _a;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        const task = project.addTask('gitpod-test', { exec: 'text' });
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addCustomTask({
            init: 'echo Initializing',
            openIn: gitpod_1.GitpodOpenIn.LEFT,
            openMode: gitpod_1.GitpodOpenMode.SPLIT_BOTTOM,
            command: `npx projen ${task.name}`,
        });
        // THEN
        const snapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
        expect(snapshot).toContain('init: echo Initializing');
        expect(snapshot).toContain('openIn: left');
        expect(snapshot).toContain('openMode: split-bottom');
        expect(snapshot).toContain('command: npx projen gitpod-test');
    });
});
test('dev environment ports', () => {
    var _a, _b;
    // GIVEN
    const project = new util_1.TestProject({
        gitpod: true,
        devContainer: true,
    });
    // WHEN
    (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addPorts('8080', '3000-3999');
    (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addPorts('8080', '3000');
    // THEN
    const outdir = util_1.synthSnapshot(project);
    const gitpodSnapshot = outdir[GITPOD_FILE];
    expect(gitpodSnapshot).toContain('port: "8080"');
    expect(gitpodSnapshot).toContain('port: 3000-3999');
    const devContainerSnapshot = outdir[DEVCONTAINER_FILE];
    expect(devContainerSnapshot).toStrictEqual({
        '//': file_1.FileBase.PROJEN_MARKER,
        'forwardPorts': ['8080', '3000'],
    });
});
test('gitpod prebuilds config', () => {
    var _a;
    // GIVEN
    const project = new util_1.TestProject({
        gitpod: true,
        devContainer: false,
    });
    // WHEN
    (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addPrebuilds({
        master: true,
        branches: true,
        pullRequestsFromForks: true,
        addBadge: false,
    });
    // THEN
    const gitpodSnapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
    expect(gitpodSnapshot).toContain('github');
    expect(gitpodSnapshot).toContain('prebuilds');
    expect(gitpodSnapshot).toContain('master');
    expect(gitpodSnapshot).toContain('branches');
    expect(gitpodSnapshot).toContain('pullRequestsFromForks');
    expect(gitpodSnapshot).toContain('addBadge');
});
test('dev environment vscode extensions', () => {
    var _a, _b;
    // GIVEN
    const project = new util_1.TestProject({
        gitpod: true,
        devContainer: true,
    });
    // WHEN
    (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addVscodeExtensions('dbaeumer.vscode-eslint@2.1.13:5sYlSD6wJi5s3xqD8hupUw==');
    (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addVscodeExtensions('dbaeumer.vscode-eslint');
    // THEN
    const outdir = util_1.synthSnapshot(project);
    const gitpodSnapshot = outdir[GITPOD_FILE];
    expect(gitpodSnapshot).toContain('extensions:');
    expect(gitpodSnapshot).toContain('dbaeumer.vscode-eslint@2.1.13:5sYlSD6wJi5s3xqD8hupUw==');
    const devContainerSnapshot = outdir[DEVCONTAINER_FILE];
    expect(devContainerSnapshot).toStrictEqual({
        '//': file_1.FileBase.PROJEN_MARKER,
        'extensions': ['dbaeumer.vscode-eslint'],
    });
});
//# sourceMappingURL=data:application/json;base64,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