"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const __1 = require("..");
const util_1 = require("./util");
describe('author', () => {
    test('authorEmail and authorAddress can be the same value', () => {
        const project = new TestJsiiProject({
            authorAddress: 'hello@hello.com',
            authorEmail: 'hello@hello.com',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'project',
            defaultReleaseBranch: 'master',
        });
        const pkgjson = util_1.synthSnapshot(project)['package.json'];
        expect(pkgjson.author).toStrictEqual({
            email: 'hello@hello.com',
            name: 'My Name',
            organization: false,
        });
    });
    test('authorUrl and authorAddress can be the same value', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'project',
            defaultReleaseBranch: 'master',
        });
        const pkgjson = util_1.synthSnapshot(project)['package.json'];
        expect(pkgjson.author).toStrictEqual({
            name: 'My Name',
            organization: false,
            url: 'https://foo.bar',
        });
    });
});
describe('maven repository options', () => {
    test('use maven central as repository', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            publishToMaven: {
                javaPackage: 'com.github.eladb.watchful',
                mavenGroupId: 'com.github.eladb',
                mavenArtifactId: 'cdk-watchful',
            },
            defaultReleaseBranch: 'master',
        });
        const outdir = util_1.synthSnapshot(project);
        expect(outdir['.projen/tasks.json'].tasks['publish:maven']).toStrictEqual({
            name: 'publish:maven',
            description: 'Publish this package to Maven Central',
            requiredEnv: [
                'MAVEN_GPG_PRIVATE_KEY',
                'MAVEN_GPG_PRIVATE_KEY_PASSPHRASE',
                'MAVEN_PASSWORD',
                'MAVEN_USERNAME',
                'MAVEN_STAGING_PROFILE_ID',
            ],
            steps: [{ exec: 'npx -p jsii-release@latest jsii-release-maven' }],
        });
        const workflow = outdir['.github/workflows/release.yml'];
        expect(workflow).toContain('run: npx -p jsii-release@latest jsii-release-maven');
        expect(workflow).toContain('MAVEN_USERNAME: ${{ secrets.MAVEN_USERNAME }}');
        expect(workflow).not.toContainEqual('MAVEN_SERVER_ID');
        expect(workflow).not.toContainEqual('MAVEN_REPOSITORY_URL');
        expect(workflow).not.toContain('packages: write');
    });
    test('use nexus repo new endpoint', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            publishToMaven: {
                javaPackage: 'com.github.eladb.watchful',
                mavenGroupId: 'com.github.eladb',
                mavenArtifactId: 'cdk-watchful',
                mavenEndpoint: 'https://s01.oss.sonatype.org',
            },
            defaultReleaseBranch: 'master',
        });
        const outdir = util_1.synthSnapshot(project);
        expect(outdir['.projen/tasks.json'].tasks['publish:maven']).toStrictEqual({
            name: 'publish:maven',
            description: 'Publish this package to Maven Central',
            env: {
                MAVEN_ENDPOINT: 'https://s01.oss.sonatype.org',
            },
            requiredEnv: [
                'MAVEN_GPG_PRIVATE_KEY',
                'MAVEN_GPG_PRIVATE_KEY_PASSPHRASE',
                'MAVEN_PASSWORD',
                'MAVEN_USERNAME',
                'MAVEN_STAGING_PROFILE_ID',
            ],
            steps: [{ exec: 'npx -p jsii-release@latest jsii-release-maven' }],
        });
        const workflow = outdir['.github/workflows/release.yml'];
        expect(workflow).toContain('run: npx -p jsii-release@latest jsii-release-maven');
        expect(workflow).toContain('MAVEN_ENDPOINT: https://s01.oss.sonatype.org');
        expect(workflow).toContain('MAVEN_USERNAME: ${{ secrets.MAVEN_USERNAME }}');
        expect(workflow).not.toContainEqual('MAVEN_SERVER_ID');
        expect(workflow).not.toContainEqual('MAVEN_REPOSITORY_URL');
        expect(workflow).not.toContain('packages: write');
    });
    test('use github as repository', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            publishToMaven: {
                javaPackage: 'com.github.eladb.watchful',
                mavenGroupId: 'com.github.eladb',
                mavenArtifactId: 'cdk-watchful',
                mavenRepositoryUrl: 'https://maven.pkg.github.com/eladb',
            },
            defaultReleaseBranch: 'master',
        });
        const outdir = util_1.synthSnapshot(project);
        expect(outdir['.projen/tasks.json'].tasks['publish:maven']).toStrictEqual({
            name: 'publish:maven',
            description: 'Publish this package to Maven Central',
            env: {
                MAVEN_SERVER_ID: 'github',
                MAVEN_REPOSITORY_URL: 'https://maven.pkg.github.com/eladb',
            },
            requiredEnv: [
                'MAVEN_PASSWORD',
                'MAVEN_USERNAME',
            ],
            steps: [{ exec: 'npx -p jsii-release@latest jsii-release-maven' }],
        });
        const workflow = outdir['.github/workflows/release.yml'];
        expect(workflow).toContain('MAVEN_SERVER_ID: github');
        expect(workflow).toContain('MAVEN_REPOSITORY_URL: https://maven.pkg.github.com/eladb');
        expect(workflow).toContain('MAVEN_USERNAME: ${{ github.actor }}');
        expect(workflow).toContain('MAVEN_PASSWORD: ${{ secrets.GITHUB_TOKEN }}');
        expect(workflow).toContain('packages: write');
    });
    test('using github as repository with incorrect server id should throw', () => {
        expect(() => new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            publishToMaven: {
                javaPackage: 'com.github.eladb.watchful',
                mavenGroupId: 'com.github.eladb',
                mavenArtifactId: 'cdk-watchful',
                mavenServerId: 'something-else',
                mavenRepositoryUrl: 'https://maven.pkg.github.com/eladb',
            },
            defaultReleaseBranch: 'master',
        })).toThrow('publishing to GitHub Packages requires the "mavenServerId" to be "github"');
    });
});
describe('publish to go', () => {
    test('defaults', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            publishToGo: {
                moduleName: 'github.com/foo/bar',
            },
            defaultReleaseBranch: 'master',
        });
        const output = util_1.synthSnapshot(project);
        const targets = output['package.json'].jsii.targets;
        expect(targets).toStrictEqual({
            go: {
                moduleName: 'github.com/foo/bar',
            },
        });
        expect(output['.github/workflows/release.yml']).toMatchSnapshot();
    });
    test('release to npm undefined', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            defaultReleaseBranch: 'main',
        });
        const output = util_1.synthSnapshot(project);
        expect(output['.github/workflows/release.yml']).toContain('release_npm');
    });
    test('release to npm true', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            defaultReleaseBranch: 'main',
            releaseToNpm: true,
        });
        const output = util_1.synthSnapshot(project);
        expect(output['.github/workflows/release.yml']).toContain('release_npm');
    });
    test('release to npm false', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            defaultReleaseBranch: 'main',
            releaseToNpm: false,
        });
        const output = util_1.synthSnapshot(project);
        expect(output['.github/workflows/release.yml']).not.toContain('release_npm');
    });
    test('customizations', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            publishToGo: {
                moduleName: 'github.com/foo/bar',
                gitBranch: 'custom-branch',
                gitCommitMessage: 'custom commit message',
                gitUserEmail: 'custom@email.com',
                gitUserName: 'custom user',
                githubRepo: 'github.com/foo/bar',
                githubTokenSecret: 'CUSTOM_SECRET',
            },
            defaultReleaseBranch: 'master',
            excludeTypescript: ['src/**/test/*.ts', 'src/**/__tests__/*.ts'],
        });
        const output = util_1.synthSnapshot(project);
        expect(output['package.json'].jsii.targets.go).toStrictEqual({ moduleName: 'github.com/foo/bar' });
        expect(output['.github/workflows/release.yml']).toMatchSnapshot();
        expect(output['package.json'].jsii.excludeTypescript).toStrictEqual(['src/**/test/*.ts', 'src/**/__tests__/*.ts']);
    });
});
test('docgen: true should just work', () => {
    const project = new TestJsiiProject({
        author: 'My name',
        name: 'testproject',
        authorAddress: 'https://foo.bar',
        defaultReleaseBranch: 'main',
        repositoryUrl: 'https://github.com/foo/bar.git',
        docgen: true,
    });
    const output = util_1.synthSnapshot(project);
    expect(output['.projen/tasks.json'].tasks.docgen.steps[0].exec).toStrictEqual('jsii-docgen');
});
class TestJsiiProject extends __1.JsiiProject {
    constructor(options) {
        super({
            outdir: util_1.mkdtemp(),
            logging: {
                level: __1.LogLevel.OFF,
            },
            ...options,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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