"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// tests for `projen new`: we run `projen new` for each supported project type
// and compare against a golden snapshot.
const child_process_1 = require("child_process");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const inventory = require("../inventory");
const util_1 = require("../util");
const util_2 = require("./util");
for (const type of inventory.discover()) {
    test(`projen new ${type.pjid}`, () => {
        withProjectDir(projectdir => {
            // execute `projen new PJID --no-synth` in the project directory
            util_2.execProjenCLI(projectdir, ['new', '--no-synth', type.pjid]);
            // compare generated .projenrc.js to the snapshot
            const actual = util_2.directorySnapshot(projectdir, {
                excludeGlobs: [
                    '.git/**',
                ],
            });
            expect(actual).toMatchSnapshot();
        });
    });
}
test('post-synthesis option enabled', () => {
    const project = new util_2.TestProject();
    expect(util_2.synthSnapshotWithPost(project)['.postsynth']).toContain('postsynth');
});
test('post-synthesis option disabled', () => {
    const project = new util_2.TestProject();
    expect(util_2.synthSnapshot(project)['.postsynth']).toBeUndefined();
});
test('projen new --from external', () => {
    withProjectDir(projectdir => {
        // execute `projen new --from cdk-appsync-project` in the project directory
        util_2.execProjenCLI(projectdir, ['new', '--from', 'cdk-appsync-project@1.1.2', '--no-post']);
        // patch the projen version in package.json to match the current version
        // otherwise, every bump would need to update these snapshots.
        util_2.sanitizeOutput(projectdir);
        // compare generated .projenrc.js to the snapshot
        const actual = util_2.directorySnapshot(projectdir, {
            excludeGlobs: [
                '.git/**',
                '.github/**',
                'node_modules/**',
                'yarn.lock',
            ],
        });
        expect(actual).toMatchSnapshot();
        expect(actual['schema.graphql']).toBeDefined();
    });
});
test('options are not overwritten when creating from external project types', () => {
    withProjectDir(projectdir => {
        // execute `projen new --from cdk-appsync-project` in the project directory
        util_2.execProjenCLI(projectdir, ['new', '--from', 'cdk-appsync-project@1.1.2', '--no-synth', '--cdk-version', '1.63.0']);
        // compare generated .projenrc.js to the snapshot
        const actual = util_2.directorySnapshot(projectdir, {
            excludeGlobs: [
                '.git/**',
                '.github/**',
                'node_modules/**',
                'yarn.lock',
            ],
        });
        expect(actual['.projenrc.js']).toContain('cdkVersion: \'1.63.0\'');
    });
});
test('can choose from one of multiple external project types', () => {
    withProjectDir(projectdir => {
        // execute `projen new --from cdk-appsync-project` in the project directory
        util_2.execProjenCLI(projectdir, ['new', '--from', '@taimos/projen@0.0.121', 'taimos-ts-lib', '--no-post']);
        // patch the projen version in package.json to match the current version
        // otherwise, every bump would need to update these snapshots.
        util_2.sanitizeOutput(projectdir);
        // compare generated .projenrc.js to the snapshot
        const actual = util_2.directorySnapshot(projectdir, {
            excludeGlobs: [
                '.git/**',
                '.github/**',
                'node_modules/**',
                'yarn.lock',
            ],
        });
        expect(actual).toMatchSnapshot();
    });
});
test('projen new --no-comments', () => {
    withProjectDir(projectdir => {
        util_2.execProjenCLI(projectdir, ['new', 'node', '--no-comments', '--no-synth']);
        const projenrc = util_2.directorySnapshot(projectdir)['.projenrc.js'];
        expect(projenrc).toBeDefined();
        expect(projenrc).not.toMatch('//');
    });
});
test('creating node project with enum-typed CLI arg', () => {
    withProjectDir(projectdir => {
        util_2.execProjenCLI(projectdir, ['new', 'node', '--package-manager', 'npm', '--no-synth']);
        const projenrc = util_2.directorySnapshot(projectdir)['.projenrc.js'];
        expect(projenrc).toMatchSnapshot();
    });
});
test('creating python project with enum-typed CLI arg', () => {
    withProjectDir(projectdir => {
        util_2.execProjenCLI(projectdir, ['new', 'python', '--project-type', 'lib', '--projenrc-python', '--no-synth']);
        const projenrc = util_2.directorySnapshot(projectdir)['.projenrc.py'];
        expect(projenrc).toMatchSnapshot();
    });
});
test('creating java project with enum-typed CLI arg', () => {
    withProjectDir(projectdir => {
        util_2.execProjenCLI(projectdir, ['new', 'java', '--project-type', 'lib', '--projenrc-java', '--no-synth']);
        const projenrc = util_2.directorySnapshot(projectdir)['src/test/java/projenrc.java'];
        expect(projenrc).toMatchSnapshot();
    });
});
describe('git', () => {
    test('--git (default) will initialize a git repo and create a commit', () => {
        withProjectDir(projectdir => {
            util_2.execProjenCLI(projectdir, ['new', 'project']);
            expect(util_1.execCapture('git log', { cwd: projectdir }).toString('utf8').includes('chore: project created with projen')).toBeTruthy();
        });
    });
    test('--no-git will not create a git repo', () => {
        withProjectDir(projectdir => {
            util_2.execProjenCLI(projectdir, ['new', 'project', '--no-git']);
            expect(fs_extra_1.pathExistsSync(path_1.join(projectdir, '.git'))).toBeFalsy();
        }, { git: false });
    });
});
function withProjectDir(code, options = {}) {
    var _a;
    const outdir = util_2.mkdtemp();
    try {
        // create project under "my-project" so that basedir is deterministic
        const projectdir = path_1.join(outdir, 'my-project');
        fs_extra_1.mkdirSync(projectdir);
        const shell = (command) => child_process_1.execSync(command, { cwd: projectdir });
        if ((_a = options.git) !== null && _a !== void 0 ? _a : true) {
            shell('git init');
            shell('git remote add origin git@boom.com:foo/bar.git');
            shell('git config user.name "My User Name"');
            shell('git config user.email "my@user.email.com"');
        }
        else if (process.env.CI) {
            // if "git" is set to "false", we still want to make sure global user is defined
            // (relevant in CI context)
            shell('git config user.name || git config --global user.name "My User Name"');
            shell('git config user.email || git config --global user.email "my@user.email.com"');
        }
        code(projectdir);
    }
    finally {
        fs_extra_1.removeSync(outdir);
    }
}
//# sourceMappingURL=data:application/json;base64,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