"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const yaml = require("yaml");
const __1 = require("..");
const deps_1 = require("../deps");
const workflows_model_1 = require("../github/workflows-model");
const logging = require("../logging");
const node_package_1 = require("../node-package");
const node_project_1 = require("../node-project");
const tasks_1 = require("../tasks");
const util_1 = require("./util");
logging.disable();
test('license file is added by default', () => {
    // WHEN
    const project = new TestNodeProject();
    // THEN
    expect(util_1.synthSnapshot(project).LICENSE).toContain('Apache License');
});
test('license file is not added if licensed is false', () => {
    // WHEN
    const project = new TestNodeProject({
        licensed: false,
    });
    // THEN
    const snapshot = util_1.synthSnapshot(project);
    expect(snapshot.LICENSE).toBeUndefined();
    expect(snapshot['.gitignore']).not.toContain('LICENSE');
    expect(snapshot['package.json'].license).toEqual('UNLICENSED');
});
describe('deps', () => {
    test('runtime deps', () => {
        // GIVEN
        const project = new TestNodeProject({
            deps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.RUNTIME);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.dependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        expect(pkgjson.peerDependencies).toStrictEqual({});
    });
    test('dev dependencies', () => {
        // GIVEN
        const project = new TestNodeProject({
            devDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addDevDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.TEST);
        project.deps.addDependency('eee@^1', deps_1.DependencyType.DEVENV);
        project.deps.addDependency('fff@^2', deps_1.DependencyType.BUILD);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.devDependencies.aaa).toStrictEqual('^1.2.3');
        expect(pkgjson.devDependencies.bbb).toStrictEqual('~4.5.6');
        expect(pkgjson.devDependencies.ccc).toStrictEqual('*');
        expect(pkgjson.devDependencies.ddd).toStrictEqual('*');
        expect(pkgjson.devDependencies.eee).toStrictEqual('^1');
        expect(pkgjson.devDependencies.fff).toStrictEqual('^2');
        expect(pkgjson.peerDependencies).toStrictEqual({});
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('peerDependencies', () => {
        // GIVEN
        const project = new TestNodeProject({
            peerDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addPeerDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.PEER);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.peerDependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        // devDependencies are added with pinned versions
        expect(pkgjson.devDependencies.aaa).toStrictEqual('1.2.3');
        expect(pkgjson.devDependencies.bbb).toStrictEqual('4.5.6');
        expect(pkgjson.devDependencies.ccc).toStrictEqual('*');
        expect(pkgjson.devDependencies.ddd).toStrictEqual('*');
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('peerDependencies without pinnedDevDep', () => {
        // GIVEN
        const project = new TestNodeProject({
            peerDependencyOptions: {
                pinnedDevDependency: false,
            },
            peerDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addPeerDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.PEER);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.peerDependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        // sanitize
        ['npm-check-updates', 'jest', 'jest-junit', 'projen', 'standard-version'].forEach(d => delete pkgjson.devDependencies[d]);
        expect(pkgjson.devDependencies).toStrictEqual({});
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('bundled deps are automatically added as normal deps', () => {
        // GIVEN
        const project = new TestNodeProject({
            bundledDeps: ['hey@2.1.1'],
        });
        // WHEN
        project.addBundledDeps('foo@^1.2.3');
        project.deps.addDependency('bar@~1.0.0', deps_1.DependencyType.BUNDLED);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.dependencies).toStrictEqual({
            hey: '2.1.1',
            foo: '^1.2.3',
            bar: '~1.0.0',
        });
        expect(pkgjson.bundledDependencies).toStrictEqual([
            'bar',
            'foo',
            'hey',
        ]);
    });
});
test('throw when \'autoApproveProjenUpgrades\' is used with \'projenUpgradeAutoMerge\'', () => {
    expect(() => {
        new TestNodeProject({ autoApproveProjenUpgrades: true, projenUpgradeAutoMerge: true });
    }).toThrow("'projenUpgradeAutoMerge' cannot be configured together with 'autoApproveProjenUpgrades'");
});
describe('deps upgrade', () => {
    test('throws when trying to auto approve projen but auto approve is not defined', () => {
        const message = 'Autoamtic approval of projen upgrades requires configuring `autoApproveOptions`';
        expect(() => { new TestNodeProject({ autoApproveProjenUpgrades: true }); }).toThrow(message);
        expect(() => { new TestNodeProject({ projenUpgradeAutoMerge: true }); }).toThrow(message);
    });
    test('throws when trying to auto approve deps but auto approve is not defined', () => {
        expect(() => {
            new TestNodeProject({ autoApproveUpgrades: true });
        }).toThrow('Autoamtic approval of dependencies upgrades requires configuring `autoApproveOptions`');
    });
    test('workflow can be auto approved', () => {
        var _a;
        const project = new TestNodeProject({
            autoApproveOptions: {
                allowedUsernames: ['dummy'],
                secret: 'dummy',
            },
            autoApproveUpgrades: true,
        });
        const snapshot = yaml.parse(util_1.synthSnapshot(project)['.github/workflows/upgrade-dependencies.yml']);
        expect(snapshot.jobs.pr.steps[3].with.labels).toStrictEqual((_a = project.autoApprove) === null || _a === void 0 ? void 0 : _a.label);
    });
    test('dependabot can be auto approved', () => {
        const project = new TestNodeProject({
            dependabot: true,
            autoApproveOptions: {
                allowedUsernames: ['dummy'],
                secret: 'dummy',
            },
            autoApproveUpgrades: true,
        });
        const snapshot = yaml.parse(util_1.synthSnapshot(project)['.github/dependabot.yml']);
        expect(snapshot.updates[0].labels).toStrictEqual(['auto-approve']);
    });
    test('default - with projen secret', () => {
        const project = new TestNodeProject({ projenUpgradeSecret: 'PROJEN_GITHUB_TOKEN' });
        const snapshot = util_1.synthSnapshot(project);
        expect(snapshot['.github/workflows/upgrade-dependencies.yml']).toBeDefined();
        expect(snapshot['.github/workflows/upgrade-projen.yml']).toBeUndefined();
        // make sure yarn upgrade all deps, including projen.
        const tasks = snapshot[tasks_1.Tasks.MANIFEST_FILE].tasks;
        expect(tasks['upgrade-dependencies'].steps[2].exec).toStrictEqual('yarn upgrade');
    });
    test('default - no projen secret', () => {
        const project = new TestNodeProject();
        const snapshot = util_1.synthSnapshot(project);
        expect(snapshot['.github/workflows/upgrade-dependencies.yml']).toBeDefined();
        expect(snapshot['.github/workflows/upgrade-projen.yml']).toBeUndefined();
    });
    test('dependabot - with projen secret', () => {
        const project = new TestNodeProject({
            depsUpgrade: node_project_1.DependenciesUpgradeMechanism.dependabot(),
            projenUpgradeSecret: 'PROJEN_GITHUB_TOKEN',
        });
        const snapshot = util_1.synthSnapshot(project);
        expect(snapshot['.github/dependabot.yml']).toBeDefined();
        expect(snapshot['.github/workflows/upgrade-projen.yml']).toBeDefined();
    });
    test('dependabot - no projen secret', () => {
        const project = new TestNodeProject({
            depsUpgrade: node_project_1.DependenciesUpgradeMechanism.dependabot(),
        });
        const snapshot = util_1.synthSnapshot(project);
        expect(snapshot['.github/dependabot.yml']).toBeDefined();
        expect(snapshot['.github/workflows/upgrade-projen.yml']).toBeUndefined();
    });
    test('github actions - with projen secret', () => {
        const project = new TestNodeProject({
            depsUpgrade: node_project_1.DependenciesUpgradeMechanism.githubWorkflow(),
            projenUpgradeSecret: 'PROJEN_GITHUB_TOKEN',
        });
        const snapshot = util_1.synthSnapshot(project);
        expect(snapshot['.github/workflows/upgrade-dependencies.yml']).toBeDefined();
        expect(snapshot['.github/workflows/upgrade-projen.yml']).toBeDefined();
    });
    test('github actions - no projen secret', () => {
        const project = new TestNodeProject({
            depsUpgrade: node_project_1.DependenciesUpgradeMechanism.githubWorkflow(),
        });
        const snapshot = util_1.synthSnapshot(project);
        expect(snapshot['.github/workflows/upgrade-dependencies.yml']).toBeDefined();
        expect(snapshot['.github/workflows/upgrade-projen.yml']).toBeUndefined();
    });
    test('throws when depracated dependabot is configued with dependenciesUpgrade', () => {
        expect(() => {
            new TestNodeProject({ dependabot: true, depsUpgrade: node_project_1.DependenciesUpgradeMechanism.githubWorkflow() });
        }).toThrow("'dependabot' cannot be configured together with 'depsUpgrade'");
    });
});
describe('npm publishing options', () => {
    test('defaults', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'my-package',
        });
        // THEN
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.PUBLIC);
        expect(npm.npmDistTag).toStrictEqual('latest');
        expect(npm.npmRegistry).toStrictEqual('registry.npmjs.org');
        expect(npm.npmRegistryUrl).toStrictEqual('https://registry.npmjs.org/');
        expect(npm.npmTokenSecret).toStrictEqual('NPM_TOKEN');
        // since these are all defaults, publishConfig is not defined.
        expect(util_1.synthSnapshot(project)['package.json'].publishConfig).toBeUndefined();
    });
    test('scoped packages default to RESTRICTED access', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
        });
        // THEN
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.RESTRICTED);
        // since these are all defaults, publishConfig is not defined.
        expect(packageJson(project).publishConfig).toBeUndefined();
    });
    test('non-scoped package cannot be RESTRICTED', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // THEN
        expect(() => new node_package_1.NodePackage(project, {
            packageName: 'my-package',
            npmAccess: node_package_1.NpmAccess.RESTRICTED,
        })).toThrow(/"npmAccess" cannot be RESTRICTED for non-scoped npm package/);
    });
    test('custom settings', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
            npmDistTag: 'next',
            npmRegistryUrl: 'https://foo.bar',
            npmAccess: node_package_1.NpmAccess.PUBLIC,
            npmTokenSecret: 'GITHUB_TOKEN',
        });
        // THEN
        expect(npm.npmDistTag).toStrictEqual('next');
        expect(npm.npmRegistry).toStrictEqual('foo.bar');
        expect(npm.npmRegistryUrl).toStrictEqual('https://foo.bar/');
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.PUBLIC);
        expect(npm.npmTokenSecret).toStrictEqual('GITHUB_TOKEN');
        expect(packageJson(project).publishConfig).toStrictEqual({
            access: 'public',
            registry: 'https://foo.bar/',
            tag: 'next',
        });
    });
    test('deprecated npmRegistry can be used instead of npmRegistryUrl and then https:// is assumed', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
            npmRegistry: 'foo.bar.com',
        });
        // THEN
        expect(npm.npmRegistry).toStrictEqual('foo.bar.com');
        expect(npm.npmRegistryUrl).toStrictEqual('https://foo.bar.com/');
        expect(packageJson(project).publishConfig).toStrictEqual({
            registry: 'https://foo.bar.com/',
        });
    });
});
test('extend github release workflow', () => {
    var _a;
    const project = new TestNodeProject();
    (_a = project.release) === null || _a === void 0 ? void 0 : _a.addJobs({
        publish_docker_hub: {
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
            },
            runsOn: 'ubuntu-latest',
            env: {
                CI: 'true',
            },
            steps: [
                {
                    name: 'Check out the repo',
                    uses: 'actions/checkout@v2',
                },
                {
                    name: 'Push to Docker Hub',
                    uses: 'docker/build-push-action@v1',
                    with: {
                        username: '${{ secrets.DOCKER_USERNAME }}',
                        password: '${{ secrets.DOCKER_PASSWORD }}',
                        repository: 'projen/projen-docker',
                        tag_with_ref: 'true',
                    },
                },
            ],
        },
    });
    const workflow = util_1.synthSnapshot(project)['.github/workflows/release.yml'];
    expect(workflow).toContain('publish_docker_hub:\n    runs-on: ubuntu-latest\n');
    expect(workflow).toContain('username: ${{ secrets.DOCKER_USERNAME }}\n          password: ${{ secrets.DOCKER_PASSWORD }}');
});
describe('scripts', () => {
    test('addTask and setScript', () => {
        const p = new TestNodeProject();
        p.addTask('chortle', { exec: 'echo "frabjous day!"' });
        p.setScript('slithy-toves', 'gyre && gimble');
        const pkg = packageJson(p);
        expect(pkg.scripts).toHaveProperty('chortle');
        expect(pkg.scripts).toHaveProperty('slithy-toves');
    });
    test('removeScript will remove tasks and scripts', () => {
        const p = new TestNodeProject();
        p.addTask('chortle', { exec: 'echo "frabjous day!"' });
        p.setScript('slithy-toves', 'gyre && gimble');
        p.removeScript('chortle');
        p.removeScript('slithy-toves');
        const pkg = packageJson(p);
        expect(pkg.scripts).not.toHaveProperty('chortle');
        expect(pkg.scripts).not.toHaveProperty('slithy-toves');
    });
});
test('mutableBuild will push changes to PR branches', () => {
    // WHEN
    const project = new TestNodeProject({
        mutableBuild: true,
    });
    // THEN
    const workflowYaml = util_1.synthSnapshot(project)['.github/workflows/build.yml'];
    const workflow = yaml.parse(workflowYaml);
    expect(workflow.jobs.build.steps).toMatchSnapshot();
});
test('projenDuringBuild can be used to disable "projen synth" during build', () => {
    const enabled = new TestNodeProject({
        projenDuringBuild: true,
    });
    const disabled = new TestNodeProject({
        projenDuringBuild: false,
    });
    const buildTaskEnabled = util_1.synthSnapshot(enabled)['.projen/tasks.json'].tasks.build;
    const buildTaskDisabled = util_1.synthSnapshot(disabled)['.projen/tasks.json'].tasks.build;
    expect(buildTaskEnabled.steps[0].exec).toEqual('npx projen');
    expect(buildTaskDisabled.steps).toBeUndefined();
});
test('projen synth is only executed for subprojects', () => {
    // GIVEN
    const root = new TestNodeProject();
    // WHEN
    new TestNodeProject({ parent: root, outdir: 'child' });
    // THEN
    const snapshot = util_1.synthSnapshot(root);
    const rootBuildTask = snapshot['.projen/tasks.json'].tasks.build;
    const childBuildTask = snapshot['child/.projen/tasks.json'].tasks.build;
    expect(rootBuildTask).toStrictEqual({
        description: 'Full release build (test+compile)',
        name: 'build',
        steps: [{ exec: 'npx projen' }],
    });
    expect(childBuildTask).toStrictEqual({
        description: 'Full release build (test+compile)',
        name: 'build',
    });
});
test('enable anti-tamper', () => {
    // WHEN
    const project = new TestNodeProject({
        packageManager: node_package_1.NodePackageManager.NPM,
        releaseToNpm: true,
        mutableBuild: false,
        antitamper: true,
    });
    // THEN
    const workflowYaml = util_1.synthSnapshot(project)['.github/workflows/build.yml'];
    const workflow = yaml.parse(workflowYaml);
    expect(workflow.jobs.build.steps).toMatchSnapshot();
    expect(workflow.jobs.build.steps).toEqual(expect.arrayContaining([
        expect.objectContaining({
            name: 'Anti-tamper check',
        }),
    ]));
});
test('enabling dependabot does not overturn mergify: false', () => {
    // WHEN
    const project = new TestNodeProject({
        dependabot: true,
        mergify: false,
    });
    // THEN
    const snapshot = util_1.synthSnapshot(project);
    // Note: brackets important, they prevent "." in filenames to be interpreted
    //       as JSON object path delimiters.
    expect(snapshot).not.toHaveProperty(['.mergify.yml']);
    expect(snapshot).toHaveProperty(['.github/dependabot.yml']);
});
test('github: false disables github integration', () => {
    // WHEN
    const project = new TestNodeProject({
        github: false,
    });
    // THEN
    const output = util_1.synthSnapshot(project);
    expect(Object.keys(output).filter(p => p.startsWith('.github/'))).toStrictEqual([]);
});
function packageJson(project) {
    return util_1.synthSnapshot(project)['package.json'];
}
class TestNodeProject extends __1.NodeProject {
    constructor(options = {}) {
        super({
            outdir: util_1.mkdtemp(),
            name: 'test-node-project',
            logging: {
                level: __1.LogLevel.OFF,
            },
            defaultReleaseBranch: 'master',
            ...options,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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