"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const YAML = require("yaml");
const workflows_model_1 = require("../../github/workflows-model");
const release_1 = require("../../release");
const util_1 = require("../util");
test('minimal', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('build');
    // WHEN
    new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: 'main',
    });
    const outdir = util_1.synthSnapshot(project);
    expect(outdir).toMatchSnapshot();
});
test('with major version filter', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('build');
    // WHEN
    new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: '10.x',
        majorVersion: 10,
        releaseWorkflowName: 'release',
    });
    // THEN
    const outdir = util_1.synthSnapshot(project);
    expect(outdir['.github/workflows/release.yml']).toBeDefined();
    expect(outdir).toMatchSnapshot();
});
test('addBranch() can be used for additional release branches', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('build');
    const release = new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: 'main',
        majorVersion: 1,
    });
    // WHEN
    release.addBranch('2.x', { majorVersion: 2 });
    release.addBranch('10.x', { majorVersion: 10 });
    // THEN
    const outdir = util_1.synthSnapshot(project);
    expect(outdir['.github/workflows/release.yml']).toBeDefined();
    expect(outdir['.github/workflows/release-2.x.yml']).toBeDefined();
    expect(outdir['.github/workflows/release-10.x.yml']).toBeDefined();
    expect(outdir).toMatchSnapshot();
});
test('if multiple branches are defined, the default branch requires a "majorVersion"', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('build');
    const release = new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: 'main',
    });
    // WHEN
    const addBranch = () => release.addBranch('2.x', { majorVersion: 2 });
    // THEN
    expect(addBranch).toThrow(/you must specify \"majorVersion\" for the default branch when adding multiple release branches/);
});
test('publisher (defaults)', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('build');
    const release = new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: 'main',
    });
    // WHEN
    release.publisher.publishToGo();
    release.publisher.publishToMaven();
    release.publisher.publishToNpm();
    release.publisher.publishToNuget();
    release.publisher.publishToPyPi();
    // THEN
    const outdir = util_1.synthSnapshot(project);
    expect(outdir['.github/workflows/release.yml']).toMatchSnapshot();
    expect(outdir['.projen/tasks.json']).toMatchSnapshot();
});
test('publishers are added as jobs to all release workflows', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('build');
    const release = new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: 'main',
        majorVersion: 1,
    });
    // WHEN
    release.addBranch('2.x', { majorVersion: 2 });
    release.publisher.publishToNpm();
    // THEN
    const outdir = util_1.synthSnapshot(project);
    const wf1 = YAML.parse(outdir['.github/workflows/release.yml']);
    expect(wf1).toMatchObject({
        on: { push: { branches: ['main'] } },
        jobs: {
            release: {
                steps: expect.any(Array),
            },
            release_npm: {},
        },
    });
    expect(wf1.jobs.release.steps.length).toBe(5);
    const wf2 = YAML.parse(outdir['.github/workflows/release-2.x.yml']);
    expect(wf2).toMatchObject({
        on: { push: { branches: ['2.x'] } },
        jobs: {
            release: {
                steps: expect.any(Array),
            },
            release_npm: {},
        },
    });
    expect(wf2.jobs.release.steps.length).toBe(5);
});
test('addJobs() can be used to add arbitrary jobs to the release workflows', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('build');
    const release = new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: 'main',
        majorVersion: 0,
    });
    release.addBranch('foo', { majorVersion: 4, workflowName: 'foo-workflow' });
    release.publisher.publishToPyPi();
    // WHEN
    release.addJobs({
        random_job: {
            runsOn: 'foo',
            permissions: {
                actions: workflows_model_1.JobPermission.NONE,
            },
            steps: [],
        },
    });
    // THEN
    const outdir = util_1.synthSnapshot(project);
    expect(outdir).toMatchSnapshot();
});
test('majorVersion can be 0', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('build');
    // WHEN
    new release_1.Release(project, {
        task: task,
        versionFile: 'goo.json',
        branch: 'main',
        majorVersion: 0,
    });
    // THEN
    const outdir = util_1.synthSnapshot(project);
    expect(outdir['.github/workflows/release.yml']).toMatchSnapshot();
    expect(outdir['.projen/tasks.json']).toMatchSnapshot();
});
test('prerelease can be specified per branch', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('build');
    // WHEN
    const release = new release_1.Release(project, {
        task: task,
        versionFile: 'goo.json',
        branch: 'main',
        majorVersion: 0,
    });
    release.addBranch('10.x', { majorVersion: 10, prerelease: 'pre' });
    // THEN
    const outdir = util_1.synthSnapshot(project);
    expect(outdir['.github/workflows/release.yml']).toMatchSnapshot();
    expect(outdir['.github/workflows/release.10.x.yml']).toMatchSnapshot();
    expect(outdir['.projen/tasks.json']).toMatchSnapshot();
});
test('releaseBranches can be use to define additional branches', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('build');
    // WHEN
    new release_1.Release(project, {
        task: task,
        versionFile: 'goo.json',
        branch: 'main',
        majorVersion: 1,
        releaseBranches: {
            '3.x': { majorVersion: 3 },
            'next': { majorVersion: 4, prerelease: 'pre' },
        },
    });
    const outdir = util_1.synthSnapshot(project);
    expect(outdir).toMatchSnapshot();
});
test('releaseBranches as an array throws an error since type was changed', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('build');
    // WHEN
    expect(() => new release_1.Release(project, {
        task: task,
        versionFile: 'goo.json',
        branch: 'main',
        majorVersion: 1,
        releaseBranches: ['10.x', '2.x'],
    })).toThrow(/\"releaseBranches\" is no longer an array. See type annotations/);
});
test('github packages are supported by npm', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('build');
    const release = new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: 'main',
    });
    // WHEN
    release.publisher.publishToNpm({
        registry: 'npm.pkg.github.com',
    });
    // THEN
    const outdir = util_1.synthSnapshot(project);
    expect(outdir).toMatchSnapshot();
});
//# sourceMappingURL=data:application/json;base64,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