"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const _resolve_1 = require("../_resolve");
test('null', () => {
    expect(_resolve_1.resolve(null)).toStrictEqual(null);
});
test('undefined', () => {
    expect(_resolve_1.resolve(undefined)).toStrictEqual(undefined);
});
test('string', () => {
    expect(_resolve_1.resolve('')).toStrictEqual('');
    expect(_resolve_1.resolve('hello')).toStrictEqual('hello');
    expect(_resolve_1.resolve('0')).toStrictEqual('0');
    expect(_resolve_1.resolve('hello world')).toStrictEqual('hello world');
    expect(_resolve_1.resolve('  ')).toStrictEqual('  ');
});
test('number', () => {
    expect(_resolve_1.resolve(0)).toStrictEqual(0);
    expect(_resolve_1.resolve(10)).toStrictEqual(10);
    expect(_resolve_1.resolve(-102)).toStrictEqual(-102);
});
test('boolean', () => {
    expect(_resolve_1.resolve(true)).toStrictEqual(true);
    expect(_resolve_1.resolve(false)).toStrictEqual(false);
});
test('array', () => {
    expect(_resolve_1.resolve([])).toStrictEqual([]);
    expect(_resolve_1.resolve([1, 2, 3])).toStrictEqual([1, 2, 3]);
});
test('object', () => {
    expect(_resolve_1.resolve(Object.create(null))).toStrictEqual({});
    expect(_resolve_1.resolve({})).toStrictEqual({});
    expect(_resolve_1.resolve({ foo: 123, hello: [1, 2, { bar: 3 }] })).toStrictEqual({
        foo: 123,
        hello: [1, 2, { bar: 3 }],
    });
});
test('functions are resolved', () => {
    expect(_resolve_1.resolve(() => 123)).toStrictEqual(123);
    expect(_resolve_1.resolve(() => 'hello')).toStrictEqual('hello');
    expect(_resolve_1.resolve(() => undefined)).toStrictEqual(undefined);
});
test('resolvable objects are resolved', () => {
    const resolvedContent = 'Content';
    const resolvable = { toJSON: () => resolvedContent };
    expect(_resolve_1.resolve({ foo: resolvable })).toStrictEqual({ foo: resolvedContent });
});
test('recursive resolve', () => {
    expect(_resolve_1.resolve(() => [1, 2, () => 30])).toStrictEqual([1, 2, 30]);
    expect(_resolve_1.resolve(() => ({ foo: 123, bar: () => 'bar' }))).toStrictEqual({ foo: 123, bar: 'bar' });
    expect(_resolve_1.resolve(() => ({ foo: 123, bar: () => ['bar', 'baz', { hello: () => 'world' }] }))).toStrictEqual({
        foo: 123,
        bar: ['bar', 'baz', { hello: 'world' }],
    });
});
test('context is passed to functions', () => {
    expect(_resolve_1.resolve((x) => x + 1, { args: [10] })).toStrictEqual(11);
    const calculator = {
        add: (a, b) => a + b,
        sub: (a, b) => a - b,
        mul: (a, b) => a * b,
        div: (a, b) => a / b,
        fun: (a1, b1, text) => [
            `this is ${text}`,
            (a2) => a1 * a2,
            (_, b2) => b1 + b2,
            {
                another: {
                    level: (a3, b3, text3) => `hello ${text3} ${b3} ${a3}`,
                },
            },
        ],
    };
    expect(_resolve_1.resolve(calculator, { args: [10, 2, 'fun'] })).toStrictEqual({
        add: 12,
        sub: 8,
        mul: 20,
        div: 5,
        fun: [
            'this is fun',
            100,
            4,
            { another: { level: 'hello fun 2 10' } },
        ],
    });
});
test('"undefined" values are omitted', () => {
    const r = (o) => _resolve_1.resolve(o);
    expect(r({ foo: undefined })).toStrictEqual({});
    expect(r({ foo: { bar: undefined } })).toStrictEqual({ foo: {} });
    expect(r({ foo: { bar: undefined, hello: 123 } })).toStrictEqual({ foo: { hello: 123 } });
    expect(r({ foo: [undefined] })).toStrictEqual({ foo: [] });
});
test('omitEmpty', () => {
    const r = (o) => _resolve_1.resolve(o, { omitEmpty: true });
    expect(r({ foo: {} })).toStrictEqual(undefined);
    expect(r({ foo: { bar: {} } })).toStrictEqual(undefined);
    expect(r({ foo: [] })).toStrictEqual(undefined);
    expect(r({ bar: { foo: [] } })).toStrictEqual(undefined);
    expect(r({ foo: { hello: 'world', boom: { bar: [] } } })).toStrictEqual({ foo: { hello: 'world' } });
});
//# sourceMappingURL=data:application/json;base64,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